/* dvireader.h - functions for reading DVI files
   $Id: dvireader.h,v 0.2 1997/03/28 03:16:45 tjchol01 Exp $
   Authors: Andrew Trevorrow, Ian Dall, Geoffrey Tobin, Tomasz J. Cholewo
 */

#ifndef __DVIREADER_H__
#define __DVIREADER_H__

#include "dvgt.h"

#define ruletablesize   300	/* maximum number of rules in a ruletable    */
#define chartablesize   3000	/* maximum number of chars in a chartable    */
#define maxfontspec     maxstring
    /* max length of a font file specification   */
#define maxspeciallen   maxstring
    /* maximum length of \special string         */
#define maxTeXchar      255	/* ignore character codes > 255              */

typedef struct _REC_ruletable
  {
    int hp, vp;			/* pixel coords of rule's ref point */
    int wd, ht;			/* dimensions of rule in pixels     */
  }
_REC_ruletable;

typedef struct _REC_chartable
  {
    int hp, vp;			/* pixel coords of char's ref point */
    unsigned char code;		/* char's code and pixeltable index */
  }
_REC_chartable;

typedef struct _REC_pixeltable
  {
    int wd, ht;
    /* glyph width and height in pixels; they
       define the size of the smallest box
       containing all the black pixels           */
    int xo, yo;
    /* x and y offsets from top left corner
       of glyph to character's reference point   */
    int dwidth;
    /* advance width in DVI units computed from
       fix width stored in font file             */
    int pwidth;
    /* advance width in pixels computed from
       fix width stored in font file             */
    int mapadr;			/* byte offset of bitmap info in font file   */
    int_or_mptr bitmap;		/* SYSDEP: starting address of bitmap        */
  }
_REC_pixeltable;

/* Information about the rules and characters appearing on a page is stored
   in dynamic one-way lists to avoid imposing any limit on their numbers.
   To reduce pointer overheads, the nodes in these lists contain large
   tables (the values of ruletablesize and chartablesize have been chosen
   so that the vast majority of DVI pages will only require one-node lists).
   When interpreting a DVI page, DVIReader adds a new rule or character node
   to the TAIL of the relevant list.  This is done so that when DVItoVDU
   accesses such lists (starting at the head), rules and characters will be
   processed in somewhat the same sequence as seen in the DVI file; i.e.,
   top-to-bottom and left-to-right across the page.
   Since a character list is part of the information stored for a font,
   the precise sequence in which DVI characters are seen is not remembered.
   Font information is also linked together in a one-way list, but the
   ordering is more or less random (see, however, the SortFonts routine).
 */

typedef struct ruleinfo
  {
    int rulecount;		/* number of rules in ruletable     */
    _REC_ruletable ruletable[ruletablesize];
    struct ruleinfo *nextrule;	/* next node in rule list           */
  }
ruleinfo;			/* a node in a list of ruletables   */

typedef struct charinfo
  {
    int charcount;		/* number of chars in chartable     */
    _REC_chartable chartable[chartablesize];
    struct charinfo *nextchar;	/* next node in char list           */
  }
charinfo;			/* a node in list of chartables     */

/* DVIReader uses wd, ht, xo and yo to calculate minhp, minvp, maxhp and
   maxvp.  dwidth and pwidth are used to advance horizontally.
   The mapadr and bitmap fields are not used by DVIReader; they are used
   by FontReader and the main program to load character bitmaps.
 */
typedef _REC_pixeltable pixeltable[maxTeXchar + 1];

typedef struct fontinfo
  {
    boolean pkfont;		/* is this a PK bitmap font?          */
    boolean fontused;		/* is font used on current page?      */
    int fontnum;		/* DVI font number: -2^31 .. 2^30 - 1 */
    int scaledsize;		/* scaled font size in DVI units      */
    int designsize;		/* design size in DVI units           */
    String fontarea;		/* explicit font directory            */
    int fontarealen;		/* length of fontarea                 */
    String fontname;		/* font name; e.g., "cmr10"           */
    int fontnamelen;		/* length of font name                */
    String fontspec;		/* complete font file pathname        */
    int fontspeclen;		/* length of fontspec                 */
    boolean fontexists;		/* can fontspec be opened?         */
    boolean honest;		/* true (and not substitute) font    */
    int totalchars;		/* number of chars from font on page  */
    charinfo *charlist;		/* head of char information list  */
    charinfo *chartail;		/* tail of char information list  */
    _REC_pixeltable *pixelptr;	/* allocated once: 1st time font used */
    struct fontinfo *nextfont;	/* next node in font list  */
  }
fontinfo;			/* a node in list of fonts            */

/* For the parameter in MoveToTeXPage: */
typedef int TeXcounters[10];

typedef struct TeXpageinfo
  {
    TeXcounters value;		/* \count0..\count9 values    */
    boolean present[10];	/* is counter relevant?       */
    char lastvalue;		/* last relevant counter      */
  }
TeXpageinfo;

/* DVIReader also builds a list of \special info: */

typedef struct specialinfo
  {
    String special;
    int hp, vp;
    struct specialinfo *nextspecial;
  }
specialinfo;


/* Most of these should be treated as read-only parameters:                */
extern int DVImag;		/* magnification stored in DVI file        */
extern int totalpages;		/* number of pages in DVI file             */
extern int totalfonts;		/* number of fonts in DVI file             */
extern int currDVIpage;		/* updated by MoveTo... calls              */
extern TeXcounters currTeXpage;
    /* ditto                                   */
extern int totalrules;		/* number of rules on current page         */
extern ruleinfo *rulelist;	/* head of rule information list           */
extern ruleinfo *ruletail;	/* tail of rule information list           */
extern specialinfo *speciallist;
    /* head of \special information list       */
extern fontinfo *fontlist;	/* head of font information list           */
extern fontinfo *currfont;	/* InterpretPage's current font info       */
extern boolean pageempty;	/* is page empty of rules and chars?       */
extern int minhp;		/* minimum horizontal pixel coordinate     */
extern int minvp;		/* minimum vertical pixel coordinate       */
extern int maxhp;		/* maximum horizontal pixel coordinate     */
extern int maxvp;		/* maximum vertical pixel coordinate       */

extern void InitDVIReader ();
extern void OpenDVIFile PARAMS ((char *name));
extern void SetConversionFactors PARAMS ((double xres, double yres, double magnification));
extern void MoveToDVIPage PARAMS ((int n));
extern void MoveToNextPage PARAMS ((boolean ascending));
extern boolean MoveToTeXPage PARAMS ((TeXpageinfo * newTeXpage));
extern int XPixelRound PARAMS ((int DVIunits));
extern int YPixelRound PARAMS ((int DVIunits));
extern void InterpretPage ();
extern void SortFonts PARAMS ((fontinfo ** unusedlist));
extern void CloseDVIFile ();

#endif /* __DVIREADER_H__ */

/* end dvireader.h */
