/* -*-C++-*-
 * ###################################################################
 *  Cpptcl - connecting C++ with Tcl
 * 
 *  FILE: "meta_type.cc"
 *                                    created: 2/11/97 {9:21:16 pm} 
 *                                last update: 09/05/98 {22:54:42 PM} 
 *  Author: Vince Darley
 *  E-mail: <darley@fas.harvard.edu>
 *    mail: Division of Engineering and Applied Sciences, Harvard University
 *          Oxford Street, Cambridge MA 02138, USA
 *     www: <http://www.fas.harvard.edu/~darley/>
 *  
 * ========================================================================
 *               Copyright (c) 1997 Vince Darley
 * ========================================================================
 *  See header file for further information
 * ###################################################################
 */

#include "meta_type.h"
#include "meta_object.h"
#include "tcl_object.h"

/* Hash table for external access to data */
static Tcl_HashTable cpptcl_hash_table;	

/* Type for a pointer to a C++ object */
Tcl_DupInternalRepProc Cpptcl_dupIntRepProc;
Tcl_UpdateStringProc Cpptcl_updateStringProc;
Tcl_SetFromAnyProc Cpptcl_setFromAnyProc;

static Tcl_ObjType Cpptclobjtype = {
	"Cpptclobj",
	(Tcl_FreeInternalRepProc*) NULL,
	Cpptcl_dupIntRepProc, 
	Cpptcl_updateStringProc, 
	Cpptcl_setFromAnyProc
};

/* Type for a C++ member */
Tcl_FreeInternalRepProc Cpptclmember_freeIntRepProc;
Tcl_DupInternalRepProc Cpptclmember_dupIntRepProc;
Tcl_UpdateStringProc Cpptclmember_updateStringProc;
Tcl_SetFromAnyProc Cpptclmember_setFromAnyProc;

static Tcl_ObjType Cpptclmembertype = {
	"Cpptclmember",
	Cpptclmember_freeIntRepProc,
	Cpptclmember_dupIntRepProc, 
	Cpptclmember_updateStringProc, 
	Cpptclmember_setFromAnyProc
};

Tcl_FreeInternalRepProc Cpptclobjmember_freeIntRepProc;
Tcl_DupInternalRepProc Cpptclobjmember_dupIntRepProc;
Tcl_UpdateStringProc Cpptclobjmember_updateStringProc;
Tcl_SetFromAnyProc Cpptclobjmember_setFromAnyProc;

static Tcl_ObjType Cpptclobjmembertype = {
	"Cpptclobjmember",
	Cpptclobjmember_freeIntRepProc,
	Cpptclobjmember_dupIntRepProc, 
	Cpptclobjmember_updateStringProc, 
	Cpptclobjmember_setFromAnyProc
};

void meta_object::register_tcl_types(void) {
	// have Tcl Object representation and compiler, so declare my types
	Tcl_RegisterObjType(&Cpptclobjtype);
	Tcl_RegisterObjType(&Cpptclmembertype);
	Tcl_RegisterObjType(&Cpptclobjmembertype);
	Tcl_InitHashTable(&cpptcl_hash_table, TCL_STRING_KEYS);
}

tcl_interaction* Cpptcl_ConvertToObject(Tcl_Interp* interp, Tcl_Obj* obj) {
	if(Tcl_ConvertToType(interp,obj,&Cpptclobjtype) == TCL_OK) {
		return (tcl_interaction*) obj->internalRep.otherValuePtr;
	} else {
		return 0;
	}
}

tcl_interaction* Cpptcl_ConvertToTypedObject(Tcl_Interp* interp, 
								Tcl_Obj* obj, const meta_object* type) {
	if(obj != 0) {
	if(type->is_a(cpp_mem::_type)) {
		if(Tcl_ConvertToType(interp,obj,&Cpptclobjmembertype) == TCL_OK) {
			cpp_wrapped_objmember* wmem = (cpp_wrapped_objmember*)obj->internalRep.otherValuePtr;
			tcl_interaction *o = (tcl_interaction*)wmem->member;
			// put container in the static member
			wmem->member->operator()(wmem->in);
			if(o->meta_info().is_a(*type)) {
				return o;
			}
		}
	} else {
		if(Tcl_ConvertToType(interp,obj,&Cpptclobjtype) == TCL_OK) {
			tcl_interaction* o = (tcl_interaction*)obj->internalRep.otherValuePtr;
			if(o->meta_info().is_a(*type)) {
				return o;
			}
		}
	}
	}
	return 0;
}

const cpp_mem* meta_object::configuration_option(tcl_args& arg, 
  							const char* member_type) const {
	// have a peek (even if there aren't any left)
	const Tcl_Obj* objPtr = arg[0];
	if(!arg.empty()) {
		if(objPtr->typePtr == &Cpptclmembertype) {
			// must check this member is a member of the same type as we
			// actually want.  Otherwise we might do something like
			//    A a ; B b ; set cf [list configure size 5]
			//    eval a $cf ; eval b $cf
			// with very strange results, if 'size' as applied to A,B
			// is different.
			cpp_wrapped_member* wmem = 
				(cpp_wrapped_member*) objPtr->internalRep.otherValuePtr;
			// if this (meta_object) is of type wmem->of, we're ok.
			if(is_a(*(wmem->of))) {
				// we have to check that we requesting a member of the
				// right type
				if(!member_type && wmem->member->ensemble_behaviour() || 
				   tcl_interaction::metaobject->is_of_type(wmem->member->type(),member_type)) {
					const cpp_mem* mem = wmem->member;
					arg.matched_cmd(mem->name());
					return mem;
				} else {
					// We know it's going to fail, but to get a good
					// error message, we continue, so no return
					// return 0;
				}
			}
		}
	}
	// If any of the above failed, we know nothing extra and must just compare
	// strings 
	arg.match_configuration_options = true;
	const cpp_wrapped_member* r = string_configuration_option(arg,member_type);
	arg.match_configuration_options = false;
	// We convert the object to our inbuilt type for future efficiency
	if(r) {
		// convert obj to cpptclmember type
		Tcl_ObjType *oldTypePtr = objPtr->typePtr;
		if ((oldTypePtr != NULL) &&	(oldTypePtr->freeIntRepProc != NULL)) {
			oldTypePtr->freeIntRepProc((Tcl_Obj*)objPtr);
		}
		((Tcl_Obj*)objPtr)->typePtr = &Cpptclmembertype;
		((Tcl_Obj*)objPtr)->internalRep.otherValuePtr = (VOID *)r;
		return r->member;
	} else {
		return (const cpp_mem*) NULL;
	}
}

const cpp_wrapped_member* meta_object::string_configuration_option(tcl_args& arg, 
  							const char* member_type) const {
  								
	for(int i = 0; i< member_info_size(); i++) {
	     const cpp_member_info* mm = member_info();
		if(!member_type && mm[i].ensemble_behaviour() || 
			tcl_interaction::metaobject->is_of_type(member_info()[i].type(),member_type)) {
			member_info()[i].member_info->get_syntax(arg);
			if(arg == member_info()[i].name()) {
				return new cpp_wrapped_member(member_info()[i].member_info,this);
			}
		}  
	}
	// look through parents
	for (list_pos<meta_object*> p = parent_list.headConst(); p; ++p) {
		const cpp_wrapped_member* r = p.item()->string_configuration_option(arg,member_type);
		if(r != 0) {
			return r;
		}
	}
	return (const cpp_wrapped_member*) NULL;
}
const cpp_mem* meta_object::find_member(const cpp_private& data, 
		const char* member_type) const {
	for(int i = 0; i< member_info_size(); i++) {
		if(tcl_interaction::metaobject->is_of_type(member_info()[i].type(),member_type)
		  && member_info()[i].private_data(data)) {
			return member_info()[i].member_info;
		}  
	}
	// look through parents
	for (list_pos<meta_object*> p = parent_list.headConst(); p; ++p) {
		const cpp_mem* r = p.item()->find_member(data,member_type);
		if(r != 0) {
			return r;
		}
	}
	return (const cpp_mem*) NULL;
}

const meta_object* meta_object::needs_container(void) const {
	if(container_type) {
		return container_type;
	} 
	const meta_object* ret = 0;
	for(list_pos<meta_object*> p(parent_list);p;++p){
		ret = p.item()->needs_container();
		if(ret) return ret;
	}
	return 0;
}

void meta_object::check_for_members(void) const {
	if(members) {
		_has_members = true;
		return;
	}
	for(list_pos<meta_object*> p(parent_list);p;++p){
		if(p.item()->has_members()) {
			_has_members = true;
			return;
		}
	}
}

bool meta_object::is_a(const meta_object& o) const {
    if(this == &o) {
	return true;
    } else {
	for(list_pos<meta_object*> p(parent_list);p;++p){
	    if(p.item()->is_a(o)) {
		return true;
	    }
	}
	return false;
    }		
}

#ifdef CPPTCL_USE_SCOPED_OBJECTS
extern "C" {
    char* Cpptcl_ObjectScopeVarDeleted(ClientData clientData,
				       Tcl_Interp *interp, CONST char *name1, CONST char *name2, int flags);
}
#endif

/* 
 * A hidden class we use to instantiate class creation commands.
 * When you issue a 'cppmeta attach' command, many instances of
 * this class are created.
 */
class tcl_class : public tcl_base {
  public:
    /// Constructor
    tcl_class(tcl_obj& i, const meta_object* _meta):
      tcl_base(i, _meta->tcl_command), meta_obj(_meta) {};
      
    /// Called when we are invoked; passes arg on to the make function.
    int parse_tcl_command(tcl_args& arg) {
	return meta_obj->parse_meta_commands(tcl_,arg, !tcl_base::metaobject->createAtGlobalScope);
    }
    /// The class's type is considered to be the type of the object it creates.
    const meta_object& meta_info(void) const {return *meta_obj;}
    /// The class's type is considered to be the type of the object it creates.
    cpx_type type(void) const {return meta_obj->tcl_command;}
  protected:
    ///
    const meta_object* meta_obj;
};

bool meta_object::attach_to_tcl_interp(tcl_obj& tcl_,bool attach) const {
    if(instantiable()) {
	if(attach) {
	    if(new tcl_class(tcl_,this)) 
		return true;
	} else {
	    tcl_class* o = (tcl_class*) Cpptcl_getObjectByName(tcl_,tcl_command);
	    if(o) {
		delete o;
		return true;
	    }
	}			
    }
    return false;
}

int meta_object::parse_meta_commands(tcl_obj& tcl_, tcl_args& arg, bool scoped) const {
    Tcl_Obj *name;
    const meta_object* required_container;
    // No match if (a) not instantiable or (b) no argument given
    if (!instantiable() || !((required_container = needs_container()) ?
			     arg("object location ?construction arguments? ?configuration options?"
				 ,"make an embedded object")=="createIn"
		:
			     arg("?construction arguments? ?configuration options?",
				 "make an object").match_into(name,"'name-of-new-object'"))) {
	return arg.no_match();
    }
    // To investigate: remove this line and let command completion
    // extend into the innards of object creation.
    if(arg.haveErr)
	return TCL_ERROR;
    tcl_object** put_here = 0;
    if(!required_container) { 
	// this is duplicated when we make the actual object
	arg.setName(name);
	arg.container = 0;
    } else {
	arg("location ?more construction arguments? ?configuration options?");
	object_reader reader(*required_container);
	arg >> reader;
	NO_EXCEPTIONS(arg);
	if(((tcl_object*)reader.obj)->contains(arg,put_here) != TCL_OK) {
	    // need to make a decent error message
	    arg >> done;
	    NO_EXCEPTIONS(arg);
	    return TCL_ERROR;
	}
	arg.container = (tcl_base*) reader.obj;
	// currently the object's 'name' is the name of the last
	// argument read, which works ok for members, but perhaps
	// not for other stuff.
	arg.setName(arg[-1]);
    }
	
#ifdef CPPTCL_USE_SCOPED_OBJECTS
    if(!required_container) { 
	if(scoped) {	
	    if(!tcl_base::could_make_a_trace(tcl_, name)) {
	        return TCL_ERROR;
	    }
	}
    } else {
	scoped = false;
    }
	
#endif
    // yes, so call the make function	
    if (tcl_object* o = make_new(arg)){
	// check if construction argument handling went wrong
	if(arg.haveErr) {
	    delete o;
	    // not 100% sure we want this line
	    arg >> done;
	    return TCL_ERROR;
	}
	if(required_container) { 
	    if(put_here) {
		*put_here = o;
	    }
	    // we've read this now
	    arg.container = 0;
	}
	tcl_ << o << result;
	if(o->parse_configuration_values(arg) == TCL_ERROR)
	    return TCL_ERROR;
	if(!required_container) { 
	    // Create hash-table entry
	    if(Cpptcl_CreatedNewObject(tcl_,o) == TCL_ERROR)
		return TCL_ERROR;
	    			
#ifdef CPPTCL_USE_SCOPED_OBJECTS
	    if (scoped) {
		o->make_a_trace();
	    }
#endif
	}
	return tcl_;
    } else {
	// We expect the constructor to return whatever 
	// error message is appropriate, or signal an
	// exception if they're supported.
	return TCL_ERROR;
    }    
}

#ifdef CPPTCL_USE_SCOPED_OBJECTS
char* Cpptcl_ObjectScopeVarDeleted(ClientData clientData,
	Tcl_Interp * /*interp*/, CONST char */*name1*/, 
	CONST char */*name2*/, int /*flags*/){
    tcl_base* o = (tcl_base*)clientData;
    Tcl_CmdInfo infoPtr;
    #define DEBUG
#ifdef DEBUG
    char *name;
#endif
	
    if (o->tracing_for_scope) {
	o->tracing_for_scope = false;
#ifdef DEBUG
	name = (char *) ckalloc(strlen(o->char_tcl_command())+1);
	strcpy(name, o->char_tcl_command());
		
	if (Tcl_GetCommandInfo(o->get_interp(), o->char_tcl_command(), &infoPtr)) {
	    if (Tcl_DeleteCommand(o->get_interp(), o->char_tcl_command()) == TCL_OK) 
		fprintf(stderr, "Deleted command %s\n", name);
	    else
		fprintf(stderr, "Unable to delete command %s\n", name);
	}
	ckfree((char *) name);
#else
	if (Tcl_GetCommandInfo(o->get_interp(), o->char_tcl_command(), &infoPtr))
	    Tcl_DeleteCommand(o->get_interp(), o->char_tcl_command());
#endif
    }
    return (char *) NULL;
}
#endif


void Cpptcl_dupIntRepProc(Tcl_Obj* srcPtr, Tcl_Obj *dupPtr) {
    // just copy the pointer
    dupPtr->internalRep.otherValuePtr = srcPtr->internalRep.otherValuePtr;
    dupPtr->typePtr = &Cpptclobjtype;
}

void Cpptclmember_freeIntRepProc(Tcl_Obj* objPtr) {
    cpp_wrapped_member* wmem = (cpp_wrapped_member*)objPtr->internalRep.otherValuePtr;
    delete wmem;
}
void Cpptclobjmember_freeIntRepProc(Tcl_Obj* objPtr) {
    cpp_wrapped_objmember* wmem = (cpp_wrapped_objmember*)objPtr->internalRep.otherValuePtr;
    delete wmem;
}

void Cpptclmember_dupIntRepProc(Tcl_Obj* srcPtr, Tcl_Obj *dupPtr) {
    // duplicate the wrapped member
    cpp_wrapped_member* wmem = (cpp_wrapped_member*)srcPtr->internalRep.otherValuePtr;
    dupPtr->internalRep.otherValuePtr = (VOID *) new cpp_wrapped_member(*wmem); 
    dupPtr->typePtr = &Cpptclmembertype;
}
void Cpptclobjmember_dupIntRepProc(Tcl_Obj* srcPtr, Tcl_Obj *dupPtr) {
    // duplicate the wrapped objmember
    cpp_wrapped_objmember* wmem = (cpp_wrapped_objmember*)srcPtr->internalRep.otherValuePtr;
    dupPtr->internalRep.otherValuePtr = (VOID *) new cpp_wrapped_objmember(*wmem); 
    dupPtr->typePtr = &Cpptclobjmembertype;
}

void Cpptcl_updateStringProc(Tcl_Obj *objPtr) {
    tcl_base* t = (tcl_base*) objPtr->internalRep.otherValuePtr;
    mystrdup(objPtr->bytes, objPtr->length, t->char_tcl_command());
}

void Cpptclmember_updateStringProc(Tcl_Obj *objPtr) {
    cpp_wrapped_member* wmem = (cpp_wrapped_member*)objPtr->internalRep.otherValuePtr;
    register Tcl_Obj* n = wmem->member->name();
    objPtr->bytes = n->bytes;
    objPtr->length = n->length;
    Tcl_IncrRefCount(objPtr);
}

void Cpptclobjmember_updateStringProc(Tcl_Obj *objPtr) {
    cpp_wrapped_objmember* wmem = (cpp_wrapped_objmember*)objPtr->internalRep.otherValuePtr;
    tcl_obj to;
    const cpp_mem * mem = wmem->member;
    mem->operator()(wmem->in);
    to << mem;
    objPtr = to;
    Tcl_IncrRefCount(objPtr);
}

int Cpptclmember_setFromAnyProc(Tcl_Interp* interp, Tcl_Obj *objPtr) {
    if(objPtr->typePtr == &Cpptclmembertype) {
	return TCL_OK;
    } else {
	// we can't actually set one of these from _any_ string, without
	// a context.  They're set internally by my configure code.
	if(interp != NULL) {
	    int length;
	    char* string = Tcl_GetStringFromObj(objPtr, &length);
	    Tcl_ResetResult(interp);
	    Tcl_Obj * resultPtr = Tcl_GetObjResult(interp);
	    Tcl_AppendToObj(resultPtr, "expected Cpptclmember but got \"", -1);
	    Tcl_AppendToObj(resultPtr, string, length);
	    Tcl_AppendToObj(resultPtr, "\"", -1);			
	}
	return TCL_ERROR;		
    }	
}

/* 
 * -------------------------------------------------------------------------
 * 
 * "Cpptcl_setFromAnyProc" --
 * 
 *  Convert a string to a tcl_object.  The string is either the name
 *  of the toplevel object, or a list whose elements show the nesting
 *  of the desired object.  
 * -------------------------------------------------------------------------
 */
int Cpptcl_setFromAnyProc(Tcl_Interp* interp, Tcl_Obj *objPtr) {
    if(objPtr->typePtr == &Cpptclobjtype) {
	return TCL_OK;
    }
    return meta_object::Cpptcl_setToType(interp,objPtr,true);
}

int Cpptclobjmember_setFromAnyProc(Tcl_Interp* interp, Tcl_Obj *objPtr) {
    if(objPtr->typePtr == &Cpptclobjmembertype) {
	return TCL_OK;
    }
    return meta_object::Cpptcl_setToType(interp,objPtr,false);
}

int meta_object::Cpptcl_setToType(Tcl_Interp* interp, Tcl_Obj* objPtr, bool want_object) {
    /*
     * Get the string representation. Make it up-to-date if necessary.
     */

    int length;
    char* string = Tcl_GetStringFromObj(objPtr, &length);
	// Zero-length causes some problems with the list scan below,
	// so just dump out now.
	if(length) {
		tcl_interaction* t = 0;
		// if we're looking for an object rather than a member
		if(want_object) {
			// Try to convert to a cpptcl object
			if(interp) {
				t = Cpptcl_getObjectByName(interp, string);
			} else {
				Tcl_HashEntry *hPtr = Tcl_FindHashEntry(&cpptcl_hash_table, (char*) string);
				if (hPtr != NULL)
					t = (tcl_interaction*) Tcl_GetHashValue(hPtr);
			}
		}
		if(!t) {
			// We didn't succeed directly.  Perhaps the object is nested.
			// In this case we actually have a list down which we have to
			// scan!
			int llength = 0;
			if((Tcl_ListObjLength(NULL,objPtr,&llength) == TCL_OK) && (llength>1)) {
				// now scan through the elements
				Tcl_Obj* elt = 0;
				tcl_object* scan_obj;
				tcl_object** embed = 0;
				int lloop = 0;
				tcl_obj o;
				// this will return TCL_OK
				Tcl_ListObjIndex(NULL,objPtr,lloop,&elt);
				if(Cpptcl_setFromAnyProc(NULL,elt) == TCL_OK) {
					llength --;
					scan_obj = (tcl_object*)elt->internalRep.otherValuePtr;
					for (lloop = 1;lloop<=llength;lloop++) {
						Tcl_ListObjIndex(NULL,objPtr,lloop,&elt);
						tcl_args arg(o,1,&elt);
						if(lloop < llength || want_object) {
							if(scan_obj->contains(arg,embed) != TCL_OK) {
								break;
							}
							// we got a sub-object
							scan_obj = *embed;
						} else {
							// we've reached the last list element,
							// and we want it to be a member (not object)
						    arg.match_configuration_options = true;
						    const cpp_wrapped_member* r = scan_obj->meta_info().string_configuration_option(arg,cpp_mem::_type.type);
							arg.match_configuration_options = false;
							if(r) {
					            cpp_wrapped_objmember * wmem = new cpp_wrapped_objmember(scan_obj,r);
								delete r;
								if(objPtr->typePtr) {
									Tcl_FreeInternalRepProc* fproc = objPtr->typePtr->freeIntRepProc;
									if(fproc) {
										(*fproc)(objPtr);
									}
								}
								objPtr->internalRep.otherValuePtr = (VOID *) wmem;
								objPtr->typePtr = &Cpptclobjmembertype;
								return TCL_OK;
							} else {
								// break so we don't set the object below
								break;
							}
						}
					}
					// if we got 'llength' nested sub-objects, all ok.
					if(lloop > llength) {
						t = scan_obj;
					}
					// otherwise t is still zero, or set by lmem above
				}
			}
		}
		if(t) {
			if(objPtr->typePtr) {
				Tcl_FreeInternalRepProc* fproc = objPtr->typePtr->freeIntRepProc;
				if(fproc) {
					(*fproc)(objPtr);
				}
			}
			objPtr->internalRep.otherValuePtr = (VOID *) t;
			objPtr->typePtr = &Cpptclobjtype;
			return TCL_OK;
		}
	}
	if(interp != NULL) {
		Tcl_ResetResult(interp);
		Tcl_Obj * resultPtr = Tcl_GetObjResult(interp);
		Tcl_AppendToObj(resultPtr,"expected Cpptclobject but got \"", -1);
		Tcl_AppendToObj(resultPtr, string, length);
		Tcl_AppendToObj(resultPtr, "\"", -1);			
	}
	return TCL_ERROR;		
}

/* 
 * -------------------------------------------------------------------------
 * 
 * "Cpptcl_UpdatedObject" --
 * 
 *  We call this when we want to use the 'blocked' has entry for the
 *  command of an object which seems to point to 'update' but doesn't,
 *  so it must have been renamed.
 *  
 *  This procedure tries to update the hash entry corresponding to
 *  the object 'update', hence freeing the 'blocked' entry for use
 *  by the calling procedure.  Returns true if successful.
 * -------------------------------------------------------------------------
 */
bool Cpptcl_UpdatedObject(tcl_base *update) {
	Tcl_HashEntry *hPtr = Tcl_FindHashEntry(&cpptcl_hash_table, 
											  update->char_tcl_command());
	if(hPtr != NULL) {
		// another is blocked...
		if(!Cpptcl_UpdatedObject((tcl_base*)Tcl_GetHashValue(hPtr))) {
			// failed to fix it
			return false;
		}
	} else {
		// we're not colliding with anything else, so we can just add an
		// entry and let our caller use our old hPtr
		int made_it;
		hPtr = Tcl_CreateHashEntry(&cpptcl_hash_table, 
						update->char_tcl_command(), &made_it);
		// very bizarre for this to happen ever
		assert(made_it);
	}	
	// hPtr is ours to use
	Tcl_SetHashValue(hPtr,update);
	return true;
}

int Cpptcl_CreatedNewObject(tcl_obj& tcl, tcl_base *o) {
	int made_it;
	char * temp = o->char_tcl_command();
	Tcl_HashEntry *hPtr = Tcl_CreateHashEntry(&cpptcl_hash_table, 
		temp, &made_it);
	if (!made_it) {
		// perhaps the old version was renamed
		if(hPtr) {
			if(Cpptcl_UpdatedObject((tcl_base*)Tcl_GetHashValue(hPtr))) {
				Tcl_SetHashValue(hPtr,o);
				return TCL_OK;
			}
		}
		tcl << "Unable to create hash table entry for object command \"" 
			<< o << '"' << tcl_error;
		return TCL_ERROR;
	}
	Tcl_SetHashValue(hPtr, o);
	return TCL_OK;
}

int Cpptcl_DeletedObject(tcl_base *o) {
	Tcl_HashEntry *hPtr = Tcl_FindHashEntry(&cpptcl_hash_table, 
							o->char_tcl_command());
	if (hPtr != NULL) {
		Tcl_DeleteHashEntry(hPtr);
		return TCL_OK;
	}
	return TCL_ERROR;
}
