#define DEBUG 1
/*
 *	pcmciad
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <regex.h>
#include <syslog.h>
#ifdef	SYSINSTALL
#include "sysinstall.h"
#endif	/* SYSINSTALL */
#include "cardd.h"

struct card_config *assign_driver(struct card *);
int	setup_slot(struct slot *);
void	read_ether(struct slot *);
void	dump_config_file();
void	pr_cmd(struct cmd *);
void	readslots();
void	slot_change(struct slot *);
void	card_removed(struct slot *);
void	card_inserted(struct slot *);
int	assign_io(struct slot *sp);

/*
 *	Dump configuration file data.
 */
void
dump_config_file()
{
struct card *cp;
struct card_config *confp;

	for (cp = cards; cp; cp = cp->next)
		{
		printf("Card manuf %s, vers %s\n", cp->manuf, cp->version);
		printf("Configuration entries:\n");
		for (confp = cp->config; confp; confp = confp->next)
			printf("\tIndex code = 0x%x, driver name = %s\n",
				confp->index, confp->driver->name);
		if (cp->insert)
			{
			printf("Insert commands are:\n");
			pr_cmd(cp->insert);
			}
		if (cp->remove)
			{
			printf("Remove commands are:\n");
			pr_cmd(cp->remove);
			}
		}
#if 0
	for (devp = devlist; devp; devp = devp->next)
		{
		if (devp->insert)
			{
			printf("Insert commands are:\n");
			pr_cmd(devp->insert);
			}
		if (devp->remove)
			{
			printf("Remove commands are:\n");
			pr_cmd(devp->remove);
			}
		}
#endif
}
void
pr_cmd(struct cmd *cp)
{
	while (cp)
		{
		printf("\t%s\n", cp->line);
		cp = cp->next;
		}
}
/*
 *	readslots - read all the PCMCIA slots, and build
 *	a list of the slots.
 */
void
readslots()
{
char	name[128];
int	i, fd;
struct slot *sp;

	for (i = 0; i < MAXSLOT; i++)
		{
		sprintf(name, CARD_DEVICE, i);
		fd = open(name, 2);
		if (fd < 0)
			continue;
#ifndef	SYSINSTALL
		fprintf(stderr, "opened %s\n", name);
#endif	/* SYSINSTALL */
		sp = xmalloc(sizeof(*sp));
		sp->fd = fd;
		sp->name = newstr(name);
		sp->slot = i;
		sp->state = empty;
/*
 *	Check to see if the controller memory has been set up.
 */
		if (slots == 0)
			{
			unsigned long mem = 0;

			if (ioctl(fd, PIOCRWMEM, &mem)) 
				{
#ifndef	SYSINSTALL
				perror("ioctl (PIOCRWMEM)");
#else	/* SYSINSTALL */
				dialog_clear();
				msgConfirm("ioctl (PIOCRWMEM)");
#endif	/* SYSINSTALL */
				}
#ifndef	SYSINSTALL
			fprintf(stderr, "mem=%lx\n", mem);
#else	/* SYSINSTALL */
			{
			char	s[200];
			sprintf(s, "PC-card: PCIC memory 0x%x\n", mem);
			dialog_clear();
			msgNotify(s);
			sleep(1);
			}
#endif	/* SYSINSTALL */
			if (mem == 0)
				{
				mem = alloc_memory(4*1024);
				if (mem == 0)
		die("Can't allocate memory for controller access");

				if (ioctl(fd, PIOCRWMEM, &mem))
					{
#ifndef	SYSINSTALL
					perror("ioctl (PIOCRWMEM)");
#else	/* SYSINSTALL */
					dialog_clear();
					msgConfirm("ioctl (PIOCRWMEM)");
#endif	/* SYSINSTALL */
					}
				}
			}
#ifndef	SYSINSTALL
		fprintf(stderr, "%p %p\n", sp, &sp->next);
#endif	/* SYSINSTALL */
		sp->next = slots;
		slots = sp;
		slot_change(sp);
		}
}
/*
 *	slot_change - Card status has changed.
 *	read new state and process.
 */
void
slot_change(struct slot *sp)
{
	struct slotstate state;

	current_slot = sp;
	if (ioctl(sp->fd, PIOCGSTATE, &state))
		{
#ifndef	SYSINSTALL
		perror("ioctl (PIOCGSTATE)");
#else	/* SYSINSTALL */
		dialog_clear();
		msgConfirm("ioctl (PIOCGSTATE)");
#endif	/* SYSINSTALL */
		return;
		}
#ifndef	SYSINSTALL
	printf("%p %p %d %d\n",sp,&sp->state,state.state,sp->state);
#endif	/* SYSINSTALL */
	if (state.state == sp->state)
		return;
	sp->state = state.state;
	switch (sp->state)
		{
	case empty:
	case noslot:
		card_removed(sp);
		break;
	case filled:
		card_inserted(sp);
		break;
		}
}
/*
 *	card_removed - card has been removed from slot.
 *	Execute the remove commands, and clear the slot's state.
 *	Execute the device commands, then the driver commands
 *	and then the card commands. This is the reverse
 *	order to the insertion commands
 */
void
card_removed(struct slot *sp)
{
struct card *cp;

	if (sp->cis)
		freecis(sp->cis);
	if (sp->config)
		{
		sp->config->inuse = 0;
		sp->config->driver->inuse = 0;
		}
	if ((cp = sp->card) != 0)
		execute(cp->remove);
	sp->cis = 0;
	sp->config = 0;
}


/* regex CIS string comparison (by hosokawa) */

#define	REGCOMP_FLAGS	(REG_EXTENDED | REG_NOSUB)
#define	REGEXEC_FLAGS	(0)

static int 
cis_strcmp(char *db, char *cis)
{
	int		res, err;
	char		buf[256];
	regex_t		rx;

	if ((err = regcomp(&rx, db, REGCOMP_FLAGS))) {
		regerror(err, &rx, buf, sizeof buf);
		fprintf(stderr, "Warning: REGEX error for"
			"\"%s\" -- %s\n",
			db, buf);
		regfree(&rx);
		return -1;
	}
	res = regexec(&rx, cis, 0, NULL, REGEXEC_FLAGS);
	regfree(&rx);
	return res;
}

/*
 * card_inserted - Card has been inserted;
 *	- Read the CIS
 *	- match the card type.
 *	- Match the driver and allocate a driver instance.
 *	- Allocate I/O ports, memory and IRQ.
 *	- Set up the slot.
 *	- assign the driver (if failed, then terminate).
 *	- Run the card commands.
 *	- Run the driver commands
 *	- Run the device commands
 */
void
card_inserted(struct slot *sp)
{
struct card *cp;
int err;

	sleep(5);
	sp->cis = readcis(sp->fd);
	if (sp->cis == 0)
		{
#ifndef	SYSINSTALL
		log_1s("Error reading CIS on %s\n", sp->name);
#else	/* SYSINSTALL */
		log_1s("PC-card: Error reading CIS on %s\n", sp->name);
#endif	/* SYSINSTALL */
		return;
		}
/*
	dumpcis(sp->cis);
*/
	for (cp = cards; cp; cp = cp->next)
		if (cis_strcmp(cp->manuf, sp->cis->manuf) == 0 &&
			cis_strcmp(cp->version, sp->cis->vers) == 0)
			{
#ifdef	DEBUG
#ifndef	SYSINSTALL
			fprintf(stderr, 
			"Card \"%s\" (\"%s\") matched \"%s\" (\"%s\")\n",
			sp->cis->manuf, sp->cis->vers,
			cp->manuf, cp->version);
#endif	/* SYSINSTALL */
#endif	/* DEBUG */
			break;
			}
	sp->card = cp;
/*
	reset_slot(sp);
 */
	if (cp == 0)
		{
#ifndef	SYSINSTALL
		log_1s("No card in database for \"%s\"", sp->cis->manuf);
		log_1s("vers: \"%s\"", sp->cis->vers);
#else	/* SYSINSTALL */
		char	s[200];
		sprintf(s, "PC-card: No card in database for \"%s\" (\"%s\")",
			sp->cis->manuf, sp->cis->vers);
		dialog_clear();
		msgConfirm(s);
#endif	/* SYSINSTALL */
		return;
		}
	if (cp->ether)
		read_ether(sp);
	sp->config = assign_driver(cp);
	if (sp->config == 0)
		{
		execute(cp->insert);
		return;
		}
	if ((err = assign_io(sp)))
		{
		char *reason = "Unknown";

		switch (err) 
			{
		case -1:
			reason = "CIS not found";
			break;
		case -2:
			reason = "memory block allocation failed";
			break;
		case -3:
			reason = "I/O block allocation failed";
			break;
			}
#ifndef	SYSINSTALL
		log_1s("Resource allocation failure for %s", sp->cis->manuf);
		log_1s("Reason: %s", reason);
#else	/* SYSINSTALL */
		{
		char	s[200];
		sprintf(s, "PC-card: Resource allocation failure for\n"
			"\"%s\" (\"%s\")\n"
			"Reason: %s",
			sp->cis->manuf, sp->cis->vers, reason);
		dialog_clear();
		msgConfirm(s);
		}
#endif	/* SYSINSTALL */

		return;
		}
/*
 *	Once assigned, then set up the I/O & mem contexts, and
 *	set up the windows, and then attach the driver.
 */
	if (setup_slot(sp))
		execute(cp->insert);
#if 0
	else
		reset_slot(sp);
#endif
}
/*
 *	read_ether - read ethernet address from card. Offset is
 *	the offset into the attribute memory of the card.
 */
void
read_ether(struct slot *sp)
{
unsigned char net_addr[12];
int rw_flags;

	rw_flags = MDF_ATTR;
	ioctl(sp->fd, PIOCRWFLAG, &rw_flags);
	lseek(sp->fd, (off_t)sp->card->ether, SEEK_SET);
	if (read(sp->fd, net_addr, sizeof(net_addr)) != sizeof(net_addr))
		{
		logerr("read err on net addr");
		return;
		}
	sp->eaddr[0] = net_addr[0];
	sp->eaddr[1] = net_addr[2];
	sp->eaddr[2] = net_addr[4];
	sp->eaddr[3] = net_addr[6];
	sp->eaddr[4] = net_addr[8];
	sp->eaddr[5] = net_addr[10];
#ifndef	SYSINSTALL
	printf("Ether=%02x:%02x:%02x:%02x:%02x:%02x\n",
		sp->eaddr[0], sp->eaddr[1], sp->eaddr[2],
		sp->eaddr[3], sp->eaddr[4], sp->eaddr[5]);
#endif	/* SYSINSTALL */
}
/*
 *	assign_driver - Assign driver to card.
 *	First, see if an existing driver is already setup.
 */
struct card_config *
assign_driver(struct card *cp)
{
struct driver *drvp;
struct card_config *conf;

	for (conf = cp->config; conf; conf = conf->next)
		if (conf->inuse == 0 && conf->driver->card == cp &&
		    conf->driver->config == conf)
			{
#ifdef	DEBUG
#ifndef	SYSINSTALL
			fprintf(stderr, "Found existing driver (%s) for %s\n",
				conf->driver->name, cp->manuf);
#else	/* SYSINSTALL */
			char s[200];
			sprintf(s, "PC-card: Found existing driver (%s) for %s\n",
				conf->driver->name, cp->manuf);
			dialog_clear();
			msgConfirm(s);
#endif	/* SYSINSTALL */
#endif /* DEBUG */
			return(conf);
			}
/*
 *	New driver must be allocated. Find one that matches the
 *	any configurations not in use.
 */
	for (conf = cp->config; conf; conf = conf->next)
		if (conf->inuse == 0 && conf->driver->card == 0)
			break;
	if (conf == 0)
		{
#ifndef	SYSINSTALL
		log_1s("No free configuration for card %s", cp->manuf);
#else	/* SYSINSTALL */
		log_1s("PC-Card: No free configuration for card %s", cp->manuf);
#endif	/* SYSINSTALL */
		return(0);
		}
/*
 *	Now we have a free driver and a matching configuration.
 *	Before assigning and allocating everything, check to
 *	see if a device class can be allocated to this.
 */
	drvp = conf->driver;
/*
 *	If none available, then we can't use this card.
 */
	if (drvp->inuse)
		{
#ifndef	SYSINSTALL
		log_1s("Driver already being used for %s", cp->manuf);
#else	/* SYSINSTALL */
		log_1s("PC-card: Driver already being used for %s", cp->manuf);
#endif	/* SYSINSTALL */
		return(0);
		}
#if 0
/*
 *	Allocate I/O, memory and IRQ resources.
 */
	for (ap = drvp->io; ap; ap = ap->next)
		{
		if (ap->addr == 0 && ap->size)
			{
			int i = bit_fns(io_avail, IOPORTS, ap->size);
	
			if (i < 0)
				{
#ifndef	SYSINSTALL
				log_1s("Failed to allocate I/O ports for %s\n",
					cp->manuf);
#else	/* SYSINSTALL */
				log_1s("PC-card: Failed to allocate I/O ports for %s\n",
					cp->manuf);
#endif	/* SYSINSTALL */
				return(0);
				}
			ap->addr = i;
			bit_nclear(io_avail, i, i + ap->size - 1);
			}
		}
	if (drvp->irq == 0)
		{
		int i;
		for (i = 1; i < 16; i++)
			if (pool_irq[i])
				{
				drvp->irq = i;
				pool_irq[i] = 0;
				break;
				}
		if (drvp->irq == 0)
			{
#ifndef	SYSINSTALL
			log_1s("Failed to allocate IRQ for %s\n", cp->manuf);
#else	/* SYSINSTALL */
			log_1s("PC-card: Failed to allocate IRQ for %s\n", cp->manuf);
#endif	/* SYSINSTALL */
			return(0);
			}
		}
	for (ap = drvp->mem; ap; ap = ap->next)
		{
		if (ap->addr == 0 && ap->size)
			{
			ap->addr = alloc_memory(ap->size);
			if (ap->addr == 0)
				{
#ifndef	SYSINSTALL
				log_1s("Failed to allocate memory for %s\n",
					cp->manuf);
#else	/* SYSINSTALL */
				log_1s("PC-card: Failed to allocate memory for %s\n",
					cp->manuf);
#endif	/* SYSINSTALL */
				return(0);
				}
			}
		}
#endif /* 0 */
	drvp->card = cp;
	drvp->config = conf;
	drvp->inuse = 1;
	conf->inuse = 1;
	return(conf);
}
/*
 *	assign_io - Allocate resources to slot matching the
 *	configuration index selected.
 */
int
assign_io(struct slot *sp)
{
struct cis *cis;
struct cis_config *cisconf, *defconf;

	cis = sp->cis;
	defconf = cis->def_config;
	for (cisconf = cis->conf; cisconf; cisconf = cisconf->next)
		if (cisconf->id == sp->config->index)
			break;
	if (cisconf == 0)
		return(-1);
	sp->card_config = cisconf;
/*
 *	Found a matching configuration. Now look at the I/O, memory and IRQ
 *	to create the desired parameters. Look at memory first.
 */
	/* 
	 * XXX -- dirty hack for some 'ed' cards that have incomplete
	 * iomem CIS tupples
	 */
	if (!(strncmp(sp->config->driver->name, "ed", 2) == 0
		&& (sp->config->flags & 0x10))
		&& (cisconf->memspace || (defconf && defconf->memspace)))

		{
		struct cis_memblk *mp;

		mp = cisconf->mem;
		if (!cisconf->memspace)
			mp = defconf->mem;
		sp->mem.size = mp->length;
		sp->mem.cardaddr = mp->address;
/*
 *	For now, we allocate our own memory from the pool.
 */
		sp->mem.addr = sp->config->driver->mem;
/*
 *	Host memory address is required. Allocate one
 *	from our pool.
 */
		if (sp->mem.size && sp->mem.addr == 0)
			{
			sp->mem.addr = alloc_memory(mp->length);
			if (sp->mem.addr == 0)
				return(-2);
			sp->config->driver->mem = sp->mem.addr;
			}
#ifdef	DEBUG
#ifndef	SYSINSTALL
		fprintf(stderr, "Using mem addr 0x%x, size %d, card addr 0x%x\n",
			sp->mem.addr, sp->mem.size, sp->mem.cardaddr);
#else	/* SYSINSTALL */
		{
		char	s[200];
		sprintf(s, "PC-card: Using mem addr 0x%x, size %d, card addr 0x%x\n",
			sp->mem.addr, sp->mem.size, sp->mem.cardaddr);
		dialog_clear();
		msgNotify(s);
		sleep(1);
		}
#endif	/* SYSINSTALL */
#if 0
		sp->mem.cardaddr = 0x4000;
#else
		sp->mem.cardaddr = 0x4000;
		sp->mem.size = 0x4000;
#endif
#endif /* DEBUG */
		}
/*
 *	Now look at I/O.
 */
	bzero(&sp->io, sizeof(sp->io));
	if (cisconf->iospace || (defconf && defconf->iospace))
		{
		struct cis_config *cp;
		struct cis_ioblk *cio;
		struct allocblk *sio;
		int x, xmax;

		cp = cisconf;
		if (!cisconf->iospace)
			cp = defconf;
/*
 *	If # of I/O lines decoded == 10, then card does its
 *	own decoding.
 */
/*
 *	If an I/O block exists, then use it.
 *	If no address (but a length) is available, allocate
 *	from the pool.
 */
#if 1
		xmax = cp->io ? cisconf->io_blks : 1;
		for (x = 0, cio = cp->io, sio = &(sp->io); x < xmax; x++)
			{
			if (cio)
				{
#if 0
				int i;
#endif
				sio->addr = cio->addr;
				sio->size = cio->size;
#if 0
/*
 * XXX: this feature is temporary closed due to lack of release
 * mechanism of I/O space, which should be implemented in card_released()
 */
				for (i = sio->addr; i < sio->addr + sio->size - 1; i++)
					if (!bit_test(io_avail, i))
						return(-3);
#endif /* 0 */
				}
/*
 *     No I/O block, assume the address lines decode gives the size.
 */
			else
				{
				int i;
				sio->size = 1 << cp->io_addr;
				i = bit_fns(io_avail, IOPORTS, sio->size);
				if (i < 0)
					return(-3);
				sio->addr = i;
				}
			bit_nclear(io_avail, sio->addr, sio->addr + sio->size - 1);
/*
 *     Set up the size to take into account the decode lines.
 */
			sio->cardaddr = cp->io_addr; 
			switch(cp->io_bus)
				{
			case 0:
				break;
			case 1:
				sio->flags = IODF_WS;
				break;
			case 2:
				sio->flags = IODF_WS|IODF_CS16;
				break;
			case 3:
				sio->flags = IODF_WS|IODF_CS16|IODF_16BIT;
			break;
				}
#ifdef        DEBUG
#ifndef SYSINSTALL      
			fprintf(stderr, "Using I/O addr 0x%x, size %d\n",   
				sio->addr, sio->size); 
#else   /* SYSINSTALL */        
				{
				char    s[200];
				sprintf(s, "PC-Card: Using I/O addr 0x%x, size %d\n",                           
				sio->addr, sio->size);
				dialog_clear();
				msgNotify(s);
				}
#endif  /* SYSINSTALL */
#endif /* DEBUG */
			if (cio && cio->next) 
				{
				sio->next = xmalloc(sizeof(*sio));
				bzero(sio->next, sizeof(*sio));
				sio = sio->next;
				cio = cio->next;
				}
                        }
#else
		if (!cp->io) 
			{
			int i;
			sp->io.size = 1 << cp->io_addr;
			i = bit_fns(io_avail, IOPORTS, sp->io.size);
			if (i < 0)
				return(-3);
			sp->io.addr = i;
			bit_nclear(io_avail, sp->io.addr, sp->io.addr + sp->io.size - 1);
			}
		else
			{
			for (x = 0, cio = cp->io, sio = &(sp->io); x < cisconf->io_blks; x++)
				{
				if (cio)
					{
					sio->addr = cio->addr;
					sio->size = cio->size;
					}
/*
 *	No I/O block, assume the address lines decode gives the size.
 */
				else
					sio->size = 1 << cio->addr;
				bit_nclear(io_avail, sio->addr, sio->addr + sio->size - 1);
/*
 *	Set up the size to take into account the decode lines.
 */
				sio->cardaddr = cp->io_addr; 
				switch(cp->io_bus)
					{
				case 0:
					break;
				case 1:
#if 0
					sio->flags = IODF_WS;
#endif
					break;
				case 2:
					sio->flags = IODF_WS|IODF_CS16;
					break;
				case 3:
					sio->flags = IODF_WS|IODF_CS16|IODF_16BIT;
					break;
					}
#ifdef	DEBUG
#ifndef	SYSINSTALL
				fprintf(stderr, "Using I/O addr 0x%x, size %d\n",
					sio->addr, sio->size);
#else	/* SYSINSTALL */
				{
				char	s[200];
				sprintf(s, "PC-Card: Using I/O addr 0x%x, size %d\n",
					sio->addr, sio->size);
				dialog_clear();
				msgNotify(s);
				}
#endif	/* SYSINSTALL */
#endif /* DEBUG */
				if (cio->next) 
					{
					sio->next = xmalloc(sizeof(*sio));
					bzero(sio->next, sizeof(*sio));
					sio = sio->next;
					cio = cio->next;
					}
				}
			}
#endif
		}
	sp->irq = sp->config->irq;
	if (sp->irq == 16) sp->irq = 0;
	return(0);
}
/*
 *	setup_slot - Allocate the I/O and memory contexts
 *	return true if completed OK.
 */
int
setup_slot(struct slot *sp)
{
struct mem_desc mem;
struct io_desc io;
struct drv_desc drv;
struct driver *drvp = sp->config->driver;
struct slotstate state;
struct allocblk *sio;
char c;
off_t offs;
int	rw_flags;
int	iowin;

	memset(&io,0,sizeof io);
	memset(&drv,0,sizeof drv);
	offs = sp->cis->reg_addr;
	rw_flags = MDF_ATTR;
	ioctl(sp->fd, PIOCRWFLAG, &rw_flags);
	lseek(sp->fd, offs, SEEK_SET);
	c = 0x80;
	write(sp->fd, &c, sizeof(c));
	usleep(sp->card->reset_time*1000);
	lseek(sp->fd, offs, SEEK_SET);
	c = 0x00;
	write(sp->fd, &c, sizeof(c));
	usleep(sp->card->reset_time*1000);
	lseek(sp->fd, offs, SEEK_SET);
	c = sp->config->index;
	c |= 0x40;
	write(sp->fd, &c, sizeof(c));
#ifdef	DEBUG
#ifndef	SYSINSTALL
	printf("Setting config reg at offs 0x%lx", (unsigned long)offs);
	printf(" to 0x%x\n",c); 
	printf("Reset time = %d ms\n", sp->card->reset_time);
#else	/* SYSINSTALL */
	{
	char	s[200];
	sprintf(s,
		"PC-card: Setting config reg at offs 0x%lx to 0x%x\nReset time = %d ms\n",
		(unsigned long)offs, c, sp->card->reset_time);
	dialog_clear();
	msgNotify(s);
	sleep(1);
	}
#endif	/* SYSINSTALL */
#endif	/* DEBUG */
	sleep(5);
	usleep(sp->card->reset_time*1000);
/*
 *	If other config registers exist, set them up.
 */
	if (sp->cis->ccrs & 2)		/* CCSR */
		{
		c = 0;
		if (sp->cis->def_config && sp->cis->def_config->misc_valid &&
		    (sp->cis->def_config->misc & 0x8))
			c |= 0x08;
		if (sp->card_config->io_bus == 1)
			c |= 0x20;
		lseek(sp->fd, offs+2, SEEK_SET);
		write(sp->fd, &c, sizeof(c));
#ifdef	DEBUG
#ifndef	SYSINSTALL
		printf("Setting CCSR reg to 0x%x\n", c);
#endif	/* SYSINSTALL */
#endif	/* DEBUG */
		}
	mem.window = 0;
#ifdef DEBUG
#ifndef	SYSINSTALL
		printf("Mem@ %x %d %x\n",sp->mem.addr, sp->mem.size, sp->mem.cardaddr);
#endif	/* SYSINSTALL */
#endif	/* DEBUG */
	if (sp->mem.addr)
		{
		mem.window = 0;
		mem.flags = sp->mem.flags | MDF_ACTIVE | MDF_16BITS;
		mem.start = (caddr_t)sp->mem.addr;
		mem.card = sp->mem.cardaddr;
		mem.size = sp->mem.size;
		if (ioctl(sp->fd, PIOCSMEM, &mem))
			{
			logerr("ioctl (PIOCSMEM)");
			return(0);
			}
		}
/*	io.window = 0; */
	ioctl( sp->fd, PIOCGSTATE, &state );
	for (iowin = 0, sio = &(sp->io); iowin <= 1 ; iowin++)
		{
		io.window = iowin;
		if (sio->size)
			{
			io.flags = sio->flags;
			io.start = sio->addr;
			io.size = sio->size;
/*
			io.start = sp->io.addr & ~((1 << sp->io.cardaddr)-1);
			io.size = 1 << sp->io.cardaddr;
			if (io.start < 0x100)
				{
				io.start = 0x100;
				io.size = 0x300;
				}
 */
			}
#ifdef	DEBUG
#ifndef	SYSINSTALL
		fprintf(stderr, "Assigning I/O window %d, start 0x%x, size 0x%x flags 0x%x\n",
			io.window, io.start, io.size,io.flags);
#else	/* SYSINSTALL */
		{
		char	s[200];
		sprintf(s, "PC-card: Assigning I/O window %d, start 0x%x, size 0x%x flags 0x%x\n",
			io.window, io.start, io.size,io.flags);
		dialog_clear();
		msgNotify(s);
		sleep(1);
		}
#endif	/* SYSINSTALL */
#endif	/* DEBUG */
		io.flags |= IODF_ACTIVE;
		if (ioctl(sp->fd, PIOCSIO, &io))
			{
			logerr("ioctl (PIOCSIO)");
			return(0);
			}
		if (sio->next) 
			sio = sio->next;
		else
			break;
		}
	strcpy(drv.name, drvp->kernel);
	drv.unit = drvp->unit;
	drv.irqmask = sp->irq ? (1 << sp->irq) : 0;
	drv.flags = sp->config->flags;
	/* XXX */
	if (strncmp(drvp->name,"sio",3)==0)
		{
#if 0
		printf("Set Irqmask = 0\n");
		drv.irqmask = 0;
#endif
		drv.flags |= 0x80;
		}
	if (sp->mem.size)
		{
		drv.mem = sp->mem.addr;
		drv.memsize = sp->mem.size;
		}
	else
		{
		drv.mem = 0;
		drv.memsize = 0;
		}
	if (sp->io.size)
		drv.iobase = sp->io.addr;
	else
		drv.iobase = 0;
#ifdef	DEBUG
#ifndef	SYSINSTALL
	fprintf(stderr, "Assign %s%d, io 0x%x, mem 0x%lx, %d bytes, irq %d, flags %x\n",
	   drv.name, drv.unit, drv.iobase, drv.mem, drv.memsize, sp->irq, drv.flags);
#else	/* SYSINSTALL */
	{
	char	s[200];
	sprintf(s, "PC-Card: Assign %s%d, io 0x%x, mem 0x%lx, %d bytes,\n         irq %d, flags %x\n",
	   drv.name, drv.unit, drv.iobase, drv.mem, drv.memsize, sp->irq, drv.flags);
	dialog_clear();
	msgNotify(s);
	sleep(1);
	}
#endif	/* SYSINSTALL */
#endif /* DEBUG */

	memcpy(drv.misc,sp->eaddr,6);
/*
 *	Pass the CIS string to enabler (the enabler can use it for
 *	card/vendor-specific initialization)
 */
	strncpy(drv.id.manuf, sp->cis->manuf, CIS_MAXSTR - 1);
	drv.id.manuf[CIS_MAXSTR - 1] = 0;
	strncpy(drv.id.vers, sp->cis->vers, CIS_MAXSTR - 1);
	drv.id.vers[CIS_MAXSTR - 1] = 0;
	strncpy(drv.id.add_info1, sp->cis->add_info1, CIS_MAXSTR - 1);
	drv.id.add_info1[CIS_MAXSTR - 1] = 0;
	strncpy(drv.id.add_info2, sp->cis->add_info2, CIS_MAXSTR - 1);
	drv.id.add_info2[CIS_MAXSTR - 1] = 0;

/*
 *	If the driver fails to be connected to the device,
 *	then it may mean that the driver did not recognise it.
 */
	if (ioctl(sp->fd, PIOCSDRV, &drv))
		{
#ifdef	DEBUG
#ifndef	SYSINSTALL
		perror(sp->card->manuf);
#endif	/* SYSINSTALL */
#endif	/* DEBUG */
#ifndef	SYSINSTALL
		log_1s("driver allocation failed for %s", sp->card->manuf);
#else	/* SYSINSTALL */
		log_1s("PC-card: driver allocation failed for %s", sp->card->manuf);
#endif	/* SYSINSTALL */
		return(0);
		}
	return(1);
}
