/*
 * Copyright (c) 1995 Andrew McRae.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Code cleanup, bug-fix and extention
 * by Tatsumi Hosokawa <hosokawa@mt.cs.keio.ac.jp>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#ifdef	SYSINSTALL
#include <dialog.h>
#endif	/* SYSINSTALL */
#include "cardd.h"

static FILE    *in;
static int      pushc, pusht;
static int      lineno;
static char    *filename;

static char    *keys[] =
{
	"io",		/* 1 */
	"irq",		/* 2 */
	"memory",	/* 3 */
	"card",		/* 4 */
	"device",	/* 5 */
	"config",	/* 6 */
	"__EOF__",	/* 7 */
	"reset",	/* 8 */
	"ether",	/* 9 */
	"insert",	/* 10 */
	"remove",	/* 11 */
	"iosize",	/* 12 */
	"memsize",	/* 13 */
	"function",	/* 14 */
	0
};

struct flags {
	char           *name;
	int             mask;
};

static void     parsefile(void);
static char    *getline(void);
static char    *next_tok(void);
static int      num_tok(void);
static void     error(char *);
static int      keyword(char *);
static struct allocblk *ioblk_tok(int);
static struct allocblk *memblk_tok(int);
static int      irq_tok(int);
static int      config_tok(void);
static int      func_tok(void);
static int      ether_mac_tok(struct ether_mac *);
static struct driver  *new_driver(char *);

static void     addcmd(struct cmd **);
static void     parse_card(int);

/*
 * Read a file and parse the pcmcia configuration data.
 * After parsing, verify the links.
 */
int
readfile(char *name)
{
	struct card    *cp;

	in = fopen(name, "r");
	if (in == 0) {
#ifndef	SYSINSTALL
		perror(name);
		exit(1);
#else	/* SYSINSTALL */
		return 1;
#endif	/* SYSINSTALL */
	}
	parsefile();
	for (cp = cards; cp; cp = cp->next) {
		if (cp->config == 0) {
#ifndef	SYSINSTALL
			fprintf(stderr,
			"warning: card %s(%s) has no valid configuration\n",
				cp->manuf, cp->version);
#else	/* SYSINSTALL */
			char            s[200];
			sprintf(s, "warning: card %s(%s) has no valid configuration\n",
				cp->manuf, cp->version);
			dialog_clear();
			msgConfirm(s);
#endif	/* SYSINSTALL */
		}
	}
	return 0;
}

static void
parsefile(void)
{
	int             i;
	struct allocblk *bp;

	pushc = 0;
	lineno = 1;
	for (;;)
		switch (keyword(next_tok())) {
		default:
			error("Syntax error");
			pusht = 0;
			break;
		case 7:
			return;
/*
 *	reserved I/O blocks
 */
		case 1:
			while ((bp = ioblk_tok(0)) != 0) {
				if (bp->size == 0 || bp->addr == 0) {
					free(bp);
					continue;
				}
				bit_nset(io_avail, bp->addr, bp->addr + bp->size - 1);
				bp->next = pool_ioblks;
				pool_ioblks = bp;
			}
			pusht = 1;
			break;
/*
 *	reserved irqs
 */
		case 2:
			while ((i = irq_tok(0)) > 0)
				pool_irq[i] = 1;
			pusht = 1;
			break;
/*
 *	reserved memory blocks.
 */
		case 3:
			while ((bp = memblk_tok(0)) != 0) {
				if (bp->size == 0 || bp->addr == 0) {
					free(bp);
					continue;
				}
				bit_nset(mem_avail, MEM2BIT(bp->addr),
					 MEM2BIT(bp->addr + bp->size) - 1);
				bp->next = pool_mem;
				pool_mem = bp;
			}
			pusht = 1;
			break;
/*
 *	Card definition.
 */
		case 4:
			parse_card(DT_VERS);
			break;
		case 14:
			parse_card(DT_FUNC);
			break;
		}
}

/*
 *	Parse a card definition.
 */
static void
parse_card(int deftype)
{
	char           *man, *vers;
	struct card    *cp;
	int             i;
	struct card_config *confp, *lastp;
	struct ether_mac ether_mac;
	struct ether   *ether;

	cp = xmalloc(sizeof(*cp));
	cp->deftype = deftype;
	cp->ether = 0;
	switch (deftype) {
	case DT_VERS:
		man = newstr(next_tok());
		vers = newstr(next_tok());
		cp->manuf = man;
		cp->version = vers;
		cp->func_id = 0;
		break;
	case DT_FUNC:
		cp->manuf = "";
		cp->version = "";
		cp->func_id = (u_char) func_tok();
		break;
	default:
		fprintf(stderr, "parse_card: unknown deftype %d\n", deftype);
		exit(1);
	}
#if 1
	cp->reset_time = 100;	/* 50ms is insufficient for some cards */
#else
	cp->reset_time = 50;
#endif
#if 1
	cp->next = 0;
	if (!last_card)
		cards = last_card = cp;
	else {
		last_card->next = cp;
		last_card = cp;
	}
#else
	cp->next = cards;
	cards = cp;
#endif
	for (;;) {
		switch (keyword(next_tok())) {
		default:
			pusht = 1;
			return;
		case 8:
			i = num_tok();
			if (i == -1) {
				error("Illegal card reset time");
				break;
			}
			cp->reset_time = i;
			break;
		case 6:
			i = config_tok();
			if (i == -1) {
				error("Illegal card config index");
				break;
			}
			confp = xmalloc(sizeof(*confp));
			man = next_tok();
			confp->driver = new_driver(man);
			confp->irq = irq_tok(1);
			confp->flags = num_tok();
			if (confp->flags == -1) {
				pusht = 1;
				confp->flags = 0;
			}
#if 1	/* quick hack: irq == 16 means PIO mode (hosokawa) */
			if (confp->irq < 0 || confp->irq > 16)
#else
			if (confp->irq < 0 || confp->irq > 15)
#endif
			{
				error("Illegal card IRQ value");
				break;
			}
			confp->index = i & 0x3F;
/*
 *	If no valid driver for this config, then do not save
 *	this configuration entry.
 */
			if (confp->driver) {
				if (cp->config == 0)
					cp->config = confp;
				else {
					for (lastp = cp->config; lastp->next;
					     lastp = lastp->next);
					lastp->next = confp;
				}
			} else
				free(confp);
			break;
		case 9:
			ether = xmalloc(sizeof(*ether));
			ether->type = ETHTYPE_GENERIC;
			if (strcmp("megahertz", next_tok()) == 0) {
				ether->type = ETHTYPE_MEGAHERTZ;
			}
			else {
				pusht = 1;
				if (strcmp("wavelan", next_tok()) == 0) {
					ether->type = ETHTYPE_WAVELAN;
				}
				else {
					pusht = 1;
				}
			}
			if (ether->type == ETHTYPE_GENERIC) {
				ether->attr = num_tok();
				if (ether->attr == -1) {
					error("Illegal ether address offset");
					free(ether);
					break;
				}
			}
			while (ether_mac_tok(&ether_mac) == 0) {
				struct ether_mac *em;
				em = xmalloc(sizeof(*em));
				for (i = 0; i < 3; i++) {
					em->addr[i] = ether_mac.addr[i];
				}
				em->next = ether->ether_mac;
				ether->ether_mac = em;
			}
			pusht = 1;
			ether->next = cp->ether;
			cp->ether = ether;
			break;
		case 10:
			addcmd(&cp->insert);
			break;
		case 11:
			addcmd(&cp->remove);
			break;
		}
	}
}
/*
 *	Generate a new driver structure. If one exists, use
 *	that one after confirming the correct class.
 */
static struct driver  *
new_driver(char *name)
{
	struct driver  *drvp;
	char           *p;

	for (drvp = drivers; drvp; drvp = drvp->next)
		if (strcmp(drvp->name, name) == 0)
			return (drvp);
	drvp = xmalloc(sizeof(*drvp));
	drvp->next = drivers;
	drivers = drvp;
	drvp->name = newstr(name);
	drvp->kernel = newstr(name);
	p = drvp->kernel;
	while (*p++)
		if (*p >= '0' && *p <= '9') {
			drvp->unit = atoi(p);
			*p = 0;
			break;
		}
#ifdef	DEBUG
	if (verbose)
		printf("Drv %s%d created\n", drvp->kernel, drvp->unit);
#endif
	return (drvp);
}

/*
 *	Parse one I/O block.
 */
static struct allocblk *
ioblk_tok(int force)
{
	struct allocblk *io;
	int             i, j;

	if ((i = num_tok()) >= 0) {
		if (strcmp("-", next_tok()) || (j = num_tok()) < 0 || j < i) {
			error("I/O block format error");
			return (0);
		}
		io = xmalloc(sizeof(*io));
		io->addr = i;
		io->size = j - i + 1;
		if (j > IOPORTS) {
			error("I/O port out of range");
			if (force) {
				free(io);
				io = 0;
			} else
				io->addr = io->size = 0;
		}
		return (io);
	}
	if (force)
		error("Illegal or missing I/O block spec");
	return (0);
}

/*
 *	Parse a memory block.
 */
static struct allocblk *
memblk_tok(int force)
{
	struct allocblk *mem;
	int             i, j;

	if ((i = num_tok()) >= 0)
		if ((j = num_tok()) < 0)
			error("Illegal memory block");
		else {
			mem = xmalloc(sizeof(*mem));
			mem->addr = i & ~(MEMUNIT - 1);
			mem->size = (j + MEMUNIT - 1) & ~(MEMUNIT - 1);
			if (i < MEMSTART || (i + j) > MEMEND) {
				error("Memory address out of range");
				if (force) {
					free(mem);
					mem = 0;
				} else
					mem->addr = mem->size = 0;
			}
			return (mem);
		}
	if (force)
		error("Illegal or missing memory block spec");
	return (0);
}

/*
 *	IRQ token. Must be number > 0 && < 16.
 *	If force is set, IRQ must exist, and can also be '?'.
 */
static int
irq_tok(int force)
{
	int             i;

	if (strcmp("?", next_tok()) == 0 && force)
		return (0);
	pusht = 1;
	if (strcmp("any", next_tok()) == 0 && force)
		return (0);
	pusht = 1;
	if (strcmp("pio", next_tok()) == 0 && force)
		return (16);
	pusht = 1;
	i = num_tok();
	if (i > 0 && i < 16)
		return (i);
	if (force)
		error("Illegal IRQ value");
	return (-1);
}

/*
 *	Config index token
 */
static int
config_tok(void)
{
	if (strcmp("default", next_tok()) == 0)	
		return 0;
	pusht = 1;
	return num_tok();
}

/*
 *	Function ID token
 */
static int
func_tok(void)
{
	if (strcmp("serial", next_tok()) == 0)	
		return 2;
	pusht = 1;
	if (strcmp("fixed_disk", next_tok()) == 0)	
		return 4;
	pusht = 1;
	return num_tok();
}

/*
 *	Ethernet address token
 */
static int
ether_mac_tok(struct ether_mac *ether_mac)
{
	int	i, x;
	u_char *t;
	static int	hex[] = {1, 1, 0, 1, 1, 0, 1, 1};

	t = next_tok();
	if (strlen(t) != 8) {
		return -1;
	}
	for (i = 0; i < 8; i++) {
		if (!(hex[i] ? isxdigit(t[i]) : (t[i] == ':'))) {
			return -1;
		}
	}
	for (i = 0; i < 3; i++) {
		sscanf(t + i * 3, "%x", &x);
		ether_mac->addr[i] = x;
	}
	return 0;
}

/*
 *	search the table for a match.
 */
static int
keyword(char *str)
{
	char          **s;
	int             i = 1;

	for (s = keys; *s; s++, i++)
		if (strcmp(*s, str) == 0)
			return (i);
	return (0);
}

/*
 *	addcmd - Append the command line to the list of
 *	commands.
 */
static void
addcmd(struct cmd ** cp)
{
	char           *s = getline();
	struct cmd     *ncp;

	if (*s) {
		ncp = xmalloc(sizeof(*ncp));
		ncp->line = s;
		while (*cp)
			cp = &(*cp)->next;
		*cp = ncp;
	}
}

static void
error(char *msg)
{
	pusht = 1;
#ifndef	SYSINSTALL
	fprintf(stderr, "%s: %s at line %d, near %s\n",
		filename, msg, lineno, next_tok());
#else	/* SYSINSTALL */
	{
		char            s[200];
		sprintf(s, "%s: %s at line %d, near %s\n",
			filename, msg, lineno, next_tok());
		dialog_clear();
		msgConfirm(s);
	}
#endif	/* SYSINSTALL */
	pusht = 1;
}

static int             last_char;

static int
get()
{
	int             c;

	if (pushc)
		c = pushc;
	else
		c = getc(in);
	pushc = 0;
	while (c == '\\') {
		c = getc(in);
		switch (c) {
		case '#':
			return (last_char = c);
		case '\n':
			lineno++;
			c = getc(in);
			continue;
		}
		pushc = c;
		return ('\\');
	}
	if (c == '\n')
		lineno++;
	if (c == '#') {
		while (get() != '\n');
		return (last_char = '\n');
	}
	return (last_char = c);
}
/*
 *	num_tok - expecting a number token. If not a number,
 *	return -1.
 *	Handles octal (who uses octal anymore?)
 *		hex
 *		decimal
 *	Looks for a 'k' at the end of decimal numbers
 *	and multiplies by 1024.
 */
static int
num_tok()
{
	char           *s = next_tok(), c;
	int             val = 0, base;

	base = 10;
	c = *s++;
	if (c == '0') {
		base = 8;
		c = *s++;
		if (c == 'x' || c == 'X') {
			c = *s++;
			base = 16;
		}
	}
	do {
		switch (c) {
		case 'k':
		case 'K':
			if (val && base == 10 && *s == 0)
				return (val * 1024);
			return (-1);
		default:
			return (-1);
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
			val = val * base + c - '0';
			break;

		case '8':
		case '9':
			if (base == 8)
				return (-1);
			else
				val = val * base + c - '0';
			break;
		case 'a':
		case 'b':
		case 'c':
		case 'd':
		case 'e':
		case 'f':
			if (base == 16)
				val = val * base + c - 'a' + 10;
			else
				return (-1);
			break;
		case 'A':
		case 'B':
		case 'C':
		case 'D':
		case 'E':
		case 'F':
			if (base == 16)
				val = val * base + c - 'A' + 10;
			else
				return (-1);
			break;
		}
	} while ((c = *s++) != 0);
	return (val);
}

static char           *_next_tok();

static char           *
next_tok()
{
	char           *s = _next_tok();
#if 0
	printf("Tok = %s\n", s);
#endif
	return (s);
}

/*
 *	get one token. Handles string quoting etc.
 */
static char           *
_next_tok()
{
	static char     buf[1024];
	char           *p = buf, instr = 0;
	int             c;

	if (pusht) {
		pusht = 0;
		return (buf);
	}
	for (;;) {
		c = get();
		switch (c) {
		default:
			*p++ = c;
			break;
		case '"':
			if (instr) {
				*p++ = 0;
				return (buf);
			}
			instr = 1;
			break;
		case '\n':
			if (instr) {
				error("Unterminated string");
				break;
			}
/*
 *	Eat whitespace unless in a string.
 */
		case ' ':
		case '\t':
			if (!instr) {
				if (p != buf) {
					*p++ = 0;
					return (buf);
				}
			} else
				*p++ = c;
			break;
/*
 *	Special characters that must be tokens on their own.
 */
		case '-':
		case '?':
		case '*':
			if (instr)
				*p++ = c;
			else {
				if (p != buf)
					pushc = c;
				else
					*p++ = c;
				*p++ = 0;
				return (buf);
			}
			break;
		case EOF:
			if (p != buf) {
				*p++ = 0;
				return (buf);
			}
			strcpy(buf, "__EOF__");
			return (buf);
		}
	}
}

/*
 *	get the rest of the line. If the
 *	last character scanned was a newline, then
 *	return an empty line. If this isn't checked, then
 *	a getline may incorrectly return the next line.
 */
static char           *
getline()
{
	char            buf[1024], *p = buf;
	int             c, i = 0;

	if (last_char == '\n')
		return (newstr(""));
	do {
		c = get();
	} while (c == ' ' || c == '\t');
	for (; c != '\n' && c != EOF; c = get())
		if (i++ < sizeof(buf) - 10)
			*p++ = c;
	*p = 0;
	return (newstr(buf));
}
