/*
 * FreeBSD sysinstall internationalization support
 *
 * Copyright (c) 1996 
 *   Tatsumi Hosokawa <hosokawa@jp.FreeBSD.org>   All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    verbatim and that no modifications are made prior to this
 *    point in the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TATSUMI HOSOKAWA ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JORDAN HUBBARD OR HIS PETS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, LIFE OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*
 *                         -- GUIDELINE --
 *
 * Public interface of this file is "i18n()".  It translates messages
 * into another language specified by variable "language" defined in
 * sysinstall.h.  The translation is based on message files.  Each
 * message have its own ID number expressed in five decimal
 * characters.  Message file have to be placed on /stand/I18N/messages 
 * and its filename is "messages.(LOCALE)".  Currently messages.ja_JP
 * (Japanese, EUC code) and messages.zh_TW (Traditional Chinese, Big5
 * code) are supported.  If you want to support another languages,
 * please read these message files.
 *
 * Messages translated by "i18n()" have to satisfy the following
 * format. If messages don't satisfy following format, "i18n()" simply
 * be echoed back to its callers.
 *
 * Format:
 * "%%xxxxx English message" (xxxxx: ID number)
 * (don't forget to put a space character between ID and English
 *  message) 
 *
 * Currently, first two digits are allocated to each source file, and
 * last three digits are allocated for each message.
 *
 * If variable "lanuguage" is set to LANG_ENGLISH, or corresponding
 * message does not exist on message file, it simply returns the
 * English message follows ID string.
 * 
 * If you want to extend the functionality of sysinstall, you don't
 * have to worry about I18N support.  If you put new messages in code, 
 * it simply ignored by "i18n()".  You don't have to put "%%xxxxx" new 
 * ID numbers.  Messages without "%%xxxxx" simply echoed back by
 * "i18n()", so English messages are displayed after all.
 *
 * Again, DON'T WORRY ABOUT I18N.  It affects you nothing, even if you
 * can't translated messages into any other languages.  We'll do that.
 *
 * Notes:
 * Currently, ad-hoc implementation of support of multibyte characters 
 * for libncurses and libdialog is not smart, and it will affect the
 * iso8859 European charset support.  It should be replaced with
 * another implementation.
 *
 * Tatsumi Hosokawa <hosokawa@jp.FreeBSD.org>, Tokyo, Japan.
 */

#include "sysinstall.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#undef	HASH_DEBUG

#ifdef I18N

/* sysinstall messages are stored to these files */
#define	I18N_MSG_FILE	"/stand/I18N/messages/messages.%s"

/* default language is English */
enum lang language = LANG_ENGLISH;

/* definition of hash table to which the messages are stored */

#define	N_I18N_MSG_HASH	256

typedef struct i18n_msg_hash {
    struct i18n_msg_hash *next;
    char	*ptr;
} i18n_msg_hash;

typedef struct i18n_msg_data {
    char	*msg;
    size_t	size;
    i18n_msg_hash *hash[N_I18N_MSG_HASH];
} i18n_msg_data;

/* this structure contains messages themselves and hash tables for messages */
struct i18n_msg_data i18n_msgs[LANG_NLANG];

/* number of ring buffers i18n() uses */
#define	N_I18N_RBUF	256


/* calculate hash table */
static int
hash(char *msg)
{
    int	i, d, offset = 0;
    char buf[8];
    
    if (strncmp(msg, "%%", 2) == 0) {
	offset = 2; /* strlen("%%") == 2 */
    }
    else if (strncmp(msg, "MES#", 4) == 0) {
	offset = 4; /* strlen("MSG#") == 4 */
    }
    else {
	return -1;
    }
    for (i = 0; i < 5; i++) {
	buf[i] = msg[i + offset];
    }
    buf[5] = 0;
    sscanf(buf, "%d", &d);

    return d % N_I18N_MSG_HASH;
}

/* register messages into hash table */
static void
register_msg(char *msg, i18n_msg_data *data)
{
    int hash_val;
    static i18n_msg_hash *p, *q;

    p = safe_malloc(sizeof(*p));
    hash_val = hash(msg);
#ifdef	HASH_DEBUG
    {
	char buf[10];
	
	strncpy(buf, msg, 9);
	buf[9] = 0;
	msgDebug("Registering: %s (%d)\n", buf, hash_val);
    }
#endif	/* HASH_DEBUG */
    q = data->hash[hash_val];
    data->hash[hash_val] = p;
    p->next = q;
    p->ptr = msg;
}

/* load messages from file and store them into hash table */
static int 
load_msg(char *i18n, i18n_msg_data *data) 
{
    int		i, fd;
    char	msgfile[64];
    struct stat	sb;
 
    bzero(data, sizeof(*data));
    
    sprintf(msgfile, I18N_MSG_FILE, i18n);
    
    if ((fd = open(msgfile, O_RDONLY)) < 0) {
	msgDebug("Can't open I18N message file %s\n", msgfile);
	return -1;
    }

    if (fstat(fd, &sb) < 0) {
	msgDebug("%s: fstat failed", msgfile);
	return -1;
    }

    data->size = sb.st_size;

    data->msg = safe_malloc(data->size + 32);
    if (read(fd, data->msg, data->size) < 0) {
	msgDebug("Can't read I18N message file %s\n", msgfile);
	safe_free(data->msg);
	data->msg = 0;
	return -1;
    }

    for (i = 0; i < data->size; i++) {
	if (strncmp(data->msg + i, "MES#", 4) == 0 &&
	    isdigit(data->msg[i + 4]) && isdigit(data->msg[i + 5]) && 
	    isdigit(data->msg[i + 6]) && isdigit(data->msg[i + 7]) && 
	    isdigit(data->msg[i + 8])) {
	    register_msg(data->msg + i, data);
	}
    }
    
    return 0;
}

static char	*rbuf[N_I18N_RBUF] = {0};

static char *
i18n_alloc(int len)
{
    static int	idx = 0;
    char *r;

    if (rbuf[idx]) {
	safe_free(rbuf[idx]);
	rbuf[idx] = 0;
    }
    r = rbuf[idx] = safe_malloc(len);
    idx++;
    if (idx >= N_I18N_RBUF) {
	idx = 0;
    }

    return r;
}

/* private routine called from i18n() function:
   get coressponding messages from hash table for current language */

static char *
private_i18n(char *str, i18n_msg_data *data)
{
    int		hash_val;
    static i18n_msg_hash *p;
    static char	buffer[1024];

    if (!str) {
	return str;
    }

    if (!data->size || str[0] != '%' || str[1] != '%' || str[7] != ' ') {
	return str;
    }

    if ((hash_val = hash(str)) < 0) {
	return str + 8;
    }
    
    for (p = data->hash[hash_val]; p; p = p->next) {
	if (strncmp(str + 2, p->ptr + 4, 5) == 0) {
	    char *q, *r;
	    int skip;
	    
	    skip = 1;
#ifdef	HASH_DEBUG
	    msgDebug("Found %s (%d).\n", str, hash_val);
#endif	/* HASH_DEBUG */
	    r = buffer;

	    for (q = p->ptr + 9; !(q[0] == '\\' && q[1] == '\\'); q++) {
		if (skip && !isspace(*q)) {
		    skip = 0;
		}
		if (!skip) {
		    *r = *q;
		    r++;
		}
	    }
	    *r = 0;

	    return buffer;
	}
    }
#ifdef	HASH_DEBUG
    msgDebug("Not Found %s (%d).\n", str, hash_val);
#endif	/* HASH_DEBUG */
    return str + 8;
}

/* 'public' interfaces for I18N message translation */

char *
i18n(char *msg)
{
    char *translated;

    if (!msg) {
	return msg;
    }
    if (language == LANG_ENGLISH) {
	if (msg[0] == '%' || msg[1] == '%' || msg[7] == ' ') {
	    return msg + 8;
	}
    }
    translated = private_i18n(msg, &i18n_msgs[language]);
    if (!translated || !strlen(translated)) {
	translated = "--";	/* translation failed */
    }
    else {
	char *tmp;
	
	tmp = i18n_alloc(strlen(translated) + 1);
	strcpy(tmp, translated);
	translated = tmp;
    }
    return translated;
}


int
i18nlen(char *str)
{
    return strlen(i18n(str));
}

/* create a string that repeat 'c' strlen(i18n(str)) times */

char *
i18nrep(char *str, int c)
{
    int	 i, l;
    static char	buf[128];
    
    buf[0] = 0;
    l = i18nlen(str);

    if (l < sizeof(buf)) {
	for (i = 0; i < l; i++) {
	    buf[i] = (char)c;
	}
	buf[i] = 0;
    }
    
    return buf;
}

/* initialize I18N support code */
void 
init_i18n(void)
{
    /* Initialize ring buffer */
    bzero(rbuf, sizeof(rbuf));
    
    /* Load message files */
#ifdef JAPANESE
    load_msg("ja_JP", &i18n_msgs[LANG_JAPANESE]);
#endif	/* JAPANESE */
#ifdef CHINESE
    load_msg("zh_TW", &i18n_msgs[LANG_CHINESE]);
#endif	/* CHINESE */
#ifdef KOREAN
    load_msg("ko_KR", &i18n_msgs[LANG_KOREAN]);
#endif	/* CHINESE */

    /* I18N replacement of static variables */
    menus_i18n();
    options_i18n();
    anonFTP_i18n();
    index_i18n();
    tcpip_i18n();
    user_i18n();
    register_i18n();
    tzsetup_i18n();
}
#endif /* I18N */
