/*
 * The new sysinstall program.
 *
 * This is probably the last program in the `sysinstall' line - the next
 * generation being essentially a complete rewrite.
 *
 * $Id: menus.c,v 1.180.2.23 1999/05/15 14:38:42 jkh Exp $
 *
 * Copyright (c) 1995
 *	Jordan Hubbard.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    verbatim and that no modifications are made prior to this
 *    point in the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY JORDAN HUBBARD ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JORDAN HUBBARD OR HIS PETS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, LIFE OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "sysinstall.h"

#ifdef __alpha__
#define _AS(str) str "alpha/"
#else	/* i386 */
#define _AS(str) str "i386/"
#endif
#define _AP(str) _AS(str "/pub/FreeBSD/releases/")

/* Miscellaneous work routines for menus */
static int
setSrc(dialogMenuItem *self)
{
    Dists |= DIST_SRC;
    SrcDists = DIST_SRC_ALL;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
clearSrc(dialogMenuItem *self)
{
    Dists &= ~DIST_SRC;
    SrcDists = 0;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
setX11All(dialogMenuItem *self)
{
    XF86Dists = DIST_XF86_ALL;
    XF86ServerDists = DIST_XF86_SERVER_ALL;
    XF86FontDists = DIST_XF86_FONTS_ALL;
    Dists |= DIST_XF86;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
clearX11All(dialogMenuItem *self)
{
    XF86Dists = 0;
    XF86ServerDists = 0;
    XF86FontDists = 0;
    Dists &= ~DIST_XF86;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
setX11Misc(dialogMenuItem *self)
{
    XF86Dists |= DIST_XF86_MISC_ALL;
    Dists |= DIST_XF86;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
clearX11Misc(dialogMenuItem *self)
{
    XF86Dists &= ~DIST_XF86_MISC_ALL;
    if (!XF86ServerDists && !XF86FontDists)
	Dists &= ~DIST_XF86;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
setX11Servers(dialogMenuItem *self)
{
    XF86Dists |= DIST_XF86_SERVER;
    XF86ServerDists = DIST_XF86_SERVER_ALL;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
clearX11Servers(dialogMenuItem *self)
{
    XF86Dists &= ~DIST_XF86_SERVER;
    XF86ServerDists = 0;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
setX11Fonts(dialogMenuItem *self)
{
    XF86Dists |= DIST_XF86_FONTS;
    XF86FontDists = DIST_XF86_FONTS_ALL;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

static int
clearX11Fonts(dialogMenuItem *self)
{
    XF86Dists &= ~DIST_XF86_FONTS;
    XF86FontDists = 0;
    return DITEM_SUCCESS | DITEM_REDRAW;
}

#define _IS_SET(dist, set) (((dist) & (set)) == (set))

#define IS_DEVELOPER(dist, extra) (_IS_SET(dist, _DIST_DEVELOPER | extra) || \
	_IS_SET(dist, _DIST_DEVELOPER | DIST_DES | extra))

#define IS_USER(dist, extra) (_IS_SET(dist, _DIST_USER | extra) || \
	_IS_SET(dist, _DIST_USER | DIST_DES | extra))

static int
checkDistDeveloper(dialogMenuItem *self)
{
    return IS_DEVELOPER(Dists, 0) && _IS_SET(SrcDists, DIST_SRC_ALL);
}

static int
checkDistXDeveloper(dialogMenuItem *self)
{
    return IS_DEVELOPER(Dists, DIST_XF86) && _IS_SET(SrcDists, DIST_SRC_ALL);
}

static int
checkDistKernDeveloper(dialogMenuItem *self)
{
    return IS_DEVELOPER(Dists, 0) && _IS_SET(SrcDists, DIST_SRC_SYS);
}

static int
checkDistXKernDeveloper(dialogMenuItem *self)
{
    return IS_DEVELOPER(Dists, DIST_XF86) && _IS_SET(SrcDists, DIST_SRC_SYS);
}

static int
checkDistUser(dialogMenuItem *self)
{
    return IS_USER(Dists, 0);
}

static int
checkDistXUser(dialogMenuItem *self)
{
    return IS_USER(Dists, DIST_XF86);
}

static int
checkDistMinimum(dialogMenuItem *self)
{
    return Dists == DIST_BIN;
}

static int
checkDistEverything(dialogMenuItem *self)
{
    return Dists == DIST_ALL && _IS_SET(SrcDists, DIST_SRC_ALL) && \
	_IS_SET(XF86Dists, DIST_XF86_ALL) && \
	_IS_SET(XF86ServerDists, DIST_XF86_SERVER_ALL) && \
	_IS_SET(XF86FontDists, DIST_XF86_FONTS_ALL);
}

static int
DESFlagCheck(dialogMenuItem *item)
{
    return DESDists;
}

static int
srcFlagCheck(dialogMenuItem *item)
{
    return SrcDists;
}

static int
x11FlagCheck(dialogMenuItem *item)
{
    return Dists & DIST_XF86;
}

static int
checkTrue(dialogMenuItem *item)
{
    return TRUE;
}

/* All the system menus go here.
 *
 * Hardcoded things like version number strings will disappear from
 * these menus just as soon as I add the code for doing inline variable
 * expansion.
 */

#ifdef I18N
DMenu MenuI18N = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "FreeBSD sysinstall - I18N Menu",
    "Please choose a language you want to use in FreeBSD installation.
Select it from this list by using the arrow keys and [Enter] key.",
    "Press F1 for more HELP",
    "languages",
    { { "English",  "English (US-ASCII)",
	  NULL, dmenuSetVariable, NULL, "language=english"},
#ifdef JAPANESE
	{ "Japanese", "Japanese (EUC-JP)",
	  NULL, dmenuSetVariable, NULL, "language=japanese"},
#endif
#ifdef CHINESE
	{ "Chinese (TW)", "Traditional Chinese (Big5)",
	    NULL,  dmenuSetVariable, NULL, "language=chinese"},
#endif
#ifdef KOREAN
	{ "Korean", "Korean (EUC-KR / Experimental)",
	    NULL,  dmenuSetVariable, NULL, "language=korean"},
#endif
	{ NULL } },
};
#endif /* I18N */

DMenu MenuIndex = {
    DMENU_NORMAL_TYPE,
    "%%00001 Glossary of functions",
    "%%00002 This menu contains an alphabetized index of the top level functions in\n"
    "this program (sysinstall).  Invoke an option by pressing [ENTER].\n"
    "Leave the index page by selecting Cancel [TAB-ENTER].",
    "%%00003 Use PageUp or PageDown to move through this menu faster!",
    NULL,
    { { "%%00006 Anon FTP",		"%%00007 Configure anonymous FTP logins.",	dmenuVarCheck, configAnonFTP, NULL, "anon_ftp" },
      { "%%00008 Commit",		"%%00009 Commit any pending actions (dangerous!)", NULL, installCustomCommit },
      { "%%00010 Console settings",	"%%00011 Customize system console behavior.",	NULL, dmenuSubmenu, NULL, &MenuSyscons },
      { "%%00012 Configure",		"%%00013 The system configuration menu.",	NULL, dmenuSubmenu, NULL, &MenuConfigure },
      { "%%32060 Defaults, Load",	"%%32061 Load default settings.",		NULL, dispatch_load_floppy },
      { "%%00014 Device, Mouse",	"%%00015 The mouse configuration menu.",	NULL, dmenuSubmenu, NULL, &MenuMouse },
      { "%%32000 Disklabel",		"%%32001 The disk Label editor",		NULL, diskLabelEditor },
      { "%%00016 Dists, All",		"%%00017 Root of the distribution tree.",	NULL, dmenuSubmenu, NULL, &MenuDistributions },
      { "%%00018 Dists, Basic",		"%%00019 Basic FreeBSD distribution menu.",	NULL, dmenuSubmenu, NULL, &MenuSubDistributions },
      { "%%00020 Dists, DES",		"%%00021 DES distribution menu.",		NULL, dmenuSubmenu, NULL, &MenuDESDistributions },
      { "%%00022 Dists, Developer",	"%%00023 Select developer's distribution.",	checkDistDeveloper, distSetDeveloper },
      { "%%00024 Dists, Src",		"%%00025 Src distribution menu.",		NULL, dmenuSubmenu, NULL, &MenuSrcDistributions },
      { "%%00026 Dists, X Developer",	"%%00027 Select X developer's distribution.",	checkDistXDeveloper, distSetXDeveloper },
      { "%%00028 Dists, Kern Developer", "%%00029 Select kernel developer's distribution.", checkDistKernDeveloper, distSetKernDeveloper },
      { "%%00030 Dists, User",		"%%00031 Select average user distribution.",	checkDistUser, distSetUser },
      { "%%00032 Dists, X User",	"%%00033 Select average X user distribution.",	checkDistXUser, distSetXUser },
      { "%%32002 Distributions, Adding", "%%32003 Installing additional distribution sets", NULL, distExtractAll },
      { "%%00034 Distributions, XFree86","%%00035 XFree86 distribution menu.",		NULL, distSetXF86 },
      { "%%00036 Documentation",	"%%00037 Installation instructions, README, etc.", NULL, dmenuSubmenu, NULL, &MenuDocumentation },
      { "%%00038 Doc, README",		"%%00039 The distribution README file.",	NULL, dmenuDisplayFile, NULL, "README" },
      { "%%00040 Doc, Hardware",	"%%00041 The distribution hardware guide.",	NULL, dmenuDisplayFile,	NULL, "HARDWARE" },
      { "%%00042 Doc, Install",		"%%00043 The distribution installation guide.",	NULL, dmenuDisplayFile,	NULL, "INSTALL" },
      { "%%00044 Doc, Copyright",	"%%00045 The distribution copyright notices.",	NULL, dmenuDisplayFile,	NULL, "COPYRIGHT" },
      { "%%00046 Doc, Release",		"%%00047 The distribution release notes.",	NULL, dmenuDisplayFile, NULL, "RELNOTES" },
      { "%%00048 Doc, HTML",		"%%00049 The HTML documentation menu.",		NULL, docBrowser },
      { "%%00050 Dump Vars",		"%%00051 (debugging) dump out internal variables.", NULL, dump_variables },
      { "%%32004 Emergency shell",	"%%32005 Start an Emergency Holographic shell.",	NULL, installFixitHoloShell },
      { "%%32006 Fdisk",		"%%32007 The disk Partition Editor",		NULL, diskPartitionEditor },
      { "%%00052 Fixit",		"%%00053 Repair mode with CDROM or fixit floppy.",	NULL, dmenuSubmenu, NULL, &MenuFixit },
      { "%%00054 FTP sites",		"%%00055 The FTP mirror site listing.",		NULL, dmenuSubmenu, NULL, &MenuMediaFTP },
      { "%%00058 Gateway",		"%%00059 Set flag to route packets between interfaces.", dmenuVarCheck, dmenuToggleVariable, NULL, "gateway=YES" },
      { "%%00060 HTML Docs",		"%%00061 The HTML documentation menu",		NULL, docBrowser },
      { "%%00062 Install, Novice",	"%%00063 A novice system installation.",	NULL, installNovice },
      { "%%00064 Install, Express",	"%%00065 An express system installation.",	NULL, installExpress },
      { "%%00066 Install, Custom",	"%%00067 The custom installation menu",		NULL, dmenuSubmenu, NULL, &MenuInstallCustom },
      { "%%00068 Label",		"%%00069 The disk Label editor",		NULL, diskLabelEditor },
      { "%%00070 Media",		"%%00071 Top level media selection menu.",	NULL, dmenuSubmenu, NULL, &MenuMedia },
      { "%%00072 Media, Tape",		"%%00073 Select tape installation media.",	NULL, mediaSetTape },
      { "%%00074 Media, NFS",		"%%00075 Select NFS installation media.",	NULL, mediaSetNFS },
      { "%%00076 Media, Floppy",	"%%00077 Select floppy installation media.",	NULL, mediaSetFloppy },
      { "%%00078 Media, CDROM",		"%%00079 Select CDROM installation media.",	NULL, mediaSetCDROM },
      { "%%00080 Media, DOS",		"%%00081 Select DOS installation media.",	NULL, mediaSetDOS },
      { "%%00082 Media, UFS",		"%%00083 Select UFS installation media.",	NULL, mediaSetUFS },
      { "%%00084 Media, FTP",		"%%00085 Select FTP installation media.",	NULL, mediaSetFTP },
      { "%%00086 Media, FTP Passive",	"%%00087 Select passive FTP installation media.", NULL, mediaSetFTPPassive },
      { "%%00088 Network Interfaces",	"%%00089 Configure network interfaces",		NULL, tcpMenuSelect },
      { "%%00090 Networking Services",	"%%00091 The network services menu.",		NULL, dmenuSubmenu, NULL, &MenuNetworking },
      { "%%00092 NFS, client",		"%%00093 Set NFS client flag.",			dmenuVarCheck, dmenuToggleVariable, NULL, "nfs_client_enable=YES" },
      { "%%00094 NFS, server",		"%%00095 Set NFS server flag.",			dmenuVarCheck, configNFSServer, NULL, "nfs_server_enable=YES" },
      { "%%00096 NTP Menu",		"%%00097 The NTP configuration menu.",		NULL, dmenuSubmenu, NULL, &MenuNTP },
      { "%%00098 Options",		"%%00099 The options editor.",			NULL, optionsEditor },
      { "%%00100 Packages",		"%%00101 The packages collection",		NULL, configPackages },
      { "%%00102 Partition",		"%%00103 The disk Slice (PC-style partition) Editor",	NULL, diskPartitionEditor },
      { "%%00104 PCNFSD",		"%%00105 Run authentication server for PC-NFS.", dmenuVarCheck, configPCNFSD, NULL, "pcnfsd" },
      { "%%00108 Root Password",	"%%00109 Set the system manager's password.",
#ifdef I18N
	NULL, passwdRoot },
#else
        NULL, dmenuSystemCommand, NULL, "passwd root" },
#endif
      { "%%00110 Router",		"%%00111 Select routing daemon (default: routed)", NULL, configRouter, NULL, "router" },
      { "%%00114 Syscons",		"%%00115 The system console configuration menu.", NULL, dmenuSubmenu, NULL, &MenuSyscons },
      { "%%00960 Syscons, Font",	"%%00961 The console screen font.",	  NULL, dmenuSubmenu, NULL, &MenuSysconsFont },
      { "%%00116 Syscons, Keymap",	"%%00117 The console keymap configuration menu.", NULL, dmenuSubmenu, NULL, &MenuSysconsKeymap },
      { "%%00118 Syscons, Keyrate",	"%%00119 The console key rate configuration menu.", NULL, dmenuSubmenu, NULL, &MenuSysconsKeyrate },
      { "%%00120 Syscons, Saver",	"%%00121 The console screen saver configuration menu.",	NULL, dmenuSubmenu, NULL, &MenuSysconsSaver },
      { "%%00962 Syscons, Screenmap",	"%%00963 The console screenmap configuration menu.", NULL, dmenuSubmenu, NULL, &MenuSysconsScrnmap },
      { "%%00122 Time Zone",		"%%00123 Set the system's time zone.",
#ifdef I18N
	NULL, tzsetup },
#else
        NULL, dmenuSystemCommand, NULL, "tzsetup" },
#endif
      { "%%00124 Upgrade",		"%%00125 Upgrade an existing system.",		NULL, installUpgrade },
      { "%%00126 Usage",		"%%00127 Quick start - How to use this menu system.",	NULL, dmenuDisplayFile, NULL, "usage" },
      { "%%00975 User Management",	"%%00976 Add user and group information.",	NULL, dmenuSubmenu, NULL, &MenuUsermgmt },
      { "%%00130 XFree86, Fonts",	"%%00131 XFree86 Font selection menu.",		NULL, dmenuSubmenu, NULL, &MenuXF86SelectFonts },
      { "%%00132 XFree86, Server",	"%%00133 XFree86 Server selection menu.",	NULL, dmenuSubmenu, NULL, &MenuXF86SelectServer },
      { "%%00964 XFree86, PC98 Server",	"%%00965 XFree86 PC98 Server selection menu.",	NULL, dmenuSubmenu, NULL, &MenuXF86SelectPC98Server },
      { NULL } },
};

/* The initial installation menu */
DMenu MenuInitial = {
    DMENU_NORMAL_TYPE,
    "%%00135 /stand/sysinstall Main Menu",			/* title */
    "%%00136 Welcome to the FreeBSD installation and configuration tool.  Please\n" /* prompt */
    "select one of the options below by using the arrow keys or typing the\n"
    "first character of the option name you're interested in.  Invoke an\n"
    "option by pressing [ENTER] or [TAB-ENTER] to exit the installation.", 
    "%%00137 Press F1 for Installation Guide",			/* help line */
    "install",						/* help file */
    { { "%%00138 Select" },
      { "%%00139 Exit Install",	NULL, NULL, dmenuExit },
      { "%%00140 1 Usage",	"%%00141 Quick start - How to use this menu system",	NULL, dmenuDisplayFile, NULL, "usage" },
      { "%%00146 2 Novice",	"%%00147 Begin a novice installation (for beginners)",	NULL, installNovice },
      { "%%00148 3 Express",	"%%00149 Begin a quick installation (for the impatient)", NULL, installExpress },
      { "%%00150 4 Custom",	"%%00151 Begin a custom installation (for experts)",	NULL, dmenuSubmenu, NULL, &MenuInstallCustom },
      { "%%00156 5 Configure",	"%%00157 Do post-install configuration of FreeBSD",	NULL, dmenuSubmenu, NULL, &MenuConfigure },
      { "%%00142 D Doc",	"%%00143 Installation instructions, README, etc.",	NULL, dmenuSubmenu, NULL, &MenuDocumentation },
      { "%%00955 K Keymap",	"%%00956 Select keyboard type",				NULL, dmenuSubmenu, NULL, &MenuSysconsKeymap },
      { "%%00144 O Options",	"%%00145 View/Set various installation options",	NULL, optionsEditor },
      { "%%00152 F Fixit",	"%%00153 Enter repair mode with CDROM/floppy or start shell",	NULL, dmenuSubmenu, NULL, &MenuFixit },
      { "%%00154 U Upgrade",	"%%00155 Upgrade an existing system",			NULL, installUpgrade },
      { "%%32062 L Load Config","%%32063 Load default install configuration",		NULL, dispatch_load_floppy },
      { "%%00158 I Index",	"%%00159 Glossary of functions",			NULL, dmenuSubmenu, NULL, &MenuIndex },
      { NULL } },
};

/* The main documentation menu */
DMenu MenuDocumentation = {
    DMENU_NORMAL_TYPE,
    "%%00171 FreeBSD Documentation Menu",
    "%%00172 If you are at all unsure about the configuration of your hardware\n"
    "or are looking to build a system specifically for FreeBSD, read the\n"
    "Hardware guide!  New users should also read the Install document for\n"
    "a step-by-step tutorial on installing FreeBSD.  For general information,\n"
    "consult the README file.",
    "%%00173 Confused?  Press F1 for help.",
    "usage",
    { { "%%00174 1 README",	"%%00175 A general description of FreeBSD.  Read this!", NULL, dmenuDisplayFile, NULL, "README" },
      { "%%00176 2 Hardware",	"%%00177 The FreeBSD survival guide for PC hardware.",	NULL, dmenuDisplayFile,	NULL, "HARDWARE" },
      { "%%00178 3 Install",	"%%00179 A step-by-step guide to installing FreeBSD.",	NULL, dmenuDisplayFile,	NULL, "INSTALL" },
      { "%%00180 4 Copyright",	"%%00181 The FreeBSD Copyright notices.",		NULL, dmenuDisplayFile,	NULL, "COPYRIGHT" },
      { "%%00182 5 Release"	,"%%00183 The release notes for this version of FreeBSD.", NULL, dmenuDisplayFile, NULL, "RELNOTES" },
      { "%%00184 6 Shortcuts",	"%%00185 Creating shortcuts to sysinstall.",		NULL, dmenuDisplayFile, NULL, "shortcuts" },
      { "%%00186 7 HTML Docs",	"%%00187 Go to the HTML documentation menu (post-install).", NULL, docBrowser },
#if defined(PCCARD)
      { "%%00858 P PC-card"	,"%%00859 PC-card (PAO package) installation guide.", NULL, dmenuDisplayFile, NULL, "pccard" },
#endif	/* PCCARD */
#if defined(JAPANESE) || defined(CHINESE)
      { "%%00188 I I18N", "%%00189 Sysinstall internationalization.",	NULL, dmenuDisplayFile, NULL, "i18n" },
#endif
      { "%%00190 0 Exit",	"%%00191 Exit this menu (returning to previous)",	NULL, dmenuExit },
      { NULL } },
};

DMenu MenuMouseType = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%32141 Select a protocol type for your mouse",
    "%%32142 If you are not sure, choose \"Auto\".  It should always work for bus\n"
    "and PS/2 style mice.  It may not work for the serial mouse if the mouse\n"
    "does not support the PnP standard.  But, it won't hurt.  Many 2-button\n"
    "serial mice are compatible with \"Microsoft\" or \"MouseMan\".  3-button\n"
    "serial mice may be compatible with \"MouseSystems\" or \"MouseMan\".  If\n"
    "the mouse has a wheel, it may be compatible with \"IntelliMouse\".",
    NULL,
    NULL,
    { { "Auto",		"%%32143 Bus mouse, PS/2 style mouse or PnP serial mouse",	
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=auto" },
      { "GlidePoint",	"%%32144 ALPS GlidePoint pad (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=glidepoint" },
      { "Hitachi","%%32145 Hitachi tablet (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=mmhittab" },
      { "IntelliMouse",	"%%32146 Microsoft IntelliMouse (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=intellimouse" },
      { "Logitech",	"%%32147 Logitech protocol (old models) (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=logitech" },
      { "Microsoft",	"%%32148 Microsoft protocol (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=microsoft" },
      { "MM Series","%%32149 MM Series protocol (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=mmseries" },
      { "MouseMan",	"%%32150 Logitech MouseMan/TrackMan models (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=mouseman" },
      { "MouseSystems",	"%%32151 MouseSystems protocol (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=mousesystems" },
      { "ThinkingMouse","%%32152 Kensington ThinkingMouse (serial)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_TYPE "=thinkingmouse" },
      { NULL } },
};

DMenu MenuMousePort = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%32160 Select your mouse port from the following menu",
    "%%32161 The built-in pointing device of laptop/notebook computers is usually\n"
    "a PS/2 style device.",
    NULL,
    NULL,
    { { "COM1",	"%%32162 Serial mouse on COM1 (/dev/cuaa0)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_PORT "=/dev/cuaa0" },
      { "COM2",	"%%32163 Serial mouse on COM2 (/dev/cuaa1)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_PORT "=/dev/cuaa1" },
      { "COM3",	"%%32164 Serial mouse on COM3 (/dev/cuaa2)",
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_PORT "=/dev/cuaa2" },
      { "COM4",	"%%32165 Serial mouse on COM4 (/dev/cuaa3)", 
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_PORT "=/dev/cuaa3" },
      { "BusMouse", "%%32166 Logitech, ATI or MS bus mouse (/dev/mse0)", 
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_PORT "=/dev/mse0" },
      { "PS/2",	"%%32167 PS/2 style mouse (/dev/psm0)", 
	dmenuVarCheck, dmenuSetVariable, NULL, VAR_MOUSED_PORT "=/dev/psm0" },
      { NULL } },
};

DMenu MenuMouse = {
    DMENU_NORMAL_TYPE,
    "%%32170 Please configure your mouse",
    "%%32252 You can cut and paste text in the text console by running the mouse\n"
    "daemon.  Specify a port and a protocol type of your mouse and enable\n"
    "the mouse daemon.  If you don't want this feature, select 4 to disable\n"
    "the daemon.\n"
    "Once you've enabled the mouse daemon, you can specify \"/dev/sysmouse\"\n"
    "as your mouse device and \"SysMouse\" or \"MouseSystems\" as mouse\n"
    "protocol when running the X configuration utility (see Configuration\n"
    "menu).\n"
    "Please note that if you're using multibyte installer (Japanese, Korean,\n"
    "and Chinese), mouse cursor won't be displayed on the console at once,\n"
    "but it'll be displayed after you reboot the machine.",
    NULL,
    NULL,
    { { "%%32171 1 Type",	"%%32172 Select mouse protocol type", NULL, dmenuSubmenu, NULL, &MenuMouseType },
      { "%%32173 2 Port",	"%%32174 Select mouse port", NULL, dmenuSubmenu, NULL, &MenuMousePort },
      { "%%32175 3 Enable",	"%%32176 Test and run the mouse daemon", NULL, mousedTest, NULL, NULL },
      { "%%32177 4 Disable",	"%%32178 Disable the mouse daemon", NULL, mousedDisable, NULL, NULL },
      { "%%32179 0 Exit", "%%32180 Exit this menu (returning to previous)", NULL, dmenuExit },
      { NULL } },
};


DMenu MenuXF86Config = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%32010 Please select the XFree86 configuration tool you want to use.",
    "%%32011 The first tool, XF86Setup, is fully graphical and requires the\n"
    "VGA16 server in order to work (should have been selected by\n"
    "default, but if you de-selected it then you won't be able to\n"
    "use this fancy setup tool).  The second tool, xf86config, is\n"
    "a more simplistic shell-script based tool and less friendly to\n"
    "new users, but it may work in situations where the fancier one\n"
    "does not.",
    NULL,
    NULL,
    { { "XF86Setup",	"%%32013 Fully graphical XFree86 configuration tool.",
	NULL, dmenuSetVariable, NULL, VAR_XF86_CONFIG "=XF86Setup" },
      { "xf86config",	"%%32014 Shell-script based XFree86 configuration tool.",
	NULL, dmenuSetVariable, NULL, VAR_XF86_CONFIG "=xf86config" },
      { "XF98Setup",	"%%32190 Fully graphical XFree86 configuration tool (PC98).",
	NULL, dmenuSetVariable, NULL, VAR_XF86_CONFIG "=XF98Setup" },
      { "XDesktop",	"%%32191 X already set up, just do desktop configuration.",
	NULL, dmenuSubmenu, NULL, &MenuXDesktops },
      { NULL } },
};

DMenu MenuXDesktops = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%32280 Please select the default X desktop to use.",
    "%%32281 By default, XFree86 comes with a fairly vanilla desktop which\n"
    "is based around the twm(1) window manager and does not offer\n"
    "much in the way of features.  It does have the advantage of\n"
    "being a standard part of X so you don't need to load anything\n"
    "extra in order to use it.  If, however, you have access to a\n"
    "reasonably full packages collection on your installation media,\n"
    "you can choose any one of the following desktops as alternatives.",
    NULL,
    NULL,
    { { "KDE",		"%%32282 The K Desktop Environment.",
	NULL, dmenuSetVariable, NULL, VAR_DESKSTYLE "=kde" },
      { "GNOME",	"%%32283 The GNOME desktop environment.",
	NULL, dmenuSetVariable, NULL, VAR_DESKSTYLE "=gnome" },
      { "Afterstep",	"%%32284 The Afterstep Window manager",
	NULL, dmenuSetVariable, NULL, VAR_DESKSTYLE "=afterstep" },
      { "Windowmaker",	"%%32285 The Windowmaker Window manager",
	NULL, dmenuSetVariable, NULL, VAR_DESKSTYLE "=windowmaker" },
      { "Enlightenment","%%32286 The E Window manager (24 bit recommended)",
	NULL, dmenuSetVariable, NULL, VAR_DESKSTYLE "=enlightenment" },
      { NULL } },
};

DMenu MenuMediaCDROM = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00211 Choose a CDROM type",
    "%%00212 FreeBSD can be installed directly from a CDROM containing a valid\n"
    "FreeBSD distribution.  If you are seeing this menu it is because\n"
    "more than one CDROM drive was found on your system.  Please select one\n"
    "of the following CDROM drives as your installation drive.",
    "%%00213 Press F1 to read the installation guide",
    "install",
    { { NULL } },
};

DMenu MenuMediaFloppy = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00215 Choose a Floppy drive",
    "%%00216 You have more than one floppy drive.  Please chose which drive\n"
    "you would like to use.",
    NULL,
    NULL,
    { { NULL } },
};

DMenu MenuMediaDOS = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00221 Choose a DOS partition",
    "%%00222 FreeBSD can be installed directly from a DOS partition\n"
    "assuming, of course, that you have copied the relevant\n"
    "distributions into your DOS partition before starting this\n"
    "installation.  If this is not the case then you should reboot\n"
    "DOS at this time and copy the distributions you wish to install\n"
    "into a \"FREEBSD\" subdirectory on one of your DOS partitions.\n"
    "Otherwise, please select the DOS partition containing the FreeBSD\n"
    "distribution files.",
    "%%00223 Press F1 to read the installation guide",
    "install",
    { { NULL } },
};

DMenu MenuMediaFTP = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00225 Please select a FreeBSD FTP distribution site",
    "%%00226 Please select the site closest to you or \"other\" if you'd like to\n"
    "specify a different choice.  Also note that not every site listed here\n"
    "carries more than the base distribution kits. Only the Primary site is\n"
    "guaranteed to carry the full range of possible distributions.",
    "%%00227 Select a site that's close!",
    "install",
    { { "%%00228 Primary Site",	"ftp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AS("=ftp://ftp.freebsd.org/pub/FreeBSD/releases/") },
      { "%%00230 URL", "%%00231 Specify some other ftp site by URL", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH "=other" },
      { "%%32080 4.0 SNAP Server", "current.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AS("=ftp://current.freebsd.org/pub/FreeBSD/snapshots/") },
      { "%%32081 3.0 SNAP Server", "releng3.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AS("=ftp://releng3.freebsd.org/pub/FreeBSD/snapshots/") },
      { "%%00970 Argentina",	"ftp.ar.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.ar.freebsd.org") },
      { "%%00232 Australia",	"ftp.au.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.au.freebsd.org") },
      { "%%00234 Australia #2",	"ftp2.au.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.au.freebsd.org") },
      { "%%00236 Australia #3",	"ftp3.au.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.au.freebsd.org") },
      { "%%00238 Australia #4",	"ftp4.au.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.au.freebsd.org") },
      { "%%32020 Australia #5",	"ftp5.au.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp5.au.freebsd.org") },
      { "%%00240 Brazil",	"ftp.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.br.freebsd.org") },
      { "%%00242 Brazil #2",	"ftp2.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.br.freebsd.org") },
      { "%%00244 Brazil #3",	"ftp3.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.br.freebsd.org") },
      { "%%00246 Brazil #4",	"ftp4.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.br.freebsd.org") },
      { "%%00248 Brazil #5",	"ftp5.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp5.br.freebsd.org") },
      { "%%32021 Brazil #6",	"ftp6.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp6.br.freebsd.org") },
      { "%%32022 Brazil #7",	"ftp7.br.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp7.br.freebsd.org") },
      { "%%00250 Canada",	"ftp.ca.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.ca.freebsd.org") },
      { "%%00252 Czech Republic", "ftp.cz.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.cz.freebsd.org") },
      { "%%32200 Denmark",	"ftp.dk.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.dk.freebsd.org") },
      { "%%32201 Denmark #2",	"ftp2.dk.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.dk.freebsd.org") },
      { "%%00254 Estonia",	"ftp.ee.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.ee.freebsd.org") },
      { "%%00256 Finland",	"ftp.fi.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.fi.freebsd.org") },
      { "%%00258 France",	"ftp.fr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.fr.freebsd.org") },
      { "%%32023 France #2",	"ftp2.fr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.fr.freebsd.org") },
      { "%%00260 Germany",	"ftp.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.de.freebsd.org") },
      { "%%00262 Germany #2",	"ftp2.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.de.freebsd.org") },
      { "%%00264 Germany #3",	"ftp3.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.de.freebsd.org") },
      { "%%00266 Germany #4",	"ftp4.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.de.freebsd.org") },
      { "%%00268 Germany #5",	"ftp5.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp5.de.freebsd.org") },
      { "%%00270 Germany #6",	"ftp6.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp6.de.freebsd.org") },
      { "%%00272 Germany #7",	"ftp7.de.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp7.de.freebsd.org") },
      { "%%00274 Holland",	"ftp.nl.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.nl.freebsd.org") },
      { "%%00276 Hong Kong",	"ftp.hk.super.net", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.hk.super.net") },
      { "%%32064 Iceland",	"ftp.is.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.is.freebsd.org") },
      { "%%00278 Ireland",	"ftp.ie.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.ie.freebsd.org") },
      { "%%00280 Israel",	"ftp.il.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.il.freebsd.org") },
      { "%%00282 Israel #2",	"ftp2.il.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.il.freebsd.org") },
      { "%%00284 Japan",	"ftp.jp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.jp.freebsd.org") },
      { "%%00286 Japan #2",	"ftp2.jp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.jp.freebsd.org") },
      { "%%00288 Japan #3",	"ftp3.jp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.jp.freebsd.org") },
      { "%%00290 Japan #4",	"ftp4.jp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.jp.freebsd.org") },
      { "%%00292 Japan #5",	"ftp5.jp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp5.jp.freebsd.org") },
      { "%%00294 Japan #6",	"ftp6.jp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp6.jp.freebsd.org") },
      { "%%00296 Korea",	"ftp.kr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.kr.freebsd.org") },
      { "%%00297 Korea #2",	"ftp2.kr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.kr.freebsd.org") },
      { "%%00298 Korea #3",	"ftp3.kr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.kr.freebsd.org") },
      { "%%00299 Korea #4",	"ftp4.kr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.kr.freebsd.org") },
      { "%%00300 Korea #5",	"ftp5.kr.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp5.kr.freebsd.org") },
      { "%%00302 Poland",	"ftp.pl.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.pl.freebsd.org") },
      { "%%00304 Portugal",	"ftp.pt.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.pt.freebsd.org") },
      { "%%32024 Portugal #2",	"ftp2.pt.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.pt.freebsd.org") },
      { "%%00306 Russia",	"ftp.ru.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.ru.freebsd.org") },
      { "%%32025 Russia #2",	"ftp2.ru.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.ru.freebsd.org") },
      { "%%32026 Russia #3",	"ftp3.ru.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.ru.freebsd.org") },
      { "%%00307 Russia #4",    "ftp4.ru.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.ru.freebsd.org") },
      { "%%00308 South Africa",	"ftp.za.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.za.freebsd.org") },
      { "%%00310 South Africa #2", "ftp2.za.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.za.freebsd.org") },
      { "%%00312 South Africa #3", "ftp3.za.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.za.freebsd.org") },
      { "%%32027 South Africa #4", "ftp4.za.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.za.freebsd.org") },
      { "%%00313 Spain",	"ftp.es.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.es.freebsd.org") },
      { "%%00315 Spain #2",	"ftp2.es.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.es.freebsd.org") },
      { "%%00314 Sweden",	"ftp.se.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.se.freebsd.org") },
      { "%%32028 Sweden #2",	"ftp2.se.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.se.freebsd.org") },
      { "%%32029 Sweden #3",	"ftp3.se.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.se.freebsd.org") },
      { "%%00316 Taiwan",	"ftp.tw.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.tw.freebsd.org") },
      { "%%00318 Taiwan #2",	"ftp2.tw.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.tw.freebsd.org") },
      { "%%00320 Taiwan #3",	"ftp3.tw.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.tw.freebsd.org") },
      { "%%00322 Thailand",	"ftp.nectec.or.th", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AS("=ftp://ftp.nectec.or.th/pub/mirrors/FreeBSD/") },
      { "%%00324 UK",		"ftp.uk.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.uk.freebsd.org") },
      { "%%00326 UK #2",	"ftp2.uk.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.uk.freebsd.org") },
      { "%%00328 UK #3",	"ftp3.uk.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.uk.freebsd.org") },
      { "%%32083 UK #4",	"ftp4.uk.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.uk.freebsd.org") },
      { "%%00330 USA",		"ftp.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp.freebsd.org") },
      { "%%00332 USA #2",	"ftp2.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp2.freebsd.org") },
      { "%%00334 USA #3",	"ftp3.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp3.freebsd.org") },
      { "%%00336 USA #4",	"ftp4.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp4.freebsd.org") },
      { "%%00338 USA #5",	"ftp5.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp5.freebsd.org") },
      { "%%00340 USA #6",	"ftp6.freebsd.org", NULL, dmenuSetVariable, NULL,
	VAR_FTP_PATH _AP("=ftp://ftp6.freebsd.org") },
      { NULL } }
};

DMenu MenuMediaTape = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00345 Choose a tape drive type",
    "%%00346 FreeBSD can be installed from tape drive, though this installation\n"
    "method requires a certain amount of temporary storage in addition\n"
    "to the space required by the distribution itself (tape drives make\n"
    "poor random-access devices, so we extract _everything_ on the tape\n"
    "in one pass).  If you have sufficient space for this, then you should\n"
    "select one of the following tape devices detected on your system.",
    "%%00347 Press F1 to read the installation guide",
    "install",
    { { NULL } },
};

DMenu MenuNetworkDevice = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00351 Network interface information required",
    "%%00352 If you are using PPP over a serial device, as opposed to a direct\n"
    "ethernet connection, then you may first need to dial your Internet\n"
    "Service Provider using the ppp utility we provide for that purpose.\n"
    "If you're using SLIP over a serial device then the expectation is\n"
    "that you have a HARDWIRED connection.\n\n"
    "You can also install over a parallel port using a special \"laplink\"\n"
    "cable to another machine running a fairly recent (2.0R or later)\n"
    "version of FreeBSD.",
    "%%00353 Press F1 to read network configuration manual",
    "network_device",
    { { NULL } },
};

/* The media selection menu */
DMenu MenuMedia = {
    DMENU_NORMAL_TYPE | DMENU_SELECTION_RETURNS,
    "%%00355 Choose Installation Media",
    "%%00356 FreeBSD can be installed from a variety of different installation\n"
    "media, ranging from floppies to an Internet FTP server.  If you're\n"
    "installing FreeBSD from a supported CDROM drive then this is generally\n"
    "the best media to use if you have no overriding reason for using other\n"
    "media.",
    "%%00357 Press F1 for more information on the various media types",
    "media",
    { { "%%00358 1 CDROM",		"%%00359 Install from a FreeBSD CDROM",		NULL, mediaSetCDROM },
      { "%%00360 2 FTP",		"%%00361 Install from an FTP server",		NULL, mediaSetFTPActive },
      { "%%00362 3 FTP Passive",	"%%00363 Install from an FTP server through a firewall", NULL, mediaSetFTPPassive },
      { "%%00364 4 DOS",		"%%00365 Install from a DOS partition",		NULL, mediaSetDOS },
      { "%%00366 5 NFS",		"%%00367 Install over NFS",			NULL, mediaSetNFS },
      { "%%00368 6 File System",	"%%00369 Install from an existing filesystem",	NULL, mediaSetUFS },
      { "%%00370 7 Floppy",		"%%00371 Install from a floppy disk set",	NULL, mediaSetFloppy },
      { "%%00372 8 Tape",		"%%00373 Install from SCSI or QIC tape",	NULL, mediaSetTape },
      { "%%32084 9 Options",		"%%32085 Go to the Options screen",		NULL, optionsEditor },
      { NULL } },
};

/* The distributions menu */
DMenu MenuDistributions = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00375 Choose Distributions",
    "%%00376 As a convenience, we provide several \"canned\" distribution sets.\n"
    "These select what we consider to be the most reasonable defaults for the\n"
    "type of system in question.  If you would prefer to pick and choose the\n"
    "list of distributions yourself, simply select \"Custom\".  You can also\n"
    "pick a canned distribution set and then fine-tune it with the Custom item.\n\n"
    "Choose an item by pressing [SPACE]. When you are finished, chose the Exit\n"
    "item or press [ENTER].",
    "%%00377 Press F1 for more information on these options.",
    "distributions",
    { { "%%00378 1 Developer",		"%%00379 Full sources, binaries and doc but no games", 
	checkDistDeveloper,	distSetDeveloper },
      { "%%00380 2 X-Developer",	"%%00381 Same as above + X Window System",
	checkDistXDeveloper,	distSetXDeveloper },
      { "%%00382 3 Kern-Developer",	"%%00383 Full binaries and doc, kernel sources only",
	checkDistKernDeveloper, distSetKernDeveloper },
      { "%%00387 4 X-Kern-Developer",	"%%00381 Same as above + X Window System",
	checkDistXKernDeveloper, distSetXKernDeveloper },
      { "%%00384 5 User",		"%%00385 Average user - binaries and doc only",
	checkDistUser,		distSetUser },
      { "%%00386 6 X-User",		"%%00381 Same as above + X Window System",
	checkDistXUser,		distSetXUser },
      { "%%00388 7 Minimal",		"%%00389 The smallest configuration possible",
	checkDistMinimum,	distSetMinimum },
      { "%%00392 8 Custom",		"%%00393 Specify your own distribution set",
	NULL,			dmenuSubmenu, NULL, &MenuSubDistributions, '>', '>', '>' },
      { "%%00390 8 All",		"%%00391 All sources and binaries (incl X Window System)",
	checkDistEverything,	distSetEverything },
      { "%%00394 9 Clear",		"%%00395 Reset selected distribution list to nothing",
	NULL,			distReset, NULL, NULL, ' ', ' ', ' ' },
      { "%%00396 0 Exit",		"%%00397 Exit this menu (returning to previous)",
	checkTrue,		dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuSubDistributions = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00401 Select the distributions you wish to install.",
    "%%00402 Please check off the distributions you wish to install.  At the\n"
    "very minimum, this should be \"bin\".  WARNING:  Do not export the\n"
    "DES distribution out of the U.S.!  It is for U.S. customers only.",
    NULL,
    NULL,
    { { "bin",		"%%00404 Binary base distribution (required)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_BIN },
      { "compat1x",	"%%00408 FreeBSD 1.x binary compatibility",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_COMPAT1X },
      { "compat20",	"%%00410 FreeBSD 2.0 binary compatibility",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_COMPAT20 },
      { "compat21",	"%%00412 FreeBSD 2.1 binary compatibility",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_COMPAT21 },
      { "compat22",	"%%00413 FreeBSD 2.2.x and 3.0 a.out binary compatibility",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_COMPAT22 },
#if __FreeBSD__ > 3
      { "compat3x",	"%%00406 FreeBSD 3.x binary compatibility",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_COMPAT3X },
#endif
      { "DES",		"%%00414 DES encryption code - NOT FOR EXPORT!",
	DESFlagCheck,	distSetDES },
      { "dict",		"%%00416 Spelling checker dictionary files",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_DICT },
      { "doc",		"%%00418 Miscellaneous FreeBSD online docs",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_DOC },
      { "games",	"%%00420 Games (non-commercial)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_GAMES },
      { "info",		"%%00422 GNU info files",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_INFO },
      { "man",		"%%00424 System manual pages - recommended",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_MANPAGES },
      { "catman",	"%%32030 Preformatted system manual pages",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_CATPAGES },
      { "proflibs",	"%%00426 Profiled versions of the libraries",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_PROFLIBS },
      { "src",		"%%00428 Sources for everything but DES",
	srcFlagCheck,	distSetSrc },
      { "ports",	"The FreeBSD Ports collection",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_PORTS },
      { "XFree86",	"%%00430 The XFree86 3.3.2.3 distribution",
	x11FlagCheck,	distSetXF86 },
#ifdef PCCARD
      { "paobin",	"%%00431 PAO binariy distribution",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_PAOBIN },
      { "paosrc",	"%%00432 PAO source distribution",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &Dists, '[', 'X', ']', DIST_PAOSRC },
#endif	/* PCCARD */
      { "%%00433 All",		"%%00434 All sources, binaries and X Window System binaries",
	NULL, distSetEverything, NULL, NULL, ' ', ' ', ' ' },
      { "%%00435 Clear",	"%%00436 Reset all of the above",
	NULL, distReset, NULL, NULL, ' ', ' ', ' ' },
      { "%%00437 Exit",		"%%00438 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuDESDistributions = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00440 Select the encryption facilities you wish to install.",
    "%%00442 Please check off any special DES-based encryption distributions\n"
    "you would like to install.  Please note that these services are NOT FOR\n"
    "EXPORT from the United States.  For information on non-U.S. FTP\n"
    "distributions of this software, please consult the release notes.",
    NULL,
    NULL,
    { { "des",		"%%00444 Basic DES encryption services",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &DESDists, '[', 'X', ']', DIST_DES_DES, },
      { "krb",		"%%00446 Kerberos encryption services",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &DESDists, '[', 'X', ']', DIST_DES_KERBEROS },
      { "skerbero",	"%%00448 Sources for Kerberos IV",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &DESDists, '[', 'X', ']', DIST_DES_SKERBEROS },
      { "ssecure",	"%%00450 Sources for DES",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &DESDists, '[', 'X', ']', DIST_DES_SSECURE },
      { "scrypto",	"%%32210 Export controlled crypto sources",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &DESDists, '[', 'X', ']', DIST_DES_SCRYPTO },
      { "%%00451 Exit",		"%%00452 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuSrcDistributions = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS ,
    "%%00455 Select the sub-components of src you wish to install.",
    "%%00456 Please check off those portions of the FreeBSD source tree\n"
    "you wish to install (remember to use SPACE, not ENTER!).",
    NULL,
    NULL,
    { { "%%00489 All",		"%%00490 Select all of the below",
	NULL,		setSrc, NULL, NULL, ' ', ' ', ' ' },
      { "%%00491 Clear",	"%%00492 Reset all of the below",
	NULL,		clearSrc, NULL, NULL, ' ', ' ', ' ' },
      { "base",		"%%00458 top-level files in /usr/src",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_BASE },
      { "contrib",	"%%00880 /usr/src/contrib (contributed software)",
	dmenuFlagCheck,	dmenuSetFlag,	NULL, &SrcDists, '[', 'X', ']', DIST_SRC_CONTRIB },
      { "gnu",		"%%00460 /usr/src/gnu (software from the GNU Project)",
	dmenuFlagCheck,	dmenuSetFlag,	NULL, &SrcDists, '[', 'X', ']', DIST_SRC_GNU },
      { "etc",		"%%00462 /usr/src/etc (miscellaneous system files)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_ETC },
      { "games",	"%%00464 /usr/src/games (the obvious!)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_GAMES },
      { "include",	"%%00466 /usr/src/include (header files)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_INCLUDE },
      { "lib",		"%%00468 /usr/src/lib (system libraries)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_LIB },
      { "libexec",	"%%00470 /usr/src/libexec (system programs)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_LIBEXEC },
      { "release",	"%%00474 /usr/src/release (release-generation tools)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_RELEASE },
      { "bin",		"%%00476 /usr/src/bin (system binaries)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_BIN },
      { "sbin",		"%%00478 /usr/src/sbin (system binaries)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_SBIN },
      { "share",	"%%00480 /usr/src/share (documents and shared files)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_SHARE },
      { "sys",		"%%00482 /usr/src/sys (FreeBSD kernel)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_SYS },
      { "ubin",		"%%00484 /usr/src/usr.bin (user binaries)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_UBIN },
      { "usbin",	"%%00486 /usr/src/usr.sbin (aux system binaries)",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &SrcDists, '[', 'X', ']', DIST_SRC_USBIN },
      { "%%00493 Exit",		"%%00494 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuXF86Select = {
    DMENU_NORMAL_TYPE,
    "%%00495 XFree86 3.3.3.1 Distribution",
    "%%00496 Please select the components you need from the XFree86 3.3.3.1\n"
    "distribution sets.",
    NULL,
    NULL,
    { { "%%00498 Basic",	"%%00499 Basic component menu (required)",	NULL, dmenuSubmenu, NULL, &MenuXF86SelectCore },
      { "%%00500 Server",	"%%00501 X server menu",			NULL, dmenuSubmenu, NULL, &MenuXF86SelectServer },
      { "%%00502 Fonts",	"%%00503 Font set menu",			NULL, dmenuSubmenu, NULL, &MenuXF86SelectFonts },
      { "%%00504 All",		"%%00505 Select all XFree86 distribution sets", NULL, setX11All },
      { "%%00506 Clear",	"%%00507 Reset XFree86 distribution list",	NULL, clearX11All },
      { "%%00508 Exit",		"%%00509 Exit this menu (returning to previous)", checkTrue, dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuXF86SelectCore = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00511 XFree86 3.3.3.1 base distribution types",
    "%%00512 Please check off the basic XFree86 components you wish to install.\n"
    "Bin, lib, and set are recommended for a minimum installaion.",
    NULL,
    NULL,
    { { "bin",		"%%00515 Client applications and shared libs",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_BIN },
      { "cfg",		"%%00885 Configuration files",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_CFG },
      { "doc",		"%%00525 READMEs and release notes",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_DOC },
      { "html",		"%%00886 HTML documentation files",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_HTML },
      { "lib",		"%%00517 Data files needed at runtime",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_LIB },
      { "lk98",		"%%00887 Server link kit for PC98 machines",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_LKIT98 },
      { "lkit",		"%%00888 Server link kit for all other machines",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_LKIT },
      { "man",		"%%00527 Manual pages",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_MAN },
      { "prog",		"%%00531 Programmer's header and library files",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_PROG },
      { "set",		"%%00890 XFree86 Setup Utility",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_SET },
      { "9set",		"%%32220 XFree86 Setup Utility for PC98 machines",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_9SET },
      { "sources",	"%%00891 XXFree86 3.3.3.1 standard sources",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_SRC },
      { "csources",	"%%00892 XFree86 3.3.3.1 contrib sources",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86Dists, '[', 'X', ']', DIST_XF86_CSRC },
      { "%%00540 All",		"%%00541 Select all of the above",
	NULL,		setX11Misc, NULL, NULL, ' ', ' ', ' ' },
      { "%%00542 Clear",	"%%00543 Reset all of the above",
	NULL,		clearX11Misc, NULL, NULL, ' ', ' ', ' ' },
      { "%%00544 Exit",		"%%00545 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuXF86SelectFonts = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS ,
    "%%00551 Font distribution selection.",
    "%%00552 Please check off the individual font distributions you wish to\n\
install.  At the minimum, you should install the standard\n\
75 DPI and misc fonts if you're also installing a server\n\
(these are selected by default).",
    NULL,
    NULL,
    { { "fnts",		"%%00555 Standard 75 DPI and miscellaneous fonts",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86FontDists, '[', 'X', ']', DIST_XF86_FONTS_MISC },
      { "f100",		"%%00557 100 DPI fonts",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86FontDists, '[', 'X', ']', DIST_XF86_FONTS_100 },
      { "fcyr",		"%%00559 Cyrillic Fonts",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86FontDists, '[', 'X', ']', DIST_XF86_FONTS_CYR },
      { "fscl",		"%%00561 Speedo and Type scalable fonts",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86FontDists, '[', 'X', ']', DIST_XF86_FONTS_SCALE },
      { "non",		"%%00563 Japanese, Chinese and other non-english fonts",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86FontDists, '[', 'X', ']', DIST_XF86_FONTS_NON },
      { "server",	"%%00565 Font server",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86FontDists, '[', 'X', ']', DIST_XF86_FONTS_SERVER },
      { "%%00566 All",		"%%00567 All fonts",
	NULL,		setX11Fonts, NULL, NULL, ' ', ' ', ' ' },
      { "%%00568 Clear",	"%%00569 Reset font selections",
	NULL,		clearX11Fonts, NULL, NULL, ' ', ' ', ' ' },
      { "%%00570 Exit",		"%%00571 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuXF86SelectServer = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00575 X Server selection.",
    "%%00576 Please check off the types of X servers you wish to install.\n"
    "If you are unsure as to which server will work for your graphics card,\n"
    "it is recommended that try the SVGA or VGA16 servers or, for PC98\n"
    "machines, the 9EGC or 9840 servers.",
    NULL,
    NULL,
    { { "SVGA",		"%%00579 Standard VGA or Super VGA card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_SVGA },
      { "VGA16",	"%%00581 Standard 16 color VGA card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_VGA16 },
      { "Mono",		"%%00583 Standard Monochrome card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_MONO },
      { "3DL",		"%%00584 8, 16 and 24 bit color 3D Labs boards",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_3DL },
      { "8514",		"%%00585 8-bit (256 color) IBM 8514 or compatible card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_8514 },
      { "AGX",		"%%00587 8-bit AGX card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_AGX },
      { "I128",		"%%00894 8, 16 and 24-bit #9 Imagine I128 card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_I128 },
      { "Ma8",		"%%00589 8-bit ATI Mach8 card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_MACH8 },
      { "Ma32",		"%%00591 8 and 16-bit (65K color) ATI Mach32 card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_MACH32 },
      { "Ma64",		"%%00593 8 and 16-bit (65K color) ATI Mach64 card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_MACH64 },
      { "P9K",		"%%00595 8, 16, and 24-bit color Weitek P9000 based boards",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_P9000 },
      { "S3",		"%%00597 8, 16 and 24-bit color S3 based boards",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_S3 },
      { "S3V",		"%%00895 8, 16 and 24-bit color S3 Virge based boards",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_S3V },
      { "W32",		"%%00599 8-bit ET4000/W32, /W32i and /W32p cards",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_W32 },
      { "PC98",		"%%00893 Select an X server for a NEC PC98 [Submenu]",
	NULL,		dmenuSubmenu,  NULL, &MenuXF86SelectPC98Server, '>', ' ', '>', 0 },
      { "%%00602 All",		"%%00603 Select all of the above",
	NULL,		setX11Servers, NULL, NULL, ' ', ' ', ' ' },
      { "%%00604 Clear",	"%%00605 Reset all of the above",
	NULL,		clearX11Servers, NULL, NULL, ' ', ' ', ' ' },
      { "%%00606 Exit",		"%%00607 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuXF86SelectPC98Server = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00900 PC98 X Server selection.",
    "%%00901 Please check off the types of NEC PC98 X servers you wish to install.\n\
If you are unsure as to which server will work for your graphics card,\n\
it is recommended that try the SVGA or VGA16 servers (the VGA16 and\n\
Mono servers are particularly well-suited to most LCD displays).",
    NULL,
    NULL,
    { { "9480",		"%%00903 PC98 8-bit (256 color) PEGC-480 card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9480 },
      { "9EGC",		"%%00904 PC98 4-bit (16 color) EGC card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9EGC },
      { "9GA9",		"%%00905 PC98 GA-968V4/PCI (S3 968) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9GA9 },
      { "9GAN",		"%%00906 PC98 GANB-WAP (cirrus) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9GAN },
      { "9LPW",		"%%00907 PC98 PowerWindowLB (S3) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9LPW },
      { "9MGA",		"%%32230 PC98 MGA (Matrox) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9MGA },
      { "9NKV",		"%%00908 PC98 NKV-NEC (cirrus) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9NKV },
      { "9NS3",		"%%00909 PC98 NEC (S3) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9NS3 },
      { "9SPW",		"%%00910 PC98 SKB-PowerWindow (S3) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9SPW },
      { "9SVG",		"%%32231 PC98 generic SVGA card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9SVG },
      { "9TGU",		"%%00911 PC98 Cyber9320 and TGUI9680 cards",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9TGU },
      { "9WEP",		"%%00912 PC98 WAB-EP (cirrus) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9WEP },
      { "9WS",		"%%00913 PC98 WABS (cirrus) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9WS },
      { "9WSN",		"%%00914 PC98 WSN-A2F (cirrus) card",
	dmenuFlagCheck,	dmenuSetFlag, NULL, &XF86ServerDists, '[', 'X', ']', DIST_XF86_SERVER_9WSN },
      { "%%00915 Exit",		"%%00916 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } }
};

DMenu MenuDiskDevices = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00611 Select Drive(s)",
    "%%00612 Please select the drive, or drives, on which you wish to perform\n"
    "this operation.  If you are attempting to install a boot partition\n"
    "on a drive other than the first one or have multiple operating\n"
    "systems on your machine, you will have the option to install a boot\n"
    "manager later.  To select a drive, use the arrow keys to move to it\n"
    "and press [SPACE].  To de-select it, press [SPACE] again.\n\n"
    "Select OK or Cancel to leave this menu.",
    "%%00613 Press F1 for important information regarding disk geometry!",
    "drives",
    { { NULL } },
};

DMenu MenuHTMLDoc = {
    DMENU_NORMAL_TYPE,
    "%%00615 Select HTML Documentation pointer",
    "%%00616 Please select the body of documentation you're interested in, the main\n"
    "ones right now being the FAQ and the Handbook.  You can also chose \"other\"\n"
    "to enter an arbitrary URL for browsing.",
    "%%00617 Press F1 for more help on what you see here.",
    "html",
    { { "Handbook",	"%%00619 The FreeBSD Handbook.",				NULL, docShowDocument },
      { "FAQ",		"%%00621 The Frequently Asked Questions guide.",		NULL, docShowDocument },
      { "Home",		"%%00623 The Home Pages for the FreeBSD Project (requires net)", NULL, docShowDocument },
      { "Other",	"%%00625 Enter a URL.",						NULL, docShowDocument },
      { NULL } },
};

/* The main installation menu */
DMenu MenuInstallCustom = {
    DMENU_NORMAL_TYPE,
    "%%00631 Choose Custom Installation Options",
    "%%00632 This is the custom installation menu. You may use this menu to specify\n"
    "details on the type of distribution you wish to have, where you wish\n"
    "to install it from and how you wish to allocate disk storage to FreeBSD.",
    "%%00633 Press F1 to read the installation guide",
    "install",
    { { "%%00634 1 Options",		"%%00635 View/Set various installation options", NULL, optionsEditor },
#ifdef __alpha__
      { "2 Label",		"Label disk partitions",		NULL, diskLabelEditor },
      { "3 Distributions",	"Select distribution(s) to extract",	NULL, dmenuSubmenu, NULL, &MenuDistributions },
      { "4 Media",		"Choose the installation media type",	NULL, dmenuSubmenu, NULL, &MenuMedia },
      { "5 Commit",		"Perform any pending Partition/Label/Extract actions", NULL, installCustomCommit },
#else
      { "%%00636 2 Partition",		"%%00637 Allocate disk space for FreeBSD",	NULL, diskPartitionEditor },
      { "%%00638 3 Label",		"%%00639 Label allocated disk partitions",	NULL, diskLabelEditor },
      { "%%00640 4 Distributions",	"%%00641 Select distribution(s) to extract",	NULL, dmenuSubmenu, NULL, &MenuDistributions },
      { "%%00642 5 Media",		"%%00643 Choose the installation media type",	NULL, dmenuSubmenu, NULL, &MenuMedia },
      { "%%00644 6 Commit",		"%%00645 Perform any pending Partition/Label/Extract actions", NULL, installCustomCommit },
#endif
      { "%%00648 0 Exit",		"%%00649 Exit this menu (returning to previous)", NULL,	dmenuExit },
      { NULL } },
};

/* MBR type menu */
DMenu MenuMBRType = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "overwrite me",		/* will be disk specific label */
    "%%00652 FreeBSD comes with a boot selector that allows you to easily\n"
    "select between FreeBSD and any other operating systems on your machine\n"
    "at boot time.  If you have more than one drive and want to boot\n"
    "from the second one, the boot selector will also make it possible\n"
    "to do so (limitations in the PC BIOS usually prevent this otherwise).\n"
    "If you do not want a boot selector, or wish to replace an existing\n"
    "one, select \"standard\".  If you would prefer your Master Boot\n"
    "Record to remain untouched then select \"None\".\n\n"
    "  NOTE:  PC-DOS users will almost certainly require \"None\"!",
    "%%00653 Press F1 to read about drive setup",
    "drives",
    { { "%%00654 BootMgr",	"%%00655 Install the FreeBSD Boot Manager",
	dmenuRadioCheck, dmenuSetValue, NULL, &BootMgr },
      { "%%00656 Standard",	"%%00657 Install a standard MBR (no boot manager)",
	dmenuRadioCheck, dmenuSetValue, NULL, &BootMgr, '(', '*', ')', 1 },
      { "%%00658 None",		"%%00659 Leave the Master Boot Record untouched",
	dmenuRadioCheck, dmenuSetValue, NULL, &BootMgr, '(', '*', ')', 2 },
      { NULL } },
};

#ifdef I18N
/* snatched from passwd (1) */

static unsigned char itoa64[] =         /* 0 ... 63 => ascii - 64 */
        "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
#include <ctype.h>

static void
to64(char *s, long v, int n)
{
    while (--n >= 0) {
	*s++ = itoa64[v&0x3f];
	v >>= 6;
    }
}


int
passwdRoot(dialogMenuItem *self)
{
    struct timeval tv;
    int i;
	
    if (language == LANG_ENGLISH) {
	if (systemExecute("passwd root")) {
	    variable_set2("root_password", "YES", 0);
	}
    }
    else {
	char *p1, *p2;
	char pw[256], cmd[256], salt[16];
	
	while (1) {
	    variable_set2(ROOT_PASS1, "", 0);
	    variable_set2(ROOT_PASS2, "", 0);
	    DialogInputAttrs |= DITEM_NO_ECHO;
	    dialog_clear_norefresh();
	    p1 = variable_get_value(ROOT_PASS1, 
				    "%%32065 Please type the password for\n"
				    "\"root\" user (system manager):", 0);
	    if (!p1) {
		    DialogInputAttrs &= ~DITEM_NO_ECHO;
		    return 1;
	    }
	    dialog_clear_norefresh();
	    p2 = variable_get_value(ROOT_PASS2, 
				    "%%32066 Please retype the password for\n"
				    "\"root\" user (system manager) again:", 0);
	    DialogInputAttrs &= ~DITEM_NO_ECHO;
	    if (!p2)
		    goto tryagain;
	    p1 = variable_get(ROOT_PASS1);
	    p2 = variable_get(ROOT_PASS2);
	    if (p1 && p2 && (strcmp(p1, p2) == 0)) {
		int len = strlen(p1);
		int alllower = 1, alldigit = 1;
		
		if (len < 5) {
		    msgConfirm("%%32069 Please enter a longer password.");
		    goto tryagain;
		}
		for (i = 0; i < len; i++) {
		    if (!islower(p1[i])) {
			alllower = 0;
		    }
		    if (!isdigit(p1[i])) {
			alldigit = 0;
		    }
		}
		if (alllower || alldigit) {
		    msgConfirm("%%32070 Please don't use an all-lower case or all-digit password.\n"
			       "Unusual capitalization, control characters or digits are suggested.");
		    goto tryagain;
		}
		if (len < 5) {
		    msgConfirm("%%32069 Please enter a longer password.");
		    goto tryagain;
		}
		break;
	    }
	    dialog_clear_norefresh();
	    msgConfirm("%%32067 Mismatch. Try Again.");
tryagain:
	}

	gettimeofday(&tv,0);
	(void)srandom((int)time((time_t *)NULL));

	/* XXX - imply MD5 */
	strncpy(&salt[0], "$1$", 3);
	to64(&salt[3], random(), 3);
	to64(&salt[6], tv.tv_usec, 3);
	salt[8] = '\0';

	strcpy(pw, p1);
	sprintf(cmd, "chpass -p '%s' root", crypt(pw, salt));
	systemExecute(cmd);
	msgNotify("%%32068 Root password changed.");
	variable_set2(ROOT_PASS1, "", 0);
	variable_set2(ROOT_PASS2, "", 0);
	variable_set2("root_password", "YES", 0);
	sleep(2);
    }

    return 0;
}
#endif /* I18N */

/* Final configuration menu */
DMenu MenuConfigure = {
    DMENU_NORMAL_TYPE,
    "%%00661 FreeBSD Configuration Menu",	/* title */
    "%%00662 If you've already installed FreeBSD, you may use this menu to customize\n"
    "it somewhat to suit your particular configuration.  Most importantly,\n"
    "you can use the Packages utility to load extra \"3rd party\"\n"
    "software not provided in the base distributions.",
    "%%00663 Press F1 for more information on these options",
    "configure",
    { { "%%32035 D Distributions", "%%32036 Install additional distribution sets",
	NULL, distExtractAll },
      { "%%00678 P Packages",	"%%00679 Install pre-packaged software for FreeBSD",
	NULL, configPackages },
      { "%%00682 R Root Password", "%%00683 Set the system manager's password",
#ifdef I18N
	NULL, passwdRoot },
#else
        NULL, dmenuSystemCommand, NULL, "passwd root" },
#endif
      { "%%32037 L Label",	"%%32038 The disk Label editor",
	NULL, diskLabelEditor },
      { "%%32039 F Fdisk",	"%%32040 The disk Slice (PC-style partition) Editor",
	NULL, diskPartitionEditor },
      { "%%00664 1 User Management",	"%%00665 Add user and group information",
	NULL, dmenuSubmenu, NULL, &MenuUsermgmt },
      { "%%00666 2 Console",	"%%00667 Customize system console behavior",
	NULL,	dmenuSubmenu, NULL, &MenuSyscons },
      { "%%00668 3 Time Zone",	"%%00669 Set which time zone you're in",
#ifdef I18N
	NULL,	tzsetup },
#else
	NULL,	dmenuSystemCommand, NULL, "tzsetup" },
#endif
      { "%%00670 4 Media",	"%%00671 Change the installation media type",
	NULL,	dmenuSubmenu, NULL, &MenuMedia	},
      { "%%00672 5 Mouse",	"%%00673 Configure your mouse",
	NULL,	dmenuSubmenu, NULL, &MenuMouse, NULL },
      { "%%00674 6 Networking",	"%%00675 Configure additional network services",
	NULL,	dmenuSubmenu, NULL, &MenuNetworking },
      { "%%32086 7 Startup",	"%%32087 Configure system startup options",
	NULL,	dmenuSubmenu, NULL, &MenuStartup },
      { "%%00676 8 Options",	"%%00677 View/Set various installation options",
	NULL, optionsEditor },
      { "%%00686 X XFree86",	"%%00687 Configure XFree86 Server",
	NULL, configXSetup },
      { "%%32041 D Desktop",	"%%32042 Configure XFree86 Desktop",
	NULL, configXDesktop },
      { "%%00684 H HTML Docs",	"%%00685 Go to the HTML documentation menu (post-install)",
	NULL, docBrowser },
      { "%%00688 E Exit",		"%%00689 Exit this menu (returning to previous)",
	NULL,	dmenuExit },
      { NULL } },
};

DMenu MenuStartup = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%32100 Startup Services Menu",
    "%%32101 This menu allows you to configure various aspects of your system's\n"
    "startup configuration.  Remember to use SPACE to select items!  The\n"
    "RETURN key will leave this menu (as with all checkbox menus).",
    NULL,
    NULL,
    { { "%%32102 APM",		"%%32103 Auto-power management services (typically laptops)",
	dmenuVarCheck,	dmenuToggleVariable, NULL, "apm_enable=YES" },
      { "%%32104 pccard",	"%%32105 Enable PCCARD (AKA PCMCIA) services (also laptops)",
	dmenuVarCheck, dmenuToggleVariable, NULL, "pccard_enable=YES" },
      { "%%32106 pccard mem",	"%%32107 Set PCCARD memory address (if enabled)",
	dmenuVarCheck, dmenuISetVariable, NULL, "pccard_mem" },
      { "%%32108 pccard ifconfig",	"%%32109 List of PCCARD ethernet devices to configure",
	dmenuVarCheck, dmenuISetVariable, NULL, "pccard_ifconfig" },
      { " ",		" -- ", NULL,	NULL, NULL, NULL, ' ', ' ', ' ' },
      { "%%32110 startup dirs",	"%%32111 Set the list of dirs to look for startup scripts",
	dmenuVarCheck, dmenuISetVariable, NULL, "local_startup" },
      { "%%32112 named",	"%%32113 Run a local name server on this host",
	dmenuVarCheck, dmenuToggleVariable, NULL, "named_enable=YES" },
      { "%%32114 named flags",	"%%32115 Set default flags to named (if enabled)",
	dmenuVarCheck, dmenuISetVariable, NULL, "named_flags" },
      { "%%32116 nis client",	"%%32117 This host wishes to be an NIS client.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "nis_client_enable=YES" },
      { "%%32118 nis server",	"%%32119 This host wishes to be an NIS server.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "nis_server_enable=YES" },
      { " ",		" -- ", NULL,	NULL, NULL, NULL, ' ', ' ', ' ' },
      { "%%32120 accounting",	"%%32121 This host wishes to run process accounting.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "accounting_enable=YES" },
      { "%%32122 lpd",		"%%32123 This host has a printer and wants to run lpd.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "lpd_enable=YES" },
      { "%%32124 linux",	"%%32125 This host wants to be able to run linux binaries.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "linux_enable=YES" },
      { "%%32126 quotas",	"%%32127 This host wishes to check quotas on startup.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "check_quotas=YES" },
      { "%%32128 SCO",		"%%32129 This host wants to be able to run IBCS2 binaries.",
	dmenuVarCheck, dmenuToggleVariable, NULL, "ibcs2_enable=YES" },
      { "%%32130 Exit",		"%%32131 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuNetworking = {
    DMENU_CHECKLIST_TYPE | DMENU_SELECTION_RETURNS,
    "%%00695 Network Services Menu",
    "%%00696 You may have already configured one network device (and the other\n"
    "various hostname/gateway/name server parameters) in the process\n"
    "of installing FreeBSD.  This menu allows you to configure other\n"
    "aspects of your system's network configuration.",
    NULL,
    NULL,
    { { "%%00697 Interfaces",	"%%00698 Configure additional network interfaces",
	NULL, tcpMenuSelect },
      { "%%00699 NFS client",	"%%00700 This machine will be an NFS client",
	dmenuVarCheck,	dmenuToggleVariable, NULL, "nfs_client_enable=YES" },
      { "%%00701 NFS server",	"%%00702 This machine will be an NFS server",
	dmenuVarCheck, configNFSServer, NULL, "nfs_server_enable=YES" },
      { "%%32088 AMD",	"%%32089 This machine wants to run the auto-mounter service",
	dmenuVarCheck, dmenuToggleVariable, NULL, "amd_enable=YES" },
      { "%%32090 AMD Flags",	"%%32091 Set flags to AMD service (if enabled)",
	dmenuVarCheck, dmenuISetVariable, NULL, "amd_flags" },
      { "%%32092 TCP Extensions",       "%%32093 Allow RFC1323 and RFC1644 TCP extensions?",
	dmenuVarCheck, dmenuToggleVariable, NULL, "tcp_extensions=YES" },
      { "%%00703 Gateway",	"%%00704 This machine will route packets between interfaces",
	dmenuVarCheck,	dmenuToggleVariable, NULL, "gateway_enable=YES" },
      { "%%00709 Ntpdate",	"%%00710 Select a clock-synchronization server",
	dmenuVarCheck,	dmenuSubmenu, NULL, &MenuNTP, '[', 'X', ']', "ntpdate_enable=YES" },
      { "%%00711 router",	"%%00712 Select routing daemon (default: routed)",
	dmenuVarCheck, configRouter, NULL, "router" },
      { "%%00713 Rwhod",	"%%00714 This machine wants to run the rwho daemon",
	dmenuVarCheck,	dmenuToggleVariable, NULL, "rwhod_enable=YES" },
      { "%%00715 Anon FTP",	"%%00716 This machine wishes to allow anonymous FTP.",
	dmenuVarCheck, configAnonFTP, NULL, "anon_ftp" },
      { "%%00721 PCNFSD",	"%%00722 Run authentication server for clients with PC-NFS.",
	dmenuVarCheck, configPCNFSD, NULL, "pcnfsd" },
      { "%%00723 Exit",		"%%00724 Exit this menu (returning to previous)",
	checkTrue,	dmenuExit, NULL, NULL, '<', '<', '<' },
      { NULL } },
};

DMenu MenuNTP = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%00725 NTPDATE Server Selection",
    "%%00726 There are a number of time synchronization servers available\n"
    "for public use around the Internet.  Please select one reasonably\n"
    "close to you to have your system time synchronized accordingly.",
    "%%00727 These are the primary open-access NTP servers",
    NULL,
    { { "%%32094 None",		        "%%32095 No ntp server",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=NO,ntpdate_flags=none" },
      { "%%00728 Other",		"%%00729 Select a site not on this list",
	dmenuVarsCheck, configNTP, NULL, NULL },
      { "%%00730 Australia",		"%%00731 ntp.syd.dms.csiro.au (HP 5061 Cesium Beam)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=ntp.syd.dms.csiro.au" },
      { "%%00732 Canada",		"%%00733 tick.usask.ca (GOES clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=tick.usask.ca" },
      { "%%00734 France",		"%%00735 canon.inria.fr (TDF clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=canon.inria.fr" },
      { "%%00736 Germany",		"%%00737 ntps1-{0,1,2}.uni-erlangen.de (GPS)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=ntps1-0.uni-erlangen.de" },
      { "%%00738 Germany #2",		"%%00739 ntps1-0.cs.tu-berlin.de (GPS)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=ntps1-0.cs.tu-berlin.de" },
      { "%%00740 Japan",		"%%00741 clock.nc.fukuoka-u.ac.jp (GPS clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=clock.nc.fukuoka-u.ac.jp" },
      { "%%00742 Japan #2",		"%%00743 clock.tl.fukuoka-u.ac.jp (GPS clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=clock.tl.fukuoka-u.ac.jp" },
      { "%%00744 Netherlands",		"%%00745 ntp0.nl.net (GPS clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=ntp0.nl.net" },
      { "%%00746 Norway",		"%%00747 timehost.ifi.uio.no (NTP clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=timehost.ifi.uio.no" },
      { "%%00748 Sweden",		"%%00749 Time1.Stupi.SE (Cesium/GPS)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=Time1.Stupi.SE" },
      { "%%00750 Switzerland",		"%%00751 swisstime.ethz.ch (DCF77 clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=swisstime.ethz.ch" },
      { "%%00752 U.S. East Coast",	"%%00753 bitsy.mit.edu (WWV clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=bitsy.mit.edu" },
      { "%%00754 U.S. East Coast #2",	"%%00755 otc1.psu.edu (WWV clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=otc1.psu.edu" },
      { "%%00756 U.S. West Coast",	"%%00757 apple.com (WWV clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=apple.com" },
      { "%%00758 U.S. West Coast #2",	"%%00759 clepsydra.dec.com (GOES clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=clepsydra.dec.com" },
      { "%%00760 U.S. West Coast #3",	"%%00761 clock.llnl.gov (WWVB clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=clock.llnl.gov" },
      { "%%00762 U.S. Midwest",		"%%00763 ncar.ucar.edu (WWVB clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=ncar.ucar.edu" },
      { "%%00764 U.S. Pacific",		"%%00765 chantry.hawaii.net (WWV/H clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=chantry.hawaii.net" },
      { "%%00766 U.S. Southwest",	"%%00767 shorty.chpc.utexas.edu (WWV clock)",
	dmenuVarsCheck,	dmenuSetVariables, NULL, 
	"ntpdate_enable=YES,ntpdate_flags=shorty.chpc.utexas.edu" },
      { NULL } },
};

DMenu MenuSyscons = {
    DMENU_NORMAL_TYPE,
    "%%00781 System Console Configuration",
    "%%00782 The default system console driver for FreeBSD (syscons) has a\n"
    "number of configuration options which may be set according to\n"
    "your preference.\n\n"
    "When you are done setting configuration options, select Cancel.",
    "%%00783 Configure your system console settings",
    NULL,
    { { "%%00922 Font",		"%%00923 Choose an alternate screen font",	NULL, dmenuSubmenu, NULL, &MenuSysconsFont },
      { "%%00784 Keymap",	"%%00785 Choose an alternate keyboard map",	NULL, dmenuSubmenu, NULL, &MenuSysconsKeymap },
      { "%%00786 Repeat",	"%%00787 Set the rate at which keys repeat",	NULL, dmenuSubmenu, NULL, &MenuSysconsKeyrate },
      { "%%00788 Saver",	"%%00789 Configure the screen saver",		NULL, dmenuSubmenu, NULL, &MenuSysconsSaver },
      { "%%00920 Screenmap",	"%%00921 Choose an alternate screenmap",	NULL, dmenuSubmenu, NULL, &MenuSysconsScrnmap },
      { "%%00790 Exit",		"%%00791 Exit this menu (returning to previous)", NULL, dmenuExit },
      { NULL } },
};

DMenu MenuSysconsKeymap = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%00795 System Console Keymap",
    "%%00796 The default system console driver for FreeBSD (syscons) defaults\n"
    "to a standard \"American\" keyboard map.  Users in other countries\n"
    "(or with different keyboard preferences) may wish to choose one of\n"
    "the other keymaps below.\n"
    "Note that sysinstall itself only uses the part of the keyboard map\n"
    "which is required to generate the ANSI character subset, but your\n"
    "choice of keymap will also be saved for later (fuller) use.",
    "%%00797 Choose a keyboard map",
    NULL,
    { { "Belgian",	"%%00932 Belgian ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=be.iso" },
      { "Brazil CP850",	"%%00924 Brazil CP850 keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=br275.cp850" },
      { "Brazil ISO (accent)",	"%%32240 Brazil ISO keymap (accent keys)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=br275.iso.acc" },
      { "Brazil ISO",	"%%00925 Brazil ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=br275.iso" },
      { "Croatian ISO",	"%%32270 Croatian ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=hr.iso" },
      { "Danish CP865",	"%%00799 Danish Code Page 865 keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=danish.cp865" },
      { "Danish ISO",	"%%00801 Danish ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=danish.iso" },
      { "Finnish CP850","%%32271 Finnish Code Page 850 keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=finnish.cp850" },
      { "Finnish ISO",  "%%32272 Finnish ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=finnish.iso" },
      { "French ISO (accent)", "%%00802 French ISO keymap (accent keys)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=fr.iso.acc" },
      { "French ISO",	"%%00803 French ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=fr.iso" },
      { "German CP850",	"%%00805 German Code Page 850 keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=german.cp850"	},
      { "German ISO",	"%%00807 German ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=german.iso" },
      { "Hungarian 101", "%%32273 Hungarian ISO keymap (101 key)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=hu.iso2.101keys" },
      { "Hungarian 102", "%%32274 Hungarian ISO keymap (102 key)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=hu.iso2.102keys" },
      { "Icelandic (accent)", "%%32241 Icelandic ISO keymap (accent keys)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=icelandic.iso.acc" },
      { "Icelandic",	"%%32242 Icelandic ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=icelandic.iso" },
      { "Italian",	"%%00809 Italian ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=it.iso" },
      { "Latin American", "%%32243 Latin American ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=lat-amer" },
      { "Japanese 106",	"%%00811 Japanese 106 keymap",  dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=jp.106" },
      { "Norway ISO",	"%%00926 Norwegian ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=norwegian.iso" },
      { "Polish ISO",	"%%32275 Polish ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=pl_PL.ISO_8859-2" },
      { "Portuguese (accent)",	"%%32276 Portuguese ISO keymap (accent keys)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=pt.iso.acc" },
      { "Portuguese",	"%%32277 Portuguese ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=pt.iso" },
      { "Russia CP866",	"%%00927 Russian CP866 keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=ru.cp866" },
      { "Russia KOI8-R", "%%00928 Russian KOI8-R keymap", dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=ru.koi8-r" },
      { "Slovenian", "%%32244 Slovenian ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=si.iso.acc" },
      { "Spanish (accent)", "%%32245 Spanish ISO keymap (accent keys)",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=spanish.iso.acc" },
      { "Spanish",	"%%00929 Spanish ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=spanish.iso" },
      { "Swedish CP850", "%%00819 Swedish Code Page 850 keymap", dmenuVarCheck,	dmenuSetKmapVariable, NULL, "keymap=swedish.cp850" },
      { "Swedish ISO",	"%%00821 Swedish ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=swedish.iso" },
      { "Swiss French",	"%%32278 Swiss French ISO keymap", dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=swissfrench.iso" },
      { "Swiss German",	"%%00930 Swiss German ISO keymap", dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=swissgerman.iso" },
      { "U.K. CP850",	"%%00823 United Kingdom Code Page 850 keymap", dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=uk.cp850" },
      { "U.K. ISO",	"%%00825 United Kingdom ISO keymap", dmenuVarCheck,	dmenuSetKmapVariable, NULL, "keymap=uk.iso" },
      { "U.S. Dvorak",	"%%00931 United States Dvorak keymap", dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=us.dvorak" },
      { "U.S. ISO",	"%%00827 United States ISO keymap",	dmenuVarCheck, dmenuSetKmapVariable, NULL, "keymap=us.iso" },
      { NULL } },
};

DMenu MenuSysconsKeyrate = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%00831 System Console Keyboard Repeat Rate",
    "%%00832 This menu allows you to set the speed at which keys repeat\n"
    "when held down.",
    "%%00833 Choose a keyboard repeat rate",
    NULL,
    { { "%%00834 Slow",	"%%00835 Slow keyboard repeat rate",	dmenuVarCheck,	dmenuSetVariable, NULL, "keyrate=slow" },
      { "%%00836 Normal", "%%00837 \"Normal\" keyboard repeat rate",	dmenuVarCheck,	dmenuSetVariable, NULL, "keyrate=normal" },
      { "%%00838 Fast",	"%%00839 Fast keyboard repeat rate",	dmenuVarCheck,	dmenuSetVariable, NULL, "keyrate=fast" },
      { "%%00840 Default", "%%00841 Use default keyboard repeat rate",	dmenuVarCheck,	dmenuSetVariable, NULL, "keyrate=NO" },
      { NULL } },
};

DMenu MenuSysconsSaver = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%00845 System Console Screen Saver",
    "%%00846 By default, the console driver will not attempt to do anything\n"
    "special with your screen when it's idle.  If you expect to leave your\n"
    "monitor switched on and idle for long periods of time then you should\n"
    "probably enable one of these screen savers to prevent phosphor burn-in.",
    "%%00847 Choose a nifty-looking screen saver",
    NULL,
    { { "%%00848 Blank",	"%%00849 Simply blank the screen",
	dmenuVarCheck, configSaver, NULL, "saver=blank" },
      { "%%32250 Daemon",	"%%32251 \"BSD Daemon\" animated screen saver (text)",
	dmenuVarCheck, configSaver, NULL, "saver=daemon" },
      { "%%32254 Fade",		"%%32255 Fade out effect screen saver",
	dmenuVarCheck, configSaver, NULL, "saver=fade" },
      { "%%32262 Fire",		"%%32263 Flames effect screen saver",
	dmenuVarCheck, configSaver, NULL, "saver=fire" },
      { "%%00850 Green",	"%%00851 \"Green\" power saving mode (if supported by monitor)",
	dmenuVarCheck, configSaver, NULL, "saver=green" },
      { "%%32256 Logo",		"%%32257 \"BSD Daemon\" animated screen saver (graphics)",
	dmenuVarCheck, configSaver, NULL, "saver=logo" },
      { "%%32258 Rain",		"%%32259 Rain drops screen saver",
	dmenuVarCheck, configSaver, NULL, "saver=rain" },
      { "%%00852 Snake",	"%%00853 Draw a FreeBSD \"snake\" on your screen",
	dmenuVarCheck, configSaver, NULL, "saver=snake" },
      { "%%00854 Star",	"%%00855 A \"twinkling stars\" effect",
	dmenuVarCheck, configSaver, NULL, "saver=star" },
      { "%%32260 Warp",	"%%32261 A \"stars warping\" effect",
	dmenuVarCheck, configSaver, NULL, "saver=warp" },
      { "%%00856 Timeout",	"%%00857 Set the screen saver timeout interval",
	NULL, configSaverTimeout, NULL, NULL, ' ', ' ', ' ' },
      { NULL } },
};

DMenu MenuSysconsScrnmap = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%00935 System Console Screenmap",
    "%%00936 Unless you load a specific font, most PC hardware defaults to\n"
    "displaying characters in the IBM 437 character set.  However,\n"
    "in the Unix world, this character set is very rarely used.  Most\n"
    "Western European countries, for example, prefer ISO 8859-1.\n"
    "American users won't notice the difference since the bottom half\n"
    "of all these character sets is ANSI anyway.\n"
    "If your hardware is capable of downloading a new display font,\n"
    "you should probably choose that option.  However, for hardware\n"
    "where this is not possible (e.g. monochrome adapters), a screen\n"
    "map will give you the best approximation that your hardware can\n"
    "display at all.",
    "%%00937 Choose a screen map",
    NULL,
    { { "None",			"%%00939 No screenmap, use default font", dmenuVarCheck, dmenuSetVariable, NULL, "scrnmap=NO" },
      { "KOI8-R to IBM866",	"%%00940 Russian KOI8-R to IBM 866 screenmap", dmenuVarCheck, dmenuSetVariable, NULL, "scrnmap=koi8-r2cp866" },
      { "ISO 8859-1 to IBM437",	"%%00941 W-Europe ISO 8859-1 to IBM 437 screenmap", dmenuVarCheck, dmenuSetVariable, NULL, "scrnmap=iso-8859-1_to_cp437" },
      { NULL } },
};

DMenu MenuSysconsFont = {
    DMENU_RADIO_TYPE | DMENU_SELECTION_RETURNS,
    "%%00945 System Console Font",
    "%%00946 Most PC hardware defaults to displaying characters in the\n"
    "IBM 437 character set.  However, in the Unix world, this\n"
    "character set is very rarely used.  Most Western European\n"
    "countries, for example, prefer ISO 8859-1.\n"
    "American users won't notice the difference since the bottom half\n"
    "of all these charactersets is ANSI anyway.  However, they might\n"
    "want to load a font anyway to use the 30- or 50-line displays.\n"
    "If your hardware is capable of downloading a new display font,\n"
    "you can select the appropriate font below.",
    "Choose a font",
    NULL,
    { { "None", "%%00947 Use default font",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=NO,font8x14=NO,font8x16=NO" },
      { "IBM 437", "%%00948 English",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=cp437-8x8,font8x14=cp437-8x14,font8x16=cp437-8x16" },
      { "IBM 850", "%%00949 Western Europe, IBM encoding",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=cp850-8x8,font8x14=cp850-8x14,font8x16=cp850-8x16" },
      { "IBM 865", "%%00950 Norwegian, IBM encoding",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=cp865-8x8,font8x14=cp865-8x14,font8x16=cp865-8x16" },
      { "IBM 866", "%%00951 Russian, IBM encoding",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=cp866-8x8,font8x14=cp866-8x14,font8x16=cp866-8x16" },
      { "ISO 8859-1", "%%00952 Western Europe, ISO encoding",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=iso-8x8,font8x14=iso-8x14,font8x16=iso-8x16" },
      { "KOI8-R", "%%00953 Russian, KOI8-R encoding",	dmenuVarCheck,	dmenuSetVariables, NULL,
	"font8x8=koi8-r-8x8,font8x14=koi8-r-8x14,font8x16=koi8-r-8x16" },
      { NULL } },
};


DMenu MenuUsermgmt = {
    DMENU_NORMAL_TYPE,
    "%%00980 User and group management",
    "%%00981 The submenus here allow to manipulate user groups and\n"
    "login accounts.\n",
    "%%00982 Configure your user groups and users",
    NULL,
    { { "Add user",	"%%00983 Add a new user to the system.",	NULL, userAddUser },
      { "Add group",	"%%00984 Add a new user group to the system.",	NULL, userAddGroup },
      { "%%00985 Exit",		"%%00986 Exit this menu (returning to previous)", NULL, dmenuExit },
      { NULL } },
};


DMenu MenuFixit = {
    DMENU_NORMAL_TYPE,
    "%%32050 Please choose a fixit option",
    "%%32051 There are three ways of going into \"fixit\" mode:\n"
    "- you can use the 2nd FreeBSD CDROM, in which case there will be\n"
    "  full access to the complete set of FreeBSD commands and utilities,\n"
    "- you can use the more limited (but perhaps customized) fixit floppy,\n"
    "- or you can start an Emergency Holographic Shell now, which is\n"
    "  limited to the subset of commands that is already available right now.",
    "%%32052 Press F1 for more detailed repair instructions",
    "fixit",
{ { "1 CDROM",	"%%32053 Use the 2nd \"live\" CDROM from the distribution",	NULL, installFixitCDROM },
  { "2 Floppy",	"%%32054 Use a floppy generated from the fixit image",		NULL, installFixitFloppy },
  { "3 Shell",	"%%32055 Start an Emergency Holographic Shell",			NULL, installFixitHoloShell },
  { NULL } },
};

#ifdef I18N
static void 
convert_i18n(char **str)
{
    char *s;
    if (!*str) {
	return;
    }
    s = i18n(*str);
    if (s != *str) {
	*str = strdup(s);
    }
}

void 
menus_i18n(void)
{
    int i;
    static DMenu *menus[] = {
	&MenuIndex,
	&MenuInitial,
	&MenuDocumentation,
	&MenuMouse,
	&MenuMouseType,
	&MenuMousePort,
	&MenuMediaCDROM,
	&MenuMediaFloppy,
	&MenuXF86Config,
	&MenuXDesktops,
	&MenuMediaDOS,
	&MenuMediaFTP,
	&MenuMediaTape,
	&MenuNetworkDevice,
	&MenuMedia,
	&MenuDistributions,
	&MenuSubDistributions,
	&MenuDESDistributions,
	&MenuSrcDistributions,
	&MenuXF86Select,
	&MenuXF86SelectCore,
	&MenuXF86SelectFonts,
	&MenuXF86SelectServer, 
	&MenuXF86SelectPC98Server,
	&MenuDiskDevices, 
	&MenuHTMLDoc,
	&MenuInstallCustom,
	&MenuMBRType, 
	&MenuConfigure, 
	&MenuNetworking, 
	&MenuNTP, 
	&MenuStartup, 
	&MenuSyscons, 
	&MenuSysconsKeymap, 
	&MenuSysconsKeyrate,
	&MenuSysconsSaver, 
	&MenuSysconsScrnmap, 
	&MenuSysconsFont, 
	&MenuFixit, 
	&MenuUsermgmt,
#ifdef PCCARD
	&MenuPCICMem,
	&MenuCardIRQ,
#endif
	0
    };
    
    for (i = 0; menus[i]; i++) {
	DMenu *menu;
	dialogMenuItem *d;
	
	menu = menus[i];
	convert_i18n(&menu->title);
	convert_i18n(&menu->prompt);
	convert_i18n(&menu->helpline);

	for (d = menu->items; d->prompt; d++) {
	    convert_i18n(&d->prompt);
	    convert_i18n(&d->title);
	}
    }
}
#endif
