#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#if __cplusplus >= 202002L
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BOX
#define CXXBRIDGE1_RUST_BOX
template <typename T>
class Box final {
public:
  using element_type = T;
  using const_pointer =
      typename std::add_pointer<typename std::add_const<T>::type>::type;
  using pointer = typename std::add_pointer<T>::type;

  Box() = delete;
  Box(Box &&) noexcept;
  ~Box() noexcept;

  explicit Box(const T &);
  explicit Box(T &&);

  Box &operator=(Box &&) & noexcept;

  const T *operator->() const noexcept;
  const T &operator*() const noexcept;
  T *operator->() noexcept;
  T &operator*() noexcept;

  template <typename... Fields>
  static Box in_place(Fields &&...);

  void swap(Box &) noexcept;

  static Box from_raw(T *) noexcept;

  T *into_raw() noexcept;

  /* Deprecated */ using value_type = element_type;

private:
  class uninit;
  class allocation;
  Box(uninit) noexcept;
  void drop() noexcept;

  friend void swap(Box &lhs, Box &rhs) noexcept { lhs.swap(rhs); }

  T *ptr;
};

template <typename T>
class Box<T>::uninit {};

template <typename T>
class Box<T>::allocation {
  static T *alloc() noexcept;
  static void dealloc(T *) noexcept;

public:
  allocation() noexcept : ptr(alloc()) {}
  ~allocation() noexcept {
    if (this->ptr) {
      dealloc(this->ptr);
    }
  }
  T *ptr;
};

template <typename T>
Box<T>::Box(Box &&other) noexcept : ptr(other.ptr) {
  other.ptr = nullptr;
}

template <typename T>
Box<T>::Box(const T &val) {
  allocation alloc;
  ::new (alloc.ptr) T(val);
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::Box(T &&val) {
  allocation alloc;
  ::new (alloc.ptr) T(std::move(val));
  this->ptr = alloc.ptr;
  alloc.ptr = nullptr;
}

template <typename T>
Box<T>::~Box() noexcept {
  if (this->ptr) {
    this->drop();
  }
}

template <typename T>
Box<T> &Box<T>::operator=(Box &&other) & noexcept {
  if (this->ptr) {
    this->drop();
  }
  this->ptr = other.ptr;
  other.ptr = nullptr;
  return *this;
}

template <typename T>
const T *Box<T>::operator->() const noexcept {
  return this->ptr;
}

template <typename T>
const T &Box<T>::operator*() const noexcept {
  return *this->ptr;
}

template <typename T>
T *Box<T>::operator->() noexcept {
  return this->ptr;
}

template <typename T>
T &Box<T>::operator*() noexcept {
  return *this->ptr;
}

template <typename T>
template <typename... Fields>
Box<T> Box<T>::in_place(Fields &&...fields) {
  allocation alloc;
  auto ptr = alloc.ptr;
  ::new (ptr) T{std::forward<Fields>(fields)...};
  alloc.ptr = nullptr;
  return from_raw(ptr);
}

template <typename T>
void Box<T>::swap(Box &rhs) noexcept {
  using std::swap;
  swap(this->ptr, rhs.ptr);
}

template <typename T>
Box<T> Box<T>::from_raw(T *raw) noexcept {
  Box box = uninit{};
  box.ptr = raw;
  return box;
}

template <typename T>
T *Box<T>::into_raw() noexcept {
  T *raw = this->ptr;
  this->ptr = nullptr;
  return raw;
}

template <typename T>
Box<T>::Box(uninit) noexcept {}
#endif // CXXBRIDGE1_RUST_BOX

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_ERROR
#define CXXBRIDGE1_RUST_ERROR
class Error final : public std::exception {
public:
  Error(const Error &);
  Error(Error &&) noexcept;
  ~Error() noexcept override;

  Error &operator=(const Error &) &;
  Error &operator=(Error &&) & noexcept;

  const char *what() const noexcept override;

private:
  Error() noexcept = default;
  friend impl<Error>;
  const char *msg;
  std::size_t len;
};
#endif // CXXBRIDGE1_RUST_ERROR

#ifndef CXXBRIDGE1_RUST_OPAQUE
#define CXXBRIDGE1_RUST_OPAQUE
class Opaque {
public:
  Opaque() = delete;
  Opaque(const Opaque &) = delete;
  ~Opaque() = delete;
};
#endif // CXXBRIDGE1_RUST_OPAQUE

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

namespace repr {
struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
template <typename T, typename = void *>
struct operator_new {
  void *operator()(::std::size_t sz) { return ::operator new(sz); }
};

template <typename T>
struct operator_new<T, decltype(T::operator new(sizeof(T)))> {
  void *operator()(::std::size_t sz) { return T::operator new(sz); }
};
} // namespace detail

template <typename T>
union MaybeUninit {
  T value;
  void *operator new(::std::size_t sz) { return detail::operator_new<T>{}(sz); }
  MaybeUninit() {}
  ~MaybeUninit() {}
};

namespace {
template <>
class impl<Error> final {
public:
  static Error error(repr::PtrLen repr) noexcept {
    Error error;
    error.msg = static_cast<char const *>(repr.ptr);
    error.len = repr.len;
    return error;
  }
};
} // namespace
} // namespace cxxbridge1
} // namespace rust

namespace cssparser {
  namespace rust {
    enum class Unit : ::std::uint8_t;
    enum class ColorType : ::std::uint8_t;
    struct Rgba;
    struct CustomColor;
    enum class ColorOperationType : ::std::uint8_t;
    struct SetColorOperationValues;
    struct MixColorOperationValues;
    struct ModifiedColor;
    enum class ValueType : ::std::uint8_t;
    enum class AttributeOperator : ::std::uint8_t;
    enum class SelectorKind : ::std::uint8_t;
    struct Dimension;
    struct ValueConversionError;
    struct StyleSheetError;
    struct ColorOperation;
    struct Color;
    struct Value;
    struct SelectorPart;
    struct Selector;
    struct Property;
    struct StyleRule;
    struct StyleSheet;
  }
}

namespace cssparser {
namespace rust {
#ifndef CXXBRIDGE1_ENUM_cssparser$rust$Unit
#define CXXBRIDGE1_ENUM_cssparser$rust$Unit
enum class Unit : ::std::uint8_t {
  Unknown = 0,
  Unsupported = 1,
  Number = 2,
  Px = 3,
  Em = 4,
  Rem = 5,
  Pt = 6,
  Percent = 7,
  Degrees = 8,
  Radians = 9,
  Seconds = 10,
  Milliseconds = 11,
};
#endif // CXXBRIDGE1_ENUM_cssparser$rust$Unit

#ifndef CXXBRIDGE1_ENUM_cssparser$rust$ColorType
#define CXXBRIDGE1_ENUM_cssparser$rust$ColorType
enum class ColorType : ::std::uint8_t {
  Empty = 0,
  Rgba = 1,
  Custom = 2,
  Modified = 3,
};
#endif // CXXBRIDGE1_ENUM_cssparser$rust$ColorType

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$Rgba
#define CXXBRIDGE1_STRUCT_cssparser$rust$Rgba
struct Rgba final {
  ::std::uint8_t r;
  ::std::uint8_t g;
  ::std::uint8_t b;
  ::std::uint8_t a;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$Rgba

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$CustomColor
#define CXXBRIDGE1_STRUCT_cssparser$rust$CustomColor
struct CustomColor final {
  ::rust::String source;
  ::rust::Vec<::rust::String> arguments;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$CustomColor

#ifndef CXXBRIDGE1_ENUM_cssparser$rust$ColorOperationType
#define CXXBRIDGE1_ENUM_cssparser$rust$ColorOperationType
enum class ColorOperationType : ::std::uint8_t {
  Set = 0,
  Add = 1,
  Subtract = 2,
  Multiply = 3,
  Mix = 4,
};
#endif // CXXBRIDGE1_ENUM_cssparser$rust$ColorOperationType

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$SetColorOperationValues
#define CXXBRIDGE1_STRUCT_cssparser$rust$SetColorOperationValues
struct SetColorOperationValues final {
  ::std::int16_t r;
  ::std::int16_t g;
  ::std::int16_t b;
  ::std::int16_t a;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$SetColorOperationValues

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$MixColorOperationValues
#define CXXBRIDGE1_STRUCT_cssparser$rust$MixColorOperationValues
struct MixColorOperationValues final {
  ::rust::Box<::cssparser::rust::Color> other;
  float amount;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$MixColorOperationValues

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$ModifiedColor
#define CXXBRIDGE1_STRUCT_cssparser$rust$ModifiedColor
struct ModifiedColor final {
  ::rust::Box<::cssparser::rust::Color> color;
  ::rust::Box<::cssparser::rust::ColorOperation> operation;

  ::cssparser::rust::ColorOperationType operation_type() const noexcept;
  ::rust::Box<::cssparser::rust::Color> color_value() const;
  ::cssparser::rust::SetColorOperationValues set_values() const;
  ::cssparser::rust::MixColorOperationValues mix_values() const;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$ModifiedColor

#ifndef CXXBRIDGE1_ENUM_cssparser$rust$ValueType
#define CXXBRIDGE1_ENUM_cssparser$rust$ValueType
enum class ValueType : ::std::uint8_t {
  Empty = 0,
  Dimension = 1,
  String = 2,
  Color = 3,
  Image = 4,
  Url = 5,
  Integer = 6,
};
#endif // CXXBRIDGE1_ENUM_cssparser$rust$ValueType

#ifndef CXXBRIDGE1_ENUM_cssparser$rust$AttributeOperator
#define CXXBRIDGE1_ENUM_cssparser$rust$AttributeOperator
enum class AttributeOperator : ::std::uint8_t {
  None = 0,
  Exists = 1,
  Equals = 2,
  Includes = 3,
  Prefixed = 4,
  Suffixed = 5,
  Substring = 6,
  DashMatch = 7,
};
#endif // CXXBRIDGE1_ENUM_cssparser$rust$AttributeOperator

#ifndef CXXBRIDGE1_ENUM_cssparser$rust$SelectorKind
#define CXXBRIDGE1_ENUM_cssparser$rust$SelectorKind
enum class SelectorKind : ::std::uint8_t {
  Unknown = 0,
  AnyElement = 1,
  Type = 2,
  Class = 3,
  Id = 4,
  PseudoClass = 5,
  Attribute = 6,
  RelativeParent = 7,
  DocumentRoot = 8,
  DescendantCombinator = 9,
  ChildCombinator = 10,
};
#endif // CXXBRIDGE1_ENUM_cssparser$rust$SelectorKind

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$Dimension
#define CXXBRIDGE1_STRUCT_cssparser$rust$Dimension
struct Dimension final {
  float value;
  ::cssparser::rust::Unit unit;

  ::rust::String to_string() const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$Dimension

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$ValueConversionError
#define CXXBRIDGE1_STRUCT_cssparser$rust$ValueConversionError
struct ValueConversionError final {
  ::rust::String message;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$ValueConversionError

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$StyleSheetError
#define CXXBRIDGE1_STRUCT_cssparser$rust$StyleSheetError
struct StyleSheetError final {
  ::rust::String file;
  ::std::uint32_t line;
  ::std::uint32_t column;
  ::rust::String message;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$StyleSheetError

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$ColorOperation
#define CXXBRIDGE1_STRUCT_cssparser$rust$ColorOperation
struct ColorOperation final : public ::rust::Opaque {
  ~ColorOperation() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$ColorOperation

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$Color
#define CXXBRIDGE1_STRUCT_cssparser$rust$Color
struct Color final : public ::rust::Opaque {
  ::cssparser::rust::ColorType color_type() const noexcept;
  ::rust::String to_string() const noexcept;
  ::cssparser::rust::Rgba to_rgba() const;
  ::cssparser::rust::CustomColor to_custom() const;
  ::cssparser::rust::ModifiedColor to_modified() const;
  ~Color() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$Color

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$Value
#define CXXBRIDGE1_STRUCT_cssparser$rust$Value
struct Value final : public ::rust::Opaque {
  ::cssparser::rust::ValueType value_type() const noexcept;
  ::cssparser::rust::Dimension to_dimension() const;
  ::rust::String to_string() const noexcept;
  ::rust::Box<::cssparser::rust::Color> to_color() const;
  ::rust::Str to_image() const;
  ::rust::Str to_url() const;
  ::std::int32_t to_integer() const;
  ~Value() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$Value

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$SelectorPart
#define CXXBRIDGE1_STRUCT_cssparser$rust$SelectorPart
struct SelectorPart final : public ::rust::Opaque {
  ::cssparser::rust::SelectorKind kind() const noexcept;
  ::cssparser::rust::Value const &value() const noexcept;
  ::rust::String attribute_name() const noexcept;
  ::cssparser::rust::AttributeOperator attribute_operator() const noexcept;
  ::cssparser::rust::Value const &attribute_value() const noexcept;
  ~SelectorPart() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$SelectorPart

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$Selector
#define CXXBRIDGE1_STRUCT_cssparser$rust$Selector
struct Selector final : public ::rust::Opaque {
  ::rust::Vec<::cssparser::rust::SelectorPart> parts() const noexcept;
  ~Selector() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$Selector

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$Property
#define CXXBRIDGE1_STRUCT_cssparser$rust$Property
struct Property final : public ::rust::Opaque {
  ::rust::String name() const noexcept;
  ::rust::Vec<::cssparser::rust::Value> values() const noexcept;
  ~Property() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$Property

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$StyleRule
#define CXXBRIDGE1_STRUCT_cssparser$rust$StyleRule
struct StyleRule final : public ::rust::Opaque {
  ::cssparser::rust::Selector const &selector() const noexcept;
  ::rust::Vec<::cssparser::rust::Property> properties() const noexcept;
  ~StyleRule() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$StyleRule

#ifndef CXXBRIDGE1_STRUCT_cssparser$rust$StyleSheet
#define CXXBRIDGE1_STRUCT_cssparser$rust$StyleSheet
struct StyleSheet final : public ::rust::Opaque {
  ::rust::Vec<::cssparser::rust::StyleRule> rules() const noexcept;
  ::rust::Vec<::cssparser::rust::StyleSheetError> errors() const noexcept;
  void set_root_path(::rust::Str root_path) noexcept;
  void parse_file(::rust::Str file_name);
  void parse_string(::rust::Str data, ::rust::Str origin);
  ~StyleSheet() = delete;

private:
  friend ::rust::layout;
  struct layout {
    static ::std::size_t size() noexcept;
    static ::std::size_t align() noexcept;
  };
};
#endif // CXXBRIDGE1_STRUCT_cssparser$rust$StyleSheet

extern "C" {
void cssparser$rust$cxxbridge1$Dimension$to_string(::cssparser::rust::Dimension const &self, ::rust::String *return$) noexcept;

::cssparser::rust::ColorOperationType cssparser$rust$cxxbridge1$ModifiedColor$operation_type(::cssparser::rust::ModifiedColor const &self) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$ModifiedColor$color_value(::cssparser::rust::ModifiedColor const &self, ::rust::Box<::cssparser::rust::Color> *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$ModifiedColor$set_values(::cssparser::rust::ModifiedColor const &self, ::cssparser::rust::SetColorOperationValues *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$ModifiedColor$mix_values(::cssparser::rust::ModifiedColor const &self, ::cssparser::rust::MixColorOperationValues *return$) noexcept;
::std::size_t cssparser$rust$cxxbridge1$ColorOperation$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$ColorOperation$operator$alignof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$Color$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$Color$operator$alignof() noexcept;

::cssparser::rust::ColorType cssparser$rust$cxxbridge1$Color$color_type(::cssparser::rust::Color const &self) noexcept;

void cssparser$rust$cxxbridge1$Color$to_string(::cssparser::rust::Color const &self, ::rust::String *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Color$to_rgba(::cssparser::rust::Color const &self, ::cssparser::rust::Rgba *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Color$to_custom(::cssparser::rust::Color const &self, ::cssparser::rust::CustomColor *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Color$to_modified(::cssparser::rust::Color const &self, ::cssparser::rust::ModifiedColor *return$) noexcept;
::std::size_t cssparser$rust$cxxbridge1$Value$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$Value$operator$alignof() noexcept;

::cssparser::rust::ValueType cssparser$rust$cxxbridge1$Value$value_type(::cssparser::rust::Value const &self) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Value$to_dimension(::cssparser::rust::Value const &self, ::cssparser::rust::Dimension *return$) noexcept;

void cssparser$rust$cxxbridge1$Value$to_string(::cssparser::rust::Value const &self, ::rust::String *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Value$to_color(::cssparser::rust::Value const &self, ::rust::Box<::cssparser::rust::Color> *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Value$to_image(::cssparser::rust::Value const &self, ::rust::Str *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Value$to_url(::cssparser::rust::Value const &self, ::rust::Str *return$) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$Value$to_integer(::cssparser::rust::Value const &self, ::std::int32_t *return$) noexcept;
::std::size_t cssparser$rust$cxxbridge1$SelectorPart$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$SelectorPart$operator$alignof() noexcept;

::cssparser::rust::SelectorKind cssparser$rust$cxxbridge1$SelectorPart$kind(::cssparser::rust::SelectorPart const &self) noexcept;

::cssparser::rust::Value const *cssparser$rust$cxxbridge1$SelectorPart$value(::cssparser::rust::SelectorPart const &self) noexcept;

void cssparser$rust$cxxbridge1$SelectorPart$attribute_name(::cssparser::rust::SelectorPart const &self, ::rust::String *return$) noexcept;

::cssparser::rust::AttributeOperator cssparser$rust$cxxbridge1$SelectorPart$attribute_operator(::cssparser::rust::SelectorPart const &self) noexcept;

::cssparser::rust::Value const *cssparser$rust$cxxbridge1$SelectorPart$attribute_value(::cssparser::rust::SelectorPart const &self) noexcept;
::std::size_t cssparser$rust$cxxbridge1$Selector$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$Selector$operator$alignof() noexcept;

void cssparser$rust$cxxbridge1$Selector$parts(::cssparser::rust::Selector const &self, ::rust::Vec<::cssparser::rust::SelectorPart> *return$) noexcept;
::std::size_t cssparser$rust$cxxbridge1$Property$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$Property$operator$alignof() noexcept;

void cssparser$rust$cxxbridge1$Property$name(::cssparser::rust::Property const &self, ::rust::String *return$) noexcept;

void cssparser$rust$cxxbridge1$Property$values(::cssparser::rust::Property const &self, ::rust::Vec<::cssparser::rust::Value> *return$) noexcept;
::std::size_t cssparser$rust$cxxbridge1$StyleRule$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$StyleRule$operator$alignof() noexcept;

::cssparser::rust::Selector const *cssparser$rust$cxxbridge1$StyleRule$selector(::cssparser::rust::StyleRule const &self) noexcept;

void cssparser$rust$cxxbridge1$StyleRule$properties(::cssparser::rust::StyleRule const &self, ::rust::Vec<::cssparser::rust::Property> *return$) noexcept;
::std::size_t cssparser$rust$cxxbridge1$StyleSheet$operator$sizeof() noexcept;
::std::size_t cssparser$rust$cxxbridge1$StyleSheet$operator$alignof() noexcept;

void cssparser$rust$cxxbridge1$StyleSheet$rules(::cssparser::rust::StyleSheet const &self, ::rust::Vec<::cssparser::rust::StyleRule> *return$) noexcept;

void cssparser$rust$cxxbridge1$StyleSheet$errors(::cssparser::rust::StyleSheet const &self, ::rust::Vec<::cssparser::rust::StyleSheetError> *return$) noexcept;

void cssparser$rust$cxxbridge1$StyleSheet$set_root_path(::cssparser::rust::StyleSheet &self, ::rust::Str root_path) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$StyleSheet$parse_file(::cssparser::rust::StyleSheet &self, ::rust::Str file_name) noexcept;

::rust::repr::PtrLen cssparser$rust$cxxbridge1$StyleSheet$parse_string(::cssparser::rust::StyleSheet &self, ::rust::Str data, ::rust::Str origin) noexcept;

::cssparser::rust::StyleSheet *cssparser$rust$cxxbridge1$create_stylesheet() noexcept;
} // extern "C"

::rust::String Dimension::to_string() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  cssparser$rust$cxxbridge1$Dimension$to_string(*this, &return$.value);
  return ::std::move(return$.value);
}

::cssparser::rust::ColorOperationType ModifiedColor::operation_type() const noexcept {
  return cssparser$rust$cxxbridge1$ModifiedColor$operation_type(*this);
}

::rust::Box<::cssparser::rust::Color> ModifiedColor::color_value() const {
  ::rust::MaybeUninit<::rust::Box<::cssparser::rust::Color>> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$ModifiedColor$color_value(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::cssparser::rust::SetColorOperationValues ModifiedColor::set_values() const {
  ::rust::MaybeUninit<::cssparser::rust::SetColorOperationValues> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$ModifiedColor$set_values(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::cssparser::rust::MixColorOperationValues ModifiedColor::mix_values() const {
  ::rust::MaybeUninit<::cssparser::rust::MixColorOperationValues> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$ModifiedColor$mix_values(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::size_t ColorOperation::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$ColorOperation$operator$sizeof();
}

::std::size_t ColorOperation::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$ColorOperation$operator$alignof();
}

::std::size_t Color::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$Color$operator$sizeof();
}

::std::size_t Color::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$Color$operator$alignof();
}

::cssparser::rust::ColorType Color::color_type() const noexcept {
  return cssparser$rust$cxxbridge1$Color$color_type(*this);
}

::rust::String Color::to_string() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  cssparser$rust$cxxbridge1$Color$to_string(*this, &return$.value);
  return ::std::move(return$.value);
}

::cssparser::rust::Rgba Color::to_rgba() const {
  ::rust::MaybeUninit<::cssparser::rust::Rgba> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Color$to_rgba(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::cssparser::rust::CustomColor Color::to_custom() const {
  ::rust::MaybeUninit<::cssparser::rust::CustomColor> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Color$to_custom(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::cssparser::rust::ModifiedColor Color::to_modified() const {
  ::rust::MaybeUninit<::cssparser::rust::ModifiedColor> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Color$to_modified(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::size_t Value::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$Value$operator$sizeof();
}

::std::size_t Value::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$Value$operator$alignof();
}

::cssparser::rust::ValueType Value::value_type() const noexcept {
  return cssparser$rust$cxxbridge1$Value$value_type(*this);
}

::cssparser::rust::Dimension Value::to_dimension() const {
  ::rust::MaybeUninit<::cssparser::rust::Dimension> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Value$to_dimension(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String Value::to_string() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  cssparser$rust$cxxbridge1$Value$to_string(*this, &return$.value);
  return ::std::move(return$.value);
}

::rust::Box<::cssparser::rust::Color> Value::to_color() const {
  ::rust::MaybeUninit<::rust::Box<::cssparser::rust::Color>> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Value$to_color(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Str Value::to_image() const {
  ::rust::MaybeUninit<::rust::Str> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Value$to_image(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Str Value::to_url() const {
  ::rust::MaybeUninit<::rust::Str> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Value$to_url(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::int32_t Value::to_integer() const {
  ::rust::MaybeUninit<::std::int32_t> return$;
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$Value$to_integer(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::std::size_t SelectorPart::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$SelectorPart$operator$sizeof();
}

::std::size_t SelectorPart::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$SelectorPart$operator$alignof();
}

::cssparser::rust::SelectorKind SelectorPart::kind() const noexcept {
  return cssparser$rust$cxxbridge1$SelectorPart$kind(*this);
}

::cssparser::rust::Value const &SelectorPart::value() const noexcept {
  return *cssparser$rust$cxxbridge1$SelectorPart$value(*this);
}

::rust::String SelectorPart::attribute_name() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  cssparser$rust$cxxbridge1$SelectorPart$attribute_name(*this, &return$.value);
  return ::std::move(return$.value);
}

::cssparser::rust::AttributeOperator SelectorPart::attribute_operator() const noexcept {
  return cssparser$rust$cxxbridge1$SelectorPart$attribute_operator(*this);
}

::cssparser::rust::Value const &SelectorPart::attribute_value() const noexcept {
  return *cssparser$rust$cxxbridge1$SelectorPart$attribute_value(*this);
}

::std::size_t Selector::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$Selector$operator$sizeof();
}

::std::size_t Selector::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$Selector$operator$alignof();
}

::rust::Vec<::cssparser::rust::SelectorPart> Selector::parts() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::cssparser::rust::SelectorPart>> return$;
  cssparser$rust$cxxbridge1$Selector$parts(*this, &return$.value);
  return ::std::move(return$.value);
}

::std::size_t Property::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$Property$operator$sizeof();
}

::std::size_t Property::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$Property$operator$alignof();
}

::rust::String Property::name() const noexcept {
  ::rust::MaybeUninit<::rust::String> return$;
  cssparser$rust$cxxbridge1$Property$name(*this, &return$.value);
  return ::std::move(return$.value);
}

::rust::Vec<::cssparser::rust::Value> Property::values() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::cssparser::rust::Value>> return$;
  cssparser$rust$cxxbridge1$Property$values(*this, &return$.value);
  return ::std::move(return$.value);
}

::std::size_t StyleRule::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$StyleRule$operator$sizeof();
}

::std::size_t StyleRule::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$StyleRule$operator$alignof();
}

::cssparser::rust::Selector const &StyleRule::selector() const noexcept {
  return *cssparser$rust$cxxbridge1$StyleRule$selector(*this);
}

::rust::Vec<::cssparser::rust::Property> StyleRule::properties() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::cssparser::rust::Property>> return$;
  cssparser$rust$cxxbridge1$StyleRule$properties(*this, &return$.value);
  return ::std::move(return$.value);
}

::std::size_t StyleSheet::layout::size() noexcept {
  return cssparser$rust$cxxbridge1$StyleSheet$operator$sizeof();
}

::std::size_t StyleSheet::layout::align() noexcept {
  return cssparser$rust$cxxbridge1$StyleSheet$operator$alignof();
}

::rust::Vec<::cssparser::rust::StyleRule> StyleSheet::rules() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::cssparser::rust::StyleRule>> return$;
  cssparser$rust$cxxbridge1$StyleSheet$rules(*this, &return$.value);
  return ::std::move(return$.value);
}

::rust::Vec<::cssparser::rust::StyleSheetError> StyleSheet::errors() const noexcept {
  ::rust::MaybeUninit<::rust::Vec<::cssparser::rust::StyleSheetError>> return$;
  cssparser$rust$cxxbridge1$StyleSheet$errors(*this, &return$.value);
  return ::std::move(return$.value);
}

void StyleSheet::set_root_path(::rust::Str root_path) noexcept {
  cssparser$rust$cxxbridge1$StyleSheet$set_root_path(*this, root_path);
}

void StyleSheet::parse_file(::rust::Str file_name) {
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$StyleSheet$parse_file(*this, file_name);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void StyleSheet::parse_string(::rust::Str data, ::rust::Str origin) {
  ::rust::repr::PtrLen error$ = cssparser$rust$cxxbridge1$StyleSheet$parse_string(*this, data, origin);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::rust::Box<::cssparser::rust::StyleSheet> create_stylesheet() noexcept {
  return ::rust::Box<::cssparser::rust::StyleSheet>::from_raw(cssparser$rust$cxxbridge1$create_stylesheet());
}
} // namespace rust
} // namespace cssparser

extern "C" {
::cssparser::rust::Color *cxxbridge1$box$cssparser$rust$Color$alloc() noexcept;
void cxxbridge1$box$cssparser$rust$Color$dealloc(::cssparser::rust::Color *) noexcept;
void cxxbridge1$box$cssparser$rust$Color$drop(::rust::Box<::cssparser::rust::Color> *ptr) noexcept;

::cssparser::rust::ColorOperation *cxxbridge1$box$cssparser$rust$ColorOperation$alloc() noexcept;
void cxxbridge1$box$cssparser$rust$ColorOperation$dealloc(::cssparser::rust::ColorOperation *) noexcept;
void cxxbridge1$box$cssparser$rust$ColorOperation$drop(::rust::Box<::cssparser::rust::ColorOperation> *ptr) noexcept;

void cxxbridge1$rust_vec$cssparser$rust$SelectorPart$new(::rust::Vec<::cssparser::rust::SelectorPart> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$SelectorPart$drop(::rust::Vec<::cssparser::rust::SelectorPart> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$SelectorPart$len(::rust::Vec<::cssparser::rust::SelectorPart> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$SelectorPart$capacity(::rust::Vec<::cssparser::rust::SelectorPart> const *ptr) noexcept;
::cssparser::rust::SelectorPart const *cxxbridge1$rust_vec$cssparser$rust$SelectorPart$data(::rust::Vec<::cssparser::rust::SelectorPart> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$SelectorPart$reserve_total(::rust::Vec<::cssparser::rust::SelectorPart> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$SelectorPart$set_len(::rust::Vec<::cssparser::rust::SelectorPart> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$SelectorPart$truncate(::rust::Vec<::cssparser::rust::SelectorPart> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$cssparser$rust$Value$new(::rust::Vec<::cssparser::rust::Value> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Value$drop(::rust::Vec<::cssparser::rust::Value> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$Value$len(::rust::Vec<::cssparser::rust::Value> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$Value$capacity(::rust::Vec<::cssparser::rust::Value> const *ptr) noexcept;
::cssparser::rust::Value const *cxxbridge1$rust_vec$cssparser$rust$Value$data(::rust::Vec<::cssparser::rust::Value> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Value$reserve_total(::rust::Vec<::cssparser::rust::Value> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Value$set_len(::rust::Vec<::cssparser::rust::Value> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Value$truncate(::rust::Vec<::cssparser::rust::Value> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$cssparser$rust$Property$new(::rust::Vec<::cssparser::rust::Property> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Property$drop(::rust::Vec<::cssparser::rust::Property> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$Property$len(::rust::Vec<::cssparser::rust::Property> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$Property$capacity(::rust::Vec<::cssparser::rust::Property> const *ptr) noexcept;
::cssparser::rust::Property const *cxxbridge1$rust_vec$cssparser$rust$Property$data(::rust::Vec<::cssparser::rust::Property> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Property$reserve_total(::rust::Vec<::cssparser::rust::Property> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Property$set_len(::rust::Vec<::cssparser::rust::Property> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$Property$truncate(::rust::Vec<::cssparser::rust::Property> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$cssparser$rust$StyleRule$new(::rust::Vec<::cssparser::rust::StyleRule> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleRule$drop(::rust::Vec<::cssparser::rust::StyleRule> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$StyleRule$len(::rust::Vec<::cssparser::rust::StyleRule> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$StyleRule$capacity(::rust::Vec<::cssparser::rust::StyleRule> const *ptr) noexcept;
::cssparser::rust::StyleRule const *cxxbridge1$rust_vec$cssparser$rust$StyleRule$data(::rust::Vec<::cssparser::rust::StyleRule> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleRule$reserve_total(::rust::Vec<::cssparser::rust::StyleRule> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleRule$set_len(::rust::Vec<::cssparser::rust::StyleRule> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleRule$truncate(::rust::Vec<::cssparser::rust::StyleRule> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$new(::rust::Vec<::cssparser::rust::StyleSheetError> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$drop(::rust::Vec<::cssparser::rust::StyleSheetError> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$len(::rust::Vec<::cssparser::rust::StyleSheetError> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$capacity(::rust::Vec<::cssparser::rust::StyleSheetError> const *ptr) noexcept;
::cssparser::rust::StyleSheetError const *cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$data(::rust::Vec<::cssparser::rust::StyleSheetError> const *ptr) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$reserve_total(::rust::Vec<::cssparser::rust::StyleSheetError> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$set_len(::rust::Vec<::cssparser::rust::StyleSheetError> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$truncate(::rust::Vec<::cssparser::rust::StyleSheetError> *ptr, ::std::size_t len) noexcept;

::cssparser::rust::StyleSheet *cxxbridge1$box$cssparser$rust$StyleSheet$alloc() noexcept;
void cxxbridge1$box$cssparser$rust$StyleSheet$dealloc(::cssparser::rust::StyleSheet *) noexcept;
void cxxbridge1$box$cssparser$rust$StyleSheet$drop(::rust::Box<::cssparser::rust::StyleSheet> *ptr) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
::cssparser::rust::Color *Box<::cssparser::rust::Color>::allocation::alloc() noexcept {
  return cxxbridge1$box$cssparser$rust$Color$alloc();
}
template <>
void Box<::cssparser::rust::Color>::allocation::dealloc(::cssparser::rust::Color *ptr) noexcept {
  cxxbridge1$box$cssparser$rust$Color$dealloc(ptr);
}
template <>
void Box<::cssparser::rust::Color>::drop() noexcept {
  cxxbridge1$box$cssparser$rust$Color$drop(this);
}
template <>
::cssparser::rust::ColorOperation *Box<::cssparser::rust::ColorOperation>::allocation::alloc() noexcept {
  return cxxbridge1$box$cssparser$rust$ColorOperation$alloc();
}
template <>
void Box<::cssparser::rust::ColorOperation>::allocation::dealloc(::cssparser::rust::ColorOperation *ptr) noexcept {
  cxxbridge1$box$cssparser$rust$ColorOperation$dealloc(ptr);
}
template <>
void Box<::cssparser::rust::ColorOperation>::drop() noexcept {
  cxxbridge1$box$cssparser$rust$ColorOperation$drop(this);
}
template <>
Vec<::cssparser::rust::SelectorPart>::Vec() noexcept {
  cxxbridge1$rust_vec$cssparser$rust$SelectorPart$new(this);
}
template <>
void Vec<::cssparser::rust::SelectorPart>::drop() noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$drop(this);
}
template <>
::std::size_t Vec<::cssparser::rust::SelectorPart>::size() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$len(this);
}
template <>
::std::size_t Vec<::cssparser::rust::SelectorPart>::capacity() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$capacity(this);
}
template <>
::cssparser::rust::SelectorPart const *Vec<::cssparser::rust::SelectorPart>::data() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$data(this);
}
template <>
void Vec<::cssparser::rust::SelectorPart>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$reserve_total(this, new_cap);
}
template <>
void Vec<::cssparser::rust::SelectorPart>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$set_len(this, len);
}
template <>
void Vec<::cssparser::rust::SelectorPart>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$cssparser$rust$SelectorPart$truncate(this, len);
}
template <>
Vec<::cssparser::rust::Value>::Vec() noexcept {
  cxxbridge1$rust_vec$cssparser$rust$Value$new(this);
}
template <>
void Vec<::cssparser::rust::Value>::drop() noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Value$drop(this);
}
template <>
::std::size_t Vec<::cssparser::rust::Value>::size() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Value$len(this);
}
template <>
::std::size_t Vec<::cssparser::rust::Value>::capacity() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Value$capacity(this);
}
template <>
::cssparser::rust::Value const *Vec<::cssparser::rust::Value>::data() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Value$data(this);
}
template <>
void Vec<::cssparser::rust::Value>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Value$reserve_total(this, new_cap);
}
template <>
void Vec<::cssparser::rust::Value>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Value$set_len(this, len);
}
template <>
void Vec<::cssparser::rust::Value>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$cssparser$rust$Value$truncate(this, len);
}
template <>
Vec<::cssparser::rust::Property>::Vec() noexcept {
  cxxbridge1$rust_vec$cssparser$rust$Property$new(this);
}
template <>
void Vec<::cssparser::rust::Property>::drop() noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Property$drop(this);
}
template <>
::std::size_t Vec<::cssparser::rust::Property>::size() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Property$len(this);
}
template <>
::std::size_t Vec<::cssparser::rust::Property>::capacity() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Property$capacity(this);
}
template <>
::cssparser::rust::Property const *Vec<::cssparser::rust::Property>::data() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Property$data(this);
}
template <>
void Vec<::cssparser::rust::Property>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Property$reserve_total(this, new_cap);
}
template <>
void Vec<::cssparser::rust::Property>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$Property$set_len(this, len);
}
template <>
void Vec<::cssparser::rust::Property>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$cssparser$rust$Property$truncate(this, len);
}
template <>
Vec<::cssparser::rust::StyleRule>::Vec() noexcept {
  cxxbridge1$rust_vec$cssparser$rust$StyleRule$new(this);
}
template <>
void Vec<::cssparser::rust::StyleRule>::drop() noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$drop(this);
}
template <>
::std::size_t Vec<::cssparser::rust::StyleRule>::size() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$len(this);
}
template <>
::std::size_t Vec<::cssparser::rust::StyleRule>::capacity() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$capacity(this);
}
template <>
::cssparser::rust::StyleRule const *Vec<::cssparser::rust::StyleRule>::data() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$data(this);
}
template <>
void Vec<::cssparser::rust::StyleRule>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$reserve_total(this, new_cap);
}
template <>
void Vec<::cssparser::rust::StyleRule>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$set_len(this, len);
}
template <>
void Vec<::cssparser::rust::StyleRule>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$cssparser$rust$StyleRule$truncate(this, len);
}
template <>
Vec<::cssparser::rust::StyleSheetError>::Vec() noexcept {
  cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$new(this);
}
template <>
void Vec<::cssparser::rust::StyleSheetError>::drop() noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$drop(this);
}
template <>
::std::size_t Vec<::cssparser::rust::StyleSheetError>::size() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$len(this);
}
template <>
::std::size_t Vec<::cssparser::rust::StyleSheetError>::capacity() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$capacity(this);
}
template <>
::cssparser::rust::StyleSheetError const *Vec<::cssparser::rust::StyleSheetError>::data() const noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$data(this);
}
template <>
void Vec<::cssparser::rust::StyleSheetError>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$reserve_total(this, new_cap);
}
template <>
void Vec<::cssparser::rust::StyleSheetError>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$set_len(this, len);
}
template <>
void Vec<::cssparser::rust::StyleSheetError>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$cssparser$rust$StyleSheetError$truncate(this, len);
}
template <>
::cssparser::rust::StyleSheet *Box<::cssparser::rust::StyleSheet>::allocation::alloc() noexcept {
  return cxxbridge1$box$cssparser$rust$StyleSheet$alloc();
}
template <>
void Box<::cssparser::rust::StyleSheet>::allocation::dealloc(::cssparser::rust::StyleSheet *ptr) noexcept {
  cxxbridge1$box$cssparser$rust$StyleSheet$dealloc(ptr);
}
template <>
void Box<::cssparser::rust::StyleSheet>::drop() noexcept {
  cxxbridge1$box$cssparser$rust$StyleSheet$drop(this);
}
} // namespace cxxbridge1
} // namespace rust
