use super::audio::AudioWidget;
use super::link::LinkPreview;
use super::video::VideoWidget;
use crate::article_view::Webview;
use crate::gobject_models::GEnclosure;
use crate::image_dialog::ImageDialog;
use crate::image_widget::ImageWidget;
use crate::main_window::MainWindow;
use glib::{
    Object, Properties, SignalHandlerId, clone,
    subclass::{prelude::*, *},
};
use gtk4::{
    Align, Allocation, CompositeTemplate, ContentFit, SizeRequestMode, Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::prelude::*;
use once_cell::sync::Lazy;
use std::cell::RefCell;

mod imp {
    const VIDEO_HEIGHT: i32 = 280;
    const IMAGE_HEIGHT: i32 = 280;
    const AUDIO_HEIGHT: i32 = 114;
    const LINK_HEIGHT: i32 = 200;

    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::EnclosurePreview)]
    #[template(file = "data/resources/ui_templates/enclosures/preview.blp")]
    pub struct EnclosurePreview {
        #[property(get, set = Self::set_enclosure)]
        pub enclosure: RefCell<GEnclosure>,

        #[property(get, set = Self::set_widget)]
        pub widget: RefCell<Option<Widget>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for EnclosurePreview {
        const NAME: &'static str = "EnclosurePreview";
        type Type = super::EnclosurePreview;
        type ParentType = Widget;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for EnclosurePreview {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("remove")
                        .param_types([super::EnclosurePreview::static_type()])
                        .build(),
                ]
            });
            SIGNALS.as_ref()
        }

        fn dispose(&self) {
            if let Some(old_widget) = self.widget.take() {
                old_widget.unparent();
            }
        }
    }

    impl WidgetImpl for EnclosurePreview {
        fn request_mode(&self) -> SizeRequestMode {
            SizeRequestMode::HeightForWidth
        }

        fn size_allocate(&self, width: i32, height: i32, baseline: i32) {
            let enclosure = self.enclosure.borrow();

            if let Some(widget) = self.widget.borrow().as_ref() {
                if enclosure.is_video() {
                    let main_window = MainWindow::instance();

                    if main_window.is_fullscreen() {
                        log::debug!("allocate full screen");
                        widget.size_allocate(&Allocation::new(0, 0, width, main_window.height()), -1);
                        self.parent_size_allocate(width, main_window.height(), baseline);
                    } else {
                        widget.size_allocate(&Allocation::new(0, 0, width, VIDEO_HEIGHT), -1);
                        self.parent_size_allocate(width, VIDEO_HEIGHT, baseline);
                    }
                } else if enclosure.is_image() {
                    widget.size_allocate(&Allocation::new(0, 0, width, IMAGE_HEIGHT), -1);
                    self.parent_size_allocate(width, IMAGE_HEIGHT, baseline);
                } else if enclosure.is_audio() {
                    widget.size_allocate(&Allocation::new(0, 0, width, AUDIO_HEIGHT), -1);
                    self.parent_size_allocate(width, AUDIO_HEIGHT, baseline);
                } else {
                    widget.size_allocate(&Allocation::new(0, 0, width, LINK_HEIGHT), -1);
                    self.parent_size_allocate(width, LINK_HEIGHT, baseline);
                }
            }

            self.parent_size_allocate(width, height, baseline);
        }
    }

    #[gtk4::template_callbacks]
    impl EnclosurePreview {
        #[template_callback]
        fn on_clicked(&self) {
            let enclosure = self.obj().enclosure();

            if let Some(mime) = enclosure.mime_type()
                && mime.starts_with("image")
            {
                let dialog = ImageDialog::new_url(&enclosure.article_id().into(), enclosure.url().as_str());

                Webview::instance().set_image_dialog_visible(true);
                dialog.connect_closed(|_dialog| {
                    Webview::instance().set_image_dialog_visible(false);
                });
                dialog.present(Some(&MainWindow::instance()))
            }
        }

        fn set_widget(&self, widget: Option<Widget>) {
            if let Some(old_widget) = self.widget.take() {
                old_widget.unparent();
            }

            if let Some(new_widget) = widget.as_ref() {
                new_widget.set_parent(&*self.obj());
            }

            self.widget.replace(widget);
        }

        fn set_enclosure(&self, enclosure: GEnclosure) {
            let obj = self.obj();

            let (height, widget) = if enclosure.is_image() {
                let image_widget = ImageWidget::default();
                image_widget.load(enclosure.article_id().as_ref(), &enclosure.url());
                image_widget.set_fit(ContentFit::Cover);
                image_widget.add_css_class("image-widget");
                image_widget.connect_error(clone!(
                    #[weak]
                    obj,
                    move |_error| obj.emmit_remove()
                ));
                image_widget.connect_tiny(clone!(
                    #[weak]
                    obj,
                    move || obj.emmit_remove()
                ));
                (IMAGE_HEIGHT, image_widget.upcast())
            } else if enclosure.is_video() {
                let video_widget = VideoWidget::from(&enclosure);
                (VIDEO_HEIGHT, video_widget.upcast())
            } else if enclosure.is_audio() {
                let audio_widget = AudioWidget::from(&enclosure);
                audio_widget.set_valign(Align::Center);
                (AUDIO_HEIGHT, audio_widget.upcast())
            } else {
                let link_preview = LinkPreview::from(&enclosure);
                link_preview.set_valign(Align::Center);
                (LINK_HEIGHT, link_preview.upcast())
            };

            obj.set_height_request(height);
            obj.set_widget(&widget);

            self.enclosure.replace(enclosure);
        }
    }
}

glib::wrapper! {
    pub struct EnclosurePreview(ObjectSubclass<imp::EnclosurePreview>)
        @extends Widget;
}

impl Default for EnclosurePreview {
    fn default() -> Self {
        Object::new()
    }
}

impl EnclosurePreview {
    pub fn new(enclosure: &GEnclosure) -> Self {
        let preview = Self::default();
        preview.set_enclosure(enclosure.clone());
        preview
    }

    pub fn emmit_remove(&self) {
        log::debug!("remove preview");
        self.emit_by_name::<()>("remove", &[&self.clone()])
    }

    pub fn connect_remove<F: Fn(&Self) + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("remove", false, move |args| {
            let preview = args[1]
                .get::<Self>()
                .expect("The value needs to be of type `EnclosurePreview`");
            f(&preview);
            None
        })
    }
}
