use std::collections::HashMap;

use gio::{ActionEntry, prelude::*};
use glib::{Variant, VariantType};
use news_flash::error::{FeedApiError, NewsFlashError};
use news_flash::models::{CategoryID, CategoryMapping, FeedID, FeedMapping, TagID, Url};

use crate::app::{App, NotificationCounts};
use crate::content_page::{ArticleListColumn, ArticleViewColumn, ContentPage};
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::settings::GFeedOrder;
use crate::undo_action::UndoAction;
use crate::util::constants;

pub struct EditActions;

impl EditActions {
    pub fn setup() {
        // -------------------------
        // add feed
        // -------------------------
        let add_feed = ActionEntry::builder("add-feed")
            .parameter_type(VariantType::new("(sss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::add_feed(parameter))
            .build();

        // -------------------------
        // add category
        // -------------------------
        let add_category = ActionEntry::builder("add-category")
            .parameter_type(VariantType::new("s").as_deref().ok())
            .activate(|_win, _action, parameter| Self::add_category(parameter))
            .build();

        // -------------------------
        // add tag
        // -------------------------
        let add_tag = ActionEntry::builder("add-tag")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::add_tag(parameter))
            .build();

        // -------------------------
        // enqueue delete category
        // -------------------------
        let enqueue_delete_category = ActionEntry::builder("enqueue-delete-category")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::enqueue_delete_category(parameter))
            .build();

        // -------------------------
        // enqueue delete feed
        // -------------------------
        let enqueue_delete_feed = ActionEntry::builder("enqueue-delete-feed")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::enqueue_delete_feed(parameter))
            .build();

        // -------------------------
        // enqueue delete tag
        // -------------------------
        let enqueue_delete_tag = ActionEntry::builder("enqueue-delete-tag")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::enqueue_delete_tag(parameter))
            .build();

        // -------------------------
        // delete category
        // -------------------------
        let delete_category = ActionEntry::builder("delete-category")
            .parameter_type(Some(&String::static_variant_type()))
            .activate(|_win, _action, parameter| Self::delete_category(parameter))
            .build();

        // -------------------------
        // delete feed
        // -------------------------
        let delete_feed = ActionEntry::builder("delete-feed")
            .parameter_type(Some(&String::static_variant_type()))
            .activate(|_win, _action, parameter| Self::delete_feed(parameter))
            .build();

        // -------------------------
        // delete tag
        // -------------------------
        let delete_tag = ActionEntry::builder("delete-tag")
            .parameter_type(Some(&String::static_variant_type()))
            .activate(|_win, _action, parameter| Self::delete_tag(parameter))
            .build();

        // ---------------------------
        // rename feed
        // ---------------------------
        let rename_feed = ActionEntry::builder("rename-feed")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::rename_feed(parameter))
            .build();

        // ---------------------------
        // rename category
        // ---------------------------
        let rename_category = ActionEntry::builder("rename-category")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::rename_category(parameter))
            .build();

        // ---------------------------
        // edit tag
        // ---------------------------
        let edit_tag = ActionEntry::builder("edit-tag")
            .parameter_type(VariantType::new("(sss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::edit_tag(parameter))
            .build();

        // ---------------------------
        // edit feed url
        // ---------------------------
        let edit_feed_url = ActionEntry::builder("edit-feed-url")
            .parameter_type(VariantType::new("(ss)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::edit_feed_url(parameter))
            .build();

        // ---------------------------
        // move feed
        // ---------------------------
        let move_feed = ActionEntry::builder("move-feed")
            .parameter_type(VariantType::new("(sssi)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::move_feed(parameter))
            .build();

        // ---------------------------
        // move category
        // ---------------------------
        let move_category = ActionEntry::builder("move-category")
            .parameter_type(VariantType::new("(ssi)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::move_category(parameter))
            .build();

        MainWindow::instance().add_action_entries([
            add_feed,
            add_category,
            add_tag,
            enqueue_delete_category,
            enqueue_delete_feed,
            enqueue_delete_tag,
            delete_category,
            delete_feed,
            delete_tag,
            rename_feed,
            rename_category,
            edit_tag,
            edit_feed_url,
            move_feed,
            move_category,
        ]);
    }

    fn add_feed(parameter: Option<&Variant>) {
        let Some(variant) = parameter else {
            return;
        };

        let feed_url = variant.child_get::<String>(0);
        let title = variant.child_get::<String>(1);
        let category_id = variant.child_get::<String>(2);

        let feed_url = Url::parse(&feed_url).unwrap();
        let title = if title.is_empty() { None } else { Some(title) };
        let category_id = if category_id.is_empty() {
            None
        } else {
            Some(CategoryID::from_owned(category_id))
        };

        log::info!("add feed '{feed_url}'");

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash.add_feed(&feed_url, title, category_id, &App::client()).await
            },
            |res| match res {
                Ok((feed, _feed_mapping, _category, _category_mapping)) => {
                    ContentPage::instance().update_sidebar();
                    Self::fetch_feed(feed.feed_id);
                }
                Err(error) => {
                    log::error!("Failed to add feed: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to add feed"), error);
                }
            },
        );
    }

    fn fetch_feed(feed_id: FeedID) {
        App::default().set_is_syncing(false);
        App::set_background_status(constants::BACKGROUND_SYNC);

        let header_map = App::default().settings().get_feed_header_map(&feed_id);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                let new_article_count = news_flash.fetch_feed(&feed_id, &App::client(), header_map).await?;
                let mut new_article_map = HashMap::new();
                new_article_map.insert(feed_id, new_article_count);
                let unread_map = news_flash
                    .unread_count_feed_map(App::default().settings().article_list().hide_future_articles())?;
                let (feeds, _feed_mappings) = news_flash.get_feeds()?;
                Ok((new_article_map, unread_map, feeds))
            },
            |res| {
                App::set_background_status(constants::BACKGROUND_IDLE);
                App::default().set_is_syncing(false);

                match res {
                    Ok((new_article_map, unread_map, feeds)) => {
                        let feed_names = feeds
                            .iter()
                            .map(|f| (f.feed_id.clone(), f.label.clone()))
                            .collect::<HashMap<FeedID, String>>();

                        ContentPage::instance().update_sidebar();
                        ArticleListColumn::instance().update_list();
                        let counts = NotificationCounts {
                            new: new_article_map,
                            unread: unread_map,
                            names: feed_names,
                        };
                        App::show_notification(counts);
                    }
                    Err(error) => {
                        if let NewsFlashError::API(FeedApiError::Unsupported) = error {
                            return;
                        }

                        ContentPage::instance().newsflash_error(&i18n("Failed to fetch feed"), error);
                    }
                }
            },
        );
    }

    fn add_category(parameter: Option<&Variant>) {
        let Some(title) = parameter.and_then(Variant::str) else {
            return;
        };

        log::info!("add category '{title}'");
        let title = title.to_string();

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash.add_category(&title, None, &App::client()).await
            },
            |res| {
                if let Err(error) = res {
                    log::error!("Failed to add category: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to add category"), error);
                } else {
                    ContentPage::instance().update_sidebar();
                }
            },
        );
    }

    fn add_tag(parameter: Option<&Variant>) {
        let Some(variant) = parameter else {
            return;
        };

        let color = variant.child_get::<String>(0);
        let title = variant.child_get::<String>(1);

        log::info!("add tag '{title}'");

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash.add_tag(&title, Some(color), &App::client()).await
            },
            move |res| match res {
                Ok(_tag) => {
                    ContentPage::instance().update_sidebar();
                    ArticleViewColumn::instance().refresh_article_metadata_from_db();
                }
                Err(error) => {
                    log::error!("Failed to add tag: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to add tag"), error);
                }
            },
        );
    }

    fn delete_category(parameter: Option<&Variant>) {
        let Some(category_id) = parameter.and_then(Variant::str).map(CategoryID::new) else {
            return;
        };

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash.remove_category(&category_id, true, &App::client()).await
            },
            move |res| {
                if let Err(error) = res {
                    log::error!("Failed to delete category: {error}");
                    ContentPage::instance().update_sidebar();
                    ContentPage::instance().newsflash_error(&i18n("Failed to delete category"), error);
                }
            },
        );
    }

    fn delete_feed(parameter: Option<&Variant>) {
        let Some(feed_id) = parameter.and_then(Variant::str).map(FeedID::new) else {
            return;
        };

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash.remove_feed(&feed_id, &App::client()).await
            },
            move |res| {
                if let Err(error) = res {
                    log::error!("Failed to delete feed: {error}");
                    ContentPage::instance().update_sidebar();
                    ContentPage::instance().newsflash_error(&i18n("Failed to delete feed"), error);
                }
            },
        );
    }

    fn delete_tag(parameter: Option<&Variant>) {
        let Some(tag_id) = parameter.and_then(Variant::str).map(TagID::new) else {
            return;
        };

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash.remove_tag(&tag_id, &App::client()).await
            },
            move |res| {
                if let Err(error) = res {
                    log::error!("Failed to delete tag: {error}");
                    ContentPage::instance().update_sidebar();
                    ContentPage::instance().newsflash_error(&i18n("Failed to delete tag"), error);
                }
            },
        );
    }

    fn enqueue_delete_category(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("Delete category: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let name_value = parameter.child_value(1);

        let (Some(id_str), Some(name_str)) = (id_value.str(), name_value.str()) else {
            return;
        };

        let category_id = CategoryID::new(id_str);
        ContentPage::instance().add_undo_notification(UndoAction::DeleteCategory(category_id, name_str.into()));
        ContentPage::instance().update_sidebar();
    }

    fn enqueue_delete_feed(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("Delete category: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let name_value = parameter.child_value(1);

        let (Some(id_str), Some(name_str)) = (id_value.str(), name_value.str()) else {
            return;
        };

        let feed_id = FeedID::new(id_str);
        ContentPage::instance().add_undo_notification(UndoAction::DeleteFeed(feed_id, name_str.into()));
        ContentPage::instance().update_sidebar();
    }

    fn enqueue_delete_tag(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("Delete category: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let name_value = parameter.child_value(1);

        let (Some(id_str), Some(name_str)) = (id_value.str(), name_value.str()) else {
            return;
        };

        let tag_id = TagID::new(id_str);
        ContentPage::instance().add_undo_notification(UndoAction::DeleteTag(tag_id, name_str.into()));
        ContentPage::instance().update_sidebar();
    }

    fn rename_feed(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("Rename feed: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let new_title_value = parameter.child_value(1);

        let (Some(id_str), Some(new_title_str)) = (id_value.str(), new_title_value.str()) else {
            return;
        };

        let feed_id = FeedID::new(id_str);
        let feed_id_clone = feed_id.clone();
        let new_title = new_title_str.to_string();
        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash.rename_feed(&feed_id, &new_title, &App::client()).await
            },
            move |res| match res {
                Ok(_feed) => {
                    ArticleListColumn::instance().update_list();
                    ContentPage::instance().update_sidebar();

                    // if the visible article belongs to the renamed feed
                    // show it again to gather the updated feed name
                    if let Some(article) = ArticleViewColumn::instance().article()
                        && article.feed_id().as_ref() == &feed_id_clone
                    {
                        ArticleViewColumn::instance().show_article(article.article_id(), false);
                    }
                }
                Err(error) => {
                    log::error!("Failed to rename feed: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to rename feed"), error);
                }
            },
        );
    }

    fn rename_category(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("Rename category: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let new_title_value = parameter.child_value(1);

        let (Some(id_str), Some(new_title_str)) = (id_value.str(), new_title_value.str()) else {
            return;
        };

        let category_id = CategoryID::new(id_str);
        let new_title = new_title_str.to_string();

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash
                    .rename_category(&category_id, &new_title, &App::client())
                    .await
            },
            move |res| match res {
                Ok(_feed) => {
                    ContentPage::instance().update_sidebar();
                }
                Err(error) => {
                    log::error!("Failed to rename category: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to rename category"), error);
                }
            },
        );
    }

    fn edit_tag(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("Rename category: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let new_title_value = parameter.child_value(1);
        let color_value = parameter.child_value(2);

        let (Some(id_str), Some(new_title_str), Some(color_str)) =
            (id_value.str(), new_title_value.str(), color_value.str())
        else {
            return;
        };

        let tag_id = TagID::new(id_str);
        let new_title = new_title_str.to_string();
        let new_color = color_str.to_string();

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash
                    .edit_tag(&tag_id, &new_title, &Some(new_color), &App::client())
                    .await
            },
            move |res| match res {
                Ok(_tag) => {
                    ArticleListColumn::instance().update_list();
                    ContentPage::instance().update_sidebar();
                }
                Err(error) => {
                    let message = &i18n("Failed to edit tag");
                    log::error!("{message}");
                    ContentPage::instance().newsflash_error(message, error);
                }
            },
        );
    }

    fn edit_feed_url(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("edit feed url: no parameter");
            return;
        };

        let id_value = parameter.child_value(0);
        let new_url_value = parameter.child_value(1);

        let (Some(id_str), Some(new_url_str)) = (id_value.str(), new_url_value.str()) else {
            return;
        };

        let feed_id = FeedID::new(id_str);
        let new_url = new_url_str.to_string();

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash.edit_feed_url(&feed_id, &new_url, &App::client()).await
            },
            move |res| {
                if let Err(error) = res {
                    log::error!("Failed to rename feed: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to rename feed"), error);
                }
            },
        );
    }

    fn move_feed(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("move feed: no parameter");
            return;
        };

        let move_id_value = parameter.child_value(0);
        let from_id_value = parameter.child_value(1);
        let to_id_value = parameter.child_value(2);
        let sort_index_value = parameter.child_value(3);

        let (Some(move_id), Some(from_id), Some(to_id), Some(sort_index)) = (
            move_id_value.str(),
            from_id_value.str(),
            to_id_value.str(),
            sort_index_value.get::<i32>(),
        ) else {
            return;
        };

        let from_mapping = FeedMapping {
            feed_id: FeedID::new(move_id),
            category_id: CategoryID::new(from_id),
            sort_index: None,
        };
        let to_mapping = FeedMapping {
            feed_id: FeedID::new(move_id),
            category_id: CategoryID::new(to_id),
            sort_index: Some(sort_index),
        };

        if App::default().settings().feed_list().order() == GFeedOrder::Alphabetical {
            App::default().settings().feed_list().set_order(GFeedOrder::Manual);
        }

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                news_flash.move_feed(&from_mapping, &to_mapping, &App::client()).await
            },
            move |res| match res {
                Ok(()) => {
                    ContentPage::instance().update_sidebar();
                }
                Err(error) => {
                    log::error!("Failed to move feed: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to move feed"), error);
                }
            },
        );
    }

    fn move_category(parameter: Option<&Variant>) {
        let Some(parameter) = parameter else {
            log::warn!("move category: no parameter");
            return;
        };

        let move_id_value = parameter.child_value(0);
        let to_id_value = parameter.child_value(1);
        let sort_index_value = parameter.child_value(2);

        let (Some(move_id), Some(to_id), Some(sort_index)) =
            (move_id_value.str(), to_id_value.str(), sort_index_value.get::<i32>())
        else {
            return;
        };

        if App::default().settings().feed_list().order() == GFeedOrder::Alphabetical {
            App::default().settings().feed_list().set_order(GFeedOrder::Manual);
        }

        let to_mapping = CategoryMapping {
            parent_id: CategoryID::new(to_id),
            category_id: CategoryID::new(move_id),
            sort_index: Some(sort_index),
        };

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash.move_category(&to_mapping, &App::client()).await
            },
            move |res| match res {
                Ok(()) => {
                    ContentPage::instance().update_sidebar();
                }
                Err(error) => {
                    log::error!("Failed to move category: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Failed to move category"), error);
                }
            },
        );
    }
}
