// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A0__F307

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_160_0_F_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xeba77777u, 0x1e1810abu, 0, 103 }, // eba777771e1810ab = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xe55cf375u, 0xfc46f1fcu, 0, 40 }, // e55cf375fc46f1fc = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xc0b910f3u, 0xecbb2044u, 0, 126 }, // c0b910f3ecbb2044 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x0bfb83c3u, 0xee666032u, 74, 23 }, // 0bfb83c3ee666032 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x1b7a0ab3u, 0x46e8dae5u, 120, 80 }, // 1b7a0ab346e8dae5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x9cba0d45u, 0x58c69619u, 120, 6 }, // 9cba0d4558c69619 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__64_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x1a4e4651u, 0xf5de1685u, 120, 103 }, // 1a4e4651f5de1685 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x91357b6cu, 0xac9be9bau, 160, 80 }, // 91357b6cac9be9ba = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_0_F_T_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{0,2,3,3,1,1,3,3,1,1},
 {5,5,6,3,1,1,3,3,3,3},
 {6,6,6,3,1,3,3,3,3,3},
 {6,5,6,3,1,3,3,3,3,3},
 {5,6,6,3,1,3,3,3,3,3},
 {6,5,6,3,3,3,3,3,3,3},
 {5,5,6,3,3,7,7,7,7,7},
 {5,5,6,7,1,7,7,7,7,7},
 {4,4,4,7,7,7,7,4,7,7},
 {6,4,4,7,7,7,4,4,7,7}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A0__F307 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 160
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 1

// vim: set fileencoding=utf-8

