// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A1__F149

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_224_0_T_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1151)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x967f9baau, 0x910fc1f9u, 0, 155 }, // 967f9baa910fc1f9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1151
    { 0xfdd9e4b6u, 0xa827b76eu, 23, 155 }, // fdd9e4b6a827b76e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1151
    { 0x908d0bb5u, 0xc91807dau, 23, 6 }, // 908d0bb5c91807da = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1151
    { 0x50a2a384u, 0x9238c340u, 23, 109 }, // 50a2a3849238c340 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1151
    { 0xfdf268d1u, 0x1ec3b314u, 23, 86 }, // fdf268d11ec3b314 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1151
    { 0x681761bfu, 0x081d37d8u, 63, 138 }, // 681761bf081d37d8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1151
    { 0x0b3fe7e2u, 0xa82d77feu, 63, 69 }, // 0b3fe7e2a82d77fe = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1151
    { 0xfb78c46bu, 0x85661cf1u, 63, 29 }, // fb78c46b85661cf1 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1151
    { 0x7453b3a5u, 0x96716e9eu, 63, 46 }, // 7453b3a596716e9e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1151
    { 0x6ac592b1u, 0x204c7c46u, 126, 29 }, // 6ac592b1204c7c46 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1151
    { 0xa2832573u, 0x1de113eau, 126, 46 }, // a28325731de113ea = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_224_0_T_F_1___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1151
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx1151_mod0
{{ 4, 0, 0, 0, 1, 1, 4},
 { 4, 0, 0, 0, 0, 0, 0},
 { 3, 0, 0, 0, 0, 0, 0},
 { 3, 0, 0, 0, 0, 0, 0},
 { 2, 8, 7, 8, 0, 0, 0},
 { 5,10, 8, 8, 8, 0, 8},
 { 5, 9, 8, 5, 6, 8, 8}}
// End of GPU gfx1151_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A1__F149 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 224
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 1

// vim: set fileencoding=utf-8

