/*******************************************************************************
*                                                                              *
* Copyright (C) 1998 Steve Haehn                                               *
*                                                                              *
* This is free software; you can redistribute it and/or modify it under the    *
* terms of the GNU General Public License as published by the Free Software    *
* Foundation; either version 2 of the License, or (at your option) any later   *
* version.                                                                     *
*                                                                              *
* This software is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or        *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License *
* for more details.                                                            *
*                                                                              *
* You should have received a copy of the GNU General Public License along with *
* software; if not, write to the Free Software Foundation, Inc., 59 Temple     *
* Place, Suite 330, Boston, MA  02111-1307 USA                                 *
*                                                                              *
*******************************************************************************/

/*==============================================================================
**
** FILE NAME:       boxcomment.c
**
** FUNCTIONS INCLUDED:
**
**      start_anchored_comment, box_it_with, unbox_it, remove_file_path, main
**
** DESCRIPTION:
**
**      If the executable name begins with the letter 'b' or 'B', this program
**      will take standard input and attempt to wrap it in a comment box and
**      place the results on standard output. The default boxing environment is
**      for 'C' language code. If the executable name begins with a letter other
**      than 'b'  or 'B', the program will take the standard input and attempt
**      to remove the expected 'box' characters from the text and rewrite the
**      results to standard output. This program is meant to be used as a filter
**      for editors that allows such things.
**
** REVISION HISTORY:
**
**       Date      Author/Comment
**   Jun 12, 1998  Steven Haehn
**                 Allowed leading spaces to be a part of the begin, and body
**                 portion of a comment.
**
**   Feb 16, 1998  Steven Haehn
**                 Added concept of anchored comments.
**
**   Feb 03, 1997  Steven Haehn
**                 Original version.
**==============================================================================
-*/

/*============================================================================*/
/*====================  OPERATING SYSTEM INCLUDE FILES  ======================*/
/*============================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/*============================================================================*/
/*                              SYMBOL DEFINITIONS                            */
/*============================================================================*/

#define MAX_LINES 300
#define EOS '\0'
#define BLANX " \t"
#define SPACE ' '
#define DEFAULT_BOX_CHARACTER '-'
#define MAXLINE 150

extern char *optarg;
extern int optind;

/*============================================================================*/
/*                             VARIABLE DECLARATIONS                          */
/*============================================================================*/

char anchored = 0; /* If non-zero, comment is anchored to 1st column of line */

/*============================================================================*/
/*                             PROGRAM PROTOTYPES                             */
/*============================================================================*/

#ifdef NEED_GETOPT_PROTO_TYPE
int   getopt( int, char **, char * );
#endif

/*============================================================================*/
/*================================= PROGRAMS =================================*/
/*============================================================================*/

static int start_anchored_comment( 

    char * comment_marker, /* comment character(s)                     */
    int    invisibleLen,   /* size of whitspace area preceding comment */
    char **newComment,     /* text around which to place box           */
    int    line_position   /* line index within newComment             */
)
{
    int marker_size = strlen( comment_marker );
    int start_pos   = 0;
    int i;
    
    printf( "%s", comment_marker );

    /*---------------------------------------------------------------
    * Account for extra characters introduced by comment marker.
    * Skip over as many spaces as there are characters in marker.
    * Assuming marker is less than 8 characters, thus encountering
    * a tab character in the short line output indicates we are done.
    *---------------------------------------------------------------*/
    for( i=0; i < marker_size && i < invisibleLen; i++ )
    {
        if( newComment[line_position][i] == '\t' )
            break;

        start_pos++;
    }
    
    return start_pos;
}

/*----------------------------------------------------------------------------*/

static char * get_s( char * line, int limit )
{
    char * what = fgets( line, limit, stdin );
    
    if( what != NULL )
    {
        /*------------------------------------------------
        * Attempt to make this feel like a real gets call.
        * (strtok will wack newline at end of line. If it
        * is the only character in the line, or what is
        * read in exceeds the limit, which means it will
        * not be present at all, then strtok will return
        * with NULL. The line length check will protect
        * the exceedingly long lines, which are truncated.
        *------------------------------------------------*/
        if( strtok( what, "\n" ) == NULL )
            if( strlen( what ) <= 1 )
                *what = EOS;  /* newline was only character on line */
    }
        
    return what;
}

/*----------------------------------------------------------------------------*/

static void box_it_with(

    char   in_boxChar,
    char * comment_begin,
    char * comment_body,
    char * comment_end
)
{
    char *newComment[MAX_LINES];
    char line[MAXLINE];
    char * visible;
    int visibleLen;
    int invisibleLen;
    int longestVisible = 0, longestLine = 0;
    int shortestInvisible = 9999;
    int lineCnt = 0;
    int i, crntLine, shortLine;
    int lineLen;
    char *cbody_sans_space = strdup( comment_body );
    int box_pad = strlen( comment_body ) - strlen( comment_begin );
    
    /*----------------------------------------
    * Whack any trailing space on comment body
    * duplicate. This will be used to construct
    * the ending comment line.
    *----------------------------------------*/
    strtok( cbody_sans_space, BLANX );
    
    if( anchored || box_pad < 0 ) box_pad = 0;

    /*------------------------------------
    * Collect lines from standard input...
    *------------------------------------*/
    while( get_s( line, MAXLINE ) && lineCnt < MAX_LINES )
    {
        /*-----------------------
        * ... save a copy and ...
        *-----------------------*/
        lineLen = strlen(line);
        newComment[lineCnt] = malloc( lineLen+1 );
        memcpy( newComment[lineCnt], line, lineLen+1 );
        
        /*------------------------------------
        * ... locate the extremes of the text.
        *------------------------------------*/
	invisibleLen = strspn( line, BLANX );
	visible      = line + invisibleLen;
	visibleLen   = strlen( visible );
	
        if( lineLen      > longestLine       ) longestLine = lineLen;
        if( invisibleLen < shortestInvisible ) 
        {
            shortestInvisible = invisibleLen;
            shortLine = lineCnt;
        }
        
        lineCnt++;
    }
    
    if( lineCnt )
    {
        int start_pos  = 0;
        
        invisibleLen   = shortestInvisible;
        longestVisible = longestLine - invisibleLen;
	visible        = newComment[0] + invisibleLen;
	visibleLen     = strlen( visible );
	
	/*-----------------------
	* Put out top line of box
	*-----------------------*/
        if( anchored )
        {
            start_pos = 
               start_anchored_comment( 
                  comment_begin, invisibleLen, newComment, shortLine );
        }
        
	for( i = start_pos; i < invisibleLen; i++ )
	    putchar( newComment[shortLine][i] );
	    
        if( ! anchored )
	    printf( "%s", comment_begin );
	
	for( i = 0; i < longestVisible + box_pad; i++ )
	    putchar( in_boxChar );

	putchar( '\n' );

        /*--------------------
        * Put out body of box.
        *--------------------*/
	for( crntLine = 0; crntLine < lineCnt; crntLine++ )
	{
            start_pos = 0;
            
            if( anchored )
            {
                start_pos = 
                   start_anchored_comment( 
                      comment_body, invisibleLen, newComment, crntLine );
            }
                
	    for( i = start_pos; i < shortestInvisible; i++ )
		putchar( newComment[crntLine][i] );
		
            if( ! anchored )
                printf( "%s", comment_body );

	    printf( "%s\n", &newComment[crntLine][shortestInvisible] );
	}
    
        /*----------------------
        * Put out bottom of box.
        *----------------------*/
        start_pos    = 0;
        crntLine     = lineCnt - 1;
        invisibleLen = shortestInvisible;

        if( anchored )
        {
            start_pos = 
               start_anchored_comment( 
                  cbody_sans_space, invisibleLen, newComment, crntLine );
        }

        for( i = start_pos; i < invisibleLen; i++ )
            putchar( newComment[crntLine][i] );
        
        if( ! anchored )
            printf( "%s", cbody_sans_space );
        
        for( i = 0; i < longestVisible + box_pad; i++ )
            putchar( in_boxChar );
        
        if( comment_end != NULL )
        {
            if( anchored && strcmp( comment_begin, comment_end ) != 0 )
                printf( "\n" );

            printf("%s", comment_end );
        }
            
        putchar( '\n' );
    }
}

/*----------------------------------------------------------------------------*/

#define ADJUST_visible( x )  if( x > 0 && (visible - x) >= line )\
                                visible = visible - x;

static void
unbox_it
(
    char   boxChar,
    char * comment_begin,
    char * comment_body,
    char * comment_end
)
{
    char line[MAXLINE];
    char firstTime = 1;
    char * visible;
    char *endComment;
    char *cbody_sans_space = strdup( comment_body );
    int  len_begin = strlen( comment_begin );
    int  len_body  = strlen( comment_body );
    int  len_end   = (comment_end != NULL) ? strlen( comment_end ) : 0;
    int  adj_begin = strspn( comment_begin, BLANX );
    int  adj_body  = strspn( comment_body,  BLANX );
    int  adj_end   = (comment_end == NULL) ? strspn( comment_body, BLANX ) : 0;
    int  len_body_ss;
    
    /*----------------------------------------
    * Whack any trailing space on comment body
    * duplicate. This will be used to search
    * for ending comment line.
    *----------------------------------------*/
    strtok( cbody_sans_space, BLANX );
    len_body_ss = strlen( cbody_sans_space );
    
    while( get_s( line, MAXLINE ) )
    {
	visible = line + strspn( line, BLANX );

        if( firstTime )
        {
            firstTime = 0;
            ADJUST_visible( adj_begin );

            /*----------------------------------------------
            * If the top of the box is found, throw it away.
            *----------------------------------------------*/
            if( strncmp( visible, comment_begin, len_begin ) == 0 )
            {
                if( (int) strlen( visible ) > len_begin )
                {
                    char * cp = visible+len_begin;
                    char   boxCharacter = *cp;
                    
                    if( anchored )
                    {
                        visible = cp = cp + strspn( cp, BLANX );
                        boxCharacter = *visible;
                    }
                    
                    /*-------------------------------------------------------
                    * Try to verify that more than 1 box character in line.
                    * This gives a better confirmation that we found box top.
                    *-------------------------------------------------------*/
                    for( cp++; *cp != EOS && *cp == boxCharacter; cp++ );
                    
                    if( cp - ( visible+len_begin ) > 2 )
		    {
                        /*------------------------------
                        * Supercede given box character 
                        * with what was actually found.
                        *------------------------------*/
                        boxChar = boxCharacter;
			continue;
		    }
                }
            }
        }
        
        /*--------------------------------------------------
        * Look for bottom line of box in two different ways,
        * from the front of the line...
        *--------------------------------------------------*/
        if( comment_end == NULL )
        {
            ADJUST_visible( adj_end );

            if( strncmp( visible, cbody_sans_space, len_body_ss ) == 0 )
            {
                char * cp = visible+len_body_ss;
                
                if( anchored )
                    cp = cp + strspn( cp, BLANX );
                
                if( *cp == boxChar )
                    continue;
            }
        }
        
        /*---------------------------------
        * ... or from the back of the line.
        *---------------------------------*/
        else
        {
            int    ce_size = strlen( comment_end );
            char * ecp     = comment_end + ce_size -1;
            
            /*---------------------------------------
            * Traverse the end of the line, backward,
            * attempting to match the line characters
            * with the comment ending characters.
            *---------------------------------------*/
            if( (endComment = strrchr( line, *ecp )) != NULL )
            {
                while( ce_size > 0 && *endComment == *ecp )
                {
                    endComment--;
                    ecp--;
                    ce_size--;
                }
            }
            
            /*-------------------------------
            * If comment end found, eat line.
            *-------------------------------*/
            if( ce_size == 0 )
                continue;
        }
        
        ADJUST_visible( adj_body );

        /*--------------------------------------------------------
        * If a box body line is found, strip out leading box edge.
        *--------------------------------------------------------*/
        if ( strncmp( visible, comment_body, len_body ) == 0)
        {
            *visible = '\0';
            visible = visible + len_body;
            
            if( anchored )
            {
                if( *visible != '\t' )
                    printf( "%*s%s\n", len_body, "", visible );
            }
            else
                printf( "%s%s\n", line, visible );
        }

        else
            printf( "%s\n", line );  /* print out unaltered line */
    }
}

/*----------------------------------------------------------------------------*/

static char *
remove_file_path
(
    char * inp_file_path
)
{
    char * file_name = strrchr( inp_file_path, '/' );
    
    return ( file_name ) ? file_name+1 : inp_file_path;
}
 
/*----------------------------------------------------------------------------*/

int main( int argc, char * argv[] )
{
    char   boxChar       = DEFAULT_BOX_CHARACTER;
    char * program_nm    = remove_file_path( *argv );
    char * comment_begin = getenv( "BOX_COMMENT_BEGIN" );
    char * comment_end   = getenv( "BOX_COMMENT_END" );
    char * comment_body  = getenv( "BOX_COMMENT_BODY" );
    char   cmnt_body_buf[50]; 
    char   c;   
    
    anchored = ( getenv( "BOX_COMMENT_ANCHORED" ) ) ? 1 : 0;
    
    while( ( c = getopt( argc, argv, "ab:e:s:" )) != -1 )
    {
        switch( c )
        {
          case 'a':   /* comment anchored to 1st character position in line */
            anchored = 1;
            break;
            
          case 'b':   /* comment body leader characters */
            comment_body = optarg;
            break;
        
          case 'e':   /* comment end characters */
            comment_end = optarg;
            break;
          
          case 's':   /* comment starting characters */
            comment_begin = optarg;
            break;
          
          default:
            printf( "\nUsage: %s [options] [box_character]\n", program_nm );
            printf( "\n" );
            printf( "  where options are:\n" );
            printf( "\n" );
            printf( "    -s<tarting comment characters>\n" );
            printf( "    -b<ody comment leader characters>\n" );
            printf( "    -e<nding comment characters>\n" );
            printf( "    -a tells program that comment characters must\n" );
            printf( "       start in first character column in a line.\n" );
            printf( "\n" );
            printf( "  The following environment variables can also be used\n" );
            printf( "  to affect the results of the program.\n" );
            printf( "\n" );
            printf( "    BOX_COMMENT_BEGIN    (same as -s)\n" );
            printf( "    BOX_COMMENT_BODY     (same as -b)\n" );
            printf( "    BOX_COMMENT_END      (same as -e)\n" );
            printf( "    BOX_COMMENT_ANCHORED (same as -a)\n" );
            printf( "\n" );
            printf( "  Program options will override the environment variables.\n" );
            printf( "  The default settings for the program are for 'C' language comments.\n" );
            printf( "  The default box_character is '%c'.\n", DEFAULT_BOX_CHARACTER );
            printf( "\n" );
            printf( "  When the first letter of the program name begins with\n" );
            printf( "  'b' or 'B', the standard input is sent to standard\n" );
            printf( "  output with an open ended comment 'box' around it.\n" );
            printf( "  Otherwise, the program attempts to strip off a bounding\n" );
            printf( "  box from the given text.\n" );
            printf( "\n" );
            printf( "  Hint: By supplying only the starting comment character,\n" );
            printf( "        the program assumes that a newline terminates\n" );
            printf( "        the comment (Good for shell programming).\n" );
            printf( "\n" );
            exit( 1 );
            break;
        }
    }
    /*-------------------------------------------
    * If invalid comment end given, don't use it.
    *-------------------------------------------*/
    if( comment_end != NULL && *comment_end == EOS )
        comment_end = NULL;
        
    /*--------------------------------------
    * Default to 'C' language comment style.
    *--------------------------------------*/
    if( comment_begin == NULL || *comment_begin == EOS )
    {
        comment_begin = "/*";
        comment_body  = "* ";
        comment_end   = "*/";
    }
    
    /*-----------------------------------------------
    * When comment body not provided, use comment 
    * begin sequence with a padded space.
    * This is meant for single line comments which
    * are terminated by a newline.
    *-----------------------------------------------*/
    else if( comment_body == NULL || *comment_body == EOS )
    {
        *cmnt_body_buf = EOS;
        strcpy( cmnt_body_buf, comment_begin );
        strcat( cmnt_body_buf, " " );
        comment_body = cmnt_body_buf;
    }
    
    /*------------------------------
    * Enclosing box character given?
    *------------------------------*/
    if( optind < argc )
        boxChar = argv[optind][0];
        
    if( *program_nm == 'b' || *program_nm == 'B' )
        box_it_with( boxChar, comment_begin, comment_body, comment_end );
    else
        unbox_it( boxChar, comment_begin, comment_body, comment_end );

    return 0;
}
