/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: vorbis_codec_info.cpp,v 1.3.4.2 2005/01/21 19:30:13 nhart Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
#include "vorbis_codec_info.h"
#include "hlxclib/memory.h"

#include "debug.h"
#include "hxassert.h"

#define D_VORBIS_INFO 0 //0x400000

const unsigned char CVorbisCodecInfo::zm_blankComment[] = {
    0x03, 'v', 'o', 'r', 'b', 'i', 's',
    0x00, 0x00, 0x00, 0x00, // vender string length
    0x00, 0x00, 0x00, 0x00, // comment count
    0x01                    // End marker
};

CVorbisCodecInfo::CVorbisCodecInfo() :
    m_state(Start),
    m_pIdentBuf(NULL),
    m_uIdentBufSize(0),
    m_currentPos(0),
    m_basePos(0),
    m_dropped(0),
    m_lastBlockSize(0),
    m_uSampleRate(0)
{
#ifdef _DEBUG
    debug_level() |= D_VORBIS_INFO;
#endif // _DEBUG
    DPRINTF(D_VORBIS_INFO, ("CVorbisCodecInfo::CVorbisCodecInfo()\n"));
    
    vorbis_mode_info_init(&m_vmi);
}

CVorbisCodecInfo::~CVorbisCodecInfo()
{
    DPRINTF(D_VORBIS_INFO, ("CVorbisCodecInfo::~CVorbisCodecInfo()\n"));

    HX_VECTOR_DELETE(m_pIdentBuf);
    vorbis_mode_info_clear(&m_vmi);
}

COggCodecInfo* 
CVorbisCodecInfo::Build(ogg_packet* pPkt)
{
    COggCodecInfo* pRet = NULL;

    if (isVorbisHeader(pPkt) &&   // Is it a Vorbis header?
        (pPkt->packet[0] == 0x01)) // Is it the ident header?
    {
        pRet = new CVorbisCodecInfo;
    }

    return pRet;
}

// COggCodecInfo methods
HX_RESULT CVorbisCodecInfo::OnPacket(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_INVALID_PARAMETER;

    if (pPkt)
    {
        DPRINTF(D_VORBIS_INFO, ("CVorbisCodecInfo::OnPacket() : size %u\n",
                                pPkt->bytes));
        switch(m_state) {
        case Start:
            res = handleIdentHdr(pPkt);
                
            if (HXR_OK == res)
            {
                changeState(NeedComment);
            }
            break;
        case NeedComment:
            if (IsHeader(pPkt) && (pPkt->packet[0] == 0x03)  &&
                vorbis_mode_info_headerin(&m_vmi, pPkt))
            {
                changeState(NeedCodebook);
                res = HXR_OK;
            }
            break;
        case NeedCodebook:
            if (IsHeader(pPkt) && (pPkt->packet[0] == 0x05) &&
                vorbis_mode_info_headerin(&m_vmi, pPkt))
            {
                changeState(NeedFirstDataPkt);
            }
            break;
        case NeedFirstDataPkt:
            if (!IsHeader(pPkt))
            {
                changeState(NeedBaseTime);
            }
            else
            {
                // we don't want to pass through
                // if this isn't a data packet
                break;
            }
            // pass thru intentional
        case NeedBaseTime:
            res = updatePosition(pPkt);

            if ((HXR_OK == res) && pPkt->granulepos != -1)
            {
                if (pPkt->granulepos >= m_currentPos)
                {
                    m_basePos = GranulePosDifference(pPkt->granulepos,
                                                     m_currentPos);
                }
                else
                {
                    m_basePos = 0;
                    m_dropped = GranulePosDifference(m_currentPos,
                                                     pPkt->granulepos);
                }

                changeState(Ready);
            }
            break;
        case Ready:
            if (pPkt->packet[0] & 0x01)
            {
                // header packet
                res = HXR_OK;
            }
            else
            {
                // data packet
                res = updatePosition(pPkt);
            }
            break;
        default:
            break;
        };
    }

    return res;
}

OggCodecType CVorbisCodecInfo::Type() const
{
    return ctVorbis;
}

BOOL CVorbisCodecInfo::HaveHeaders() const
{
    return ((NeedBaseTime == m_state) ||
            (Ready == m_state));
}

BOOL CVorbisCodecInfo::HaveBaseGranulePos() const
{
    return (Ready == m_state);
}

BOOL CVorbisCodecInfo::AllowPacketOutError() const
{
    return (m_state == NeedFirstDataPkt) ? TRUE : FALSE;
}

ogg_int64_t CVorbisCodecInfo::CurrentGranulePos() const
{
    return m_currentPos;
}

BOOL 
CVorbisCodecInfo::IsHeader(ogg_packet* pPkt) const
{
    return isVorbisHeader(pPkt);
}

BOOL CVorbisCodecInfo::IsCommentHeader(ogg_packet* pPkt) const
{
    return (IsHeader(pPkt) && (pPkt->packet[0] == 0x3));
}

ogg_packet* CVorbisCodecInfo::CreateBlankCommentPacket() const
{
    ogg_packet* pRet = new ogg_packet;

    if (pRet)
    {
        pRet->b_o_s = 0;
        pRet->e_o_s = 0;
        pRet->granulepos = 0;
        pRet->packetno = 1;
        pRet->bytes = sizeof(zm_blankComment);
        pRet->packet = new unsigned char[pRet->bytes];

        if (pRet->packet)
        {
            memcpy(pRet->packet, zm_blankComment, pRet->bytes);
        }
        else
        {
            delete pRet;
            pRet = NULL;
        }

    }

    return pRet;
}

HX_RESULT 
CVorbisCodecInfo::GetTimestamp(ogg_int64_t granulePos, 
                               COggTimestamp& timestamp) const
{
    HX_RESULT res = HXR_UNEXPECTED;

    HX_ASSERT(Ready == m_state);

    if (Ready == m_state)
    {
        if (granulePos == 0)
        {
            timestamp = COggTimestamp(0, m_uSampleRate);
            timestamp += m_startTimestamp;

            res = HXR_OK;
        }
        else if (granulePos >= m_basePos)
        {
            timestamp = COggTimestamp(granulePos - m_basePos, m_uSampleRate);
            timestamp += m_startTimestamp;

            res = HXR_OK;
        }
    }

    return res;
}


HX_RESULT 
CVorbisCodecInfo::GetBaseTimestamp(COggTimestamp& timestamp) const
{
    timestamp = COggTimestamp(m_basePos, m_uSampleRate);
    return HXR_OK;
}

HX_RESULT 
CVorbisCodecInfo::SetBaseTimestamp(const COggTimestamp& timestamp)
{
    if (timestamp.SampleRate() != m_uSampleRate)
    {
        COggTimestamp tmp = timestamp;
        tmp.SetSampleRate(m_uSampleRate);
        
        m_basePos = tmp.Samples();
    }
    else
    {
        m_basePos = timestamp.Samples();
    }

    return HXR_OK;
}

HX_RESULT 
CVorbisCodecInfo::SetStartTimestamp(const COggTimestamp& timestamp)
{
    m_startTimestamp = timestamp;
    m_startTimestamp.SetSampleRate(m_uSampleRate);

    return HXR_OK;
}

HX_RESULT 
CVorbisCodecInfo::SetCurrentGranulePos(ogg_int64_t pos)
{
    HX_RESULT res = HXR_UNEXPECTED;

    HX_ASSERT(Ready == m_state);

    if (Ready == m_state)
    {
        m_currentPos = pos;
        m_lastBlockSize = 0;
        res = HXR_OK;
    }

    return res;
}

ogg_int64_t CVorbisCodecInfo::GranulePosDropped() const
{
    return m_dropped;
}

ogg_int64_t CVorbisCodecInfo::GranulePosDifference(ogg_int64_t a, 
                                                   ogg_int64_t b) const
{
    return (a - b);
}

COggCodecInfo* CVorbisCodecInfo::Clone() const
{
    COggCodecInfo* pRet = NULL;

    if (Ready == m_state)
    {
        CVorbisCodecInfo* pInfo = new CVorbisCodecInfo;

        if (pInfo)
        {
            if (HXR_OK == pInfo->copy(*this))
            {
                pRet = pInfo;
            }
            else
            {
                delete pInfo;
                pInfo = NULL;
            }
        }
    }

    return pRet;
}

HX_RESULT 
CVorbisCodecInfo::copy(const CVorbisCodecInfo& rhs)
{
    HX_RESULT res = HXR_OK;
    
    delete [] m_pIdentBuf;
    m_pIdentBuf = NULL;
    
    vorbis_mode_info_clear(&m_vmi);
    
    m_state = rhs.m_state;
    m_uIdentBufSize = rhs.m_uIdentBufSize;
    m_currentPos = rhs.m_currentPos;
    m_basePos = rhs.m_basePos;
    m_dropped = rhs.m_dropped;
    m_startTimestamp = rhs.m_startTimestamp;

    m_lastBlockSize = rhs.m_lastBlockSize;
    m_uSampleRate = rhs.m_uSampleRate;

    if (m_uIdentBufSize && rhs.m_pIdentBuf)
    {
        m_pIdentBuf = new UINT8[m_uIdentBufSize];
        if (m_pIdentBuf)
        {
            memcpy(m_pIdentBuf, rhs.m_pIdentBuf, m_uIdentBufSize);
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
    }

    if ((HXR_OK == res) && 
        !vorbis_mode_info_copy(&m_vmi, &rhs.m_vmi))
    {
        res = HXR_OUTOFMEMORY;
    }

    return res;
}

static const char* z_pStateNames[] = {
    "Start",
    "NeedComment",
    "NeedCodebook",
    "NeedFirstDataPkt",
    "NeedBaseTime",
    "Ready"
};

void CVorbisCodecInfo::changeState(State newState)
{
    DPRINTF(D_VORBIS_INFO, ("CVorbisCodecInfo::changeState() : %s -> %s\n",
                            z_pStateNames[m_state],
                            z_pStateNames[newState]));
    m_state = newState;
}

HX_RESULT CVorbisCodecInfo::handleIdentHdr(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_INVALID_PARAMETER;

    if (IsHeader(pPkt) && 
        (pPkt->packet[0] == 0x01))
    {
        HX_VECTOR_DELETE(m_pIdentBuf);
        m_pIdentBuf = new UINT8[pPkt->bytes];

        if (m_pIdentBuf)
        {
            memcpy(m_pIdentBuf, pPkt->packet, pPkt->bytes);
            m_uIdentBufSize = pPkt->bytes;

            if (vorbis_mode_info_headerin(&m_vmi, pPkt))
            {
                m_uSampleRate = (pPkt->packet[12] |
                                 (pPkt->packet[13] << 8) |
                                 (pPkt->packet[14] << 16) |
                                 (pPkt->packet[15] << 24));

                m_startTimestamp.SetSampleRate(m_uSampleRate);

                DPRINTF(D_VORBIS_INFO, 
                        ("CVorbisCodecInfo::handleIdentHdr : sampleRate %u\n",
                         m_uSampleRate));
                DPRINTF(D_VORBIS_INFO, 
                        ("CVorbisCodecInfo::handleIdentHdr : blockSize 0 %u\n",
                         m_vmi.blocksizes[0]));
                DPRINTF(D_VORBIS_INFO, 
                        ("CVorbisCodecInfo::handleIdentHdr : blockSize 1 %u\n",
                         m_vmi.blocksizes[1]));
                                             
                res = HXR_OK;
            }
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
    }

    return res;
}

HX_RESULT CVorbisCodecInfo::updatePosition(ogg_packet* pPkt)
{
    HX_RESULT res = HXR_INVALID_PARAMETER;

    if (pPkt)
    {
        int currentSize = vorbis_mode_info_getblocksize(&m_vmi, pPkt);

        if (m_lastBlockSize)
        {
            int outputSize = 
                currentSize / 4 + m_lastBlockSize / 4;
            
            m_currentPos += outputSize;
        }
        m_lastBlockSize = currentSize;

        res = HXR_OK;
    }

    return res;
}

BOOL CVorbisCodecInfo::isVorbisHeader(ogg_packet* pPkt)
{
    BOOL bRet = FALSE;

    if (pPkt && pPkt->packet && (pPkt->bytes > 7) &&
        ((pPkt->packet[0] & 0x1) == 0x1) &&
        !memcmp(pPkt->packet + 1, "vorbis", 6))
        
    {
        bRet = TRUE;
    }

    return bRet;
}

