/*****************************************************************

Copyright (c) 2001 Matthias Elter <elter@kde.org>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <math.h>

#include <qapplication.h>
#include <qdesktopwidget.h>
#include <qlayout.h>
#include <qstringlist.h>

#include <dcopclient.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kglobalaccel.h>
#include <klocale.h>
#include <kstandarddirs.h>

#include "kickerSettings.h"
#include "taskbarsettings.h"
#include "taskcontainer.h"
#include "taskmanager.h"

#include "taskbar.h"
#include "taskbar.moc"

TaskBar::TaskBar( QWidget *parent, const char *name )
    : Panner( parent, name ),
      m_currentScreen(-1),
      m_showOnlyCurrentScreen(false)
{
    setBackgroundOrigin( AncestorOrigin );

    setFrameStyle( NoFrame );

    arrowType = LeftArrow;
    blocklayout = true;

    // init
    setMinimumSize( WINDOWLISTBUTTON_SIZE, WINDOWLISTBUTTON_SIZE );
    setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding ) );
    containers.setAutoDelete( false );

    // setup animation frames
    frames = new PixmapList();

    frames->setAutoDelete( true );

    for ( int i=1; i<11; i++ )
    {
        frames->append( new QPixmap( locate("data", "kicker/pics/disk" + QString::number(i) + ".png") ) );
    }

    // configure
    configure();

    // connect manager
    connect(TaskManager::the(), SIGNAL(taskAdded(Task::Ptr)), SLOT(add(Task::Ptr)));
    connect(TaskManager::the(), SIGNAL(taskRemoved(Task::Ptr)), SLOT(remove(Task::Ptr)));
    connect(TaskManager::the(), SIGNAL(startupAdded(Startup::Ptr)), SLOT(add(Startup::Ptr)));
    connect(TaskManager::the(), SIGNAL(startupRemoved(Startup::Ptr)), SLOT(remove(Startup::Ptr)));
    connect(TaskManager::the(), SIGNAL(desktopChanged(int)), SLOT(desktopChanged(int)));
    connect(TaskManager::the(), SIGNAL(windowChanged(WId)), SLOT(windowChanged(WId)));

    isGrouping = shouldGroup();

    // register existant tasks
    {
        TaskList tasks = TaskManager::the()->tasks();
        TaskList::iterator itEnd = tasks.end();
        for (TaskList::iterator it = tasks.begin(); it != itEnd; ++it)
        {
            add( *it );
        }
    }

    // register existant startups
    {
        StartupList startups = TaskManager::the()->startups();
        StartupList::iterator itEnd = startups.end();
        for (StartupList::iterator it = startups.begin(); it != itEnd; ++it)
        {
            add( *it );
        }
    }

    blocklayout = false;

    connect(kapp, SIGNAL(settingsChanged(int)), SLOT(slotSettingsChanged(int)));
    keys = new KGlobalAccel( this );
#include "taskbarbindings.cpp"
    keys->readSettings();
    keys->updateConnections();
}

TaskBar::~TaskBar()
{
    delete frames;
}

QSize TaskBar::sizeHint() const
{
    return QSize(BUTTON_MIN_WIDTH, TaskBarSettings::minimumButtonHeight());
}

QSize TaskBar::sizeHint( KPanelExtension::Position p, QSize maxSize) const
{
    if ( p == KPanelExtension::Left || p == KPanelExtension::Right )
    {
        int actualMax = BUTTON_HEIGHT * containerCount();

        if (actualMax > maxSize.height())
        {
            return maxSize;
        }
        else
        {
            return QSize( maxSize.width(), actualMax );
        }
    }
    else
    {
        int rows = KickerSettings::conserveSpace() ?
                   contentsRect().height() / TaskBarSettings::minimumButtonHeight() :
                   1;
        if ( rows < 1 )
        {
            rows = 1;
        }

        int actualMax = TaskBarSettings::maximumButtonWidth() * (containerCount() / rows);
        if (containerCount() % rows > 0)
        {
            actualMax += TaskBarSettings::maximumButtonWidth();
        }

        if (actualMax > maxSize.width())
        {
           return maxSize;
        }
        else
        {
           return QSize( actualMax, maxSize.height() );
        }
    }
}

// These are the strings that are actually stored in the config file.
// They must be synchronized with kcmtaskbar. They are not translated.
const QStringList& TaskBar::actionList()
{
    static QStringList list(
            QStringList() << "Show Task List" << "Show Operations Menu"
            << "Activate, Raise or Minimize Task"
            << "Activate Task" << "Raise Task"
            << "Lower Task" << "Minimize Task" );
    return list;
}

// Translate from config entry name to enumeration
enum TaskBar::Action TaskBar::buttonAction( ButtonState button, const QString& actionName )
{
   int index = actionList().findIndex( actionName );
   if( index != -1 ) return static_cast<Action>(index);

   // Otherwise return the default.
   switch( button ) {
   case MidButton: return ActivateRaiseOrIconify;
   case RightButton: return ShowOperationsMenu;
   case LeftButton: // fall through
   default: return ShowTaskList;
   }
}

// These are the strings that are actually stored in the config file.
const QStringList& TaskBar::groupModeList()
{
    static QStringList list(
            QStringList() << I18N_NOOP("Never") << I18N_NOOP("When Taskbar Full")
            << I18N_NOOP("Always"));
    return list;
}

// Translate from config entry name to enumeration
enum TaskBar::GroupMode TaskBar::groupMode(const QString& groupModeName )
{
   int index = groupModeList().findIndex( groupModeName );
   if( index != -1 ) return static_cast<GroupMode>(index);

   // Translate old entries
   if( groupModeName == "true" ) return GroupAlways;
   if( groupModeName == "false" ) return GroupNever;

   // Otherwise return the default.
   return GroupWhenFull;
}

void TaskBar::configure()
{
    bool wasShowWindows = m_showAllWindows;
    bool wasSortByDesktop = m_sortByDesktop;
    bool wasShowIcon = m_showIcon;

    m_showAllWindows = TaskBarSettings::showAllWindows();
    m_sortByDesktop = TaskBarSettings::sortByDesktop();
    m_showIcon = TaskBarSettings::showIcon();

    m_currentScreen = -1;    // Show all screens or re-get our screen
    m_showOnlyCurrentScreen = TaskBarSettings::showCurrentScreenOnly() &&
                              QApplication::desktop()->isVirtualDesktop() &&
                              QApplication::desktop()->numScreens() > 1;

    // we need to watch geometry issues if we aren't showing windows when we
    // are paying attention to the current Xinerama screen
    if (m_showOnlyCurrentScreen)
    {
        // disconnect first in case we've been here before
        // to avoid multiple connections
        disconnect(TaskManager::the(), SIGNAL(windowChangedGeometry(WId)),
                    this, SLOT( windowChangedGeometry( WId ) ) );
        connect(TaskManager::the(), SIGNAL(windowChangedGeometry(WId)),
                 this, SLOT( windowChangedGeometry( WId ) ) );
    }

    if (wasShowWindows != m_showAllWindows ||
        wasSortByDesktop != m_sortByDesktop ||
        wasShowIcon != m_showIcon)
    {
        // relevant settings changed, update our task containers
        for ( TaskContainerIterator it(containers); it.current(); ++it )
        {
            it.current()->settingsChanged();
        }
    }
    Action leftButtonAction = buttonAction(LeftButton, TaskBarSettings::leftButtonAction());
    Action middleButtonAction = buttonAction(MidButton, TaskBarSettings::middleButtonAction());
    Action rightButtonAction = buttonAction(RightButton, TaskBarSettings::rightButtonAction());

    TaskContainer::setLeftButtonAction( leftButtonAction );
    TaskContainer::setMiddleButtonAction( middleButtonAction );
    TaskContainer::setRightButtonAction( rightButtonAction );

    if ( !blocklayout )
        reLayout();
}

void TaskBar::setOrientation( Orientation o )
{
    Panner::setOrientation( o );
    reLayout();
}

void TaskBar::resizeEvent( QResizeEvent* e )
{
    if (m_showOnlyCurrentScreen)
    {
        QPoint topLeft = mapToGlobal(this->geometry().topLeft());
        if (m_currentScreen != QApplication::desktop()->screenNumber(topLeft))
        {
            // we have been moved to another screen!
            m_currentScreen = -1;
            reGroup();
        }
    }

    Panner::resizeEvent( e );

    if ( !blocklayout )
    {
        reLayout();
    }
}

void TaskBar::add(Task::Ptr task)
{
    if (!task ||
        (m_showOnlyCurrentScreen && !TaskManager::isOnScreen(showScreen(), task->window())))
    {
        return;
    }

    // try to group
    if ( isGrouping )
    {
        for ( TaskContainerIterator it(containers); it.current(); ++it )
        {
            TaskContainer* c = it.current();

            if (idMatch( task->classClass().lower(), c->id().lower() ))
            {
                c->add(task);

                if (!blocklayout)
                {
                    reLayout();
                }

                return;
            }
        }
    }

    // create new container
    TaskContainer *c = new TaskContainer( task, this, viewport() );
    addChild( c );
    containers.append( c );

    if ( !blocklayout )
    {
        emit containerCountChanged();
        reLayout();
    }
}

void TaskBar::add( Startup::Ptr startup )
{
    if ( !startup ) return;

    // try to group
    if ( isGrouping )
    {
        for ( TaskContainerIterator it(containers); it.current(); ++it )
        {
            TaskContainer* c = it.current();
            if ( idMatch( startup->text().lower(), c->id().lower() ) )
            {
                c->add( startup );

                if ( !blocklayout )
                {
                    reLayout();
                }

                return;
            }
        }
    }

    // create new container
    TaskContainer *c = new TaskContainer( startup, frames, this, viewport() );
    addChild( c );
    containers.append( c );

    if (!blocklayout)
    {
        emit containerCountChanged();
        reLayout();
    }
}

void TaskBar::remove(Task::Ptr task, TaskContainer* container)
{
    if (!container)
    {
        TaskContainerIterator it(containers);
        for (; it.current(); ++it)
        {
            if (it.current()->contains(task))
            {
                break;
            }
        }

        container = it.current();

        if (!container)
        {
            return;
        }
    }

    container->remove(task);

    if (container->isEmpty())
    {
        containers.removeRef(container);
        container->deleteLater();

        if (!blocklayout)
        {
            emit containerCountChanged();
            reLayout();
        }
    }
    else if (!blocklayout && container->filteredTaskCount() < 1)
    {
        emit containerCountChanged();
        reLayout();
    }
}

void TaskBar::remove(Startup::Ptr startup, TaskContainer* container)
{
    if (!container)
    {
        TaskContainerIterator it(containers);
        for (; it.current(); ++it)
        {
            if (it.current()->contains(startup))
            {
                break;
            }
        }

        container = it.current();

        if (!container)
        {
            return;
        }
    }

    container->remove(startup);

    if (container->isEmpty())
    {
        containers.removeRef(container);
        container->deleteLater();

        if (!blocklayout)
        {
            emit containerCountChanged();
            reLayout();
        }
    }
}

void TaskBar::desktopChanged( int desktop )
{
    if ( !m_showAllWindows )
    {
        for ( TaskContainerIterator it(containers); it.current(); ++it )
        {
            TaskContainer* c = it.current();
            c->desktopChanged( desktop );
        }
        emit containerCountChanged();
        reLayout();
    }
}

void TaskBar::windowChanged( WId win )
{
    if (m_showOnlyCurrentScreen &&
        !TaskManager::isOnScreen(showScreen(), win))
    {
        return; // we don't care about this window
    }

    TaskContainer* container = 0;
    for (TaskContainerIterator it(containers); it.current(); ++it)
    {
        TaskContainer* c = it.current();
        if (c->contains(win))
        {
            container = c;
            break;
        }
    }

    if (!container)
    {
        return;
    }

    container->windowChanged(win);

    if (m_showAllWindows && !container->onCurrentDesktop() ||
        TaskBarSettings::showOnlyIconified() && container->isIconified())
    {
        return; // we still don't care about this window
    }

    if (!m_showAllWindows)
    {
        emit containerCountChanged();
    }

    reLayout();
}

void TaskBar::windowChangedGeometry( WId win)
{
    //TODO: this gets called every time a window's geom changes
    //      when we are in "show only on the same Xinerama screen"
    //      mode it would be Good(tm) to compress these events so this
    //      gets run less often, but always when needed
    TaskContainer* container = 0;
    for (TaskContainerIterator it(containers); it.current(); ++it)
    {
        TaskContainer* c = it.current();
        if (c->contains( win ))
        {
            container = c;
            break;
        }
    }

    if ((!!container) == TaskManager::isOnScreen(showScreen(), win))
    {
        // we have this window covered, so we don't need to do anything
        return;
    }

    // we don't have this window! let's find it and add it
    // TODO: would be nice to have a map of winIDs -> tasks!
    //       might make this faster?
    TaskList tasks = TaskManager::the()->tasks();
    TaskList::iterator itLast = tasks.end();
    for (TaskList::iterator it = tasks.begin(); it != itLast; ++it)
    {
        if (win == (*it)->window())
        {
            if (container)
            {
                remove(*it, container);
            }
            else
            {
                add(*it);
            }
            break;
        }
    }
}

void TaskBar::reLayout()
{
    // filter task container list
    TaskContainerList list = filteredContainers();
    if ( list.count() < 1 )
    {
        resizeContents( contentsRect().width(), contentsRect().height() );
        return;
    }

    if ( isGrouping != shouldGroup() )
    {
        reGroup();
        return;
    }

    // sort container list by desktop
    if ( m_sortByDesktop )
        list = sortContainersByDesktop( list );

    // init content size
    resizeContents( contentsRect().width(), contentsRect().height() );

    // horizontal layout
    if ( orientation() == Horizontal )
    {
        int bwidth = BUTTON_MIN_WIDTH;

        // number of rows simply depends on our height
        int rows = contentsRect().height() / TaskBarSettings::minimumButtonHeight();
        if ( rows < 1 ) rows = 1;

        // actual button height
        int bheight = contentsRect().height() / rows;

        // buttons per row
        int bpr = (int)ceil( (double)list.count() / rows);

        // adjust content size
        if ( contentsRect().width() < bpr * BUTTON_MIN_WIDTH )
        {
            resizeContents( bpr * BUTTON_MIN_WIDTH, contentsRect().height() );
        }

        // maximum number of buttons per row
        int mbpr = contentsRect().width() / BUTTON_MIN_WIDTH;

        // expand button width if space permits
        if ( mbpr > bpr )
        {
            bwidth = contentsRect().width() / bpr;
            if (bwidth > TaskBarSettings::maximumButtonWidth())
            {
                bwidth = TaskBarSettings::maximumButtonWidth();
            }
        }

        // layout containers

        // for taskbars at the bottom, we need to ensure that the bottom
        // buttons touch the bottom of the screen. since we layout from
        // top to bottom this means seeing if we have any padding and
        // popping it on the top. this preserves Fitt's Law behaviour
        // for taskbars on the bottom
        int topPadding = 0;
        if (arrowType == UpArrow)
        {
            topPadding = contentsRect().height() % (rows * bheight);
        }

        int i = 0;
        bool reverseLayout = QApplication::reverseLayout();
        for (TaskContainerIterator it(list); it.current(); ++it, i++)
        {
            TaskContainer* c = it.current();

            int row = i % rows;

            c->setArrowType(arrowType);
            c->resize( bwidth, bheight );
            c->show();

            int x = ( i / rows ) * bwidth;
            if (reverseLayout)
            {
                x = contentsRect().width() - x - bwidth;
            }

            moveChild(c, x, (row * bheight) + topPadding);
        }
    }
    else // vertical layout
    {
        // adjust content size
        if ( contentsRect().height() < (int)list.count() * BUTTON_HEIGHT )
        {
            resizeContents( contentsRect().width(), list.count() * BUTTON_HEIGHT );
        }

        // layout containers
        int i = 0;
        for ( TaskContainerIterator it(list); it.current(); ++it )
        {
            TaskContainer* c = it.current();

            c->setArrowType(arrowType);
            c->resize( contentsRect().width(), BUTTON_HEIGHT );
            c->show();

            moveChild( c, 0, i * BUTTON_HEIGHT );
            i++;
        }
    }
    QTimer::singleShot( 100, this, SLOT( publishIconGeometry() ) );
}

void TaskBar::setArrowType(Qt::ArrowType at)
{
    if (arrowType == at)
    {
        return;
    }

    arrowType = at;
    for (TaskContainerIterator it(containers); it.current(); ++it)
    {
        TaskContainer* c = it.current();
        c->setArrowType(arrowType);
    }
}

void TaskBar::publishIconGeometry()
{
    QPoint p = mapToGlobal( QPoint( 0,0 ) ); // roundtrip, don't do that too often

    for ( TaskContainerIterator it(containers); it.current(); ++it )
    {
        TaskContainer* c = it.current();
        c->publishIconGeometry( p );
    }
}

void TaskBar::viewportMousePressEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}

void TaskBar::viewportMouseReleaseEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}

void TaskBar::viewportMouseDoubleClickEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}

void TaskBar::viewportMouseMoveEvent( QMouseEvent* e )
{
    propagateMouseEvent( e );
}

void TaskBar::propagateMouseEvent( QMouseEvent* e )
{
    if ( !isTopLevel()  )
    {
        QMouseEvent me( e->type(), mapTo( topLevelWidget(), e->pos() ),
                        e->globalPos(), e->button(), e->state() );
        QApplication::sendEvent( topLevelWidget(), &me );
    }
}

bool TaskBar::idMatch( const QString& id1, const QString& id2 )
{
    if ( id1.isEmpty() || id2.isEmpty() )
        return false;

    return id1.lower() == id2.lower();
}

int TaskBar::containerCount() const
{
    int i = 0;
    TaskContainerIterator it(containers);
    for ( ; it.current(); ++it ) {
        if ( ( m_showAllWindows ||  (*it)->onCurrentDesktop() ) &&
               ( (showScreen() == -1) || ( (*it)->isOnScreen() ) ) ) {
            i++;
               }
    }
    return i;
}

int TaskBar::taskCount() const
{
    int i = 0;
    TaskContainerIterator it(containers);
    for ( ; it.current(); ++it ) {
        if ( (m_showAllWindows || (*it)->onCurrentDesktop() ) &&
              ( (showScreen() == -1) || ( (*it)->isOnScreen() ) ) ){
            i += (*it)->filteredTaskCount();
              }
    }
    return i;
}

int TaskBar::maximumButtonsWithoutShrinking() const
{
    int rows = contentsRect().height() / TaskBarSettings::minimumButtonHeight();
    if ( rows < 1 ) rows = 1;

    if ( orientation() == Horizontal ) {
       // They squash a bit before they pop, hence the 2
        return rows * (contentsRect().width() / TaskBarSettings::maximumButtonWidth()) + 2;
    } else {
        // Overlap slightly and ugly arrows appear, hence -1
        return rows - 1;
    }
}

bool TaskBar::shouldGroup() const
{
    return groupMode(TaskBarSettings::groupTasks()) == GroupAlways ||
           (groupMode(TaskBarSettings::groupTasks()) == GroupWhenFull &&
            taskCount() > maximumButtonsWithoutShrinking());
}

void TaskBar::reGroup()
{
    isGrouping = shouldGroup();
    blocklayout = true;
    TaskList tasks = TaskManager::the()->tasks();

    TaskContainerList::iterator lastContainer = containers.end();
    for (TaskContainerList::iterator it = containers.begin(); it != lastContainer; ++it )
    {
        (*it)->deleteLater();
    }
    containers.clear();

    {
        TaskList::iterator lastTask = tasks.end();
        for (TaskList::iterator it = tasks.begin(); it != lastTask; ++it)
        {
            if (showScreen() == -1 || (*it)->isOnScreen(showScreen()))
            {
                add(*it);
            }
        }
    }

    {
        StartupList startups = TaskManager::the()->startups();
        StartupList::iterator itEnd = startups.end();
        for (StartupList::iterator it = startups.begin(); it != itEnd; ++it)
        {
            add(*it);
        }
    }

    blocklayout = false;
    reLayout();
}

TaskContainerList TaskBar::filteredContainers()
{
    // filter task container list
    TaskContainerList list;
    for ( TaskContainerIterator it(containers); it.current(); ++it )
    {
        TaskContainer* c = it.current();
        if ((m_showAllWindows || c->onCurrentDesktop()) &&
            (!TaskBarSettings::showOnlyIconified() || c->isIconified()) &&
            ((showScreen() == -1) || (*it)->isOnScreen()))
        {
            list.append( c );
            c->show();
        }
        else
            c->hide();
    }
    return list;
}

void TaskBar::activateNextTask( bool forward )
{
    bool forcenext = false;
    TaskContainerList list = containers;
    if( m_sortByDesktop )
        list = sortContainersByDesktop( list );
    TaskContainerIterator it( list );
    for ( forward ? it.toFirst() : it.toLast(); it.current(); forward ? ++it : --it ) {
	TaskContainer* c = it.current();
	if ( c->activateNextTask( forward, forcenext ) )
            return;
    }

    if (forcenext)
    {
        // moving forward from the last, or backward from the first, loop around
        for (forward ? it.toFirst() : it.toLast(); it.current(); forward ? ++it : --it )
        {
            TaskContainer* c = it.current();
            if (c->activateNextTask(forward, forcenext))
            {
                return;
            }
        }

        return;
    }

    forcenext = true; // select first
    for ( forward ? it.toFirst() : it.toLast(); it.current(); forward ? ++it : --it ) {
        TaskContainer* c = it.current();
        if( m_sortByDesktop )
            if (forward ? c->desktop() < TaskManager::the()->currentDesktop()
                : c->desktop() > TaskManager::the()->currentDesktop())
                continue;
        if ( c->activateNextTask( forward, forcenext ) )
            return;
    }
}

void TaskBar::wheelEvent(QWheelEvent* e)
{
    if (e->delta() > 0)
    {
        // scroll away from user, previous task
        activateNextTask(false);
    }
    else
    {
        // scroll towards user, next task
        activateNextTask(true);
    }
}

void TaskBar::slotActivateNextTask()
{
    activateNextTask( true );
}

void TaskBar::slotActivatePreviousTask()
{
    activateNextTask( false );
}

void TaskBar::slotSettingsChanged( int category )
{
    if( category == (int) KApplication::SETTINGS_SHORTCUTS )
    {
        keys->readSettings();
        keys->updateConnections();
    }
}

int TaskBar::showScreen() const
{
    if (m_showOnlyCurrentScreen && m_currentScreen == -1)
    {
        const_cast<TaskBar*>(this)->m_currentScreen =
            QApplication::desktop()->screenNumber(mapToGlobal(this->geometry().topLeft()));
    }

    return m_currentScreen;
}

TaskContainerList TaskBar::sortContainersByDesktop( TaskContainerList list )
{
    TaskContainerList sorted;
    /* antlarr: look for commentary about "residue" in
      TaskContainer::updateFilteredTaskList() */
    TaskContainerList residue = list;
    for (int desktop = -1; desktop <= TaskManager::the()->numberOfDesktops(); desktop++)
    {
        for ( TaskContainerIterator it(list); it.current(); ++it )
        {
            TaskContainer* c = it.current();
            if (c->desktop() == desktop)
            {
                sorted.append(c);
                residue.remove(c);
            }
        }
    }

    list = sorted;

    TaskContainerList::iterator itEnd = residue.end();
    for (TaskContainerList::iterator it = residue.begin(); it != itEnd; ++it)
    {
        list.append( *it );
    }

    return list;
}

