/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kita_misc.h"
#include "qcp932codec.h"
#include "datinfo.h"  /* struct RESDAT is defined. */
#include "kita-utf8.h"
#include "kita-utf16.h"
#include "kitaconfig.h"

#include <kurl.h>
#include <klocale.h>
#include <kdebug.h>
#include <kdeversion.h>

#include <qregexp.h>
#include <qeucjpcodec.h>
#include <qdir.h>
#include <qmutex.h>
#include <qdatetime.h>

#define KITA_RESDIGIT 4


namespace Kita
{
    static QMutex codecMutex;
    static QCp932Codec* qcpCodec = NULL;
    static QTextCodec* utf8Codec = NULL;
    static QTextCodec* eucCodec = NULL;

    static QString m_weekstr[ 7 ];
    static QString m_colonstr;
    static QString m_colonnamestr;

    /* fro convertURL */
    static int m_prevConvMode;
    static QString m_prevConvURL;
    static QString m_prevConvNewURL;
    static QString m_prevConvRefstr;

    /* for ParseMachiBBSOneLine */
    static QString m_machiSubject;
    static QString m_machiLine;
}


/*---------------------------------------------------*/

/* Text codecs */


QString Kita::qcpToUnicode( const QString& str )
{

    QMutexLocker locker( & Kita::codecMutex ); /* QTextCodec is not reentrant. */

    if ( !Kita::qcpCodec ) Kita::qcpCodec = new QCp932Codec();

    return Kita::qcpCodec->toUnicode( str );
}


QString Kita::utf8ToUnicode( const QString& str )
{

    QMutexLocker locker( & Kita::codecMutex ); /* QTextCodec is not reentrant. */

    if ( !Kita::utf8Codec ) Kita::utf8Codec = QTextCodec::codecForName( "utf8" );

    return Kita::utf8Codec->toUnicode( str );
}


QString Kita::ecuToUnicode( const QString& str )
{
    QMutexLocker locker( & Kita::codecMutex ); /* QTextCodec is not reentrant. */

    if ( !Kita::eucCodec ) Kita::eucCodec = QTextCodec::codecForName( "eucJP" );

    return Kita::eucCodec->toUnicode( str );
}



/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* conversion of DAT */


/* get HTML from raw data   */
QString Kita::DatToHtml( const QString& rawData, int num )
{
    QString retHTML, subject, titleHTML;
    RESDAT resdat;

    resdat.num = num;
    resdat.linestr = rawData;
    resdat.parsed = FALSE;
    parseResDat( resdat, subject );
    createTitleHTML( resdat, titleHTML );

    retHTML = "<div class=\"res_title\">" + titleHTML + "</div>";
    retHTML += "<div class=\"res_body\">" + resdat.bodyHTML + "</div>";

    return retHTML;
}


/* get plain text from raw data   */
/*
   This function replaces "<br>" to "\n", removes HTML tags and
   replaces special chars.
*/
void Kita::DatToText(

    /* input */
    const QString &rawData,

    /* output */
    QString& text
)
{
    text = QString::null;

    unsigned int startPos, pos;
    const QChar *chpt = rawData.unicode();
    unsigned int length = rawData.length();

    for ( unsigned int i = startPos = 0 ; i < length ; i++ ) {

        switch ( chpt[ i ].unicode() ) {

        case '<':

            /* " <br> " */
            if ( chpt[ i + 1 ] == 'b' && chpt[ i + 2 ] == 'r' && chpt[ i + 3 ] == '>' ) {

                unsigned int i2 = i - startPos;
                if ( i > 0 && chpt[ i - 1 ] == ' ' ) i2--; /* remove space before <br> */
                text += rawData.mid( startPos, i2 ) + '\n';
                startPos = i + 4;
                if ( chpt[ startPos ] == ' ' ) startPos++; /* remove space after <br> */
                i = startPos - 1;
            }

            /*----------------------------------------*/

            /* remove HTML tags <[^>]*>  */
            else {

                if ( i - startPos ) text += rawData.mid( startPos, i - startPos );
                while ( chpt[ i ] != '>' && i < length ) i++;
                startPos = i + 1;
            }

            break;


            /*----------------------------------*/

        case '&':

            /* special char */
            {
                QString tmpstr;
                tmpstr = parseSpecialChar( chpt + i, pos );

                if ( tmpstr != QString::null ) {
                    text += rawData.mid( startPos, i - startPos ) + tmpstr;
                    startPos = i + pos;
                    i = startPos - 1;
                }
            }

            break;
        }
    }

    text += rawData.mid( startPos );
}


/* parsing function for special char (such as &hearts; */

/* For example, if cdat = "&amp;", then
 
   pos (= length of cdat) = 5,
   retstr = "&".                           */
QString Kita::parseSpecialChar(

    /* input */
    const QChar *cdat,

    /* output */
    unsigned int& pos )
{
    QString retstr = QString::null;

    if ( ( pos = isEqual( cdat , "&gt;" ) ) ) retstr = ">";
    else if ( ( pos = isEqual( cdat , "&lt;" ) ) ) retstr = "<";
    else if ( ( pos = isEqual( cdat , "&nbsp;" ) ) ) retstr = " ";
    else if ( ( pos = isEqual( cdat , "&amp;" ) ) ) retstr = "&";
    else if ( ( pos = isEqual( cdat , "&quot;" ) ) ) retstr = "\"";

    else if ( ( pos = isEqual( cdat , "&hearts;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_HEART );

    else if ( ( pos = isEqual( cdat , "&diams;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_DIA );

    else if ( ( pos = isEqual( cdat , "&clubs;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_CLUB );

    else if ( ( pos = isEqual( cdat , "&spades;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_SPADE );

    return retstr;
}


/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* conversion of URL */

KURL Kita::getDatURL( const KURL& url , QString& refstr )
{
    return convertURL( URLMODE_DAT, url, refstr );
}

KURL Kita::getDatURL( const KURL& url )
{
    QString refstr;
    return convertURL( URLMODE_DAT, url, refstr );
}

QString Kita::getThreadURL( const KURL& url, QString& refstr )
{
    return convertURL( URLMODE_THREAD, url, refstr );
}

QString Kita::getThreadURL( const KURL& url )
{
    QString refstr;
    return convertURL( URLMODE_THREAD, url, refstr );
}


/* convert thread URL, and get reference.
   If mode = URLMODE_DAT, output is URL of dat file.
   If mode = URLMODE_THREAD, output is URL of read.cgi .
 
   If url is NOT enrolled, return QString::null. 
   
(ex.1)
 
mode = 0
url  = http://pc5.2ch.net/linux/dat/1069738960.dat#20-30
->
return : http://pc5.2ch.net/linux/dat/1069738960.dat
retstr : 20-30
 
(ex.2)
 
mode = 1
url  = http://pc5.2ch.net/linux/dat/1069738960.dat#20-30
->
return : http://pc5.2ch.net/test/read.cgi/linux/1069738960
retstr : 20-30
 
(ex.3)
 
mode = 0
url  = http://pc5.2ch.net/test/read.cgi/linux/1069738960/-100
->
return : http://pc5.2ch.net/linux/dat/1069738960.dat
refstr : 1-100
                                                            */

QString Kita::convertURL(

    /* input */
    int mode,   /* if 0, output is dat URL. If 1, output is thread URL */
    const KURL& url ,

    /* output */
    QString& refstr )
{
    refstr = QString::null;

    if ( url.isEmpty() ) return QString::null;

    /* cache */
    if ( m_prevConvMode == mode && m_prevConvURL == url.prettyURL() ) {

        refstr = m_prevConvRefstr;
        return m_prevConvNewURL;
    }

    /* Is board enrolled ? */
    BoardData* bdata = Kita::BoardManager::getBoardData( url );
    if ( bdata == NULL ) return QString::null;

    QString urlstr = url.prettyURL();

    /* get thread & reference */
    QString thread = QString::null;
    QString refBase = QString::null;;

    if ( urlstr.contains( "/dat/" ) ) {

        /* url = (hostname)/(rootPath)/(bbsPath)/dat/(thread_ID).(ext)#(refBase)  */
        thread = url.filename().remove( bdata->ext() );
        refBase = url.ref();
    } else if ( urlstr.contains( bdata->delimiter() ) ) {

        QString tmpstr;
        switch ( bdata->type() ) {

            /* machi BBS */
            /* ex.) If url = http://kanto.machi.to/bbs/read.pl?BBS=kana&KEY=1096716679 ,
               then, thread = 1096716679                                */
        case Board_MachiBBS:
            thread = url.queryItem( "KEY" );
            refBase = QString::null;
            break;

            /* url = (hostname)/(rootPath)/(delimiter)/(bbsPath)/(thread_ID)/(refBase) */
        default:
            tmpstr = urlstr.section( bdata->delimiter() + bdata->bbsPath(), 1, 1 );
            thread = tmpstr.section( '/', 1, 1 );
            refBase = tmpstr.section( '/', 2, 2 );
            break;
        }
    }

    if ( thread == QString::null ) return QString::null;

    if ( refBase != QString::null ) {

        if ( refBase.at( 0 ) == '-' ) refstr = "1" + refBase;
        else refstr = refBase;
    }

    /* create new URL */
    QString newURL;
    if ( mode == URLMODE_DAT ) newURL = bdata->basePath() + "dat/" + thread + bdata->ext();
    else {
        newURL = bdata->cgiBasePath();

        switch ( bdata->type() ) {

        case Board_MachiBBS:
            newURL += "&KEY=" + thread;
            break;

        default:
            newURL += thread;
            break;
        }
    }

    /* cache */
    m_prevConvMode = mode;
    m_prevConvURL = url.prettyURL();
    m_prevConvNewURL = newURL;
    m_prevConvRefstr = refstr;

    return newURL;
}



/**
 * http://pc5.2ch.net/linux/dat/1089905503.dat
 * -> http://pc5.2ch.net/test/offlaw.cgi?bbs=linux&key=1089905503
 */
QString Kita::datToOfflaw( const KURL& datURL )
{
    /* TODO: not tested. */
    KURL url( datURL );
    QString root = url.host();

    QStringList list = QStringList::split( ".", url.fileName() );
    if ( list.size() != 2 ) {
        return QString::null;
    }
    QString datName = list[ 0 ];

    url.cd( ".." );
    if ( url.fileName() != "dat" ) {
        return QString::null;
    }

    url.cd( ".." );
    QString board = url.fileName();

    return QString( "http://%1/test/offlaw.cgi?raw=0.0&bbs=%2&key=%3" ).arg( root ).arg( board ).arg( datName );
}


/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* utilities */

/* create directory recursively */
bool Kita::mkdir( const QString& targetPath )
{
    QDir qdir( targetPath );
    if ( !qdir.exists() ) {

        QStringList pathList = QStringList::split( "/", targetPath );
        QString path = QString::null;

        for ( unsigned int i = 0; i < pathList.count(); ++i ) {

            path += "/" + pathList[ i ];

            qdir = path;
            if ( !qdir.exists() ) {
                if ( !qdir.mkdir( path ) ) return FALSE;
            }
        }
    }

    return TRUE;
}


QString Kita::unescape( const QString& str )
{
    QString ret = str;
    return ret.replace( "&lt;", "<" ).replace( "&gt;", ">" ).replace( "&amp;", "&" );
}


uint Kita::datToSince( const KURL& datURL )
{
    return KURL( datURL ).fileName().section( '.', 0, 0 ).toInt();
}


/* if cdat == str, return str.length() */
int Kita::isEqual( const QChar *cdat, const QString& str )
{
    int i = 0;
    while ( str.at( i ) != '\0' ) {
        if ( *cdat != str.at( i ) ) return 0;
        cdat++;i++;
    }
    return i;
}


/* convert strings to positive number.  */
/* if cdat is not number, return -1.    */

/*  For example, if cdat = "1234", then
    ret = 1234. If cdat = "abcd", then
    ret = -1.                           */
int Kita::stringToPositiveNum( const QChar *cdat, const unsigned int length )
{
    int ret = 0;

    for ( unsigned int i = 0 ; i < length ; i++ ) {

        unsigned short c = cdat[ i ].unicode();

        if ( ( c < UTF16_0 || c > UTF16_9 ) && ( c < '0' || c > '9' ) ) return -1;

        ret *= 10;
        if ( c >= UTF16_0 ) ret += c - UTF16_0;
        else ret += c - '0';
    }

    return ret;
}



/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* internal parsing functions  */


/* for Machi BBS */


/* init parser. Don't forget to call this before parsing. */
void Kita::InitParseMachiBBS()
{
    m_machiSubject = QString::null;
    m_machiLine = QString::null;
}

QString Kita::ParseMachiBBSOneLine( const QString& inputLine, int& nextNum )
{
    QString ret = QString::null;
    m_machiLine += inputLine;

    int num = 0;
    QString name = QString::null;
    QString mail = QString::null;
    QString date = QString::null;
    QString time = QString::null;
    QString id = QString::null;
    QString host = QString::null;
    QString message = QString::null;

    // Subject
    QRegExp title_regexp( "<title>(.*)</title>" );

    // pattern 1 (tokyo,kanagawa,...)
    QRegExp regexp ( "<dt>(\\d*) .*<font color=\"#......\"><b> (.*) </b></font> .* (..../../..).* (..:..:..) ID:([^<]*)<br><dd>(.*)" );
    QRegExp regexp2( "<dt>(\\d*) .*<a href=\"mailto:(.*)\"><b> (.*) </B></a> .* (..../../..).* (..:..:..) ID:([^<]*)<br><dd>(.*)" );

    // pattern 2 (hokkaido,...)
    QRegExp regexp3( "<dt>(\\d*) .*<font color=\"#......\"><b> (.*) </b></font> .* (..../../..).* (..:..:..) ID:([^<]*) <font size=.>\\[ ([^ ]*) \\]</font><br><dd>(.*)" );
    QRegExp regexp4( "<dt>(\\d*) .*<a href=\"mailto:(.*)\"><b> (.*) </B></a> .* (..../../..).* (..:..:..) ID:([^<]*) <font size=.>\\[ ([^ ]*) \\]</font><br><dd>(.*)" );

    /* abone */
    QRegExp regexp5( "<dt>(\\d*) .*<br><dd>.*" );


    if ( regexp.search( m_machiLine ) != -1 ) {

        num = regexp.cap( 1 ).toInt();
        name = regexp.cap( 2 );
        date = regexp.cap( 3 );
        time = regexp.cap( 4 );
        id = regexp.cap( 5 );
        message = regexp.cap( 6 );

    } else if ( regexp2.search( m_machiLine ) != -1 ) {

        num = regexp2.cap( 1 ).toInt();
        mail = regexp2.cap( 2 );
        name = regexp2.cap( 3 );
        date = regexp2.cap( 4 );
        time = regexp2.cap( 5 );
        id = regexp2.cap( 6 );
        message = regexp2.cap( 7 );

    } else if ( regexp3.search( m_machiLine ) != -1 ) {

        num = regexp3.cap( 1 ).toInt();
        name = regexp3.cap( 2 );
        date = regexp3.cap( 3 );
        time = regexp3.cap( 4 );
        id = regexp3.cap( 5 );
        host = regexp3.cap( 6 );
        message = regexp3.cap( 7 );

    } else if ( regexp4.search( m_machiLine ) != -1 ) {

        num = regexp4.cap( 1 ).toInt();
        mail = regexp4.cap( 2 );
        name = regexp4.cap( 3 );
        date = regexp4.cap( 4 );
        time = regexp4.cap( 5 );
        id = regexp4.cap( 6 );
        host = regexp4.cap( 7 );
        message = regexp4.cap( 8 );

    } else if ( regexp5.search( m_machiLine ) != -1 ) { /* abone */

        num = regexp5.cap( 1 ).toInt();
        m_machiLine = QString::null;
        if ( num == nextNum ) return "abone<><><>abone<>";
        else return QString::null;

    } else if ( title_regexp.search( m_machiLine ) != -1 ) { /* get title */

        m_machiSubject = title_regexp.cap( 1 );
        m_machiLine = QString::null;
        return QString::null;
    }
    else return QString::null;

    if ( num >= nextNum ) {

        if ( num != 1 ) m_machiSubject = QString::null;
        ret += name + "<><>" + date + " " + time + " ID:" + id;
        if ( host != QString::null ) ret += " HOST:" + host;
        ret += "<>" + message + "<>" + m_machiSubject;
        nextNum = num;
    }

    m_machiLine = QString::null;
    return ret;
}


/*---------------------------------*/

/* for JBBS */

QString Kita::ParseJBBSOneLine( const QString& line, int& nextNum )
{
    QString ret = QString::null;
    QStringList list = QStringList::split( "<>", line, true );
    if ( list.size() != 7 ) return QString::null;

    int num = list[ 0 ].toInt();
    QString name = list[ 1 ];
    QString mail = list[ 2 ];
    QString date = list[ 3 ];
    QString body = list[ 4 ];
    QString subject = list[ 5 ];
    QString id = list[ 6 ];

    if ( num < nextNum ) return QString::null;

    /* remove tag */
    QRegExp rex( "<[^<]*>" );
    name.remove( rex );

    /* remove week */
    rex = QRegExp( "\\(.*\\)" );
    date.remove( rex );

    ret += name + "<>" + mail + "<>" + date + " ID:" + id + "<>" + body + "<>" + subject;
    nextNum = num;

    return ret;
}


/*---------------------------------*/

/* for Flash CGI/Mini Thread  */

QString Kita::ParseFlashCGIOneLine( const QString& line )
{
    QString ret = QString::null;
    QStringList list = QStringList::split( "<>", line, true );
    if ( list.size() != 13 ) return QString::null;

    QString name = list[ 0 ];
    QString mail = list[ 1 ];
    QString date = list[ 2 ];
    QString body = list[ 3 ];
    QString subject = list[ 4 ];
    QString id = list[ 6 ];
    QString host = list[ 7 ];

    /* remove tag */
    QRegExp rex( "<[^<]*>" );
    name.remove( rex );

    ret += name + "<>" + mail + "<>" + date + " ID:" + id;
    if ( host != QString::null ) ret += " HOST:" + host;
    ret += "<>" + body + "<>" + subject;

    return ret;
}


/*-------------------------------------------------*/
/*-------------------------------------------------*/

/* parsing functions */

/*-------------------------------------------------*/
/*-------------------------------------------------*/


/* Main Parser */

/*
  struct RESDAT is defined in datinfo.h.
  This function is called from Kita::DatToHtml() and DatInfo::parseDat()
  
  input:
 
  resdat.num     ... number
  resdat.linestr ... raw line strings
 
  output:
  
  resdat.*
  subject
*/
bool Kita::parseResDat( RESDAT& resdat, QString& subject )
{
    if ( resdat.parsed ) return TRUE;

    resdat.parsed = TRUE;
    resdat.broken = FALSE;
    resdat.anclist.clear();

    /* search the staring positions of each section to split raw data. */
    const QChar *chpt = resdat.linestr.unicode();
    unsigned int length = resdat.linestr.length();
    unsigned int section = 0;
    unsigned int sectionPos[ 5 ];
    for ( unsigned int i = 0 ; i < length ; i++ ) {

        /* sections are splitted by "<>" */
        if ( chpt[ i ] == '<' && chpt[ i + 1 ] == '>' ) {
            section++;


            if ( section >= 5 ) {
                resdat.broken = TRUE;
                return TRUE;
            }

            sectionPos[ section ] = i + 2;
            i++;
        }
    }

    /* broken data */
    if ( section != 4 ) {
        resdat.broken = TRUE;
        return TRUE;
    }

    //    qDebug("[%d] %d %d %d %d",section, sectionPos[1],sectionPos[2],sectionPos[3],sectionPos[4] );

    /* name */
    length = sectionPos[ 1 ] - 2 ;
    parseName( resdat.linestr.mid( 0, length ), resdat );

    /* mail */
    length = sectionPos[ 2 ] - 2 - sectionPos[ 1 ];
    DatToText( resdat.linestr.mid( sectionPos[ 1 ], length ), resdat.address );

    /* date, ID, host  */
    length = sectionPos[ 3 ] - 2 - sectionPos[ 2 ];
    parseDateId( resdat.linestr.mid( sectionPos[ 2 ], length ), resdat );

    /* body */
    length = sectionPos[ 4 ] - 2 - sectionPos[ 3 ];
    parseBody( resdat.linestr.mid( sectionPos[ 3 ], length ), resdat );

    /* subject */
    subject = resdat.linestr.mid( sectionPos[ 4 ] );

    return TRUE;
}


/* parse name */

/* output:
  
   resdat.name
   resdat.nameHTML
   
*/
void Kita::parseName( const QString& rawStr, RESDAT& resdat )
{
    unsigned int i = 0, pos;
    int refNum[ 2 ];
    QString linkurl, linkstr;

    DatToText( rawStr, resdat.name );

    const QChar * chpt = resdat.name.unicode();
    unsigned int length = resdat.name.length();
    resdat.nameHTML = QString::null;

    /* anchor */
    while ( parseResAnchor( chpt + i, length - i, linkstr, refNum, pos ) ) {

        linkurl = QString( "#%1" ).arg( refNum[ 0 ] );
        if ( refNum[ 1 ] ) linkurl += QString( "-%1" ).arg( refNum[ 1 ] );

        resdat.nameHTML += "<a href=\"" + linkurl + "\">";
        resdat.nameHTML += linkstr;
        resdat.nameHTML += "</a>";

        ANCNUM anctmp;
        if ( refNum[ 1 ] < refNum[ 0 ] ) refNum[ 1 ] = refNum[ 0 ];
        anctmp.from = refNum[ 0 ];
        anctmp.to = refNum[ 1 ];
        resdat.anclist += anctmp;

        i += pos;
    }

    /* non-digits strings */
    if ( i < length ) {

        resdat.nameHTML += "<span class=\"name_noaddr\">";
        resdat.nameHTML += resdat.name.mid( i );
        resdat.nameHTML += "</span>";
    }

}


/* parse date, ID, host */

/* output :
   
   resdat.dateTime
   resdat.date
   resdat.id
   resdat.host
 
*/
void Kita::parseDateId( const QString& rawStr, RESDAT& resdat )
{
    resdat.date = rawStr;
    resdat.id = QString::null;
    resdat.host = QString::null;
    resdat.be = QString::null;
    resdat.bepointmark = QString::null;

    const QChar *chpt = rawStr.unicode();
    unsigned int pos = 0, startpos = 0;
    unsigned int length = rawStr.length();

    while ( chpt[ pos ] != '\0' &&
            !( chpt[ pos ] == 'I' && chpt[ pos + 1 ] == 'D' ) &&
            !( chpt[ pos ] == 'B' && chpt[ pos + 1 ] == 'E' ) ) {
        pos++;
    }
    resdat.date = rawStr.left( pos );

    /* id */
    if ( chpt[ pos ] == 'I' && chpt[ pos + 1 ] == 'D' ) {
        pos += 3;
        startpos = pos;
        while ( chpt[ pos ] != ' ' && pos++ < length );
        resdat.id = rawStr.mid( startpos, pos - startpos );
        pos++;
    }

    //    qDebug("date %s, ID %s", (const char*)resdat.date.local8Bit(), resdat.id.ascii() );

    if ( pos >= length ) return ;

    /* be */
    if ( chpt[ pos ] == 'B' && chpt[ pos + 1 ] == 'E' ) {
        pos += 3;
        startpos = pos;
        while ( chpt[ pos ] != '-' && pos++ < length );
        resdat.be = rawStr.mid( startpos, pos - startpos );
        pos++;
        if ( pos < length && chpt[ pos ] == '#') {
            startpos = pos;
            while ( chpt[ pos ] == '#' && pos++ < length );
            resdat.bepointmark = rawStr.mid( startpos, pos - startpos );
        }
    }

    if ( pos >= length ) return ;

    /* host */
    if ( chpt[ pos ] == 'H' && chpt[ pos + 1 ] == 'O' ) {
        pos += 5;
        startpos = pos;
        while ( chpt[ pos ] != ' ' && pos++ < length );
        resdat.host = rawStr.mid( startpos, pos - startpos );
        pos++;
        //	qDebug("host %s", resdat.host.ascii());
    }
}



/* parse body */

/* output :
   
   resdat.bodyHTML
 
*/
void Kita::parseBody( const QString &rawStr, RESDAT& resdat )
{
#if KDE_IS_VERSION( 3, 2, 0 )
    resdat.bodyHTML = QString::null;
#else
    /* For KDE3.1    */
    bool showAA = KitaConfig::showAA();
    resdat.bodyHTML = "<div>";
#endif

    unsigned int startPos, pos;
    QString linkstr, linkurl;
    const QChar *chpt = rawStr.unicode();
    unsigned int length = rawStr.length();

    bool ancChain = FALSE;

    /* ancChain is chain for anchor. For examle, if anchor "&gt;2"
       appeared, ancChain is set to TRUE. Moreover, if next strings
       are "=5", anchor for 5 is also set. Thus, we can obtain anchors
       for strings "&gt;2=5" as follows:

       <a href="#2">&gt;2</a><a href="#5">=5</a>
    */

    int offset = 0;
    if ( chpt[ 0 ] == ' ' ) offset = 1; /* remove one space after <> */
    for ( unsigned int i = startPos = offset ; i < length ; i++ ) {

        switch ( chpt[ i ].unicode() ) {

        case '<':

            /* " <br> " */
            if ( chpt[ i + 1 ] == 'b' && chpt[ i + 2 ] == 'r' && chpt[ i + 3 ] == '>' ) {

                /* reset anchor chain */
                ancChain = FALSE;

                unsigned int i2 = i - startPos;
                if ( i > 0 && chpt[ i - 1 ] == ' ' ) i2--; /* remove space before <br> */
                resdat.bodyHTML += rawStr.mid( startPos, i2 );

                resdat.bodyHTML += "<br>";

#if KDE_IS_VERSION( 3, 2, 0 )
#else
                /* show Ascii Art (for KDE3.1*) */
                if ( showAA ) resdat.bodyHTML += "<span style=\"color: white\">_</span>";
#endif

                startPos = i + 4;
                if ( chpt[ startPos ] == ' ' ) startPos++; /* remove space after <br> */
                i = startPos - 1;
            }

            /*----------------------------------------*/

            /* remove HTML tags <[^>]*>  */
            else {

                if ( i - startPos ) resdat.bodyHTML += rawStr.mid( startPos, i - startPos );
                while ( chpt[ i ] != '>' && i < length ) i++;
                startPos = i + 1;
            }

            break;

            /*----------------------------------------*/

        case 'h':     /* "http://" or "ttp://" or "tp:" */
        case 't':

            if ( parseLink( chpt + i, length - i, linkstr, linkurl, pos ) ) {

                resdat.bodyHTML += rawStr.mid( startPos, i - startPos );
                resdat.bodyHTML += "<a href=\"" + linkurl + "\">";
                resdat.bodyHTML += linkstr;
                resdat.bodyHTML += "</a>";

                startPos = i + pos;
                i = startPos - 1;
            }

            break;

            /*----------------------------------*/

        case '&':

            /* &gt; */
            if ( chpt[ i + 1 ] == 'g' && chpt[ i + 2 ] == 't' && chpt[ i + 3 ] == ';' )
                ancChain = createResAnchor( rawStr, resdat, chpt, i, startPos );

            break;

            /*----------------------------------------*/

            /* unicode '>'  */
        case UTF16_BRACKET:

            ancChain = createResAnchor( rawStr, resdat, chpt, i, startPos );
            break;

            /*----------------------------------*/

        default:

            if ( ancChain ) ancChain = createResAnchor( rawStr, resdat, chpt, i, startPos );
        }
    }

    resdat.bodyHTML += rawStr.mid( startPos );
#if KDE_IS_VERSION( 3, 2, 0 )
#else
    resdat.bodyHTML += "</div>";
#endif
}



/* parsing function for link   */

/* For example,
 
   cdat = "ttp://foo.com",
 
   then
 
   linkstr = "ttp://foo.com",
   linkurl = "http://foo.com",
   pos (= length of cdat) = 13,
 
   and return TRUE.
                                */
bool Kita::parseLink(

    /* input */
    const QChar *cdat, const unsigned int length,

    /* output */
    QString& linkstr, QString& linkurl, unsigned int& pos
)
{

    /*-----------------------------*/

    linkstr = QString::null;
    linkurl = QString::null;

    QString retlinkstr = QString::null;
    QString prefix = QString::null;
    QString scheme = QString::null;

    if ( isEqual( cdat , "http://" ) ) {
        prefix = "http://";
        scheme = "http://";
    } else if ( isEqual( cdat , "ttp://" ) ) {
        prefix = "ttp://";
        scheme = "http://";
    } else if ( isEqual( cdat , "tp://" ) ) {
        prefix = "tp://";
        scheme = "http://";
    } else if ( isEqual( cdat , "https://" ) ) {
        prefix = "https://";
        scheme = "https://";
    } else if ( isEqual( cdat , "ttps://" ) ) {
        prefix = "ttps://";
        scheme = "https://";
    } else if ( isEqual( cdat , "tps://" ) ) {
        prefix = "tps://";
        scheme = "https://";
    } else {
        return false;
    }

    pos = prefix.length();
    while ( cdat[ pos ] >= '!' && cdat[ pos ] <= '~' &&
            cdat[ pos ] != ' ' && cdat[ pos ] != '<' && cdat[ pos ] != '>'
            && pos < length ) {
        retlinkstr += cdat[ pos++ ];
    }
    if ( pos > length ) return FALSE;

    if ( retlinkstr != QString::null ) DatToText( retlinkstr, linkstr );

    linkurl = scheme + linkstr;
    linkstr = prefix + linkstr;

    return TRUE;
}



/* parsing function for anchor (>>digits)   */

/* This fuction parses res anchor.
 
   For example, if cdat = "&gt;12-20", then
 
   linkstr = ">12-20",
   refNum[0] = 12,
   refNum[1] = 20,
   pos (= length of cdat ) = 9,
   ret = TRUE;
 
*/
bool Kita::parseResAnchor(

    /* input */
    const QChar *cdat, const unsigned int length,

    /* output */
    QString& linkstr, int* refNum, unsigned int& pos )
{

    struct LocalFunc {
        static bool isHYPHEN( unsigned short c )
        {

            /* UTF-16 */
            if ( c == '-'
                    || ( c >= 0x2010 && c <= 0x2015 )
                    || ( c == 0x2212 )
                    || ( c == 0xFF0D )          /* UTF8: 0xEFBC8D */
               ) {
                return TRUE;
            }

            return FALSE;
        }
    };

    bool ret = FALSE;
    int i;

    if ( length == 0 ) return FALSE;

    linkstr = QString::null;
    refNum[ 0 ] = 0;
    refNum[ 1 ] = 0;
    pos = 0;

    /* check '>' twice */
    for ( i = 0;i < 2;i++ ) {

        if ( cdat[ pos ].unicode() == UTF16_BRACKET ) {
            linkstr += cdat[ pos ];
            pos++;
        } else if ( cdat[ pos ] == '&' && cdat[ pos + 1 ] == 'g'  /* &gt; */
                    && cdat[ pos + 2 ] == 't' && cdat[ pos + 3 ] == ';' ) {
            linkstr += ">";
            pos += 4;
        }

    }

    /* check ',' */
    if ( !pos ) {
        if ( cdat[ pos ] == ',' || cdat[ pos ].unicode() == UTF16_COMMA ) {
            linkstr += ",";
            pos ++;
        }
    }

    /* check '=' */
    if ( !pos ) {
        if ( cdat[ pos ] == '=' || cdat[ pos ].unicode() == UTF16_EQ ) {
            linkstr += "=";
            pos ++;
        }
    }

    /* check digits */
    int hyphen = 0;

    for ( i = 0 ; i < KITA_RESDIGIT + 1 && pos < length ; i++, pos++ ) {

        unsigned short c = cdat[ pos ].unicode();

        if ( ( c < UTF16_0 || c > UTF16_9 )
                && ( c < '0' || c > '9' )
                && ( !LocalFunc::isHYPHEN( c )
                     || ( i == 0 && LocalFunc::isHYPHEN( c ) )
                     || ( hyphen && LocalFunc::isHYPHEN( c ) ) )
           ) break;

        linkstr += cdat[ pos ];

        if ( LocalFunc::isHYPHEN( c ) ) {
            hyphen = 1;
            i = -1;
        } else {
            if ( c >= UTF16_0 ) c = '0' + cdat[ pos ].unicode() - UTF16_0;
            refNum[ hyphen ] *= 10;
            refNum[ hyphen ] += c - '0';
        }

        ret = TRUE;
    }

    return ret;
}



/* create res anchor  */
/* This function is called from parseBody internally.
   See also parseBody.                                */
bool Kita::createResAnchor( const QString &rawStr, RESDAT& resdat,
                            const QChar *chpt, unsigned int &i, unsigned int &startPos )
{
    QString linkstr, linkurl;
    int refNum[ 2 ];
    unsigned int pos;
    unsigned int length = rawStr.length();

    /* parse anchor */
    if ( !parseResAnchor( chpt + i, length - i, linkstr, refNum, pos ) ) {

        i += pos - 1;
        return FALSE;
    }

    /* create anchor */
    resdat.bodyHTML += rawStr.mid( startPos, i - startPos );
    linkurl = QString( "#%1" ).arg( refNum[ 0 ] );
    if ( refNum[ 1 ] ) linkurl += QString( "-%1" ).arg( refNum[ 1 ] );

    resdat.bodyHTML += "<a href=\"" + linkurl + "\">";
    resdat.bodyHTML += linkstr;
    resdat.bodyHTML += "</a>";

    /* add anchor to ancList */
    ANCNUM anctmp;
    if ( refNum[ 1 ] < refNum[ 0 ] ) refNum[ 1 ] = refNum[ 0 ];
    anctmp.from = refNum[ 0 ];
    anctmp.to = refNum[ 1 ];
    resdat.anclist += anctmp;

    startPos = i + pos;
    i = startPos - 1;

    return TRUE;
}


/* create HTML of title.
  
  struct RESDAT resdat should be parsed by parseResDat before  calling this function.
 
  output: titleHTML
  
*/
void Kita::createTitleHTML( RESDAT& resdat, QString& titleHTML )
{
    titleHTML = QString::null;
    if ( !resdat.parsed ) return ;

    bool showMailAddress = KitaConfig::showMailAddress();
    bool useTableTag = KitaConfig::useStyleSheet();

    if ( m_colonstr == QString::null ) {
        m_colonstr = utf8ToUnicode( KITAUTF8_COLON );
        m_colonnamestr = utf8ToUnicode( KITAUTF8_NAME );
    }

#if KDE_IS_VERSION( 3, 2, 0 )
#else
    /* For KDE3.1    */
    useTableTag = TRUE;
#endif

    if ( useTableTag ) titleHTML += "<table class=\"res_title\"><tr>";

    /* res number */
    if ( useTableTag ) titleHTML += "<td class=\"res_title_number\">";
    titleHTML += "<a href=\"#write" + QString().setNum( resdat.num ) + "\">";
    titleHTML += QString().setNum( resdat.num );
    titleHTML += "</a> ";


    /* name & mail address */
    if ( useTableTag ) titleHTML += "<td class=\"res_title_name\">";
    titleHTML += "<b>" + m_colonnamestr;

    /* show name with mail address */
    if ( showMailAddress ) {

        titleHTML += resdat.nameHTML;
        if ( resdat.address != QString::null ) titleHTML += " [" + resdat.address + "]";

    } else { /* don't show mail address */

        if ( resdat.address == QString::null ) {

            titleHTML += "<span class=\"name_noaddr\">";
            titleHTML += resdat.name;
            titleHTML += "</span>";

        } else {

            titleHTML += "<a href=\"mailto:" + resdat.address + "\"";
            titleHTML += " title=\"" + resdat.address + "\">";
            titleHTML += resdat.name;
            titleHTML += "</a>";
        }
    }

    titleHTML += "</b> ";

    /* date */
    if ( useTableTag ) titleHTML += "<td class=\"res_title_date\">";
    titleHTML += m_colonstr + resdat.date;
    if ( useTableTag ) titleHTML += "</td>";

    /* ID */
    if ( resdat.id != QString::null ) {

        if ( useTableTag ) titleHTML += "<td class=\"res_title_id\">";
        if ( resdat.id.contains( "???", true ) >= 1 ) titleHTML += " ID:" + resdat.id;
        else titleHTML += " <a href=\"#idpop" + resdat.id + "\">ID</a>" + ":" + resdat.id;
        if ( useTableTag ) titleHTML += "</td>";
    }

    /* BE */
    if ( resdat.be != QString::null ) {

        if ( useTableTag ) titleHTML += "<td class=\"res_title_be\">";
        titleHTML += " <a href=\"#bepop" + resdat.be + "\">?" + resdat.bepointmark + "</a>";
        if ( useTableTag ) titleHTML += "</td>";
    }

    /* host */
    if ( resdat.host != QString::null ) {

        if ( useTableTag ) titleHTML += "<td class=\"res_title_host\">";
        titleHTML += " HOST:" + resdat.host;
        if ( useTableTag ) titleHTML += "</td>";
    }

    if ( useTableTag ) titleHTML += "</tr></table>";
}



/*--------------------------------*/
/*--------------------------------*/
/* obsolete */

QString Kita::datToBoard( const KURL& datURL ) { return BoardManager::boardURL( datURL ); }
int Kita::boardType( const KURL& url ) { return BoardManager::type( url ); }
