/***************************************************************************
 *   Copyright (C) 2008 by Kitsune                                         *
 *   sutoka@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 or 3 of the License,   *
 *   or (at your option) any later version.                                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "cookiemonster.h"

#include <QFile>
#include <QDateTime>
#include <iostream>
#include <QTimerEvent>
#include <kstandarddirs.h>

cookieMonster::cookieMonster(QObject * parent) : QNetworkCookieJar(parent), s_cookieSaveInterval(3000), m_readOnly(false)
{
        setObjectName("cookieMonster:" + QString::number(uint(this)));
        std::cout << "CREATE:cookieMonster," << uint(this) << ";parent:" << uint(parent) << std::endl;
        loadFromDisk();
        m_timerId = startTimer(s_cookieSaveInterval); //Auto-save cookies after changes
}


cookieMonster::~cookieMonster()
{
        saveIfTainted();
        std::cout << "DESTROY:cookieMonster," << uint(this) << ";parent:" << uint(parent()) << std::endl;
}

void cookieMonster::timerEvent(QTimerEvent * event)
{
        if(event->timerId() == m_timerId && m_tained)
        {
                writeToDisk(); //TODO Only write to disk if there are changes.
        }
}

signed int cookieMonster::writeToDisk()
{
        std::cout << "ACTION:writeCookies,";

        QString cookiePath = KStandardDirs::locateLocal("appdata", "cookies.db");
        if(cookiePath.isNull())
        {
                std::cout << "NoPathGiven" << std::endl;
                return NOPATHGIVEN;
        }

        QFile file(cookiePath);
        if(!file.open(QIODevice::WriteOnly))
        {
                std::cout << "OpenFail" << std::endl;
                return OPENFAIL; //TODO Give a better error?
        }
        QDataStream out(&file);
        QList<QNetworkCookie> *cookieList = new QList<QNetworkCookie>(allCookies());
        //TODO first go through the list and delete expired cookies, then replace the normal list with that one
        out << MAGICNUMBER;
        out << VERSION_CURRENT;
        out.setVersion(QDataStream::Qt_4_0);

        //TODO Compress and hash this data to make sure it's valid?
        out << cookieList->size();
        for (int i = 0; i < cookieList->size(); ++i) {
                if(!cookieList->at(i).isSessionCookie() && (cookieList->at(i).expirationDate() > QDateTime::currentDateTime()))
                {
                        out << cookieList->at(i).expirationDate();
                        out << cookieList->at(i).domain();
                        out << cookieList->at(i).isSecure();
                        out << cookieList->at(i).name();
                        out << cookieList->at(i).path();
                        out << cookieList->at(i).value();
                }
        }
        if(!file.flush())
        {
                std::cout << "FlushFail" << std::endl;
                return FLUSHFAIL;
        }
        std::cout << "Finished" << std::endl;
        m_tained = false; //Cookies on disk are accurate now.
        return ALLOKAY;
}

signed int cookieMonster::loadFromDisk()
{
        std::cout << "ACTION:readCookies,";

        QString cookiePath = KStandardDirs::locateLocal("appdata", "cookies.db");
        if(cookiePath.isNull())
        {
                std::cout << "NoPathGiven" << std::endl;
                return NOPATHGIVEN;
        }

        QFile file(cookiePath);
        if(!file.exists())
        {
                std::cout << "NoFile" << std::endl;
                return NOFILE;
        }
        if(!file.open(QIODevice::ReadOnly))
        {
                std::cout << "OpenFail" << std::endl;
                return OPENFAIL; //TODO Give a better error?
        }
        QDataStream in(&file);

        // Do several checks to make sure everything is okay.
        quint32 magicnum;
        in >> magicnum;
        if(!magicnum == MAGICNUMBER)
        {
                //TODO Disable saving cookies? Prompt the user?
                std::cout << "WrongMagic" << std::endl;
                return WRONGMAGIC;
        }
        int size, version;
        in >> version;
        if(version > VERSION_CURRENT)
        {
                //TODO Disable saving cookies? Prompt the user?
                std::cout << "TooNew" << std::endl;
                return TOONEW;
        }
        in.setVersion(QDataStream::Qt_4_0); //Should go okay from here...
        in >> size;
        if(!size > 0) //Not an error if it's true.
        {
                std::cout << "EmptyList" << std::endl;
                return EMPTYLIST;
        }

        // Everything looks okay, now load all the cookies.
        QList<QNetworkCookie> cookieList;
        for(int i = 0; i < size; i++)
        {
                QString domain, path;
                QDateTime expiration;
                bool secure;
                QByteArray name, value;
                in >> expiration >> domain >> secure >> name >> path >> value;
                if(expiration > QDateTime::currentDateTime())
                {
                        QNetworkCookie cookie;
                        cookie.setDomain(domain);
                        cookie.setExpirationDate(expiration);
                        cookie.setSecure(secure);
                        cookie.setName(name);
                        cookie.setPath(path);
                        cookie.setValue(value);
                        cookieList.append(cookie);
                }
        }
        if(cookieList.size() > 0)
        {
                setAllCookies(cookieList);
                std::cout << "AllOkay" << std::endl;
                return ALLOKAY;
        }
        else
        {
                std::cout << "EmptyList" << std::endl;
                return EMPTYLIST;
        }
}

bool cookieMonster::setCookiesFromUrl(const QList< QNetworkCookie > & cookieList, const QUrl & url)
{
        if(m_readOnly)
                return false;
        m_tained = true;
        return QNetworkCookieJar::setCookiesFromUrl(cookieList, url);
}

signed int cookieMonster::saveIfTainted()
{
        if(m_tained)
                return writeToDisk();
        return NOTTAINTED;
}

QList<QNetworkCookie> cookieMonster::cookiesForUrl(const QUrl & url) const
{
        if(!m_parentJar)
        {
                // We must not have a parent jar, so just act normal.
                return QNetworkCookieJar::cookiesForUrl(url);
        }
        return QNetworkCookieJar::cookiesForUrl(url);
        //FIXME: Add code here to grab m_parentJar's cookie list, then merge it with the list from this object, cookies from /this/ will replace cookies from the parent.
}
