/* 
   USB firmware loader for Broadcom BCM2033 
   Copyright (C) 2001-2002 Maxim Krasnyansky <maxk@qualcomm.com>
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation;
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
   IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY CLAIM,
   OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER
   RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
   NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE
   USE OR PERFORMANCE OF THIS SOFTWARE.
   
   ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS, COPYRIGHTS,
   TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS SOFTWARE IS DISCLAIMED.
*/

/*
 * $Id: bcm_usb.c,v 1.5 2002/10/28 00:24:51 holtmann Exp $
 */

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include "usblib.h"
#include "bluefw.h"

#define INTR_IN_EP	0x81
#define BULK_IN_EP	0x82
#define BULK_OUT_EP	0x02

#define BULK_WRITE_TO	100000

static int usb_bulk_load(int dev, char *file, char *name)
{
	char buf[4096];
	int len, err, fd;

	if ((fd = open(file, O_RDONLY)) < 0) {
		syslog(LOG_ERR, "Can't open %s file %s", name, file);
		return -1;
	}

	err = 0;

	while (1) { 
		len = read(fd, buf, sizeof(buf)); 
		if (len < 0) {
			syslog(LOG_ERR, "Read failed %s (%d)", strerror(errno), errno);
			err = 1;		
			goto done;
		}	

		if (!len)
			break;

		if (usb_bulk_write(dev, BULK_OUT_EP, buf, len, BULK_WRITE_TO) < 0) {
			syslog(LOG_ERR, "Bulk write failed %s (%d)", strerror(errno), errno);
			err = 1;		
			goto done;
		}
	}

done:
	close(fd);
	return err;
}

int bcm_usb_loader(devio_t io, devinfo_t *info)
{
	char buf[100];

	if (usb_claim_interface(io.fd, 0)) {
		syslog(LOG_ERR, "Can't claim interface 0. %s (%d)", strerror(errno), errno);
		return -1;
	}

	/* Upload mini driver */
	sprintf(buf, "%s/%s", FWDIR, info->minidrv);
	if (usb_bulk_load(io.fd, buf, "Mini Driver"))
		goto failed;

	usleep(10);

	if (usb_bulk_write(io.fd, BULK_OUT_EP, "#", 1, BULK_WRITE_TO) < 0) {
		syslog(LOG_ERR, "Bulk write failed. %s (%d)", strerror(errno), errno);
		goto failed;
	}

	memset(buf, 0, 16);
	if (usb_bulk_read(io.fd, INTR_IN_EP, buf, 10, 10) < 0) {
		syslog(LOG_ERR, "Intr read #1 failed. %s (%d)", strerror(errno), errno);
		goto failed;
	}
	
	if (buf[0] != '#') {
		syslog(LOG_ERR, "Memory select failed. '%c'", buf[0]);
		goto failed;
	}
			
	/* Load FW */
	sprintf(buf, "%s/%s", FWDIR, info->firmware);
	if (usb_bulk_load(io.fd, buf, "FW"))
		goto failed;

	if (usb_bulk_read(io.fd, INTR_IN_EP, buf, 10, 10) < 0) {
		syslog(LOG_ERR, "Intr read #2 failed. %s (%d)", strerror(errno), errno);
		goto failed;
	}

	if (buf[0] == '.') {
		syslog(LOG_ERR, "FW loaded");
	} else {
		syslog(LOG_ERR, "FW load failed. '%c'", buf[0]);
		goto failed;
	}

	usleep(500000);

	usb_release_interface(io.fd, 0);
	return 0;

failed:
	usb_release_interface(io.fd, 0);
	return -1;
}
