/* 
   BlueFW USB library 
   Copyright (C) 2000-2002 Maxim Krasnyansky <maxk@qualcomm.com>
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation;
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
   IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY CLAIM,
   OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER
   RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
   NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE
   USE OR PERFORMANCE OF THIS SOFTWARE.
   
   ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS, COPYRIGHTS,
   TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS SOFTWARE IS DISCLAIMED.
*/

/*
 * $Id: usblib.c,v 1.6 2002/08/10 04:42:26 maxk Exp $
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>

#include "usblib.h"

int usb_dev_open(char *dev, struct usb_device_descriptor *desc)
{
	char file[255];
	int fd;

	if (dev[0] != '/')
		sprintf(file, "/proc/bus/usb/%s", dev);
	else
		sprintf(file, "%s", dev);

	fd = open(file, O_RDWR);
	if (fd < 0)
		return fd;

	if (desc) {
		int len = read(fd, desc, sizeof(*desc));
		if (len < 0 || len != sizeof(*desc)) {
			close(fd);
			errno = EILSEQ;
			return -1;
		}
	}

	return fd;
}

int usb_dev_close(int dev)
{
	return close(dev);
}

int usb_claim_interface(int dev, int ifnum)
{
	return ioctl(dev, USBDEVFS_CLAIMINTERFACE, (unsigned long) &ifnum);
}

int usb_release_interface(int dev, int ifnum)
{
	return ioctl(dev, USBDEVFS_RELEASEINTERFACE, (unsigned long) &ifnum);
}

int usb_intr_read(int dev, int ep, char *buf, int len)
{
	struct usbdevfs_urb urb;
	unsigned long addr = 0;

	memset(&urb, 0, sizeof(urb));

	urb.type = USBDEVFS_URB_TYPE_INTERRUPT;
	urb.endpoint = ep;
	urb.buffer   = buf;
	urb.buffer_length = len;
	if (ioctl(dev, USBDEVFS_SUBMITURB, (unsigned long)&urb) < 0)
		return -1;

	if (ioctl(dev, USBDEVFS_REAPURB, &addr) < 0)
		return -1;

	if (urb.status) {
		/* usb-uhci and ohci set urb status to CONNABORTED
		 * for one shot interrupt transfers even if transfer
		 * was successeful. */
		if (urb.status == -ECONNABORTED)
			return 0;
		
		errno = -urb.status;
		return -1;
	}
	
	return 0;
}

int usb_bulk_read(int dev, int ep, char *buf, int len, int timeout)
{
	struct usbdevfs_bulktransfer bulk;

	bulk.ep   = ep;
	bulk.len  = len;
	bulk.data = buf;
	bulk.timeout = timeout;

	return ioctl(dev, USBDEVFS_BULK, (unsigned long) &bulk);
}

int usb_bulk_write(int dev, int ep, char *buf, int len, int timeout)
{
	struct usbdevfs_bulktransfer bulk;

	bulk.ep   = ep;
	bulk.len  = len;
	bulk.data = buf;
	bulk.timeout = timeout;

	return ioctl(dev, USBDEVFS_BULK, (unsigned long) &bulk);
}

int usb_reset(int dev, int timeout)
{
	return ioctl(dev, USBDEVFS_RESET, 0);
}
