// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 2001 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif
#include "def.h"

#ifdef HAVE_LIBXML
  #include <libxml/xmlmemory.h>
  #include <libxml/parser.h>
  #include <libxml/encoding.h>
  #ifdef USE_ICONV
    #include <iconv.h>
    #include <errno.h>
  #endif
#else
#include "tinyxml.h"
#endif

#include "xmlsub.h"

#include "debug.h"
#include <cstdio>

bool XML_Initialize(void){
#if defined(HAVE_LIBXML) && !defined(LIBXML_ICONV_ENABLED)
  xmlCharEncodingHandlerPtr hp=
    (xmlCharEndodingHandlerPtr) xmlMalloc(sizeof(xmlCharEncodingHandler));
  hp->input=(xmlCharEncodingInputFunc)EucToUTF8;
  hp->output=(xmlCharEncodingOutputFunc)UTF8ToEuc;
  handler->name=strdup("EUC-JP");
#endif
  return true;
}

static std::vector<NATIVE_STRING> 
gettoken(const NATIVE_STRING& str,char delim){
  NATIVE_STRING st=str;
  std::vector<NATIVE_STRING> x;
  int len;
  if (st.length()>0 && st.substr(0,1)=="/"){
    st=st.substr(1);
  }
  while((len=st.find(delim))>=0){
    x.push_back(st.substr(0,len));
    st=st.substr(len+1);
  }
  x.push_back(st);
  return x;
}

xmlNodePtr XML_GetNodeFromXPath(xmlDocPtr doc,const NATIVE_STRING& path){
  std::vector<NATIVE_STRING> nodeName=gettoken(path.c_str(),'/');

  xmlNodePtr cur=xmlDocGetRootElement(doc);
  if (cur==NULL || cur->name==NULL) return NULL;

  std::vector<NATIVE_STRING>::iterator ii=nodeName.begin();

  if (xmlStrcmp(cur->name,(const xmlChar *)((*ii).c_str()))!=0) return NULL;
  for(++ii;ii!=nodeName.end();++ii){
    cur=cur->xmlChildrenNode;
    while(cur!=NULL && cur->name!=NULL && 
	  xmlStrcmp(cur->name,(const xmlChar*)(*ii).c_str())!=0) {
      cur=cur->next;
    }
    if (cur==NULL) {
      return NULL;
    }
  }
  return cur;
};

std::vector<xmlNodePtr> 
XML_GetNodesFromXPath(xmlDocPtr doc,const NATIVE_STRING& path){
  std::vector<xmlNodePtr> nodes;
  if (doc==NULL) return nodes;
  xmlNodePtr node=XML_GetNodeFromXPath(doc,path);
  while(node!=NULL){
    nodes.push_back(node);
    node=node->next;
  }
  return nodes;
};

xmlNodePtr XML_CreateNodesFromXPath(xmlDocPtr doc,const NATIVE_STRING& path){
  Debug::DebugOut(Debug::TEMP,"CreateNodes='%s'\n",path.c_str());
  std::vector<NATIVE_STRING> nodeName=gettoken(path,'/');
  std::vector<NATIVE_STRING>::iterator nodePtrs=nodeName.begin();
  xmlNodePtr cur=xmlDocGetRootElement(doc);
  if (cur==NULL) return NULL;
  if (cur->name!=NULL && // =======================
      xmlStrcmp(cur->name,(const xmlChar*)(*nodePtrs).c_str())!=0) return NULL;
  ++nodePtrs;
  xmlNodePtr parent=cur;
  cur=cur->xmlChildrenNode;
  while(nodePtrs!=nodeName.end()){
    while(cur!=NULL && cur->name!=NULL && // ===========
	  xmlStrcmp(cur->name,(const xmlChar*)(*nodePtrs).c_str())!=0) {
      cur=cur->next;
    }
    if (cur==NULL) {
      cur=xmlNewChild(parent,NULL,(const xmlChar*)(*nodePtrs).c_str(),NULL);
      parent=cur;
    } else {
      parent=cur;
      cur=cur->xmlChildrenNode;
    }
    ++nodePtrs;
  }
  return parent;
};

bool XML_SetString(xmlDocPtr doc,xmlNodePtr np,const NATIVE_STRING& str){
  xmlNodeSetContent(np,(xmlChar*)(str.c_str()));
  return true;
}

bool XML_Match(xmlNodePtr node,char* str2){
  if (node==NULL) return false;
  return strcmp((char*)node->name,str2)==0;
};

NATIVE_STRING XML_GetString(xmlDocPtr doc,xmlNodePtr node){
  char *str=(char*)xmlNodeGetContent(node);
  if (str==NULL) {
    return "";
  }
  NATIVE_STRING x=str;
  return x;
}

int XML_GetInt(xmlDocPtr doc,xmlNodePtr node){
  int r=strtol(XML_GetString(doc,node).c_str(),NULL,0);
  return r;
}

bool XML_GetBool(xmlDocPtr doc,xmlNodePtr node){
  bool r=(strcmp(XML_GetString(doc,node).c_str(),"true")==0);
  return r;
}


