// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#ifndef __dict_h__
#define __dict_h__

#include "def.h"
#include "catalog.h"
#include "tag.h"
#include "gaiji.h"
#include "bfile.h"
#include "codeconv.h"


// STL
#if defined(HAVE_MAP) && defined(HAVE_VECTOR) && \
              defined(HAVE_LIST) && defined(HAVE_STRING)
#ifndef INC_MAP
#define INC_MAP
#include <map>
#endif

#ifndef INC_VECTOR
#define INC_VECTOR
#include <vector>
#endif

#ifndef INC_LIST
#define INC_LIST
#include <list>
#endif

#ifndef INC_DEQUE
#define INC_DEQUE
#include <deque>
#endif

#else
#error You need STL.
#endif


extern const char *GetIndexName(int id,bool isEB);
extern bool isIndexKey(int type,bool isEB);
extern bool isMenuKey(int type,bool isEB);


/*
Ҵ  {
  ҹǿ   : Word
  ͽΰ1        : Word
  ǥåμ谷ˡ  bytes
  ͽΰ2        : 11 Bytes
  ҹ {
    ҹǼ̻         : Byte
    ͽΰ3                  : Byte
    Ƭɥ쥹               : DWord
    ΰ襵                 : DWord
    ǥåͭ : byte
    ǥå       : 3 bytes
    ͽΰ4                  : 2 Bytes
  } * ҹǿ
  ɽˡν; {
    ̵ͭե饰 : byte
    ͽΰ5      : 3bytes
    ɽɽ     : byte
    ʸɽˡ     : byte
  }
  ͽΰ6        : 10 bytes
}
*/

// ҹ
struct INDEX_HEAD {
  int type;                            // ҹǼ̻
  dword start;                         // Ƭɥ쥹
  dword length;                        // ΰ襵
  byte indexDecodeMethodAvailability;  // ǥåͭ
  CODE_CONV_MODE codeConvMode;         // ǥå
  std::string typeName;  
  INDEX_HEAD(byte* &data,int indexDecodeMethod);
  INDEX_HEAD() { type=-1; typeName=std::string(""); };
  virtual ~INDEX_HEAD() { };
  void DebugOut(int level=Debug::INDEX_DECODE,bool isEB=false);
};

struct SUBINDEX {
  byte typeName[31];
  std::map<int,INDEX_HEAD> subSubIndex;
  INDEX_HEAD* SubSubIndex(int n);
  SUBINDEX(byte* &data,int indexDecodeMethod);

  void DebugOut(int level=Debug::INDEX_DECODE);
};

struct MINDEX_HEAD {
  std::string typeName;
  std::vector<SUBINDEX> subIndex;
  
  int nSubIndex() const { return subIndex.size(); };
  std::string SubIndexName(int n) const;
  SUBINDEX* SubIndex(int n);
  MINDEX_HEAD(byte* &data,int indexDecodeMethod);

  void DebugOut(int level=Debug::INDEX_DECODE);
};

struct INDEXITEM {
  std::string keyword;
  std::string dkeyword;
  TAG adrs;
  TAG keywordTag;

  INDEXITEM();
  ~INDEXITEM() { };
  INDEXITEM(const std::string& key,const TAG& a,const std::string& dkey,const TAG& l);
  INDEXITEM(const INDEXITEM& i);
  const INDEXITEM& operator = (const INDEXITEM& i);
  bool operator==(const INDEXITEM& itm) const;
  bool operator<(const INDEXITEM& itm) const;
  int FuzzyCompare(const INDEXITEM& itm) const;
  void DebugOut(int level=Debug::INDEX_DECODE) const;
private:
  void copy(const INDEXITEM& itm);
};

typedef std::list<INDEXITEM> INDEXITEMLIST_T;

struct INDEX {
  BlockIO *pf;
  int block;

  int type;
  bool isEb;

  int blockType;
  int keyLength;
  std::vector<INDEXITEM> indexItem;

  int currentPoint;

  bool isUpperIndex();
  bool isIndexHead();
  bool isIndexTail();
  bool isGroupIndex();
  int unknownType();

  bool isShortIndex();
  bool isEB();

  bool isVariableIndex();

  const INDEXITEM& CurrentIndexItem() { return indexItem[currentPoint]; };

  INDEX(INDEX* idx);
  INDEX(BlockIO *p,int blockNumber,int type,bool iseb);
  INDEX();

  INDEX(const INDEX& idx);
  const INDEX& operator=(const INDEX& idx);

  ~INDEX();

  void DebugOut(int level=Debug::INDEX_DECODE);
  bool operator==(const INDEX& idx) const;
  bool operator<(const INDEX& idx) const;

  bool Next();
  bool Prev();
private:
  bool DecodeUpperIndex(byte* data);
  bool DecodeUpperFixedIndex(byte* data);
  bool DecodeUpperVariableIndex(byte* data);
  bool DecodeLowerIndex(BlockIO* pf,int block);
  bool DecodeLowerFixedIndex(BlockIO* pf,int block);
  bool DecodeLowerVariableIndex(BlockIO* pf,int block);
  bool ReadIndex(BlockIO *p,int blockNumber,int type);
  void Dup(const INDEX& idx);
};

struct SELECTION_MENU {
  int nReverse;
  int nForward;
  std::vector<INDEXITEM> items;

  INDEXITEM currentPoint;

  std::vector<INDEX> idx;

  SELECTION_MENU();
  SELECTION_MENU(const SELECTION_MENU& a);
  ~SELECTION_MENU();
  bool Append(INDEX* idx,bool validFlag);
  bool Clear();

  bool Prev();
  bool Next();
  void Setup(size_t nPrev,size_t nForw);

private:
  bool Prev(std::vector<INDEX>&);
  bool Next(std::vector<INDEX>&);
};

class DICTIONARY {
  DICT* dictFileStruct;
  BlockIO* pf;

  GAIJI* gaiji;

  int indexDecodeMethod;     // ǥåμ谷
  std::map<int,INDEX_HEAD> indexHead;
  std::vector<MINDEX_HEAD> mIndexHead;
  std::map<int,bool> kinsoku[3];  // Ƭ§,2ŹƬ§,§
  int defaultIndexDisplayMethod;  // ɽɽ
  int defaultHonmonDisplayMethod; // ʸɽˡ

  bool InitializeDictionary(DICT* dct);
  void SetupKinsoku(byte* buf);
  void SetupDummyKinsoku(void);
  void SetupCombinedIndexName(byte* buf);
public:
  BlockIO* BlockFile() { return pf; };
  int nIndex() const { return indexHead.size();  }; // ҹǿ
  bool IndexExist(int indexType) { 
    return (indexHead.find(indexType)!=indexHead.end());
  };
  INDEX *currentIndex;

  int charExtend,charOffset,gaijiOffset; 
  int charHeight,charWidth;

  SELECTION_MENU *sel;

  bool autoImageDisplay,isEB;

  DICTIONARY(DICT* dct) {
    /* ========================= */
    charWidth=16;
    charHeight=16;
    gaijiOffset=0;
    charOffset=2;
    charExtend=6;
    InitializeDictionary(dct);
  };
  GAIJI* Gaiji() const { return gaiji; };
  bool FindWordHead(const std::string& key,const INDEX_HEAD& iHead);
  bool FindWordHead(const std::string& key,int indexType);
  INDEXITEMLIST_T FindWord(const std::string& key,const INDEX_HEAD& iHead);
  INDEXITEMLIST_T FindWord(const std::string& key,int indexType=0x91);
  INDEXITEMLIST_T FindWord(const std::string& key,int extnum,int extnum2);

  int nmIndex() const { return mIndexHead.size(); };
  
  std::string GetLine(const TAG& t);
  ~DICTIONARY();
  
  TAG IndexStart(int type) { 
    if (IndexExist(type)) return TAG(indexHead[type].start,0); 
    else return TAG(0,0);
  };
  const char* IndexName(int type) {
    if (!IndexExist(type)) return NULL;
    if (indexHead[type].typeName==std::string("")){
      return GetIndexName(type,isEB);
    } else {
      return indexHead[type].typeName.c_str();
    }
  };
  const char* MIndexName(int num);
  MINDEX_HEAD* MIndex(int num){
    if (num>=0 && num<nmIndex()){
      return & mIndexHead[num];
    }
    return NULL;
  }

  bool IsKinsoku(int type,int code);
  TAG RandomJump(const TAG& start);
  //  TAG NextJump(const TAG& start);
};

extern INDEXITEMLIST_T And(const INDEXITEMLIST_T& a,
			   const INDEXITEMLIST_T& b);
#endif
