// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "gaiji.h"
#include "ndtp.h"
#include <cstdio>

byte GAIJI::rTbl[256];
bool GAIJI::rTblInitialized=false;

GaijiChar::GaijiChar(int w,int h,const byte* bmp) :
  width(w),
  height(h),
  size((((w-1)>>3)+1)*h)
{
  bitmap=new byte[size];
  memcpy(bitmap,bmp,size);
}

void GaijiChar::copy(const GaijiChar& g){
  width=g.width;
  height=g.height;
  size=g.size;
  bitmap=new byte[size];
  memcpy(bitmap,g.bitmap,g.size);
}

GaijiChar::GaijiChar(const GaijiChar& g){
  copy(g);
}

const GaijiChar& GaijiChar::operator=(const GaijiChar& g){
  copy(g);
  return *this;
}

GAIJI::GAIJI(BlockIO* pf,int size){
  ReadNdtp(reinterpret_cast<NdtpIO*>(pf),size);
}

GAIJI::GAIJI(BlockIO* pf,int startBlock,int startBlock2){
  ReadFile(pf,startBlock);
  ReadFile(pf,startBlock2);
}

void GAIJI::ReadFile(BlockIO* pf,int startBlock){
  byte buf[cBlockSize];
  pf->Seek(TAG(startBlock,0));
  pf->Read(buf,cBlockSize);
  int width=buf[8];
  int height=buf[9];
  
  Debug::DebugOut(Debug::GAIJI_DECODE,"   %d x %d\n",width,height);
  //  int hOffset=width<height?0x80:0;
  int hOffset=0;
  int n=256*buf[12]+buf[13];
  int code=256*buf[10]+buf[11]+hOffset;
  int wb=((width-1)>>3)+1;
  code|=(width<height)?0x10000:0;  // Hankaku Flag
  while(n>0){
    pf->Read(buf,1024);
    for(int gp=0;gp<=1024-wb*height && n>0; gp+=wb*height,n--){
      if (gaijiMap.find(code)!=gaijiMap.end()) {
	delete gaijiMap[code];
      }
      gaijiMap[code]=new GaijiChar(width,height,buf+gp);
      code++;
      if ((code & 0x7f)>0x7e) code=(code & 0xfff80)+0x121;
    }
  }
}

GAIJI::GAIJI(const char* fileName,const char* fileName2){
  //  int width,height;
  if (fileName!=NULL && fileName[0]!=0){
    BlockIO* pf=OpenDict(fileName);
    Debug::DebugOut(Debug::GAIJI_DECODE,"Open '%s'\n",fileName);
    if (pf==NULL || !pf->Initialized()){
      Debug::DebugOut(Debug::GAIJI_DECODE,"Open '%s' Fail\n",fileName);
    } else {
      ReadFile(pf,1);
      delete pf;
    }
    Debug::DebugOut(Debug::GAIJI_DECODE,"Close '%s' Ok\n",fileName);
  }

  if (fileName2!=NULL && fileName2[0]!=0){
    BlockIO* pf=OpenDict(fileName2);
    Debug::DebugOut(Debug::GAIJI_DECODE,"Open '%s'\n",fileName2);
    if (pf==NULL || !pf->Initialized()){
      Debug::DebugOut(Debug::GAIJI_DECODE,"Open '%s' Fail\n",fileName2);
    } else {
      ReadFile(pf,1);
      delete pf;
    }
    Debug::DebugOut(Debug::GAIJI_DECODE,"Close '%s' Ok\n",fileName2);
  }
}

GaijiChar* GAIJI::Gaiji(int code) {
  if (gaijiMap.size()==0) return NULL;
  std::map<int, GaijiChar* >::iterator i=gaijiMap.find(code);
  if (i==gaijiMap.end()) return NULL;
  return (*i).second;
}

GAIJI::~GAIJI(){
  std::map<int, GaijiChar* >::iterator i;
  for(i=gaijiMap.begin();i!=gaijiMap.end();++i){
    delete (*i).second;
  }
}

void GAIJI::ReadNdtp(NdtpIO* nf,int size){
  if (!rTblInitialized){
    for(int i=0;i<256;i++){
      int t=0;
      for(int j=0;j<8;j++) if ((128>>j) & i) t|=(1<<j);
      rTbl[i]=t;
    }
    rTblInitialized=true;
  }

  Debug::DebugOut(Debug::NDTP,"NDTP Gaiji Start\n");
  char buf[cTmpLength];
  int width=16,height=16,gsize=0;

  nf->SendCommand("XI");
  nf->ReadLine(buf,cTmpLength);
  if (strncmp(buf,"$I",2)!=0) {
    Debug::DebugOut(Debug::WARNINGS,"Can't read gaiji via ndtp\n");
    return;
  }
  for(int c=0;c<10;c++) {
    nf->ReadLine(buf,cTmpLength);
    if (buf[0]=='$') break;
    int s=atoi(buf);
    Debug::DebugOut(Debug::TEMP,"Gaiji Size=%d/%d\n",s,size);
    if (s>gsize && s<=size) gsize=s;
  } 
  if (buf[0]!='$'){
    throw("Ndtp I/O Error : Command XI\n");
  }
  if (gsize==0) {
    Debug::DebugOut(Debug::WARNINGS,"No Gaiji Font!\n");
    return;
  }
  nf->SendCommand("XL%d",gsize);
  nf->ReadLine(buf,cTmpLength);
  if (strncmp(buf,"$*",2)!=0) {
    Debug::DebugOut(Debug::NDTP,"Can't read %dx%d gaiji characters '%s'\n",
		    gsize,gsize,buf);
    return;
  }


  nf->SendCommand("XB");
  nf->ReadLine(buf,cTmpLength);
  if (strncmp(buf,"$I",2)!=0) {
    Debug::DebugOut(Debug::NDTP,
		    "NDTP: Server dose not support gaiji extension.\n");
    return;
  }

  byte* buf2=new byte[gsize*gsize];
  for(;;){
    if (!nf->ReadLine(buf,cTmpLength)) break;
    if (strncmp(buf,"$$",2)==0) break;
    if (strncmp(buf,"$=",2)==0){
      char* p;
      int code=strtol(buf+3,NULL,16);
      //      Debug::DebugOut(Debug::NDTP,"NDTP Gaiji Code=%X\n",code);
      if (!nf->ReadLine(buf,cTmpLength)) break;
      p=strstr(buf,"_width");
      if (p!=NULL){
	while(*p!=' ')p++;
	while(*p==' ')p++;
	width=strtol(p,NULL,0);
	//	Debug::DebugOut(Debug::NDTP,"NDTP Gaiji Width=%d\n",width);
      }
      if (!nf->ReadLine(buf,cTmpLength)) break;
      p=strstr(buf,"_height");
      if (p!=NULL){
	while(*p!=' ')p++;
	while(*p==' ')p++;
	height=strtol(p,NULL,0);
	//	Debug::DebugOut(Debug::NDTP,"NDTP Gaiji Height=%d\n",height);
      }
      if (width<height) code|=0x10000; // Hankaku Flag

      // Dummy Read "static unsigned..."
      if (!nf->ReadLine(buf,cTmpLength)) break;  
      int bwidth=((width-1)>>3)+1;
      int bytes=bwidth*height;
      p=NULL;
      int count=0;
      while(count<bytes){
	char* p2;
	if (p==NULL || (p2=strchr(p,'0'))==NULL){
	  if (!nf->ReadLine(buf,cTmpLength)) break;
	  p=buf;
	  p2=buf;
	}
	buf2[count++]=rTbl[255 & strtol(p2,&p,0)];
      }
      if (gaijiMap.find(code)!=gaijiMap.end()) {
	delete gaijiMap[code];
      }
      gaijiMap[code]=new GaijiChar(width,height,buf2);
    }
  }
  delete [] buf2;
}

