// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 2000 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "ndtp.h"
#include "codeconv.h"
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <unistd.h>
#include <cstdarg>
#include <cstdio>

NdtpPath::NdtpPath(const char* URIpath):
  host(NULL),
  port(-1),
  base(NULL),
  path(NULL),
  isEB(false)
{
  Debug::DebugOut(Debug::NDTP,"NDTP Decode Path '%s'\n",URIpath);
  char* p=strdup(URIpath);
  if (strncmp(p,"ndtp:",5)!=0) {
    Debug::DebugOut(Debug::NDTP,"NDTP Path=local file '%s'\n",path);
    base=p;
    return;
  }
  char* p2=p+5;
  if (strncmp(p2,"//",2)==0){
    // "ndtp://hostname/...." style
    p2+=2;
    char* p3=strchr(p2,'/');
    if (p3==NULL){
      free(p);
      Debug::DebugOut(Debug::NDTP,"NDTP Path decode error\n");
      return;
    }
    *p3++=0;
    char* p4=strchr(p2,':');
    if (p4==NULL){
      struct servent* se=getservbyname("ndtp","tcp");
      if (se!=NULL) {
	port=ntohs(se->s_port);
      }
    } else {
      *p4=0;
      port=atoi(p4+1);
      if (port==0){
	port=-1;
	struct servent* se=getservbyname(p4+1,"tcp");
	if (se!=NULL) {
	  port=ntohs(se->s_port);
	}
      }
    }
    host=strdup(p2);
    path=strdup(p3);
  } else {
    // "ndtp:PATH" style
    host=strdup("localhost");
    struct servent* se=getservbyname("ndtp","tcp");
    if (se!=NULL) {
      port=ntohs(se->s_port);
    }
    path=strdup(p2);
  }
  free(p);
  p2=strchr(path,'#');
  if (p2!=NULL) {
    *p2=0;
    isEB=true;
  }
  char buf[cTmpLength];
  if (snprintf(buf,cTmpLength,"ndtp://%s:%d/",host,port)<0) {
    base=NULL;
  } else {
    base=strdup(buf);
    Debug::DebugOut(Debug::NDTP,
		    "NDTP Host='%s', Port=%d, Base='%s', Path='%s'\n",
		    host,port,base,path);
  }
}

NdtpPath::~NdtpPath(){
  free(path);
  free(host);
  free(base);
  path=NULL;
  host=NULL;
  base=NULL;
}

void NdtpPath::copy(const NdtpPath& p){
  if (p.path==NULL) path=NULL; else path=strdup(p.path);
  if (p.host==NULL) host=NULL; else host=strdup(p.host);
  if (p.base==NULL) base=NULL; else base=strdup(p.base);
  port=p.port;
  isEB=p.isEB;
}

NdtpPath::NdtpPath(const NdtpPath& p){
  copy(p);
}
  
const NdtpPath& NdtpPath::operator = (const NdtpPath& p) {
  copy(p);
  return *this;
}


NdtpIO::NdtpIO(char* hostname,int port){
  int ndtpSock;
  ndtpSocket=NULL;
  type=BFileTypeError;
  if (hostname==NULL) hostname="localhost";
  if (port<0) {
    Debug::DebugOut(Debug::NDTP,"NDTP Port Number Error : %d\n",port);
    return;
  }
  Debug::DebugOut(Debug::NDTP,
		  "NDTP Open : Host='%s' : Port=%d \n",hostname,port);
  ndtpSock=socket(AF_INET,SOCK_STREAM,0);
  if (ndtpSock<0) {
    Debug::DebugOut(Debug::NDTP,"NDTP Socket Open Error\n");
    return;
  }

  struct hostent* he=gethostbyname(hostname);
  struct sockaddr_in adrs;
  adrs.sin_family=AF_INET;
  memcpy(&(adrs.sin_addr),he->h_addr,he->h_length);
  adrs.sin_port=htons(port);

  if (connect(ndtpSock,reinterpret_cast<sockaddr*>(&adrs),sizeof(adrs))<0){
    Debug::DebugOut(Debug::NDTP,"NDTP Connect Error\n");
    close(ndtpSock);
    ndtpSock=-1;
  } 
  if (ndtpSock>=0) {
    type=BFileTypeNDTP;
    Debug::DebugOut(Debug::NDTP,"NDTP open succeed.\n");
    ndtpSocket=fdopen(ndtpSock,"r+");
    GetDictList();
  }
}

NdtpIO::~NdtpIO(){
  if (ndtpSocket!=NULL){
    SendCommand("Q");
    char buf[1024];
    while(ReadLine(buf,1024));
    Debug::DebugOut(Debug::NDTP,"NDTP close\n");
    fclose(ndtpSocket);
  }
  ndtpSocket=NULL;
}

bool NdtpIO::SeekWithoutCache(int block){
  // Debug::DebugOut(Debug::NDTP,"NDTP Seek\n");
  char buf[10];
  sprintf(buf,"F%X",block);
  SendCommand(buf);
  if (!ReadRaw(buf,2) || buf[0]!='$' || buf[1]!='F') return false;
  if (!ReadRaw(reinterpret_cast<char*>(ibuffer),cBlockSize)) return false;
  return ReadRaw(buf,1);
}

bool NdtpIO::SendCommand(const char* cmd, ...){
  if (ndtpSocket==NULL) return false;

  static char buf[1024];
  va_list  ap;
  va_start(ap,cmd);
  vsnprintf(buf,1024,cmd,ap);
  va_end(ap);

  size_t len=strlen(buf);
  while(len>1 && (buf[len-1]=='\n' || buf[len-1]=='\r')) len--;
  Debug::DebugOut(Debug::NDTP,"Send: '%s' %d\n",buf,len);
  if (fwrite(buf,1,len,ndtpSocket)!=len) return false;
  if (fwrite("\n",1,1,ndtpSocket)!=1) return false;
  fflush(ndtpSocket);
  return true;
}

bool NdtpIO::ReadLine(char* buf,int maxChar){
  if (ndtpSocket==NULL) return false;
  return fgets(buf,maxChar,ndtpSocket)!=NULL;
}

bool NdtpIO::ReadRaw(char* buf,size_t len){
  if (ndtpSocket==NULL) return false;
  if (len!=fread(buf,1,len,ndtpSocket)) {
    return false;
  }
  return true;
}

bool NdtpIO::GetDictList(){
  if (ndtpSocket==NULL) return false;
  diclist.clear();
  SendCommand("t");
  char buf[1024];
  for(;;){
    ReadLine(buf,1024);
    if (buf[0]=='$' && (buf[1]=='$' || buf[1]=='*')) break;
    char* title=NULL,* path=NULL,* eol=NULL;
    title=strchr(buf,'\t');
    if (title!=NULL){
      path=strchr(title+1,'\t');
      *title++=0;
    }
    if (path!=NULL){
      eol=strchr(path+1,'\t');
      *path++=0;
    }
    if (eol!=NULL) {
      *eol++=0;
    } else {
      continue; // Invalid Line
    }
    Debug::DebugOut(Debug::NDTP,"No. %d:title=%s:path=%s:\n",
    		    atoi(buf),title,path);
    diclist.push_back(NdtpDicList(atoi(buf),title,path));
  }
  return true;
}

bool NdtpIO::SelectDict(const char* path){
  if (ndtpSocket==NULL) return false;

  char buf[cTmpLength];
  Debug::DebugOut(Debug::NDTP,"NDTP Find Dictionary '%s'\n",path);
  std::vector<NdtpDicList>::iterator i;
  for(i=diclist.begin();i!=diclist.end();++i){
    Debug::DebugOut(Debug::NDTP,"NDTP Compare :%s:%s:\n",
    		    (*i).path.c_str(),path);
    if (strcasecmp((*i).path.c_str(),path)==0) {
      Debug::DebugOut(Debug::NDTP,"NDTP Found at %d\n",(*i).number);
      sprintf(buf,"L%d",(*i).number);
      SendCommand(buf);
      ReadLine(buf,cTmpLength);
      if (buf[0]!='$' || buf[1]!='*') {
	Debug::DebugOut(Debug::NDTP,"NDTP Dictionary Change Failed\n");
	return false;
      }
      SendCommand("Aebdic");
      while(ReadLine(buf,cTmpLength) && buf[0]!='$');
      if (buf[0]!='$' || buf[1]!='A') {
	Debug::DebugOut(Debug::NDTP,"NDTP Auth Failed\n");
	return false;
      }
      currentDict=(*i).number-1;
      return true;
    }
  }
  Debug::DebugOut(Debug::NDTP,"NDTP Not Found '%s'\n",path);
  return false;
}

std::string NdtpIO::GetTitle(const char* path){
  if (ndtpSocket==NULL) return std::string("");

  Debug::DebugOut(Debug::NDTP,"NDTP Find Title '%s'\n",path);
  std::vector<NdtpDicList>::iterator i;
  for(i=diclist.begin();i!=diclist.end();++i){
    if (strcasecmp((*i).path.c_str(),path)==0) {
      Debug::DebugOut(Debug::NDTP,
		      "NDTP Title Found '%s'\n",(*i).title.c_str());
      return (*i).title;
    }
  }
  Debug::DebugOut(Debug::NDTP,"NDTP Title Not Found\n");
  return std::string("");
}

