/*
 *  Copyright (C) 2004 Morten Fjord-Larsen
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gconf/gconf-client.h>

#include "gpass/error.h"
#include "gpass/configuration.h"

#define GCONF_PATH "/apps/gpass"

static GObjectClass *parent_class = NULL;

static void
gpass_configuration_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassConfiguration *self = GPASS_CONFIGURATION(instance);
    
    self->gconf_client = NULL;
    self->window_x = 0;
    self->window_y = 0;
    self->window_width = 500;
    self->window_height = 300;
    self->toolbar_visible = TRUE;
    self->statusbar_visible = TRUE;
    self->pane_width = 250;
    g_datalist_init(&self->launchers);
    self->undo_levels = 15;
    self->visible_secrets = FALSE;
    self->lock = TRUE;
    self->lock_timeout = 15;
}

enum {
    CONFIGURATION_PROP_0,
    CONFIGURATION_PROP_GCONF_CLIENT,
    CONFIGURATION_PROP_WINDOW_X,
    CONFIGURATION_PROP_WINDOW_Y,
    CONFIGURATION_PROP_WINDOW_WIDTH,
    CONFIGURATION_PROP_WINDOW_HEIGHT,
    CONFIGURATION_PROP_TOOLBAR_VISIBLE,
    CONFIGURATION_PROP_STATUSBAR_VISIBLE,
    CONFIGURATION_PROP_PANE_WIDTH,
    CONFIGURATION_PROP_UNDO_LEVELS,
    CONFIGURATION_PROP_VISIBLE_SECRETS,
    CONFIGURATION_PROP_LOCK,
    CONFIGURATION_PROP_LOCK_TIMEOUT,
};

static void
gpass_configuration_set_property(GObject *object, guint prop_id,
                                 const GValue *value, GParamSpec *pspec)
{
    GPassConfiguration *self = GPASS_CONFIGURATION(object);
    GError *error = NULL;
    
    switch (prop_id) {
    case CONFIGURATION_PROP_GCONF_CLIENT:
        self->gconf_client = GCONF_CLIENT(g_value_get_object(value));
        gconf_client_add_dir(self->gconf_client, GCONF_PATH,
                             GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
        break;
    case CONFIGURATION_PROP_WINDOW_X:
        self->window_x = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/ui/window_x",
                                 self->window_x, &error);
        }
        break;
    case CONFIGURATION_PROP_WINDOW_Y:
        self->window_y = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/ui/window_y",
                                 self->window_y, &error);
        }
        break;
    case CONFIGURATION_PROP_WINDOW_WIDTH:
        self->window_width = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/ui/window_width",
                                 self->window_width, &error);
        }
        break;
    case CONFIGURATION_PROP_WINDOW_HEIGHT:
        self->window_height = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/ui/window_height",
                                 self->window_height, &error);
        }
        break;
    case CONFIGURATION_PROP_TOOLBAR_VISIBLE:
        self->toolbar_visible = g_value_get_boolean(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_bool(self->gconf_client,
                                  GCONF_PATH "/ui/toolbar_visible",
                                  self->toolbar_visible, &error);
        }
        break;
    case CONFIGURATION_PROP_STATUSBAR_VISIBLE:
        self->statusbar_visible = g_value_get_boolean(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_bool(self->gconf_client,
                                  GCONF_PATH "/ui/statusbar_visible",
                                  self->statusbar_visible, &error);
        }
        break;
    case CONFIGURATION_PROP_PANE_WIDTH:
        self->pane_width = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/ui/pane_width",
                                 self->pane_width, &error);
        }
        break;
    case CONFIGURATION_PROP_UNDO_LEVELS:
        self->undo_levels = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/environment/undo_levels",
                                 self->undo_levels, &error);
        }
        break;
    case CONFIGURATION_PROP_VISIBLE_SECRETS:
        self->visible_secrets = g_value_get_boolean(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_bool(self->gconf_client,
                                  GCONF_PATH "/ui/visible_secrets",
                                  self->visible_secrets, &error);
        }
        break;
    case CONFIGURATION_PROP_LOCK:
        self->lock = g_value_get_boolean(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_bool(self->gconf_client,
                                  GCONF_PATH "/ui/lock",
                                  self->lock, &error);
        }
        break;
    case CONFIGURATION_PROP_LOCK_TIMEOUT:
        self->lock_timeout = g_value_get_int(value);
        if (self->gconf_client != NULL) {
            gconf_client_set_int(self->gconf_client,
                                 GCONF_PATH "/ui/lock_timeout",
                                 self->lock_timeout, &error);
        }
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
    if (error != NULL) {
        gpass_error_show_and_exit(error);
    }
}

static void
gpass_configuration_get_property(GObject *object, guint prop_id,
                                 GValue *value, GParamSpec *pspec)
{
    GPassConfiguration *self = GPASS_CONFIGURATION(object);
    GConfValue *cval;
    const gchar *path;
    GError *error = NULL;
    
    switch (prop_id) {
    case CONFIGURATION_PROP_WINDOW_X:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/window_x";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->window_x = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
            else {
                path = GCONF_PATH "/ui/main_window_position_x";
                cval = gconf_client_get(self->gconf_client, path, &error);
                if (error == NULL && cval != NULL) {
                    self->window_x = gconf_value_get_int(cval);
                    gconf_value_free(cval);
                    gconf_client_unset(self->gconf_client, path, &error);
                }
            }
        }
        g_value_set_int(value, self->window_x);
        break;
    case CONFIGURATION_PROP_WINDOW_Y:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/window_y";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->window_y = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
            else {
                path = GCONF_PATH "/ui/main_window_position_y";
                cval = gconf_client_get(self->gconf_client, path, &error);
                if (error == NULL && cval != NULL) {
                    self->window_y = gconf_value_get_int(cval);
                    gconf_value_free(cval);
                    gconf_client_unset(self->gconf_client, path, &error);
                }
            }
        }
        g_value_set_int(value, self->window_y);
        break;
    case CONFIGURATION_PROP_WINDOW_WIDTH:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/window_width";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->window_width = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
            else {
                path = GCONF_PATH "/ui/window_width";
                cval = gconf_client_get(self->gconf_client, path, &error);
                if (error == NULL && cval != NULL) {
                    self->window_width = gconf_value_get_int(cval);
                    gconf_value_free(cval);
                    gconf_client_unset(self->gconf_client, path, &error);
                }
            }
        }
        g_value_set_int(value, self->window_width);
        break;
    case CONFIGURATION_PROP_WINDOW_HEIGHT:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/window_height";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->window_height = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
            else {
                path = GCONF_PATH "/ui/window_height";
                cval = gconf_client_get(self->gconf_client, path, &error);
                if (error == NULL && cval != NULL) {
                    self->window_height = gconf_value_get_int(cval);
                    gconf_value_free(cval);
                    gconf_client_unset(self->gconf_client, path, &error);
                }
            }
        }
        g_value_set_int(value, self->window_height);
        break;
    case CONFIGURATION_PROP_TOOLBAR_VISIBLE:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/toolbar_visible";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->toolbar_visible = gconf_value_get_bool(cval);
                gconf_value_free(cval);
            }
            else {
                path = GCONF_PATH "/ui/toolbar_hidden";
                cval = gconf_client_get(self->gconf_client, path, &error);
                if (error == NULL && cval != NULL) {
                    self->toolbar_visible = !gconf_value_get_bool(cval);
                    gconf_value_free(cval);
                    gconf_client_unset(self->gconf_client, path, &error);
                }
            }
        }
        g_value_set_boolean(value, self->toolbar_visible);
        break;
    case CONFIGURATION_PROP_STATUSBAR_VISIBLE:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/statusbar_visible";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->statusbar_visible = gconf_value_get_bool(cval);
                gconf_value_free(cval);
            }
            else {
                path = GCONF_PATH "/ui/statusbar_hidden";
                cval = gconf_client_get(self->gconf_client, path, &error);
                if (error == NULL && cval != NULL) {
                    self->statusbar_visible = !gconf_value_get_bool(cval);
                    gconf_value_free(cval);
                    gconf_client_unset(self->gconf_client, path, &error);
                }
            }
        }
        g_value_set_boolean(value, self->statusbar_visible);
        break;
    case CONFIGURATION_PROP_PANE_WIDTH:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/pane_width";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->pane_width = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
        }
        g_value_set_int(value, self->pane_width);
        break;
    case CONFIGURATION_PROP_UNDO_LEVELS:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/environment/undo_levels";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->undo_levels = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
        }
        g_value_set_int(value, self->undo_levels);
        break;
    case CONFIGURATION_PROP_VISIBLE_SECRETS:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/visible_secrets";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->visible_secrets = gconf_value_get_bool(cval);
                gconf_value_free(cval);
            }
        }
        g_value_set_boolean(value, self->visible_secrets);
        break;
    case CONFIGURATION_PROP_LOCK:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/lock";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->lock = gconf_value_get_bool(cval);
                gconf_value_free(cval);
            }
        }
        g_value_set_boolean(value, self->lock);
        break;
    case CONFIGURATION_PROP_LOCK_TIMEOUT:
        if (self->gconf_client != NULL) {
            path = GCONF_PATH "/ui/lock_timeout";
            cval = gconf_client_get(self->gconf_client, path, &error);
            if (error == NULL && cval != NULL) {
                self->lock_timeout = gconf_value_get_int(cval);
                gconf_value_free(cval);
            }
        }
        g_value_set_int(value, self->lock_timeout);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
    if (error != NULL) {
        gpass_error_show_and_exit(error);
    }
}

static void
gpass_configuration_instance_finalize(GObject *object)
{
    GPassConfiguration *self = GPASS_CONFIGURATION(object);

    if (self->gconf_client != NULL) {
        g_object_unref(self->gconf_client);
    }
    g_datalist_clear(&self->launchers);
    parent_class->finalize(object);
}

static void
gpass_configuration_class_init (gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);

    parent_class = g_type_class_peek_parent(g_class);
    gobject_class->set_property = gpass_configuration_set_property;
    gobject_class->get_property = gpass_configuration_get_property;
    gobject_class->finalize = gpass_configuration_instance_finalize;

    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_GCONF_CLIENT,
         g_param_spec_object("gconf_client", _("GConfClient"),
                             _("The pointer of GConfClient object"),
                             GCONF_TYPE_CLIENT, G_PARAM_WRITABLE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_WINDOW_X,
         g_param_spec_int("window_x", _("Window x coordinate"),
                          _("The x coordinate of main window"),
                          0, G_MAXINT, 0, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_WINDOW_Y,
         g_param_spec_int("window_y", _("Window y coordinate"),
                          _("The y coordinate of main window"),
                          0, G_MAXINT, 0, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_WINDOW_WIDTH,
         g_param_spec_int("window_width", _("Window width"),
                          _("The width of main window"),
                          0, G_MAXINT, 500, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_WINDOW_HEIGHT,
         g_param_spec_int("window_height", _("Window height"),
                          _("The height of main window"),
                          0, G_MAXINT, 300, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_TOOLBAR_VISIBLE,
         g_param_spec_boolean("toolbar_visible", _("Toolbar visible"),
                              _("Whether visible toolbar"),
                              TRUE, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_STATUSBAR_VISIBLE,
         g_param_spec_boolean("statusbar_visible", _("Statusbar visible"),
                              _("Whether visible statusbar"),
                              TRUE, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_PANE_WIDTH,
         g_param_spec_int("pane_width", _("Pane width"),
                          _("The width of entries tree pane"),
                          0, G_MAXINT, 250, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_UNDO_LEVELS,
         g_param_spec_int("undo_levels", _("Undo levels"),
                          _("The number of undo levels"),
                          0, G_MAXINT, 15, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_VISIBLE_SECRETS,
         g_param_spec_boolean("visible_secrets", _("Visible secrets"),
                              _("Whether visible secrets "
                                "at some entry and summary"),
                              FALSE, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_LOCK,
         g_param_spec_boolean("lock", _("Lock"),
                              _("Whether automatically lock the main window "
                                "after a period of inactivity"),
                              TRUE, G_PARAM_READWRITE));
    g_object_class_install_property
        (gobject_class, CONFIGURATION_PROP_LOCK_TIMEOUT,
         g_param_spec_int("lock_timeout", _("Lock timeout"),
                          _("The number of minutes of inactivity "
                            "before the main window is automatically locked."),
                          0, G_MAXINT, 15, G_PARAM_READWRITE));
}

GType
gpass_configuration_get_type(void)
{
    static GType type = 0;
    
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassConfigurationClass),
            NULL,
            NULL,
            gpass_configuration_class_init,
            NULL,
            NULL,
            sizeof(GPassConfiguration),
            0,
            gpass_configuration_instance_init
        };
        
        type = g_type_register_static(G_TYPE_OBJECT,
                                      "GPassConfiguration", &info, 0);
    }
    return type;
}

static GPassConfiguration *instance = NULL;

GPassConfiguration *
gpass_configuration_instance(void)
{
    if (instance == NULL) {
        instance = g_object_new(GPASS_TYPE_CONFIGURATION, NULL);
    }
    return instance;
}

void
gpass_configuration_finalize(void)
{
    if (instance != NULL) {
        g_object_unref(instance);
        instance = NULL;
    }
}

void
gpass_configuration_get_launcher(GPassConfiguration *self, const gchar *type,
                                 gchar **launcher)
{
    GError *error = NULL;

    if (self->gconf_client == NULL) {
        *launcher = g_strdup(g_datalist_get_data(&self->launchers, type));
    }
    else {
        gchar *path = g_strdup_printf("%s/%s", GCONF_PATH "/launcher", type);
        GConfValue *value = gconf_client_get(self->gconf_client, path, &error);

        if (error == NULL && value != NULL) {
            *launcher = g_strdup(gconf_value_get_string(value));
            gconf_value_free(value);
            g_datalist_set_data_full(&self->launchers, type,
                                     g_strdup(*launcher), g_free);
        }
        else {
            *launcher = NULL;
        }
        g_free(path);
    }
    if (error != NULL) {
        gpass_error_show_and_exit(error);
    }
}

void
gpass_configuration_set_launcher(GPassConfiguration *self, const gchar *type,
                                 const gchar *launcher)
{
    GError *error = NULL;
    
    if (self->gconf_client != NULL) {
        gchar *path = g_strdup_printf("%s/%s", GCONF_PATH "/launcher", type);
        
        gconf_client_set_string(self->gconf_client, path, launcher, &error);
        g_free(path);
    }
    g_datalist_set_data_full(&self->launchers, type,
                             g_strdup(launcher), g_free);
    if (error != NULL) {
        gpass_error_show_and_exit(error);
    }
}
