// -*- c++ -*-
/***************************************************************************
                           vbimanager.h
                           ------------
    begin                : Fri Oct 31 2003
    copyright            : (C) 2003 by Dirk Ziegelmeier
    email                : dziegel@gmx.de
 ***************************************************************************/

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef VBIMANAGER_H
#define VBIMANAGER_H

#include <qobject.h>

class KdetvVbiPlugin;
class PluginFactory;

// ---------------------------------------------- Manager

/**
 * Provides access to the current VBI decoder
 */
class VbiManager : public QObject
{
    Q_OBJECT

public:
    /// Scans for VBI plugins
    void scanPlugins();

    /// Restart plugins
    void restartPlugin();

    /// Returns whether decoder is _running_ (!= receiving data)
    virtual bool decoding() const;

    /// Returns whether decoder receives data
    virtual bool tuned() const;

    /// Returns pointer to internal decoder data. Platform specific!!!
    void* internalData() const;

    /// Call this before you are going to use the decoder
    void addClient();

    /// Call if you stop using the decoder
    void removeClient();

    /// Suspends the sending of VBI data
    void suspend();
    
    /// Resumes the sending of VBI data
    void resume();

signals:
    /// New network identification received
    void networkId(const QString& name, int& id, const QString& call);

    /// Closed caption data received
    void caption(int pgno);

    /// Teletext page received
    void ttxPage(int pgno, int subno, int pno, bool roll, bool header, bool clock);

    /// Aspect ratio changed
    void aspect(int fline, int lline, double ratio, bool mode, int subtitles);

    /// Program title info received
    void progTitle(const QString& title);

    /// Program rating information changed
    void ratingChanged(const QString& rating);

    /// Decoder state changed. If true, reinit your client. If false, don't touch internal data pointer!
    void running(bool);


protected:
    friend class Kdetv;
    VbiManager(PluginFactory* pf);
    ~VbiManager();

private:
    void customEvent(QCustomEvent* e);

    PluginFactory*  _pf;
    KdetvVbiPlugin* _plugin;
    int             _clients;
    int             _suspendCount;
};

// ---------------------------------------------- Events the manager understands

enum { 
    EventIdStationName = QEvent::User,
    EventIdCaption,
    EventIdTtx,
    EventIdAspect,
    EventIdProgTitle,
    EventIdRating,
    EventIdRunning
};

class EventStationName : public QCustomEvent
{
public:
    EventStationName(const QString& name, int id, const QString& call);
    virtual ~EventStationName();

    QString  _name;
    int      _id;
    QString  _call;
};

class EventCaption : public QCustomEvent
{
public:
    EventCaption(int pgno);
    virtual ~EventCaption();

    int _pgno;
};

class EventTtx : public QCustomEvent
{
public:
    EventTtx(int pgno, int subno, int pno, 
	     bool roll, bool header, bool clock);
    virtual ~EventTtx();

    int		   _pgno;
    int		   _subno;
    int		   _pno;
    bool           _roll;
    bool           _header;
    bool           _clock;
};

class EventAspect : public QCustomEvent
{
public:
    EventAspect(int fline, int lline, double ratio, bool mode, int subtitles);
    virtual ~EventAspect();

    int	   _fline;
    int	   _lline;
    double _ratio;
    bool   _mode;
    int	   _subtitles;
};

class EventProgTitle : public QCustomEvent
{
public:
    EventProgTitle(const QString& title);
    virtual ~EventProgTitle();

    QString  _title;
};

class EventRating : public QCustomEvent
{
public:
    EventRating(const QString& rating);
    virtual ~EventRating();

    QString _rating;
};

class EventRunning : public QCustomEvent
{
public:
    EventRunning(bool running);
    virtual ~EventRunning();

    bool _running;
};

#endif
