/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 07-24-2007
 * 
 * Copyright 2007-2008 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

#include "exif.h"

#include <QString>
#include <QByteArray>
#include <QTextCodec>
#include <QDataStream>
#include <QTextStream>
#include <QApplication>

#define TIFFSTART   10

// See http://park2.wakwak.com/~tsuruzoh/Computer/Digicams/exif-e.html
// These are the size type, unsigned byte, ascii strings,...
int sizes[] = {1,1,2,4,8,1,1,2,4,8,4,8};

class EXIF::PrivateData
{
public:
  PrivateData()
    : exif()
    , comment(QString::null)
  {
  }

  QByteArray exif;      /// storage for the EXIF data
  QString comment;      /// storage for a jpeg meta data comment string
};


EXIF::EXIF(const QString& imageFilename)
  : d(*(new EXIF::PrivateData()))
{
  QByteArray c;
  //    quint8 e;
  quint16 s;
  quint16 marker;
  quint16 dataLength;
  int numFound = 0;  // We are finding 2 things, EXIF data and Comment. Return after found.

  QFile imageFile(imageFilename);
  if ( imageFile.size() < 2 || !imageFile.open(QIODevice::ReadOnly) )
  {
    return;
  }

  // Based on data from the web EXIF data will be at the start of the JPEG
  // file. All JPEG files start with 0xFFD8. This might be followed by JFIF
  // marker (0xFFE0) and data and then possible EXIF marker (0xFFE1) if
  // there is EXIF data. The next two bytes will give the length
  // from the start of the EXIF marker to the end of the EXIF data.

  // Initialize the stream. JPEG uses Big Endian so we don't have to worry
  // about switching the endianess at this point.
  QDataStream stream(&imageFile);

  // First ensure this is a valid JPEG file.
  stream >> s;
  if( s == 0xFFD8 )
  {
    while( (imageFile.pos() + 2) <= imageFile.size() )
    {
      // Get next marker
      stream >> marker;
      switch(marker)
      {
      case 0xFFE1:    // EXIF marker
	stream >> dataLength;
	d.exif.resize(dataLength + 2);
	imageFile.seek(imageFile.pos() - 4);   // backup 4 bytes to get the marker and size
	if( imageFile.read(d.exif.data(), dataLength + 2) != (dataLength + 2))
	{
	  d.exif.resize(0);
	  Q_ASSERT(0);    // really have to look at what failed here
	  return;
	}
	if( ++numFound == 2 )
	{
	  return;
	}
	break;
      case 0xFFFE:    // JPEG comment
	stream >> dataLength;
	c.resize(dataLength - 2); // remove length as it is included initially
	if( imageFile.read(c.data(), dataLength - 2) != (dataLength - 2))
	{
	  Q_ASSERT(0);    // really have to look at what failed here
	  return;
	}
	d.comment = c;
	if( ++numFound == 2 )
	{
	  return;
	}
	break;
      case 0xFFDA:    // Start if scan
      case 0xFFD9:    // EOI (End Of Image marker).
	imageFile.seek(imageFile.size());
	break;
	/* The following is actually how to handle the start of scan, however,
	   from what I can see this is always last in the file and causes too
	   much slowdown. So I can either thread this or as I am doing for now
	   just assuming if this is hit that I have hit the end of useful data.
	   --
	   case 0xFFDA:    // Start of scan, skip to next header by scanning for it.
	   stream >> dataLength;
	   if( !imageFile.seek(imageFile.pos() + dataLength - 2) )
	   {
	   Q_ASSERT(0);    // really have to look at what failed here
	   return;
	   }
	   while( imageFile.pos() + 2 <= imageFile.size() )
	   {
	   stream >> e;
	   if( e == 0xFF )
	   {
	   stream >> e;
	   if( e != 0x00 )
	   {
	   // Ensure not a restart marker 0xD0 - 0xD7
	   if( 0xD0 > e && e > 0xD7 )
	   {
	   imageFile.seek(imageFile.pos() - 2); // backup
	   break;
	   }
	   }
	   }
	   }
	   break;
	*/
      default:
	stream >> dataLength;
	if( !imageFile.seek(imageFile.pos() + dataLength - 2) )
	{
	  Q_ASSERT(0);    // really have to look at what failed here
	  return;
	}
      }
    }
  }
}

QByteArray EXIF::rawData()
{
  return d.exif;
}

QString EXIF::userComment()
{
  int num;
  int type;
  int offset;
  QString EXIFComment;
  quint16 tag = 0x9286;

  // JPEG comment outside the EXIF will be used if avail.
  if( d.comment.trimmed() == "" )
  {
    return d.comment;
  }

  // Otherwise look for the UserComment in the EXIF
  if( findData(tag, &type, &offset, &num) )
  {
    QByteArray cstr;
    cstr.resize(8);

    qstrncpy(cstr.data(), d.exif.data() + (offset + TIFFSTART), 8);

    if( cstr == "ASCII" )
    {
      QByteArray b(num - 8, '\0');
      qstrncpy(b.data(), d.exif.data() + (offset + TIFFSTART + 8), (num - 8));
      EXIFComment = QString(b);
    }
    else if( cstr == "UNICODE" )
    {
      ///TODO Fix this to work in either byte order
      // Most likely will not handle byte order issues but I will get to that
      // when I move to QT4 and Unicode has better support.
      QByteArray b(num - 8, '\0');
      qstrncpy(b.data(), d.exif.data() + (offset + TIFFSTART + 8), (num - 8));

      QTextStream ts(b, QIODevice::ReadOnly);
      ts.setCodec("UTF-8");

      while( !ts.atEnd() )
      {
	EXIFComment += ts.readLine();
      }
    }

    if( EXIFComment.trimmed() != "" )
    {
      return EXIFComment.trimmed();
    }
  }

  // No useful comment found in JPEG or EXIF
  return QString::null;
}

bool EXIF::findData(const quint16 tag, int* type, int* offset, int* num)
{
  quint16 s;
  quint32 t;
  QString str;
  QByteArray a;
  quint16 numEntries;
  quint32 subIFDOffset;
  quint32 nextIFDOffset;
  quint32 readOffset = 0;

  QDataStream stream(&d.exif, QIODevice::ReadOnly);

  *type = 0;
  *offset = 0;
  *num = 0;

  // definitely not a valid exif
  if( d.exif.size() < 14 )
  {
    return FALSE;
  }

  // remove the first unused data: APP1 Marker, APP1 Data Size and Exif Header
  // checking to ensure the header is present as expected.
  stream >> s >> s >> t >> s;
  if( t != 0x45786966 )
  {
    return FALSE;
  }

  // next comes the tiff header with the byte order
  readOffset += 2;
  stream >> s;
  if( s == 0x4949 )
  {
    // switch endianess
    stream.setByteOrder(QDataStream::LittleEndian);
  }

  // Another check we are on track ensure next comes 42
  readOffset += 2;
  stream >> s;
  if( s != 0x002A )
  {
    return FALSE;
  }

  // Offset to IFD0
  readOffset += 4;
  stream >> t;

  // progress to IFD0
  if( t != readOffset )
  {
    a.resize(t - readOffset);
    stream.readRawData(a.data(), (t - readOffset));
    readOffset = t;
  }

  // number of directory entries
  readOffset += 2;
  stream >> numEntries;

  // cycle through entries to find the tag of interest
  while( numEntries > 0 )
  {
    readOffset += 2;
    stream >> s;
    if( s == tag )
    {
      stream >> s;
      *type = s;
      stream >> t;
      *num = t;
      stream >> t;
      *offset = t;
      return TRUE;
    }
    else if( s == 0x8769 )
    {
      readOffset += 10;
      stream >> s >> t >> subIFDOffset;
    }
    else
    {
      readOffset += 10;
      stream >> s >> t >> t;
    }
    --numEntries;
  }

  // tag was not found in IFD0, move on to SubIFD read in the
  // readOffset to next IFD however at this point in time we won't use it.
  readOffset += 4;
  stream >> nextIFDOffset;

  // move to subIFD
  if( subIFDOffset != readOffset )
  {
    a.resize(subIFDOffset - readOffset);
    stream.readRawData(a.data(), (subIFDOffset - readOffset));
    readOffset = subIFDOffset;
  }

  // number of directory entries
  readOffset += 2;
  stream >> numEntries;

  // cycle through entries to find the tag of interest
  while( numEntries > 0 )
  {
    readOffset += 2;
    stream >> s;
    if( s == tag )
    {
      stream >> s;
      *type = s;
      stream >> t;
      *num = t;
      stream >> t;
      *offset = t;
      return TRUE;
    }
    else
    {
      readOffset += 10;
      stream >> s >> t >> t;
    }
    --numEntries;
  }
  return FALSE;
}
