/**
 * @file libgalago-gtk/galago-gtk-account-combo-box.c Account ComboBox widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-account-combo-box.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago/galago.h>
#include <string.h>

struct _GalagoGtkAccountComboBoxPriv
{
	GalagoPerson *person;
	GalagoService *service;

	gboolean me_person_set;
	gboolean me_accounts;

	gboolean rebuilding_list;

	GtkWidget *service_combobox;

	gint service_changed_id;
	gint service_combo_destroyed_id;

	gulong account_added_id;
	gulong account_removed_id;
	gulong account_conn_state_changed_id;
	gulong person_added_id;
	gulong person_destroy_id;
};

enum
{
	ACCOUNT_CHANGED,
	LAST_SIGNAL
};

enum
{
	COLUMN_DATA,
	COLUMN_ICON,
	COLUMN_NAME,
	NUM_COLUMNS
};

static void galago_gtk_account_combo_box_class_init(
	GalagoGtkAccountComboBoxClass *klass);
static void galago_gtk_account_combo_box_init(
	GalagoGtkAccountComboBox *combobox);
static void galago_gtk_account_combo_box_finalize(GObject *obj);
static void galago_gtk_account_combo_box_destroy(GtkObject *obj);
static gboolean account_added_removed_cb(GSignalInvocationHint *ihint,
										 guint n_param_values,
										 const GValue *param_values,
										 gpointer data);
static gboolean account_conn_changed_cb(GSignalInvocationHint *ihint,
										guint n_param_values,
										const GValue *param_values,
										gpointer data);
static void person_destroy_cb(GalagoPerson *person,
							  GalagoGtkAccountComboBox *combobox);

static GtkComboBoxClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

GType
galago_gtk_account_combo_box_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkAccountComboBoxClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_account_combo_box_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkAccountComboBox),
			0,
			(GInstanceInitFunc)galago_gtk_account_combo_box_init
		};

		type = g_type_register_static(GTK_TYPE_COMBO_BOX,
									  "GalagoGtkAccountComboBox", &info, 0);
	}

	return type;
}

static void
galago_gtk_account_combo_box_class_init(GalagoGtkAccountComboBoxClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_account_combo_box_finalize;

	object_class->destroy = galago_gtk_account_combo_box_destroy;

	signals[ACCOUNT_CHANGED] =
		g_signal_new("account_changed",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkAccountComboBoxClass,
									 account_changed),
					 NULL, NULL,
					 gtk_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);
}

static void
combo_box_changed(GalagoGtkAccountComboBox *combobox, gpointer user_data)
{
	GalagoAccount *account;

	account = galago_gtk_account_combo_box_get_account(combobox);

	g_signal_emit(combobox, signals[ACCOUNT_CHANGED], 0, account);
}

static void
galago_gtk_account_combo_box_init(GalagoGtkAccountComboBox *combobox)
{
	GtkListStore *store;
	GtkCellRenderer *cell;

	combobox->priv = g_new0(GalagoGtkAccountComboBoxPriv, 1);

	store = gtk_list_store_new(NUM_COLUMNS, G_TYPE_POINTER,
							   GDK_TYPE_PIXBUF, G_TYPE_STRING);

	gtk_combo_box_set_model(GTK_COMBO_BOX(combobox),
							GTK_TREE_MODEL(store));

	g_object_unref(G_OBJECT(store));

	cell = gtk_cell_renderer_pixbuf_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combobox), cell, FALSE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combobox), cell,
								   "pixbuf", COLUMN_ICON,
								   NULL);

	cell = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combobox), cell, TRUE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combobox), cell,
								   "text", COLUMN_NAME,
								   NULL);

	g_signal_connect(G_OBJECT(combobox), "changed",
					 G_CALLBACK(combo_box_changed), NULL);

	/*
	 * Ensure that we have these types loaded first, or glib will yell at us.
	 */
	g_type_class_ref(GALAGO_TYPE_SERVICE);
	g_type_class_ref(GALAGO_TYPE_ACCOUNT);

	combobox->priv->account_added_id =
		g_signal_add_emission_hook(g_signal_lookup("account-added",
												   GALAGO_TYPE_SERVICE),
								   0, account_added_removed_cb, combobox, NULL);

	combobox->priv->account_removed_id =
		g_signal_add_emission_hook(g_signal_lookup("account-removed",
												   GALAGO_TYPE_SERVICE),
								   0, account_added_removed_cb, combobox, NULL);

	combobox->priv->account_conn_state_changed_id =
		g_signal_add_emission_hook(g_signal_lookup("connection-state-changed",
												   GALAGO_TYPE_ACCOUNT),
								   0, account_conn_changed_cb, combobox, NULL);
}

static void
galago_gtk_account_combo_box_finalize(GObject *obj)
{
	GalagoGtkAccountComboBox *combobox;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(obj));

	combobox = GALAGO_GTK_ACCOUNT_COMBO_BOX(obj);

	g_free(combobox->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_account_combo_box_destroy(GtkObject *obj)
{
	GalagoGtkAccountComboBox *combobox;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(obj));

	combobox = GALAGO_GTK_ACCOUNT_COMBO_BOX(obj);

	GALAGO_GTK_DISCONNECT_HANDLER(combobox->priv->service_combobox,
								  combobox->priv->service_changed_id);
	GALAGO_GTK_DISCONNECT_HANDLER(combobox->priv->service_combobox,
								  combobox->priv->service_combo_destroyed_id);
	GALAGO_GTK_DISCONNECT_HANDLER(galago_get_core(),
								  combobox->priv->person_added_id);
	GALAGO_GTK_DISCONNECT_HANDLER(galago_get_core(),
								  combobox->priv->person_destroy_id);

	GALAGO_GTK_DISCONNECT_EMISSION_HANDLER(
		GALAGO_TYPE_SERVICE, "account-added",
		combobox->priv->account_added_id);

	GALAGO_GTK_DISCONNECT_EMISSION_HANDLER(
		GALAGO_TYPE_SERVICE, "account-removed",
		combobox->priv->account_removed_id);

	GALAGO_GTK_DISCONNECT_EMISSION_HANDLER(
		GALAGO_TYPE_ACCOUNT, "connection-state-changed",
		combobox->priv->account_conn_state_changed_id);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
append_account(GtkListStore *store, GalagoAccount *account)
{
	GdkPixbuf *pixbuf;
	GtkTreeIter iter;

	pixbuf = galago_gdk_pixbuf_new_from_account_with_size(account,
		GTK_ICON_SIZE_MENU);

	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
					   COLUMN_DATA, account,
					   COLUMN_ICON, pixbuf,
					   COLUMN_NAME, galago_account_get_display_name(account),
					   -1);

	if (pixbuf != NULL)
		g_object_unref(G_OBJECT(pixbuf));
}

static void
append_accounts_from_person(GalagoGtkAccountComboBox *combobox,
							GtkListStore *store, GalagoPerson *person,
							GalagoService *service)
{
	GList *l;

	for (l = galago_person_get_accounts(person, TRUE);
		 l != NULL;
		 l = l->next)
	{
		GalagoAccount *account = (GalagoAccount *)l->data;

		if (service == NULL || galago_account_get_service(account) == service)
			append_account(store, account);
	}
}

static void
append_accounts_from_service(GalagoGtkAccountComboBox *combobox,
							 GtkListStore *store, GalagoService *service)
{
	GList *l;

	for (l = galago_service_get_accounts(service, TRUE);
		 l != NULL;
		 l = l->next)
	{
		append_account(store, (GalagoAccount *)l->data);
	}
}

static void
rebuild_list(GalagoGtkAccountComboBox *combobox)
{
	GtkListStore *store;

	if (combobox->priv->rebuilding_list)
		return;

	combobox->priv->rebuilding_list = TRUE;

	store = (GtkListStore *)gtk_combo_box_get_model(GTK_COMBO_BOX(combobox));

	gtk_list_store_clear(store);

	if (combobox->priv->person != NULL)
	{
		append_accounts_from_person(combobox, store,
									combobox->priv->person,
									combobox->priv->service);
	}
	else if (combobox->priv->person == NULL &&
			 combobox->priv->me_person_set)
	{
		goto exit;
	}
	else if (combobox->priv->service != NULL)
	{
		append_accounts_from_service(combobox, store,
									 combobox->priv->service);
	}
	else
	{
		GList *l;

		for (l = galago_get_services(GALAGO_REMOTE, TRUE);
			 l != NULL;
			 l = l->next)
		{
			append_accounts_from_service(combobox, store,
										 (GalagoService *)l->data);
		}
	}

	gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);

exit:
	combobox->priv->rebuilding_list = FALSE;
}

static gboolean
account_added_removed_cb(GSignalInvocationHint *ihint, guint n_param_values,
						 const GValue *param_values, gpointer data)
{
	GalagoGtkAccountComboBox *combobox = (GalagoGtkAccountComboBox *)data;
	GalagoService *service;
	GalagoAccount *cur_account;
	GalagoAccount *account;

	g_assert(n_param_values > 1);

	service = g_value_get_object(&param_values[0]);
	account = g_value_get_pointer(&param_values[1]);

	if (combobox->priv->service != NULL && service != combobox->priv->service)
		return TRUE;

	cur_account = galago_gtk_account_combo_box_get_account(combobox);

	rebuild_list(combobox);

	if ((ihint->signal_id == g_signal_lookup("account-added",
											 GALAGO_TYPE_SERVICE) &&
		 cur_account != NULL) ||
		(ihint->signal_id == g_signal_lookup("account-removed",
											 GALAGO_TYPE_SERVICE) &&
		 cur_account == account))
	{
		if (galago_gtk_account_combo_box_get_account(combobox) != cur_account)
			galago_gtk_account_combo_box_set_account(combobox, cur_account);
	}
	else
	{
		gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);
	}

	return TRUE;
}

static gboolean
account_conn_changed_cb(GSignalInvocationHint *ihint, guint n_param_values,
						const GValue *param_values, gpointer data)
{
	GalagoGtkAccountComboBox *combobox = (GalagoGtkAccountComboBox *)data;
	GalagoAccount *account = g_value_get_object(&param_values[0]);
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean cont;

	model = gtk_combo_box_get_model(GTK_COMBO_BOX(combobox));

	for (cont = gtk_tree_model_get_iter_first(model, &iter);
		 cont;
		 cont = gtk_tree_model_iter_next(model, &iter))
	{
		GalagoAccount *temp_account;

		gtk_tree_model_get(model, &iter,
						   COLUMN_DATA, &temp_account,
						   -1);

		if (temp_account == account)
		{
			GdkPixbuf *pixbuf;

			pixbuf = galago_gdk_pixbuf_new_from_account_with_size(account,
				GTK_ICON_SIZE_MENU);

			gtk_list_store_set(GTK_LIST_STORE(model), &iter,
							   COLUMN_ICON, pixbuf,
							   -1);

			if (pixbuf != NULL)
				g_object_unref(G_OBJECT(pixbuf));

			break;
		}
	}

	return TRUE;
}

static void
person_added_cb(GalagoObject *core, GalagoPerson *person,
				GalagoGtkAccountComboBox *combobox)
{
	if (combobox->priv->me_person_set && galago_person_is_me(person))
		galago_gtk_account_combo_box_set_person(combobox, person);
}

static void
person_destroy_cb(GalagoPerson *person, GalagoGtkAccountComboBox *combobox)
{
	combobox->priv->person = NULL;

	GALAGO_GTK_DISCONNECT_HANDLER(galago_get_core(),
								  combobox->priv->person_destroy_id);

	if (galago_person_is_me(person))
	{
		combobox->priv->person_added_id =
			g_signal_connect(galago_get_core(), "person-added",
							 G_CALLBACK(person_added_cb), combobox);
	}
}

GtkWidget *
galago_gtk_account_combo_box_new(gboolean me_accounts)
{
	GalagoGtkAccountComboBox *combobox;

	combobox = g_object_new(GALAGO_GTK_TYPE_ACCOUNT_COMBO_BOX, NULL);

	combobox->priv->me_accounts = me_accounts;

	if (me_accounts)
	{
		galago_gtk_account_combo_box_set_person(combobox,
			galago_get_me(GALAGO_REMOTE, TRUE));
	}

	rebuild_list(combobox);

	gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), 0);

	return GTK_WIDGET(combobox);
}

void
galago_gtk_account_combo_box_set_person(GalagoGtkAccountComboBox *combobox,
										GalagoPerson *person)
{
	g_return_if_fail(combobox != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox));

	if (combobox->priv->person == person)
		return;

	combobox->priv->person = person;

	if (person != NULL)
	{
		combobox->priv->me_person_set = galago_person_is_me(person);

		combobox->priv->person_destroy_id =
			g_signal_connect(G_OBJECT(person), "destroy",
							 G_CALLBACK(person_destroy_cb), combobox);
	}
	else
		combobox->priv->me_person_set = FALSE;

	rebuild_list(combobox);
}

GalagoPerson *
galago_gtk_account_combo_box_get_person(
	const GalagoGtkAccountComboBox *combobox)
{
	g_return_val_if_fail(combobox != NULL,                          NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox), NULL);

	return combobox->priv->person;
}

void
galago_gtk_account_combo_box_set_service(GalagoGtkAccountComboBox *combobox,
										 GalagoService *service)
{
	g_return_if_fail(combobox != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox));

	if (combobox->priv->service == service)
		return;

	combobox->priv->service = service;

	rebuild_list(combobox);
}

GalagoService *
galago_gtk_account_combo_box_get_service(
	const GalagoGtkAccountComboBox *combobox)
{
	g_return_val_if_fail(combobox != NULL,                          NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox), NULL);

	return combobox->priv->service;
}

void
galago_gtk_account_combo_box_set_account(GalagoGtkAccountComboBox *combobox,
										 const GalagoAccount *account)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean cont;

	g_return_if_fail(combobox != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox));
	g_return_if_fail(account != NULL);

	model = gtk_combo_box_get_model(GTK_COMBO_BOX(combobox));

	for (cont = gtk_tree_model_get_iter_first(model, &iter);
		 cont;
		 cont = gtk_tree_model_iter_next(model, &iter))
	{
		GalagoAccount *temp_account;

		gtk_tree_model_get(model, &iter,
						   COLUMN_DATA, &temp_account,
						   -1);

		if (temp_account == account)
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combobox), &iter);
	}
}

GalagoAccount *
galago_gtk_account_combo_box_get_account(
	const GalagoGtkAccountComboBox *combobox)
{
	GalagoAccount *account = NULL;
	GtkTreeModel *model;
	GtkTreeIter iter;

	g_return_val_if_fail(combobox != NULL,                          NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox), NULL);

	model = gtk_combo_box_get_model(GTK_COMBO_BOX(combobox));

	if (gtk_combo_box_get_active_iter(GTK_COMBO_BOX(combobox), &iter))
	{
		gtk_tree_model_get(model, &iter,
						   COLUMN_DATA, &account,
						   -1);
	}

	return account;
}

static void
service_changed_cb(GalagoGtkServiceComboBox *service_combobox,
				   GalagoService *service, GalagoGtkAccountComboBox *combobox)
{
	galago_gtk_account_combo_box_set_service(combobox,
		galago_gtk_service_combo_box_get_service(service_combobox));
}

static void
service_combo_box_destroyed_cb(GalagoGtkServiceComboBox *service_combobox,
							   GalagoGtkAccountComboBox *combobox)
{
	combobox->priv->service_changed_id         = 0;
	combobox->priv->service_combo_destroyed_id = 0;
	combobox->priv->service_combobox           = NULL;

	galago_gtk_account_combo_box_set_service(combobox, NULL);
}

void
galago_gtk_account_combo_box_set_services_combo_box(
	GalagoGtkAccountComboBox *combobox, GtkWidget *service_combobox)
{
	GalagoService *service = NULL;

	g_return_if_fail(combobox != NULL);
	g_return_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox));
	g_return_if_fail(service_combobox == NULL ||
					 GALAGO_GTK_IS_SERVICE_COMBO_BOX(service_combobox));

	if (service_combobox == combobox->priv->service_combobox)
		return;

	GALAGO_GTK_DISCONNECT_HANDLER(combobox->priv->service_combobox,
								  combobox->priv->service_changed_id);
	GALAGO_GTK_DISCONNECT_HANDLER(combobox->priv->service_combobox,
								  combobox->priv->service_combo_destroyed_id);

	combobox->priv->service_combobox = service_combobox;

	if (service_combobox != NULL)
	{
		combobox->priv->service_changed_id =
			g_signal_connect(G_OBJECT(service_combobox), "service_changed",
							 G_CALLBACK(service_changed_cb), combobox);
		combobox->priv->service_combo_destroyed_id =
			g_signal_connect(G_OBJECT(service_combobox), "destroy",
							 G_CALLBACK(service_combo_box_destroyed_cb),
							 combobox);

		service = galago_gtk_service_combo_box_get_service(
			GALAGO_GTK_SERVICE_COMBO_BOX(service_combobox));
	}

	galago_gtk_account_combo_box_set_service(combobox, service);
}

GtkWidget *
galago_gtk_account_combo_box_get_services_combo_box(
	const GalagoGtkAccountComboBox *combobox)
{
	g_return_val_if_fail(combobox != NULL,                          NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_ACCOUNT_COMBO_BOX(combobox), NULL);

	return combobox->priv->service_combobox;
}
