/* 
   Copyright (C) Andrew Tridgell 1998,
   Con Kolivas 2006-2008
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* multiplex N streams into a file - the streams are passed
   through different compressors */

#include "rzip.h"
#include <bzlib.h>
#include <zlib.h>

/* since we have both zlib.h and lzmalib.h, omit lzmalib.h
 * and just but the two function definitions it needs.
 * The two headers are identical and cause compiler warnings.
*/

ZEXTERN int ZEXPORT lzma_compress OF((Bytef *dest,   uLongf *destLen,
                                  const Bytef *source, uLong sourceLen,
                                  int level, unsigned short lc,
				  unsigned short lp, unsigned short pb,
				  unsigned threads));
ZEXTERN int ZEXPORT lzma_uncompress OF((Bytef *dest,   uLongf *destLen,
                                   const Bytef *source, uLong sourceLen, int level,
				   unsigned short lc, unsigned short lp, 
				   unsigned short pb));

#define CTYPE_NONE 3
#define CTYPE_BZIP2 4
#define CTYPE_LZO 5
#define CTYPE_LZMA 6
#define CTYPE_GZIP 7

#define STREAM_BUFSIZE (1024 * 1024 * 10)

typedef uint16 u16;
typedef uint32 u32;

struct stream {
	u32 last_head;
	uchar *buf;
	unsigned int buflen;
	unsigned int bufp;
};

struct stream_info {
	struct stream *s;
	int num_streams;
	int fd;
	u32 bufsize;
	u32 cur_pos;
	off_t initial_pos;
	off_t total_read;
};

/* just to keep things clean, declare function here
 * but move body to the end since it's a work function
*/
 
static int lzo_compresses( struct stream *s );

/*
  ***** COMPRESSION FUNCTIONS *****
  
  BZIP, GZIP, LZMA, LZO
  
  try to compress a buffer. If compression fails for whatever reason then
  leave uncompressed. Return the compression type in c_type and resulting
  length in c_len
*/

static void bzip2_compress_buf(struct stream *s, int *c_type, u32 *c_len)
{
	uchar *c_buf;
	unsigned int dlen = s->buflen-1;

	c_buf = malloc(dlen);
	if (!c_buf)
		return;

	if (BZ2_bzBuffToBuffCompress((char*)c_buf, &dlen, (char*)s->buf, s->buflen, 
				     control.compression_level, 0,
				     control.compression_level*10) != BZ_OK) {
		free(c_buf);
		return;
	}

	if (dlen >= *c_len) {
		/* Incompressible, leave as CTYPE_NONE */
		free(c_buf);
		return;
	}

	*c_len = dlen;
	free(s->buf);
	s->buf = c_buf;
	*c_type = CTYPE_BZIP2;
}

static void gzip_compress_buf(struct stream *s, int *c_type, u32 *c_len)
{
	uchar *c_buf;
	unsigned long dlen = s->buflen-1;

	c_buf = malloc(dlen);
	if (!c_buf)
		return;

	if (compress2(c_buf, &dlen, s->buf, s->buflen, 
				     control.compression_level) != Z_OK) {
		free(c_buf);
		return;
	}

	if (dlen >= *c_len) {
		/* Incompressible, leave as CTYPE_NONE */
		free(c_buf);
		return;
	}

	*c_len = dlen;
	free(s->buf);
	s->buf = c_buf;
	*c_type = CTYPE_GZIP;
}

static void lzma_compress_buf(struct stream *s, int *c_type, u32 *c_len)
{
	uchar *c_buf = NULL;
	unsigned long dlen = s->buflen - 1;

	if (!lzo_compresses(s))
		goto out;

	c_buf = malloc(dlen);
	if (!c_buf)
		return;

	if (control.flags & FLAG_SHOW_PROGRESS) {
		printf("\tProgress percentage pausing during lzma compression...");
		fflush(stdout);
	}
	if (lzma_compress(c_buf,&dlen,(uchar*)s->buf, s->buflen, control.compression_level, 
			control.lc, control.lp, control.pb, control.threads) != 0) {
		/* can pass -1 if not compressible! Thanks Lasse Collin */
		free(c_buf);
		goto out;
	}

	if (dlen >= *c_len) {
		/* Incompressible, leave as CTYPE_NONE */
		free(c_buf);
		goto out;
	}

	*c_len = dlen;
	free(s->buf);
	s->buf = c_buf;
	*c_type = CTYPE_LZMA;
out:
	if (control.flags & FLAG_VERBOSITY_MAX)
		printf("\n");
	else if ((control.flags & FLAG_SHOW_PROGRESS || control.flags & FLAG_VERBOSITY )) 
		printf("\r\t                                                                                      \r");
	fflush(stdout);
}

static void lzo_compress_buf(struct stream *s, int *c_type, u32 *c_len)
{
	uchar *c_buf;
	lzo_bytep wrkmem;
	lzo_uint in_len = s->buflen;
	lzo_uint dlen = in_len + in_len / 16 + 64 + 3;
	lzo_int return_var;	/* lzo1x_1_compress does not return anything but LZO_OK */

	wrkmem = (lzo_bytep) malloc(LZO1X_1_MEM_COMPRESS);
	if (wrkmem == NULL)
		return;

	c_buf = malloc(dlen);
	if (!c_buf)
		goto out_free;

	return_var = lzo1x_1_compress((uchar *)s->buf,in_len,(uchar *)c_buf,&dlen,wrkmem);

	if (dlen >= in_len){
		/* Incompressible, leave as CTYPE_NONE */
		free(c_buf);
		goto out_free;
	}

	*c_len = dlen;
	free(s->buf);
	s->buf = c_buf;
	*c_type = CTYPE_LZO;
out_free:
	free(wrkmem);
}

/*
  ***** DECOMPRESSION FUNCTIONS *****
  
  BZIP, GZIP, LZMA, LZO
  
  try to decompress a buffer. Return 0 on success and -1 on failure.
*/

static int bzip2_decompress_buf(struct stream *s, u32 c_len)
{
	uchar *c_buf;
	unsigned int dlen = s->buflen;
	int bzerr;

	c_buf = s->buf;
	s->buf = malloc(dlen);
	if (!s->buf) {
		err_msg("Failed to allocate %d bytes for decompression\n", dlen);
		return -1;
	}

	bzerr = BZ2_bzBuffToBuffDecompress((char*)s->buf, &dlen, (char*)c_buf, c_len, 0, 0);
	if (bzerr != BZ_OK) {
		err_msg("Failed to decompress buffer - bzerr=%d\n", bzerr);
		return -1;
	}

	if (dlen != s->buflen) {
		err_msg("Inconsistent length after decompression. Got %d bytes, expected %d\n", dlen, s->buflen);
		return -1;
	}

	free(c_buf);
	return 0;
}

static int gzip_decompress_buf(struct stream *s, u32 c_len)
{
	uchar *c_buf;
	unsigned long dlen = s->buflen;
	int gzerr;

	c_buf = s->buf;
	s->buf = malloc(dlen);
	if (!s->buf) {
		err_msg("Failed to allocate %d bytes for decompression\n", dlen);
		return -1;
	}

	gzerr = uncompress(s->buf, &dlen, c_buf, c_len);
	if (gzerr != Z_OK) {
		err_msg("Failed to decompress buffer - bzerr=%d\n", gzerr);
		return -1;
	}

	if (dlen != s->buflen) {
		err_msg("Inconsistent length after decompression. Got %d bytes, expected %d\n", dlen, s->buflen);
		return -1;
	}

	free(c_buf);
	return 0;
}

static int lzma_decompress_buf(struct stream *s, u32 c_len)
{
	uchar *c_buf;
	unsigned long dlen = s->buflen;
	int lzmaerr;

	c_buf = s->buf;
	s->buf = malloc(dlen);
	if (!s->buf) {
		err_msg("Failed to allocate %d bytes for decompression\n", dlen);
		return -1;
	}

	lzmaerr = lzma_uncompress((uchar*)s->buf, &dlen, (uchar*)c_buf, c_len, control.compression_level,
			control.lc, control.lp, control.pb);
	if (lzmaerr != 0) {
		err_msg("Failed to decompress buffer - lzmaerr=%d\n", lzmaerr);
		return -1;
	}

	if (dlen != s->buflen) {
		err_msg("Inconsistent length after decompression. Got %d bytes, expected %d\n", dlen, s->buflen);
		return -1;
	}

	free(c_buf);
	return 0;
}

static int lzo_decompress_buf(struct stream *s, u32 c_len)
{
	uchar *c_buf;
	lzo_uint dlen = s->buflen;
	int lzerr;

	c_buf = s->buf;
	s->buf = malloc(dlen);
	if (!s->buf) {
		err_msg("Failed to allocate %d bytes for decompression\n", dlen);
		return -1;
	}

	lzerr = lzo1x_decompress((uchar*)c_buf,c_len,(uchar*)s->buf,&dlen,NULL);
	if (lzerr != LZO_E_OK) {
		err_msg("Failed to decompress buffer - lzerr=%d\n", lzerr);
		return -1;
	}

	if (dlen != s->buflen) {
		err_msg("Inconsistent length after decompression. Got %d bytes, expected %d\n", dlen, s->buflen);
		return -1;
	}

	free(c_buf);
	return 0;
}

/* WORK FUNCTIONS */

/* write to a file, return 0 on success and -1 on failure */
static int write_buf(int f, uchar *p, int len)
{
	int ret;
	ret = write(f, p, len);
	if (ret == -1) {
		err_msg("Write of length %d failed - %s\n", len, strerror(errno));
		return -1;
	}
	if (ret != len) {
		err_msg("Partial write!? asked for %d bytes but got %d\n", len, ret);
		return -1;
	}
	return 0;
}

/* write a byte */
static int write_u8(int f, uchar v)
{
	return write_buf(f, &v, 1);
}

/* write a short */
static int write_u16(int f, u16 v)
{
	uchar p[2];
	p[0] = v&0xFF;
	p[1] = (v>>8)&0xFF;
	return write_buf(f, p, 2);
}

/* write a long */
static int write_u32(int f, u32 v)
{
	if (write_u16(f, v&0xFFFF) != 0 ||
	    write_u16(f, (v>>16)) != 0) {
		return -1;
	}
	return 0;
}

static int read_buf(int f, uchar *p, int len)
{
	int ret;
	ret = read(f, p, len);
	if (ret == -1) {
		err_msg("Read of length %d failed - %s\n", len, strerror(errno));
		return -1;
	}
	if (ret != len) {
		err_msg("Partial read!? asked for %d bytes but got %d\n", len, ret);
		return -1;
	}
	return 0;
}

static int read_u8(int f, uchar *v)
{
	return read_buf(f, v, 1);
}

static int read_u16(int f, u16 *v)
{
	uchar p[2];
	if (read_buf(f, p, 2) != 0) {
		return -1;
	}
	*v = (p[1]<<8) | p[0];
	return 0;
}

static int read_u32(int f, u32 *v)
{
	u16 v1, v2;

	if (read_u16(f, &v1) != 0) {
		return -1;
	}
	if (read_u16(f, &v2) != 0) {
		return -1;
	}
	*v = v2;
	*v <<= 16;
	*v |= v1;
	return 0;
}

/* seek to a position within a set of streams - return -1 on failure */
static int seekto(struct stream_info *sinfo, u32 pos)
{
	off_t spos = pos + sinfo->initial_pos;
	if (lseek(sinfo->fd, spos, SEEK_SET) != spos) {
		err_msg("Failed to seek to %d in stream\n", pos);
		return -1;
	}
	return 0;
}

/* open a set of output streams, compressing with the given
   compression level and algorithm */
void *open_stream_out(int f, int n, unsigned int limit)
{
	int i;
	struct stream_info *sinfo;

	sinfo = malloc(sizeof(*sinfo));
	if (!sinfo) {
		return NULL;
	}

	sinfo->num_streams = n;
	sinfo->cur_pos = 0;
	sinfo->fd = f;
	/* 10MB streams for non lzma compress. There is virtually no gain
	   in lzo and bzip2 with larger streams. With lzma, however, the
	   larger the buffer, the better the compression so we make it as
	   large as the window */
	if (control.flags & FLAG_NOT_LZMA)
		sinfo->bufsize = STREAM_BUFSIZE;
	else
		sinfo->bufsize = STREAM_BUFSIZE * 10 * control.window;
	/* No point making the stream larger than the amount of data */
	if (limit && limit < sinfo->bufsize)
		sinfo->bufsize = limit;
	sinfo->initial_pos = lseek(f, 0, SEEK_CUR);

	sinfo->s = (struct stream *)calloc(sizeof(sinfo->s[0]), n);
	if (!sinfo->s) {
		free(sinfo);
		return NULL;
	}

	for (i=0;i<n;i++) {
		sinfo->s[i].buf = malloc(sinfo->bufsize);
		if (!sinfo->s[i].buf) goto failed;
	}

	/* write the initial headers */
	for (i=0;i<n;i++) {
		sinfo->s[i].last_head = sinfo->cur_pos + 9;
		write_u8(sinfo->fd, CTYPE_NONE);
		write_u32(sinfo->fd, 0);
		write_u32(sinfo->fd, 0);
		write_u32(sinfo->fd, 0);
		sinfo->cur_pos += 13;
	}
	return (void *)sinfo;

failed:
	for (i=0;i<n;i++) {
		if (sinfo->s[i].buf) free(sinfo->s[i].buf);
	}
	free(sinfo);
	return NULL;
}

/* prepare a set of n streams for reading on file descriptor f */
void *open_stream_in(int f, int n)
{
	int i;
	struct stream_info *sinfo;

	sinfo = calloc(sizeof(*sinfo), 1);
	if (!sinfo) {
		return NULL;
	}

	sinfo->num_streams = n;
	sinfo->fd = f;
	sinfo->initial_pos = lseek(f, 0, SEEK_CUR);

	sinfo->s = (struct stream *)calloc(sizeof(sinfo->s[0]), n);
	if (!sinfo->s) {
		free(sinfo);
		return NULL;
	}

	for (i=0;i<n;i++) {
		uchar c;
		u32 v1, v2;

	again:
		if (read_u8(f, &c) != 0) {
			goto failed;
		}
		if (read_u32(f, &v1) != 0) {
			goto failed;
		}
		if (read_u32(f, &v2) != 0) {
			goto failed;
		}
		if (read_u32(f, &sinfo->s[i].last_head) != 0) {
			goto failed;
		}

		if (c == CTYPE_NONE && v1==0 && v2==0 && sinfo->s[i].last_head==0 &&
		    i == 0) {
			err_msg("Enabling stream close workaround\n");
			sinfo->initial_pos += 13;
			goto again;
		}

		sinfo->total_read += 13;

		if (c != CTYPE_NONE) {
			err_msg("Unexpected initial tag %d in streams\n", c);
			goto failed;
		}
		if (v1 != 0) {
			err_msg("Unexpected initial c_len %d in streams %d\n", v1, v2);
			goto failed;
		}
		if (v2 != 0) {
			err_msg("Unexpected initial u_len %d in streams\n", v2);
			goto failed;
		}
	}

	return (void *)sinfo;

failed:
	free(sinfo->s);
	free(sinfo);
	return NULL;
}

/* flush out any data in a stream buffer. Return -1 on failure */
static int flush_buffer(struct stream_info *sinfo, int stream)
{
	int c_type = CTYPE_NONE;
	u32 c_len = sinfo->s[stream].buflen;
	
	if (seekto(sinfo, sinfo->s[stream].last_head) != 0) {
		return -1;
	}
	if (write_u32(sinfo->fd, sinfo->cur_pos) != 0) {
		return -1;
	}

	sinfo->s[stream].last_head = sinfo->cur_pos + 9;
	if (seekto(sinfo, sinfo->cur_pos) != 0) {
		return -1;
	}

	if (!(control.flags & FLAG_NO_COMPRESS)) {
		if (LZMA_COMPRESS(control.flags))
			lzma_compress_buf(&sinfo->s[stream], &c_type, &c_len);
		else if (control.flags & FLAG_LZO_COMPRESS)
			lzo_compress_buf(&sinfo->s[stream], &c_type, &c_len);
		else if (control.flags & FLAG_BZIP2_COMPRESS)
			bzip2_compress_buf(&sinfo->s[stream], &c_type, &c_len);
		else if (control.flags & FLAG_ZLIB_COMPRESS)
			gzip_compress_buf(&sinfo->s[stream], &c_type, &c_len);
		else fatal("Dunno wtf compression to use!\n");
	}
//	printf("c_type %d, c_len %u\n",c_type, c_len);

	if (write_u8(sinfo->fd, c_type) != 0 ||
	    write_u32(sinfo->fd, c_len) != 0 ||
	    write_u32(sinfo->fd, sinfo->s[stream].buflen) != 0 ||
	    write_u32(sinfo->fd, 0) != 0) {
		return -1;
	}
	sinfo->cur_pos += 13;

	if (write_buf(sinfo->fd, sinfo->s[stream].buf, c_len) != 0) {
		return -1;
	}
	sinfo->cur_pos += c_len;

	sinfo->s[stream].buflen = 0;

	free(sinfo->s[stream].buf);
	sinfo->s[stream].buf = malloc(sinfo->bufsize);
	if (!sinfo->s[stream].buf) {
		return -1;
	}
	return 0;
}

/* fill a buffer from a stream - return -1 on failure */
static int fill_buffer(struct stream_info *sinfo, int stream)
{
	uchar c_type;
	u32 u_len, c_len;

	if (seekto(sinfo, sinfo->s[stream].last_head) != 0) {
		return -1;
	}

	if (read_u8(sinfo->fd, &c_type) != 0) {
		return -1;
	}
	if (read_u32(sinfo->fd, &c_len) != 0) {
		return -1;
	}
	if (read_u32(sinfo->fd, &u_len) != 0) {
		return -1;
	}
	if (read_u32(sinfo->fd, &sinfo->s[stream].last_head) != 0) {
		return -1;
	}

	sinfo->total_read += 13;

	if (sinfo->s[stream].buf) {
		free(sinfo->s[stream].buf);
	}
	sinfo->s[stream].buf = malloc(u_len);
	if (!sinfo->s[stream].buf) {
		return -1;
	}
	if (read_buf(sinfo->fd, sinfo->s[stream].buf, c_len) != 0) {
		return -1;
	}

	sinfo->total_read += c_len;

	sinfo->s[stream].buflen = u_len;
	sinfo->s[stream].bufp = 0;

	if (c_type != CTYPE_NONE) {
		if (c_type == CTYPE_LZMA) {
			if (lzma_decompress_buf(&sinfo->s[stream], c_len))
				return -1;
		} else if (c_type == CTYPE_LZO) {
			if (lzo_decompress_buf(&sinfo->s[stream], c_len))
				return -1;
		} else if (c_type == CTYPE_BZIP2) {
			if (bzip2_decompress_buf(&sinfo->s[stream], c_len))
				return -1;
		} else if (c_type == CTYPE_GZIP) {
			if (gzip_decompress_buf(&sinfo->s[stream], c_len))
				return -1;
		} else fatal("Dunno wtf decompression type to use!\n");
	}

	return 0;
}

/* write some data to a stream. Return -1 on failure */
int write_stream(void *ss, int stream, uchar *p, unsigned int len)
{
	struct stream_info *sinfo = ss;

	while (len) {
		int n = MIN(sinfo->bufsize - sinfo->s[stream].buflen, len);

		memcpy(sinfo->s[stream].buf+sinfo->s[stream].buflen, p, n);
		sinfo->s[stream].buflen += n;
		p += n;
		len -= n;

		if (sinfo->s[stream].buflen == sinfo->bufsize) {
			if (flush_buffer(sinfo, stream) != 0) {
				return -1;
			}
		}
	}
	return 0;
}

/* read some data from a stream. Return number of bytes read, or -1
   on failure */
int read_stream(void *ss, int stream, uchar *p, unsigned int len)
{
	struct stream_info *sinfo = ss;
	int ret=0;

	while (len) {
		int n = MIN(sinfo->s[stream].buflen-sinfo->s[stream].bufp, len);

		if (n > 0) {
			memcpy(p, sinfo->s[stream].buf+sinfo->s[stream].bufp, n);
			sinfo->s[stream].bufp += n;
			p += n;
			len -= n;
			ret += n;
		}

		if (len &&
		    sinfo->s[stream].bufp == sinfo->s[stream].buflen) {
			if (fill_buffer(sinfo, stream) != 0) {
				return -1;
			}
			if (sinfo->s[stream].bufp == sinfo->s[stream].buflen) break;
		}
	}

	return ret;
}

/* flush and close down a stream. return -1 on failure */
int close_stream_out(void *ss)
{
	struct stream_info *sinfo = ss;
	int i;
	for (i=0;i<sinfo->num_streams;i++) {
		if (sinfo->s[i].buflen != 0 &&
		    flush_buffer(sinfo, i) != 0) {
			return -1;
		}
		if (sinfo->s[i].buf) free(sinfo->s[i].buf);
	}
	free(sinfo->s);
	free(sinfo);
	return 0;
}

/* close down an input stream */
int close_stream_in(void *ss)
{
	struct stream_info *sinfo = ss;
	int i;

	if (lseek(sinfo->fd, sinfo->initial_pos + sinfo->total_read, 
		  SEEK_SET) != sinfo->initial_pos + sinfo->total_read) {
		return -1;
	}
	for (i=0;i<sinfo->num_streams;i++) {
		if (sinfo->s[i].buf) free(sinfo->s[i].buf);
	}

	free(sinfo->s);
	free(sinfo);
	return 0;
}

/* As lzma is very slow and lzo very fast, it is worth doing a quick lzo pass
   to see if there is any compression at all with lzo first. It is unlikely
   that lzma will be able to compress if lzo is unable to drop a single byte
   so do not compress any block with lzma that is incompressible by lzo. */
static int lzo_compresses(struct stream *s)
{
	lzo_bytep wrkmem=NULL;
	lzo_uint in_len, test_len = s->buflen, save_len = s->buflen;
	lzo_uint dlen;
	lzo_int return_var;	/* lzo1x_1_compress does not return anything but LZO_OK */
	uchar *c_buf = NULL, *test_buf = s->buf;
	/* set minimum buffer test size based on the length of the test stream */
	unsigned long buftest_size = (test_len > 5*STREAM_BUFSIZE ? STREAM_BUFSIZE : STREAM_BUFSIZE / 4096);
	int ret = 0;
	int workcounter = 0;	/* count # of passes */
	lzo_uint best_dlen = UINT_MAX; /* save best compression estimate */

#if 0
	/* This assumption is wrong */

	/* if buffer length > window size * threshold, abort right away */
	if ( (double) save_len > (double) ( control.window * 104857600 * control.threshold) ) {
		if ( (control.flags & FLAG_VERBOSITY) || (control.flags & FLAG_VERBOSITY_MAX) ) {
			printf("\tSkipping lzo compression test. Data chunk %ld > Window * Threshold. %5.2f%%",
				save_len, 100 * 
				((double) save_len / (double) (control.window * 104857600 * control.threshold)));
			fflush(stdout);
		}
		ret=0;
		goto out;
	}
#endif
	wrkmem = (lzo_bytep) malloc(LZO1X_1_MEM_COMPRESS);
	if (wrkmem == NULL)
		goto out;

	in_len = MIN(test_len, buftest_size);
	dlen = STREAM_BUFSIZE + STREAM_BUFSIZE / 16 + 64 + 3;

	c_buf = malloc(dlen);
	if (!c_buf)
		goto out;

	if (control.flags & FLAG_SHOW_PROGRESS) {
		printf("\tlzo testing for incompressible data...");
		fflush(stdout);
	}

	/* Test progressively larger blocks at a time and as soon as anything
	   compressible is found, jump out as a success */
	while (test_len > 0) {
		workcounter++;
		return_var = lzo1x_1_compress(test_buf,in_len,(uchar *)c_buf,&dlen,wrkmem);

		/* since this test can return false positives
		 * i.e. a compressible part followed by a long random
		 * stream of data, a threshold of 90% (-T 2) for lzo compression
		 * is set by default, but can be changed at startup with the -T option.
		 * The theory is that if lzo can't compress well, neither will lzma.
		*/
		
		if ( dlen < best_dlen )
			best_dlen = dlen;	/* save best value */

		if ( (double) dlen < (double) in_len * control.threshold ) {
			ret = 1;
			break;
		}
		/* expand and move buffer */
		test_len -= in_len;
		if (test_len) {
			test_buf += (ptrdiff_t) in_len;
			if (buftest_size < STREAM_BUFSIZE)
				buftest_size <<= 1;
			in_len = MIN(test_len, buftest_size);
		}
	}
	if ( control.flags & FLAG_VERBOSITY_MAX )
		printf("%s for chunk %ld. Compressed size = %5.2F%% of chunk, %d Passes\n", 
			(ret==0?"FAILED - below threshold":"OK"), save_len, 
			100 * ((double) best_dlen / (double) in_len), workcounter);
	else if ( control.flags & FLAG_VERBOSITY )
		printf("%s\r", (ret==0?"FAILED - below threshold":"OK"));
	else if (control.flags & FLAG_SHOW_PROGRESS)
		printf("\r\t                                                      \r");
	fflush(stdout);

out:
	if (wrkmem)
		free(wrkmem);
	if (c_buf)
		free(c_buf);

	return ret;
}
