#include "common.h"

static sMenuItem* sMenuItem_new(char* name, int key, char* cmd) {
    sMenuItem* self = GC_malloc(sizeof(sMenuItem));

    self->mName = GC_strdup(name);
    self->mKey = key;
    self->mCmd = GC_strdup(cmd);

    return self;
}

sMenu* sMenu_new(char* title)
{
    sMenu* self = GC_malloc(sizeof(sMenu));

    strcpy(self->mTitle, title);
    self->mScrollTop = 0;
    self->mCursor = 0;

    self->mMenuItems = vector_new(10);

    hash_put(gMenu, title, self);

    return self;
}

sMenu* gActiveMenu = NULL;
hash_obj* gMenu;

///////////////////////////////////////////////////////////////////////////////
// 初期化
///////////////////////////////////////////////////////////////////////////////
void menu_init()
{
    gMenu = hash_new(10);
}

///////////////////////////////////////////////////////////////////////////////
// メニューに追加
///////////////////////////////////////////////////////////////////////////////
void menu_append(sMenu* self, char* name, int key, char* cmd)
{
    char buf[256];
    sprintf(buf, " %-40s", name);
    vector_add(self->mMenuItems, sMenuItem_new(buf, key, cmd));
}

///////////////////////////////////////////////////////////////////////////////
// キー入力
///////////////////////////////////////////////////////////////////////////////
void menu_input(sMenu* self, int meta, int key)
{
    if(key == 1 || key == KEY_HOME) {        // CTRL-A
        self->mCursor = 0;
    }
    else if(key == 5 || key == KEY_END) {    // CTRL-E
        self->mCursor = vector_size(self->mMenuItems)-1;
    }
    else if(key == 14 || key == 6|| key == KEY_DOWN) {    // CTRL-N CTRL-F
        self->mCursor++;
    }
    else if(key == 16 || key == 2 || key == KEY_UP) {    // CTRL-P CTRL-B
        self->mCursor--;
    }
    else if(key == 4 || key == KEY_NPAGE) { // CTRL-D PAGE DOWN
        self->mCursor+=7;
    }
    else if(key == 21 || key == KEY_PPAGE) { // CTRL-U   PAGE UP
        self->mCursor-=7;
    }
    else if(key == 3 || key == 7 || key == 27) {    // CTRL-C CTRL-G Escape
        gActiveMenu = NULL;
        mclear();
    }
    else if(key == 10 || key == 13) {    // CTRL-M CTRL-J
        gActiveMenu = NULL;
        
        sMenuItem* item = vector_item(self->mMenuItems, self->mCursor);
        
        mclear();
        view();
        mrefresh();
        
        int rcode = shell(item->mCmd, NULL, -1, FALSE, FALSE, -1);

        mclear();
    }
    else if(key == 12) {// CTRL-L
        mclear_immediately();
    }
    else {
        gActiveMenu = NULL;
        
        int i;
        for(i=0; i<vector_size(self->mMenuItems); i++) {
            sMenuItem* item = (sMenuItem*)vector_item(self->mMenuItems, i);
            
            if(item->mKey == key) {
                mclear();
                view();
                mrefresh();
                
                int rcode = shell(item->mCmd, NULL, -1, FALSE, FALSE, -1);
            }
        }
        
        mclear();
    }

    char* env = getenv("OPTION_MENU_CYCLE");
    if(env && strcmp(env, "1") == 0) {
        while(self->mCursor < 0) {
            self->mCursor += vector_size(self->mMenuItems);
        }
        while(vector_size(self->mMenuItems) != 0 
            && self->mCursor >= vector_size(self->mMenuItems)) 
        {
           self->mCursor -= vector_size(self->mMenuItems);
        }
    }
    else {
        if(self->mCursor < 0) {
            self->mCursor = 0;
        }
        if(vector_size(self->mMenuItems) != 0 
            && self->mCursor >= vector_size(self->mMenuItems)) 
        {
            self->mCursor = vector_size(self->mMenuItems)-1;
        }
    }

    const int maxy = mgetmaxy() -2;
    
    if(self->mCursor >= (self->mScrollTop+maxy)) {
        self->mScrollTop += self->mCursor - (self->mScrollTop+maxy) + 1;
    }
    if(self->mCursor < self->mScrollTop) {
        self->mScrollTop = self->mCursor;
    }
}

///////////////////////////////////////////////////////////////////////////////
// 描写
///////////////////////////////////////////////////////////////////////////////
void menu_view(sMenu* self)
{
    const int maxx = mgetmaxx();
    const int maxy = mgetmaxy();

    int item_maxx = 0;
    int i;
    for(i=0; i<vector_size(self->mMenuItems); i++) {
        sMenuItem* item = (sMenuItem*)vector_item(self->mMenuItems, i);
        
        int len = str_termlen(item->mName);

        if(item_maxx < len) {
            item_maxx = len;
        }
    }
    item_maxx+=2;
    
    if(item_maxx < 43) {
        item_maxx = 43;
    }
    else if(item_maxx > maxx-2) {
        item_maxx = maxx-2;
    }
    
    const int size = vector_size(self->mMenuItems) + 2;
    mbox(0, 2, item_maxx, size < maxy ? size : maxy);

    mattron(kCABold);
    mmvprintw(0, 4, self->mTitle);
    mattroff();

    float f;
    if(vector_size(self->mMenuItems) == 0) {
        f = 0.0;
    }
    else {
        f = ((float)(self->mCursor+1)
                /(float)vector_size(self->mMenuItems))*100;
    }

    mmvprintw((size < maxy ? size: maxy) - 1, 2+item_maxx-8, "(%3d%%)", (int)f);
    for(i=self->mScrollTop;
        (i<vector_size(self->mMenuItems) && i-self->mScrollTop<maxy-2); i++) {
        sMenuItem* item = (sMenuItem*)vector_item(self->mMenuItems, i);
        
        char buf[1024];
        if(strlen(item->mName) > maxx-4) {
            str_cut2(item->mName, maxx-4, buf, 1024);
        }
        else {
            strcpy(buf, item->mName);
        }
        
        if(i == self->mCursor) {
            mattron(kCAReverse);
            mmvprintw(i-self->mScrollTop+1, 3, buf);
            mattroff();
        }
        else  {
            mmvprintw(i-self->mScrollTop+1, 3, buf);
        }
    }

    mmove(maxy-1, maxx-2);
}
