#define _XOPEN_SOURCE 500

#include "minato_kanji.h"
#include "minato_curses.h"
#include <wctype.h>
#include <stdlib.h>
#include <string.h>

enum eKanjiCode gKanjiCode = kUtf8;
char* gKanjiCodeString[6] = {
    "eucjp", "sjis", "utf8", "utf8-mac", "ascii", "unknown"
};

char gIconvKanjiCodeName[5][256];        // ICONV漢字コードの名前

///////////////////////////////////////////////////////////////////////////////
// 初期化
///////////////////////////////////////////////////////////////////////////////
void kanji_init()
{
    strcpy(gIconvKanjiCodeName[kEucjp], "eucjp");
//    strcpy(gIconvKanjiCodeName[kSjis], "cp932");
    strcpy(gIconvKanjiCodeName[kSjis], "sjis");
    strcpy(gIconvKanjiCodeName[kUtf8], "utf-8");
    strcpy(gIconvKanjiCodeName[kUtf8Mac], "utf-8-mac");
}

///////////////////////////////////////////////////////////////////////////////
// 解放
///////////////////////////////////////////////////////////////////////////////
void kanji_final()
{
}

///////////////////////////////////////////////////////////////////////////////
// 半角文字一文字目かどうか
///////////////////////////////////////////////////////////////////////////////
int is_hankaku(unsigned char c)
{
    if(gKanjiCode == kEucjp)
        return c == 0x8e;
    else
        return 0;
}

///////////////////////////////////////////////////////////////////////////////
// 漢字一文字目かどうか
///////////////////////////////////////////////////////////////////////////////
int is_kanji(unsigned char c)
{
    if(gKanjiCode == kEucjp) {
        return c >= 0xA1 && c <= 0xFE || c == 0x8e;
//        return c >= 161 && c <= 254;
    }
    else if(gKanjiCode == kSjis) {
        return c >= 0x81 && c <= 0x9f || c >= 0xE0;
    }
    else if(gKanjiCode == kUtf8) {
        return !(c>=' ' && c<='~');
    }
}

///////////////////////////////////////////////////////////////////////////////
// FALSE:漢字まじり TRUE:全部英数字
///////////////////////////////////////////////////////////////////////////////
int is_all_ascii(char* buf)
{
    int i;

    for(i=0; i<strlen(buf); i++) {
        if(buf[i] < ' ' || buf[i] > '~') {
            return FALSE;
        }
    }

    return TRUE;
}

///////////////////////////////////////////////////////////////////////////////
// 引数の文字列が画面で何文字か
///////////////////////////////////////////////////////////////////////////////
int str_termlen(char* mbs)
{
    if(gKanjiCode == kUtf8) {
        int result;
        wchar_t* wcs;

        wcs = (wchar_t*)malloc(sizeof(wchar_t)*((strlen(mbs)+1)*UTF_CHAR_SIZE_MAX));
        if(mbstowcs(wcs, mbs, (strlen(mbs)+1)*UTF_CHAR_SIZE_MAX) == -1) {
            free(wcs);
            return -1;
        }
        result = wcswidth(wcs, wcslen(wcs));
        free(wcs);

        return result;
    }
    else {
        return strlen(mbs);
    }
}

///////////////////////////////////////////////////////////////////////////////
// 漢字でpos文字目までの端末で何文字か
///////////////////////////////////////////////////////////////////////////////
int str_termlen2(char* mbs, int pos)
{
    if(gKanjiCode == kUtf8) {
        char* mbs2;
        int result;
        char* point;

        mbs2 = strdup(mbs);
        point = str_utfpos2pointer(mbs2, pos);
        if(point == NULL) {
            if(pos < 0) 
                result = -1;
            else
                result = str_termlen(mbs2);
        }
        else {
            *(mbs2 + (point-mbs2)) = 0;
            result = str_termlen(mbs2);
        }

        free(mbs2);

        return result;
    }
    else {
        return pos*2;
    }
}

///////////////////////////////////////////////////////////////////////////////
// 引数が漢字で何文字か
///////////////////////////////////////////////////////////////////////////////
int str_kanjilen(char* mbs)
{
    if(gKanjiCode == kUtf8) {
        int result;
        wchar_t* wcs;
        
        wcs = (wchar_t*)malloc(sizeof(wchar_t)*(strlen(mbs)+1)*UTF_CHAR_SIZE_MAX);

        if(mbstowcs(wcs, mbs, (strlen(mbs)+1)*UTF_CHAR_SIZE_MAX) == -1) {
            free(wcs);
            return -1;
        }

        result = wcslen(wcs);

        free(wcs);

        return result;
    }
    else {
        int result = 0;
        char* p = mbs;
        while(*p) {
            if(is_kanji(*p) && *(p+1) != 0) {
                p+=2;
                result++;
            }
            else {
                p++;
                result++;
            }
        }
        return result;
    }
}






///////////////////////////////////////////////////////////////////////////////
// 引数の文字列が画面で何文字か
///////////////////////////////////////////////////////////////////////////////
int wcs_termlen(wchar_t* wcs)
{
    return wcswidth(wcs, wcslen(wcs));
}

///////////////////////////////////////////////////////////////////////////////
// mbsをUTF文字列と見たときのpos文字目の位置を返す
///////////////////////////////////////////////////////////////////////////////
char* str_utfpos2pointer(char* mbs, int pos)
{
    char* result;
    wchar_t* wcs;
    char* mbs2;
    
    if(pos<0) return mbs;
    if(pos >= str_kanjilen(mbs)) return mbs + strlen(mbs);

    mbs2 = (char*)malloc(sizeof(char)*(strlen(mbs)+1));
    wcs = (wchar_t*)malloc(sizeof(wchar_t)*(strlen(mbs)+1)*UTF_CHAR_SIZE_MAX);

    if(mbstowcs(wcs, mbs, (strlen(mbs)+1)*UTF_CHAR_SIZE_MAX) == -1) {
        mbstowcs(wcs, "?????", (strlen(mbs)+1)*UTF_CHAR_SIZE_MAX);
    }

    wcs[pos] = 0;

    wcstombs(mbs2, wcs, strlen(mbs)+1);

    result = mbs + strlen(mbs2);

    free(wcs);
    free(mbs2);

    return result;
}

///////////////////////////////////////////////////////////////////////////////
// mbsをUTF文字列と見たときのpointの場所は何文字目かを返す
///////////////////////////////////////////////////////////////////////////////
int str_pointer2utfpos(char* mbs, char* point)
{
    char* mbs2;
    int result;

    mbs2 = strdup(mbs);
    *(mbs2 + (point-mbs)) = 0;

    result = str_kanjilen(mbs2);

    free(mbs2);

    return result;
}

///////////////////////////////////////////////////////////////////////////////
// mbsから端末上の文字列分(termsize)のみ残して残りを捨てた文字列をdest_mbsに返す dest_byte: dest_mbsのサイズ
///////////////////////////////////////////////////////////////////////////////
void str_cut(char* mbs, int termsize, char* dest_mbs, int dest_byte)
{
    if(gKanjiCode == kUtf8) {
        wchar_t* wcs;
        wchar_t* wcs_tmp;
        int i;

        wcs = (wchar_t*)GC_malloc(sizeof(wchar_t)*(termsize+1)*UTF_CHAR_SIZE_MAX);
        wcs_tmp = (wchar_t*)GC_malloc(sizeof(wchar_t)*(termsize+1)*UTF_CHAR_SIZE_MAX);

        if(mbstowcs(wcs, mbs, (termsize+1)*UTF_CHAR_SIZE_MAX) == -1) {
            mbstowcs(wcs, "?????", (termsize+1)*UTF_CHAR_SIZE_MAX);
        }

        for(i=0; i<wcslen(wcs); i++) {
            wcs_tmp[i] = wcs[i];
            wcs_tmp[i+1] = 0;

            if(wcs_termlen(wcs_tmp) > termsize) {
                wcs_tmp[i] = 0;
                break;
            }
        }

        wcstombs(dest_mbs, wcs_tmp, dest_byte);
    }
    else {
        int n;
        BOOL kanji = FALSE;
        for(n=0; n<termsize && n<dest_byte-1; n++) {
            if(!kanji && is_kanji(mbs[n])) {
                kanji = TRUE;
            }
            else {
                kanji = FALSE;
            }

            dest_mbs[n] = mbs[n];
        }
        
        if(kanji)
            dest_mbs[n-1] = 0;
        else
            dest_mbs[n] = 0;
    }
}
///////////////////////////////////////////////////////////////////////////////
// mbsから端末上の文字列分(termsize)のみ残して残りを捨てた文字列をdest_mbsに返す dest_byte: dest_mbsのサイズ
// 切り捨てた文字列をスペースで埋める
///////////////////////////////////////////////////////////////////////////////
void str_cut2(char* mbs, int termsize, char* dest_mbs, int dest_byte)
{
    if(gKanjiCode == kUtf8) {
        int i;
        int n;

        wchar_t* wcs;
        wchar_t* tmp;
        wcs = (wchar_t*)GC_malloc(sizeof(wchar_t)*(termsize+1)*UTF_CHAR_SIZE_MAX);
        tmp = (wchar_t*)GC_malloc(sizeof(wchar_t)*(termsize+1)*UTF_CHAR_SIZE_MAX);

        if(mbstowcs(wcs, mbs, (termsize+1)*UTF_CHAR_SIZE_MAX) == -1) {
            mbstowcs(wcs, "?????", (termsize+1)*UTF_CHAR_SIZE_MAX);
        }

        i = 0;
        while(1) {
            if(i < wcslen(wcs)) {
                tmp[i] = wcs[i];
                tmp[i+1] = 0;
            }
            else {
                tmp[i] = ' ';
                tmp[i+1] = 0;
            }

            n = wcs_termlen(tmp);
            if(n < 0) {
                strcpy(dest_mbs, "?????");
/*
                free(wcs);
                free(tmp);
*/
                return;
            }
            else {
                if(n > termsize) {
                    tmp[i] = 0;

                    if(wcs_termlen(tmp) != termsize) {
                        tmp[i] = ' ';
                        tmp[i+1] = 0;
                    }
                    break;
                }
            }

            i++;
        }

        wcstombs(dest_mbs, tmp, dest_byte);
/*
        free(wcs);
        free(tmp);
*/
    }
    else {
        int n;
        BOOL tmpend = FALSE;
        BOOL kanji = FALSE;
        for(n=0; n<termsize && n<dest_byte-1; n++) {
            if(kanji)
                kanji = FALSE;
            else if(!tmpend && is_kanji(mbs[n])) {
                kanji = TRUE;
            }

            if(!tmpend && mbs[n] == 0) tmpend = TRUE;
                        
            if(tmpend)
                dest_mbs[n] = ' ';
            else
                dest_mbs[n] = mbs[n];
        }
        
        if(kanji) dest_mbs[n-1] = ' ';
        dest_mbs[n] = 0;
    }
}

///////////////////////////////////////////////////////////////////////////////
// mbsから端末上の文字列分(termsize)のみ残して残りを捨てた文字列をdest_mbsに返す 
// dest_byte: dest_mbsのサイズ
// 切り捨てた文字列をスペースで埋める
// 切り捨てるのは文字列の前から
///////////////////////////////////////////////////////////////////////////////
void str_cut3(char* mbs, int termsize, char* dest_mbs, int dest_byte)
{
    if(gKanjiCode == kUtf8) {
        wchar_t* wcs;
        wchar_t* tmp;
        int i;
        int j;

        wcs = (wchar_t*)GC_malloc(sizeof(wchar_t)*(termsize+1)*UTF_CHAR_SIZE_MAX);
        tmp = (wchar_t*)GC_malloc(sizeof(wchar_t)*(termsize+1)*UTF_CHAR_SIZE_MAX);
    //    wcs = (wchar_t*)malloc(sizeof(wchar_t)*(strlen(mbs)+1)*UTF_CHAR_SIZE_MAX);
    //    tmp = (wchar_t*)malloc(sizeof(wchar_t)*(strlen(mbs)+1)*UTF_CHAR_SIZE_MAX);

        if(str_termlen(mbs) <= termsize) {
            str_cut2(mbs, termsize, dest_mbs, dest_byte);
        }
        else {
            if(mbstowcs(wcs, mbs, (termsize+1)*UTF_CHAR_SIZE_MAX) == -1) {
                mbstowcs(wcs, "?????", (termsize+1)*UTF_CHAR_SIZE_MAX);
            }

            i = wcslen(wcs)-1;
            j = 0;
            while(1) {
                tmp[j++] = wcs[i--];
                tmp[j] = 0;

                if(wcs_termlen(tmp) > termsize) {
                    tmp[j-1] = 0;
                    i+=2;

                    if(wcs_termlen(tmp) == termsize) {
                        wcscpy(tmp, wcs + i);
                        break;
                    }
                    else {
                        tmp[0] = ' ';
                        wcscpy(tmp + 1, wcs + i);
                        break;
                    }
                }
            }

            wcstombs(dest_mbs, tmp, dest_byte);
        }

        //free(wcs);
        //free(tmp);
    }
    else {
        BOOL* kanji;
        int i;
        const int len = strlen(mbs);
        char* buf;

        if(len < termsize) {
            strcpy(dest_mbs, mbs);
            for(i=len; i<termsize; i++) {
                strcat(dest_mbs, " ");
            }
        }
        else {
            kanji = malloc(sizeof(BOOL)*len);
            for(i=0; i<len; i++) {
                if(is_kanji(mbs[i])) {
                    kanji[i] = TRUE;
                    i++;
                    kanji[i] = FALSE;
                }
                else {
                    kanji[i] = FALSE;
                }
            }

            if(kanji[len-termsize-1]) {
                int i;
                for(i=len-termsize; i<len; i++) {
                    dest_mbs[i-len+termsize] = mbs[i];
                }
                dest_mbs[i-len+termsize] = 0;
                dest_mbs[0] = ' ';
            }
            else {
                int i;
                for(i=len-termsize; i<len; i++) {
                    dest_mbs[i-len+termsize] = mbs[i];
                }
                dest_mbs[i-len+termsize] = 0;
            }

            free(kanji);
        }
    }
}

#if defined(HAVE_ICONV_H) || defined(HAVE_BICONV_H)
///////////////////////////////////////////////////////////////////////////////
// 引数の文字列の漢字コードを返す
///////////////////////////////////////////////////////////////////////////////

#if defined(HAVE_BICONV_H)
#include <biconv.h>
#else
#include <iconv.h>
#endif

/*
enum eKanjiCode kanji_encode_type(char* buf)
{
    iconv_t ic;
    char* str_in;
    char* str_out;
    char* ptr_in;
    char* ptr_out;
    kcode_t kcode;
    const int len = strlen(buf);
    size_t bufsz_in = (size_t) len;
    size_t bufsz_out = (size_t) len * UTF_CHAR_SIZE_MAX;
    size_t result;

    if(is_all_ascii(buf)) {
        return kAscii;
    }

#if defined(__CYGWIN__)
    return ruby_kconv_guess(buf);
#else
    str_in = (char*)malloc(len*2);
    str_out = (char*)malloc(len * UTF_CHAR_SIZE_MAX);

    ptr_in = str_in;
    ptr_out = str_out;
    strcpy(str_in, buf);


    ic = iconv_open(gIconvKanjiCodeName[kUtf8Mac], gIconvKanjiCodeName[kUtf8Mac]);
    if(!(iconv(ic, &ptr_in, &bufsz_in, &ptr_out, &bufsz_out) < 0 || len-bufsz_in < len)) {
        iconv_close(ic);

        free(str_in);
        free(str_out);
        return kUtf8Mac;
    }
    else {
        ic = iconv_open(gIconvKanjiCodeName[kUtf8], gIconvKanjiCodeName[kUtf8]);
        if(!(iconv(ic, &ptr_in, &bufsz_in, &ptr_out, &bufsz_out) < 0 || len-bufsz_in < len)) {
            iconv_close(ic);

            free(str_in);
            free(str_out);
            return kUtf8;
        }
        else {
            iconv_close(ic);

            kcode = wkfGuessKanjiCodeOfString(buf);

            if(kcode == KC_SJIS) {
            //if(kcode == KC_SJIS|| KC_EUCorSJIS) {
                return kSjis;
            }
            if(kcode == KC_EUC) {
                return kEucjp;
            }
            else {
                return kUnknown;
            }
        }
    }
#endif
}
*/

///////////////////////////////////////////////////////////////////////////////
// 漢字エンコードの変換。iconvのラッパー
///////////////////////////////////////////////////////////////////////////////
int kanji_convert(char* input_buf, char* output_buf, size_t output_buf_size, enum eKanjiCode input_kanji_encode_type, enum eKanjiCode output_kanji_encode_type)
{
    iconv_t ic;
    char* ptr_in;
    char* ptr_out;
    const int len = strlen(input_buf);
    size_t result;
    size_t bufsz_in;
    size_t bufsz_out;

    if(is_all_ascii(input_buf) || input_kanji_encode_type == kAscii || output_kanji_encode_type == kAscii) {
        strcpy(output_buf, input_buf);
        return 0;
    }

/*
    if(input_kanji_encode_type == kUnknown) {
        input_kanji_encode_type = kanji_encode_type(input_buf);
    }
    else {
        if(input_kanji_encode_type != kanji_encode_type(input_buf)) {
            return -1;
        }
    }
*/
    if(input_kanji_encode_type == output_kanji_encode_type) {
        strcpy(output_buf, input_buf);
        return 0;
    }
    
    if(input_kanji_encode_type == kUnknown || output_kanji_encode_type == kUnknown) return -1;

    
    ptr_in = input_buf;
    ptr_out = output_buf;
    bufsz_in = (size_t) len;
    bufsz_out = (size_t) output_buf_size;

    ic = iconv_open(gIconvKanjiCodeName[output_kanji_encode_type], gIconvKanjiCodeName[input_kanji_encode_type]);
    if(iconv(ic, &ptr_in, &bufsz_in, &ptr_out, &bufsz_out) < 0 || len-bufsz_in < len) {
        iconv_close(ic);

        return -1;
    }
    output_buf[output_buf_size - bufsz_out] = 0;

    iconv_close(ic);

    return 0;
}

#endif

