
/* except.q: exception handling for the builtin and standard library functions
   07-26-01, 03-04-02, 02-18-03, 03-18-03 AG */

/* This file is part of the Q programming system.

   The Q programming system is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   The Q programming system is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* This script defines sample default rules for all the builtin and standard
   library operations, which throw an exception whenever such an operation
   "fails". The exception handling implemented here is fairly simplistic:
   just throw back the offending expression protected with a special
   constructor. There are different constructors, belonging to different
   Exception subtypes, for the builtins and each part of the standard library,
   such as builtin_err of type BuiltinException for the builtins, stdlib_err
   of type StdlibException for the operations in stdlib.q, etc.

   There also is a simple handler function which can be supplied as the first
   argument to `catch'. When invoked with an exception, i.e., a value of type
   Exception, it prints an appropriate error message and halts. Otherwise it
   just returns the argument value unchanged.

   NOTE: These definitions are only examples, which should be adapted for the
   particular application. Note that Q is essentially an exception-free
   language because of its rewriting semantics. Thus if you want to report
   exceptions for certain operations you *always* have to add corresponding
   rules yourself. Judicious use of exception rules, in particular for
   operations involving side-effects can help to locate certain bugs in a
   script which otherwise are hard to find. But adding exceptions to each and
   every operation in a program generally is a bad idea because it interferes
   with symbolic evaluation. Therefore exception rules are *not* included in
   the standard library by default.

   EXAMPLES: Default error message printed by the interpreter in response to
   an unhandled exception:

   ==> 17+4*3/0
   ! Exception
   builtin_err (12/0)
   >>> 17+4*3/0
               ^

   Use our own exception handler:

   ==> catch exception (17+4*3/0)
   ! Exception: builtin_err (12/0)

   You can also handle exceptions in the Q machine. E.g., let's force a stack
   overflow:

   ==> catch exception (iter 100000000 (+) 1)
   ! Exception: syserr 5

   Note that 5 is the error code for an expression stack overflow; see the
   manual for the list of all built-in exception codes.  */

public exception X;

exception X:Exception	= fwrites ERROR ("! Exception: "++str X++"\n") ||
			  halt;
exception X		= X otherwise;

/* The following rules are at the lowest possible priority level so as not to
   interfere with the standard library and user equations. */

@-0x80000000

/* Note that some operations are missing in this list, because they never fail
   unless a hard error condition occurs. */

/* builtins ******************************************************************/

public type BuiltinException : Exception = special const builtin_err X;

/* operators */

X^Y			= throw (builtin_err (X^Y));
X!Y			= throw (builtin_err (X!Y));

-X			= throw (builtin_err (-X));
#X			= throw (builtin_err (#X));
not X			= throw (builtin_err (not X));

X*Y			= throw (builtin_err (X*Y));
X/Y			= throw (builtin_err (X/Y));
X div Y			= throw (builtin_err (X div Y));
X mod Y			= throw (builtin_err (X mod Y));
X and Y			= throw (builtin_err (X and Y));
X and then Y		= throw (builtin_err (X and then Y));

X++Y			= throw (builtin_err (X++Y));
X+Y			= throw (builtin_err (X+Y));
X-Y			= throw (builtin_err (X-Y));
X or Y			= throw (builtin_err (X or Y));
X or else Y		= throw (builtin_err (X or else Y));

X<Y			= throw (builtin_err (X<Y));
X>Y			= throw (builtin_err (X>Y));
(X=Y)			= throw (builtin_err (X=Y));
X<=Y			= throw (builtin_err (X<=Y));
X>=Y			= throw (builtin_err (X>=Y));
X<>Y			= throw (builtin_err (X<>Y));

/* functions */

succ X			= throw (builtin_err (succ X));
pred X			= throw (builtin_err (pred X));
enum X Y		= throw (builtin_err (enum X Y));
enum_from X		= throw (builtin_err (enum_from X));
streamenum X Y		= throw (builtin_err (streamenum X Y));
streamenum_from X	= throw (builtin_err (streamenum_from X));

shl N COUNT		= throw (builtin_err (shl N COUNT));
shr N COUNT		= throw (builtin_err (shr N COUNT));

exp X			= throw (builtin_err (exp X));
ln X			= throw (builtin_err (ln X));
sqrt X			= throw (builtin_err (sqrt X));
sin X			= throw (builtin_err (sin X));
cos X			= throw (builtin_err (cos X));
atan X			= throw (builtin_err (atan X));
atan2 X Y		= throw (builtin_err (atan2 X Y));
seed N			= throw (builtin_err (seed N));

sub Xs I J		= throw (builtin_err (sub Xs I J));
substr S K L		= throw (builtin_err (substr S K L));
pos S1 S		= throw (builtin_err (pos S1 S));

trunc X			= throw (builtin_err (trunc X));
round X			= throw (builtin_err (round X));
float X			= throw (builtin_err (float X));
int X			= throw (builtin_err (int X));
frac X			= throw (builtin_err (frac X));
ord X			= throw (builtin_err (ord X));
chr N			= throw (builtin_err (chr N));
list Xs			= throw (builtin_err (list Xs));
tuple Xs		= throw (builtin_err (tuple Xs));
str X			= throw (builtin_err (str X));
strq X			= throw (builtin_err (strq X));
val S			= throw (builtin_err (val S));
valq S			= throw (builtin_err (valq S));

fopen NAME MODE		= throw (builtin_err (fopen NAME MODE));
popen CMD MODE		= throw (builtin_err (popen CMD MODE));
fclose F		= throw (builtin_err (fclose F));
read			= throw (builtin_err read);
readq			= throw (builtin_err readq);
readc			= throw (builtin_err readc);
reads			= throw (builtin_err reads);
fread F			= throw (builtin_err (fread F));
freadq F		= throw (builtin_err (freadq F));
freadc F		= throw (builtin_err (freadc F));
freads F		= throw (builtin_err (freads F));
write X			= throw (builtin_err (write X));
writeq X		= throw (builtin_err (writeq X));
writec X		= throw (builtin_err (writec X));
writes S		= throw (builtin_err (writes S));
fwrite F X		= throw (builtin_err (fwrite F X));
fwriteq F X		= throw (builtin_err (fwriteq F X));
fwritec F X		= throw (builtin_err (fwritec F X));
fwrites F S		= throw (builtin_err (fwrites F S));
eof			= throw (builtin_err eof);
feof F			= throw (builtin_err (feof F));
flush			= throw (builtin_err flush);
fflush F		= throw (builtin_err (fflush F));

trap ACT SIG		= throw (builtin_err (trap ACT SIG));
time			= throw (builtin_err time);
sleep X			= throw (builtin_err (sleep X));

/* clib **********************************************************************/

public type ClibException : Exception = special const clib_err X;

islower C		= throw (clib_err (islower C));
isupper C		= throw (clib_err (isupper C));
isalpha C		= throw (clib_err (isalpha C));
isdigit C		= throw (clib_err (isdigit C));
isxdigit C		= throw (clib_err (isxdigit C));
isalnum C		= throw (clib_err (isalnum C));
ispunct C		= throw (clib_err (ispunct C));
isspace C		= throw (clib_err (isspace C));
isgraph C		= throw (clib_err (isgraph C));
isprint C		= throw (clib_err (isprint C));
iscntrl C		= throw (clib_err (iscntrl C));
isascii C		= throw (clib_err (isascii C));
tolower S		= throw (clib_err (tolower S));
toupper S		= throw (clib_err (toupper S));
bytestr X		= throw (clib_err (bytestr X));
bcat Bs			= throw (clib_err (bcat Bs));
bsize B			= throw (clib_err (bsize B));
byte I B		= throw (clib_err (byte I B));
bsub B I J		= throw (clib_err (bsub B I J));
bcmp B1 B2		= throw (clib_err (bcmp B1 B2));
bint B			= throw (clib_err (bint B));
bfloat B		= throw (clib_err (bfloat B));
bstr B			= throw (clib_err (bstr B));
bytes B			= throw (clib_err (bytes B));
get_int8 B I		= throw (clib_err (get_int8 B I));
get_int16 B I		= throw (clib_err (get_int16 B I));
get_int32 B I		= throw (clib_err (get_int32 B I));
get_uint8 B I		= throw (clib_err (get_uint8 B I));
get_uint16 B I		= throw (clib_err (get_uint16 B I));
get_uint32 B I		= throw (clib_err (get_uint32 B I));
get_float B I		= throw (clib_err (get_float B I));
get_double B I		= throw (clib_err (get_double B I));
put_int8 B I X		= throw (clib_err (put_int8 B I X));
put_int16 B I X		= throw (clib_err (put_int16 B I X));
put_int32 B I X		= throw (clib_err (put_int32 B I X));
put_uint8 B I X		= throw (clib_err (put_uint8 B I X));
put_uint16 B I X	= throw (clib_err (put_uint16 B I X));
put_uint32 B I X	= throw (clib_err (put_uint32 B I X));
put_float B I X		= throw (clib_err (put_float B I X));
put_double B I X	= throw (clib_err (put_double B I X));
int8_list B		= throw (clib_err (int8_list B));
int16_list B		= throw (clib_err (int16_list B));
int32_list B		= throw (clib_err (int32_list B));
uint8_list B		= throw (clib_err (uint8_list B));
uint16_list B		= throw (clib_err (uint16_list B));
uint32_list B		= throw (clib_err (uint32_list B));
float_list B		= throw (clib_err (float_list B));
double_list B		= throw (clib_err (double_list B));
int8_vect Xs		= throw (clib_err (int8_vect Xs));
int16_vect Xs		= throw (clib_err (int16_vect Xs));
int32_vect Xs		= throw (clib_err (int32_vect Xs));
uint8_vect Xs		= throw (clib_err (uint8_vect Xs));
uint16_vect Xs		= throw (clib_err (uint16_vect Xs));
uint32_vect Xs		= throw (clib_err (uint32_vect Xs));
float_vect Xs		= throw (clib_err (float_vect Xs));
double_vect Xs		= throw (clib_err (double_vect Xs));
fdopen FD MODE		= throw (clib_err (fdopen FD MODE));
freopen NAME MODE F	= throw (clib_err (freopen NAME MODE F));
fileno F		= throw (clib_err (fileno F));
setvbuf F MODE		= throw (clib_err (setvbuf F MODE));
tmpnam			= throw (clib_err tmpnam);
tmpfile			= throw (clib_err tmpfile);
ftell F			= throw (clib_err (ftell F));
fseek F N WHENCE	= throw (clib_err (fseek F N WHENCE));
rewind F		= throw (clib_err (rewind F));
gets			= throw (clib_err gets);
fgets F			= throw (clib_err (fgets F));
fget F			= throw (clib_err (fget F));
ungetc C		= throw (clib_err (ungetc C));
fungetc C		= throw (clib_err (fungetc C));
printf FORMAT ARGS	= throw (clib_err (printf FORMAT ARGS));
fprintf F FORMAT ARGS	= throw (clib_err (fprintf F FORMAT ARGS));
sprintf FORMAT ARGS	= throw (clib_err (sprintf FORMAT ARGS));
scanf FORMAT		= throw (clib_err (scanf FORMAT));
fscanf F FORMAT		= throw (clib_err (fscanf F FORMAT));
sscanf S FORMAT		= throw (clib_err (sscanf S FORMAT));
exit N			= throw (clib_err (exit N));
thread_no THREAD	= throw (clib_err (thread_no THREAD));
this_thread		= throw (clib_err this_thread);
thread X		= throw (clib_err (thread X));
return X		= throw (clib_err (return X));
cancel THREAD		= throw (clib_err (cancel THREAD));
result THREAD		= throw (clib_err (result THREAD));
yield			= throw (clib_err yield);
active THREAD		= throw (clib_err (active THREAD));
canceled THREAD		= throw (clib_err (canceled THREAD));
setsched THREAD POL PRIO= throw (clib_err (setsched THREAD POL PRIO));
getsched THREAD		= throw (clib_err (getsched THREAD));
mutex			= throw (clib_err mutex);
errorchecking_mutex	= throw (clib_err errorchecking_mutex);
recursive_mutex		= throw (clib_err recursive_mutex);
lock MUTEX		= throw (clib_err (lock MUTEX));
unlock MUTEX		= throw (clib_err (unlock MUTEX));
try MUTEX		= throw (clib_err (try MUTEX));
condition		= throw (clib_err condition);
signal COND		= throw (clib_err (signal COND));
broadcast COND		= throw (clib_err (broadcast COND));
await COND		= throw (clib_err (await COND));
semaphore		= throw (clib_err semaphore);
bounded_semaphore MAX	= throw (clib_err (bounded_semaphore MAX));
post SEM X		= throw (clib_err (post SEM X));
try SEM			= throw (clib_err (try SEM));
get_size SEM		= throw (clib_err (get_size SEM));
get_bound SEM		= throw (clib_err (get_bound SEM));
ref X			= throw (clib_err (ref X));
sentinel X		= throw (clib_err (sentinel X));
put REF X		= throw (clib_err (put REF X));
get REF			= throw (clib_err (get REF));
pow M N			= throw (clib_err (pow M N));
root M N		= throw (clib_err (root M N));
intsqrt M		= throw (clib_err (intsqrt M));
powmod K M N		= throw (clib_err (powmod K M N));
invmod K M		= throw (clib_err (invmod K M));
isprime N		= throw (clib_err (isprime N));
gcd M N			= throw (clib_err (gcd M N));
lcm M N			= throw (clib_err (lcm M N));
remove_factor M N	= throw (clib_err (remove_factor M N));
jacobi M N		= throw (clib_err (jacobi M N));
sort P Xs		= throw (clib_err (sort P Xs));
fnmatch PATTERN S	= throw (clib_err (fnmatch PATTERN S));
glob PATTERN		= throw (clib_err (glob PATTERN));
regmatch OPTS REGEX S	= throw (clib_err (regmatch OPTS REGEX S));
regnext			= throw (clib_err regnext);
regdone			= throw (clib_err regdone);
regex OPTS REGEX S EXPR	= throw (clib_err (regex OPTS REGEX S EXPR));
regstart		= throw (clib_err regstart);
regskip			= throw (clib_err regskip);
reg N			= throw (clib_err (reg N));
regpos N		= throw (clib_err (regpos N));
regend N		= throw (clib_err (regend N));
regs			= throw (clib_err regs);

/* complex *******************************************************************/

public type ComplexException : Exception = special const complex_err X;

arg Z			= throw (complex_err (arg Z));
re Z			= throw (complex_err (re Z));
im Z			= throw (complex_err (im Z));
conj Z			= throw (complex_err (conj Z));
cis X			= throw (complex_err (cis X));
polar R X		= throw (complex_err (polar R X));

/* cond **********************************************************************/

public type CondException : Exception = special const cond_err X;

ifelse P X Y		= throw (cond_err (ifelse P X Y));
when P X		= throw (cond_err (when P X));
unless P X		= throw (cond_err (unless P X));
cond CASES		= throw (cond_err (cond CASES));
case X CASES		= throw (cond_err (case X CASES));

/* error *********************************************************************/

public type ErrorException : Exception = special const error_err X;

error S			= throw (error_err (error S));

/* graphics ******************************************************************/

/* Note that the graphics script is not by default included by the prelude,
   but will be imported into your program with this module. You might wish to
   remove this section if you don't need the graphics script in your
   program. */

from graphics import /* all */;

public type GraphicsException : Exception = special const graphics_err X;

moveto X Y		= throw (graphics_err (moveto X Y));
rmoveto DX DY		= throw (graphics_err (rmoveto DX DY));
lineto X Y		= throw (graphics_err (lineto X Y));
rlineto DX DY		= throw (graphics_err (rlineto DX DY));
curveto X1 Y1 X2 Y2 X3 Y3
			= throw (graphics_err (curveto X1 Y1 X2 Y2 X3 Y3));
rcurveto DX1 DY1 DX2 DY2 DX3 DY3
			= throw (graphics_err
				 (rcurveto DX1 DY1 DX2 DY2 DX3 DY3));
arc X Y R A1 A2		= throw (graphics_err (arc X Y R A1 A2));
narc X Y R A1 A2	= throw (graphics_err (narc X Y R A1 A2));
arct X1 Y1 X2 Y2 R	= throw (graphics_err (arct X1 Y1 X2 Y2 R));
charpath S T		= throw (graphics_err (charpath S T));
show S			= throw (graphics_err (show S));
translate TX TY		= throw (graphics_err (translate TX TY));
scale SX SY		= throw (graphics_err (scale SX SY));
rotate A		= throw (graphics_err (rotate A));
setlinewidth X		= throw (graphics_err (setlinewidth X));
setlinecap N		= throw (graphics_err (setlinecap N));
setlinejoin N		= throw (graphics_err (setlinejoin N));
setdash Xs DX		= throw (graphics_err (setdash Xs DX));
setgray N		= throw (graphics_err (setgray N));
setrgbcolor R G B	= throw (graphics_err (setrgbcolor R G B));
sethsbcolor H S B	= throw (graphics_err (sethsbcolor H S B));
setcmykcolor C M Y K	= throw (graphics_err (setcmykcolor C M Y K));
setfont S X		= throw (graphics_err (setfont S X));
setcolor C		= throw (graphics_err (setcolor C));
copies N		= throw (graphics_err (copies N));
psfile S		= throw (graphics_err (psfile S));
psstr S			= throw (graphics_err (psstr S));
ps S			= throw (graphics_err (ps S));

/* math **********************************************************************/

public type MathException : Exception = special const math_err X;

asin X			= throw (math_err (asin X));
acos X			= throw (math_err (acos X));
tan X			= throw (math_err (tan X));
lg X			= throw (math_err (lg X));
log X			= throw (math_err (log X));
sinh X			= throw (math_err (sinh X));
cosh X			= throw (math_err (cosh X));
tanh X			= throw (math_err (tanh X));
asinh X			= throw (math_err (asinh X));
acosh X			= throw (math_err (acosh X));
atanh X			= throw (math_err (atanh X));

/* rational ******************************************************************/

public type RationalException : Exception = special const rational_err X;

X%Y			= throw (rational_err (X%Y));
rational X		= throw (rational_err (rational X));
num X			= throw (rational_err (num X));
den X			= throw (rational_err (den X));
num_den X		= throw (rational_err (num_den X));

/* sort **********************************************************************/

public type SortException : Exception = special const sort_err X;

msort P Xs		= throw (sort_err (msort P Xs));
qsort P Xs		= throw (sort_err (qsort P Xs));

/* stdlib ********************************************************************/

/* Make sure that these definitions come last because some of the default
   rules here are overridden above (overloaded operations!). */

public type StdlibException : Exception = special const stdlib_err X;

abs X			= throw (stdlib_err (abs X));
all P Xs		= throw (stdlib_err (all P Xs));
any P Xs		= throw (stdlib_err (any P Xs));
append Xs Y		= throw (stdlib_err (append Xs Y));
cat Xs			= throw (stdlib_err (cat Xs));
cons X Xs		= throw (stdlib_err (cons X Xs));
do F Xs			= throw (stdlib_err (do F Xs));
drop N Xs		= throw (stdlib_err (drop N Xs));
dropwhile P Xs		= throw (stdlib_err (dropwhile P Xs));
filter P Xs		= throw (stdlib_err (filter P Xs));
foldl F A Xs		= throw (stdlib_err (foldl F A Xs));
foldl1 F Xs		= throw (stdlib_err (foldl1 F Xs));
foldr F A Xs		= throw (stdlib_err (foldr F A Xs));
foldr1 F Xs		= throw (stdlib_err (foldr1 F Xs));
fst Xs			= throw (stdlib_err (fst Xs));
hd Xs			= throw (stdlib_err (hd Xs));
hds Xss			= throw (stdlib_err (hds Xss));
init Xs			= throw (stdlib_err (init Xs));
iter N F A		= throw (stdlib_err (iter N F A));
last Xs			= throw (stdlib_err (last Xs));
map F Xs		= throw (stdlib_err (map F Xs));
mklist X N		= throw (stdlib_err (mklist X N));
null Xs			= throw (stdlib_err (null Xs));
nums N M		= throw (stdlib_err (nums N M));
numsby K N M		= throw (stdlib_err (numsby K N M));
pop Xs			= throw (stdlib_err (pop Xs));
prd Xs			= throw (stdlib_err (prd Xs));
push Xs X		= throw (stdlib_err (push Xs X));
reverse Xs		= throw (stdlib_err (reverse Xs));
scanl F A Xs		= throw (stdlib_err (scanl F A Xs));
scanl1 F Xs		= throw (stdlib_err (scanl1 F Xs));
scanr F A Xs		= throw (stdlib_err (scanr F A Xs));
scanr1 F Xs		= throw (stdlib_err (scanr1 F Xs));
sgn X			= throw (stdlib_err (sgn X));
snd Xs			= throw (stdlib_err (snd Xs));
sum Xs			= throw (stdlib_err (sum Xs));
take N Xs		= throw (stdlib_err (take N Xs));
takewhile P Xs		= throw (stdlib_err (takewhile P Xs));
tl Xs			= throw (stdlib_err (tl Xs));
tls Xss			= throw (stdlib_err (tls Xss));
top Xs			= throw (stdlib_err (top Xs));
transpose Xss		= throw (stdlib_err (transpose Xss));
trd Xs			= throw (stdlib_err (trd Xs));
tuplecat Xs		= throw (stdlib_err (tuplecat Xs));
unzip Xs:List		= throw (stdlib_err (unzip Xs));
unzip3 Xs:List		= throw (stdlib_err (unzip3 Xs));
zip Xs Ys		= throw (stdlib_err (zip Xs Ys));
zip3 Xs Ys Zs		= throw (stdlib_err (zip3 Xs Ys Zs));
zipwith F Xs Ys		= throw (stdlib_err (zipwith F Xs Ys));
zipwith3 F Xs Ys Zs	= throw (stdlib_err (zipwith3 F Xs Ys Zs));

/* stdtypes ******************************************************************/

/* As of Q 7.8, the container data structures are not part of the prelude
   anymore, but they *will* be imported into your program with this module.
   Thus you might wish to remove this section if you don't need the stdtypes
   script in your program. */

from stdtypes import /* all */;
from array import Array, array, array2, mkarray, mkarray2, members2, update2;
from heap  import Heap, heap;
from set   import Set, set;
from bag   import Bag, bag;
from dict  import Dict, dict, mkdict;
from hdict import HDict, hdict, mkhdict;

public type StdtypesException : Exception = special const stdtypes_err X;

array Xs		= throw (stdtypes_err (array Xs));
array2 Xs		= throw (stdtypes_err (array2 Xs));
bag Xs			= throw (stdtypes_err (bag Xs));
delete A X		= throw (stdtypes_err (delete A X));
dict XYs		= throw (stdtypes_err (dict XYs));
first A			= throw (stdtypes_err (first A));
hdict XYs		= throw (stdtypes_err (hdict XYs));
heap Xs			= throw (stdtypes_err (heap Xs));
insert A X		= throw (stdtypes_err (insert A X));
keys D			= throw (stdtypes_err (keys D));
last A:Array		= throw (stdtypes_err (last A));
last M:Bag		= throw (stdtypes_err (last M));
last D:Dict		= throw (stdtypes_err (last D));
last M:Set		= throw (stdtypes_err (last M));
member A X		= throw (stdtypes_err (member A X));
members A		= throw (stdtypes_err (members A));
members2 A		= throw (stdtypes_err (members2 A));
mkarray X N		= throw (stdtypes_err (mkarray X N));
mkarray2 X NM		= throw (stdtypes_err (mkarray2 X NM));
mkdict Y Xs		= throw (stdtypes_err (mkdict Y Xs));
mkhdict Y Xs		= throw (stdtypes_err (mkhdict Y Xs));
rmfirst A		= throw (stdtypes_err (rmfirst A));
rmlast A		= throw (stdtypes_err (rmlast A));
set Xs			= throw (stdtypes_err (set Xs));
update A K X		= throw (stdtypes_err (update A K X));
update2 A IJ X		= throw (stdtypes_err (update2 A IJ X));
vals D			= throw (stdtypes_err (vals D));

/* stream ********************************************************************/

public type StreamException : Exception = special const stream_err X;

stream Xs		= throw (stream_err (stream Xs));
foldl1 F Xs:Stream	= throw (stream_err (foldl1 F Xs));
foldr1 F Xs:Stream	= throw (stream_err (foldr1 F Xs));
hd Xs:Stream		= throw (stream_err (hd Xs));
init Xs:Stream		= throw (stream_err (init Xs));
last Xs:Stream		= throw (stream_err (last Xs));
pop Xs:Stream		= throw (stream_err (pop Xs));
tl Xs:Stream		= throw (stream_err (tl Xs));
top Xs:Stream		= throw (stream_err (top Xs));
numstream N		= throw (stream_err (numstream N));
numstreamby K N		= throw (stream_err (numstreamby K N));
repeatn N X		= throw (stream_err (repeatn N X));
cycle Xs		= throw (stream_err (cycle Xs));

/* string ********************************************************************/

public type StringException : Exception = special const string_err X;

chars S			= throw (string_err (chars S));
join DELIM Xs		= throw (string_err (join DELIM Xs));
split DELIM S		= throw (string_err (split DELIM S));
strcat Xs		= throw (string_err (strcat Xs));

/* system ********************************************************************/

/* Note that, as of Q 7.8, the operations listed below are not part of the
   clib module anymore. They are now provided by a separate 'system' module
   which is not included in the prelude, but they *will* be imported into your
   program with this module. Thus you might wish to remove this section if you
   don't need the system script in your program. */

from system import /* all */;

public type SystemException : Exception = special const system_err X;

rename OLD NEW		= throw (system_err (rename OLD NEW));
unlink NAME		= throw (system_err (unlink NAME));
truncate NAME LEN	= throw (system_err (truncate NAME LEN));
getcwd			= throw (system_err getcwd);
chdir NAME		= throw (system_err (chdir NAME));
mkdir NAME MODE		= throw (system_err (mkdir NAME MODE));
rmdir NAME		= throw (system_err (rmdir NAME));
readdir NAME		= throw (system_err (readdir NAME));
link OLD NEW		= throw (system_err (link OLD NEW));
symlink OLD NEW		= throw (system_err (symlink OLD NEW));
readlink NAME		= throw (system_err (readlink NAME));
mkfifo NAME MODE	= throw (system_err (mkfifo NAME MODE));
access NAME MODE	= throw (system_err (access NAME MODE));
chmod NAME MODE		= throw (system_err (chmod NAME MODE));
chown NAME UID GID	= throw (system_err (chown NAME UID GID));
lchown NAME UID GID	= throw (system_err (lchown NAME UID GID));
utime NAME TIMES	= throw (system_err (utime NAME TIMES));
umask N			= throw (system_err (umask N));
stat NAME		= throw (system_err (stat NAME));
lstat NAME		= throw (system_err (lstat NAME));
system CMD		= throw (system_err (system CMD));
fork			= throw (system_err fork);
exec PROG ARGS		= throw (system_err (exec PROG ARGS));
spawn PROG ARGS		= throw (system_err (spawn PROG ARGS));
_spawn MODE PROG ARGS	= throw (system_err (_spawn MODE PROG ARGS));
nice INC		= throw (system_err (nice INC));
pause			= throw (system_err pause);
raise SIG		= throw (system_err (raise SIG));
kill SIG PID		= throw (system_err (kill SIG PID));
getpid			= throw (system_err getpid);
getppid			= throw (system_err getppid);
wait			= throw (system_err wait);
waitpid PID OPTIONS	= throw (system_err (waitpid PID OPTIONS));
isactive STATUS		= throw (system_err (isactive STATUS));
isexited STATUS		= throw (system_err (isexited STATUS));
exitstatus STATUS	= throw (system_err (exitstatus STATUS));
issignaled STATUS	= throw (system_err (issignaled STATUS));
termsig STATUS		= throw (system_err (termsig STATUS));
isstopped STATUS	= throw (system_err (isstopped STATUS));
stopsig STATUS		= throw (system_err (stopsig STATUS));
getenv NAME		= throw (system_err (getenv NAME));
setenv NAME VAL		= throw (system_err (setenv NAME VAL));
setuid UID		= throw (system_err (setuid UID));
setgid GID		= throw (system_err (setgid GID));
seteuid UID		= throw (system_err (seteuid UID));
setegid GID		= throw (system_err (setegid GID));
setreuid RUID EUID	= throw (system_err (setreuid RUID EUID));
setregid RGID EGID	= throw (system_err (setregid RGID EGID));
getuid			= throw (system_err getuid);
geteuid			= throw (system_err geteuid);
getgid			= throw (system_err getgid);
getegid			= throw (system_err getegid);
getlogin		= throw (system_err getlogin);
getgroups		= throw (system_err getgroups);
setgroups GIDS		= throw (system_err (setgroups GIDS));
getpgid PID		= throw (system_err (getpgid PID));
setpgid PID PGID	= throw (system_err (setpgid PID PGID));
getpgrp			= throw (system_err getpgrp);
setpgrp			= throw (system_err setpgrp);
getsid PID		= throw (system_err (getsid PID));
setsid			= throw (system_err setsid);
open NAME FLAGS MODE	= throw (system_err (open NAME FLAGS MODE));
close FD		= throw (system_err (close FD));
dup FD			= throw (system_err (dup FD));
dup2 OLDFD NEWFD	= throw (system_err (dup2 OLDFD NEWFD));
pipe			= throw (system_err pipe);
fstat FD		= throw (system_err (fstat FD));
fchdir FD		= throw (system_err (fchdir FD));
fchmod FD MODE		= throw (system_err (fchmod FD MODE));
fchown FD UID GID	= throw (system_err (fchown FD UID GID));
ftruncate FD LEN	= throw (system_err (ftruncate FD LEN));
fsync FD		= throw (system_err (fsync FD));
fdatasync FD		= throw (system_err (fdatasync FD));
bread FD COUNT		= throw (system_err (bread FD COUNT));
bwrite FD MEM		= throw (system_err (bwrite FD MEM));
lseek FD POS WHENCE	= throw (system_err (lseek FD POS WHENCE));
fcntl FD CMD ARG	= throw (system_err (fcntl FD CMD ARG));
select FILES		= throw (system_err (select FILES));
isatty FD		= throw (system_err (isatty FD));
ttyname FD		= throw (system_err (ttyname FD));
ctermid			= throw (system_err ctermid);
openpty			= throw (system_err openpty);
forkpty			= throw (system_err forkpty);
tcgetattr FD		= throw (system_err (tcgetattr FD));
tcsetattr FD WHEN ATTR	= throw (system_err (tcsetattr FD WHEN ATTR));
tcsendbreak FD DURATION	= throw (system_err (tcsendbreak FD DURATION));
tcdrain FD		= throw (system_err (tcdrain FD));
tcflush FD QUEUE	= throw (system_err (tcflush FD QUEUE));
tcflow FD ACTION	= throw (system_err (tcflow FD ACTION));
tcgetpgrp FD		= throw (system_err (tcgetpgrp FD));
tcsetpgrp FD PGID	= throw (system_err (tcsetpgrp FD PGID));
errno			= throw (system_err errno);
seterrno N		= throw (system_err (seterrno N));
perror S		= throw (system_err (perror S));
strerror N		= throw (system_err (strerror N));
uname			= throw (system_err uname);
getpwuid UID		= throw (system_err (getpwuid UID));
getpwnam NAME		= throw (system_err (getpwnam NAME));
getpwent		= throw (system_err getpwent);
getgrgid GID		= throw (system_err (getgrgid GID));
getgrnam NAME		= throw (system_err (getgrnam NAME));
getgrent		= throw (system_err getgrent);
crypt KEY SALT		= throw (system_err (crypt KEY SALT));
gethostbyname HOST	= throw (system_err (gethostbyname HOST));
gethostent		= throw (system_err gethostent);
getprotobyname NAME	= throw (system_err (getprotobyname NAME));
getprotobynumber PROTO	= throw (system_err (getprotobynumber PROTO));
getprotoent		= throw (system_err getprotoent);
getservbyname NAME	= throw (system_err (getservbyname NAME));
getservbyport PORT	= throw (system_err (getservbyport PORT));
getservent		= throw (system_err getservent);
socket FAMILY TYPE PROTO= throw (system_err (socket FAMILY TYPE PROTO));
socketpair FAMILY TYPE PROTO
			= throw (system_err (socketpair FAMILY TYPE PROTO));
shutdown SOCKET HOW	= throw (system_err (shutdown SOCKET HOW));
closesocket SOCKET	= throw (system_err (closesocket SOCKET));
bind SOCKET ADDR	= throw (system_err (bind SOCKET ADDR));
listen SOCKET N		= throw (system_err (listen SOCKET N));
accept SOCKET		= throw (system_err (accept SOCKET));
connect SOCKET ADDR	= throw (system_err (connect SOCKET ADDR));
getsockname SOCKET	= throw (system_err (getsockname SOCKET));
getpeername SOCKET	= throw (system_err (getpeername SOCKET));
getsockopt SOCKET LEVEL OPT
			= throw (system_err (getsockopt SOCKET LEVEL OPT));
setsockopt SOCKET LEVEL OPT DATA
			= throw (system_err (setsockopt SOCKET LEVEL OPT DATA));
recv SOCKET FLAGS	= throw (system_err (recv SOCKET FLAGS));
send SOCKET FLAGS DATA	= throw (system_err (send SOCKET FLAGS DATA));
recvfrom SOCKET FLAGS	= throw (system_err (recvfrom SOCKET FLAGS));
sendto SOCKET FLAGS DATA= throw (system_err (sendto SOCKET FLAGS DATA));
tzname			= throw (system_err tzname);
timezone		= throw (system_err timezone);
daylight		= throw (system_err daylight);
ctime T			= throw (system_err (ctime T));
gmtime T		= throw (system_err (gmtime T));
localtime T		= throw (system_err (localtime T));
mktime TM		= throw (system_err (mktime TM));
asctime TM		= throw (system_err (asctime TM));
strftime FORMAT TM	= throw (system_err (strftime FORMAT TM));
clock			= throw (system_err clock);
times			= throw (system_err times);
setlocale CATEGORY LOCALE
			= throw (system_err (setlocale CATEGORY LOCALE));
localeconv		= throw (system_err localeconv);
nl_langinfo ITEM	= throw (system_err (nl_langinfo ITEM));
strfmon FORMAT ARGS	= throw (system_err (strfmon FORMAT ARGS));
strcoll S1 S2		= throw (system_err (strcoll S1 S2));
strxfrm S		= throw (system_err (strxfrm S));
wcswidth S		= throw (system_err (wcswidth S));
wcwidth C		= throw (system_err (wcwidth C));
iconv_open TO FROM	= throw (system_err (iconv_open TO FROM));
iconv_close IC		= throw (system_err (iconv_close IC));
iconv IC B		= throw (system_err (iconv IC B));
textdomain DOMAIN	= throw (system_err (textdomain DOMAIN));
bindtextdomain DOMAIN DIR
			= throw (system_err (bindtextdomain DOMAIN DIR));
gettext MSGID		= throw (system_err (gettext MSGID));
dgettext DOMAIN MSGID	= throw (system_err (dgettext DOMAIN MSGID));
dcgettext DOMAIN MSGID CATEGORY
			= throw (system_err (dcgettext DOMAIN MSGID CATEGORY));
ngettext MSGID1 MSGID2 N
			= throw (system_err (ngettext MSGID1 MSGID2 N));
dngettext DOMAIN MSGID1 MSGID2 N
			= throw (system_err (dngettext DOMAIN MSGID1 MSGID2 N));
dcngettext DOMAIN MSGID1 MSGID2 N CATEGORY
			= throw (system_err (dcngettext DOMAIN MSGID1 MSGID2 N CATEGORY));
