/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGACCOUNTOBJECT_H
#define SKGACCOUNTOBJECT_H
/** @file
 * This file is part of Skrooge and defines classes SKGAccountObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgnamedobject.h"
#include "skgbankmodeler_export.h"
class SKGBankObject;
class SKGOperationObject;
class SKGDocumentBank;
class SKGUnitObject;

/**
 * This class manages account object
 */
class SKGBANKMODELER_EXPORT SKGAccountObject : public SKGNamedObject
{
        Q_OBJECT;

public:
        /**
        * This structure defines type of account
         */
        enum AccountType {CURRENT, 	/**< to define a bank account*/
                          CREDITCARD, 	/**< to define a credit card account*/
                          INVESTMENT, 	/**< to define an account for investment */
                          ASSETS, 	/**< to define a assets account */
                          OTHER		/**< other kind of account */
                         };

        /**
         * Constructor
         * @param iDocument the document containing the object
         * @param iID the identifier in @p iTable of the object
         */
        explicit SKGAccountObject(const SKGDocument* iDocument=NULL, int iID=0);

        /**
         * Copy constructor
         * @param iObject the object to copy
         */
        SKGAccountObject(const SKGAccountObject& iObject);

        /**
         * Copy constructor
         * @param iObject the object to copy
         */
        SKGAccountObject(const SKGNamedObject& iObject);

        /**
         * Copy constructor
         * @param iObject the object to copy
         */
        SKGAccountObject(const SKGObjectBase& iObject);

        /**
         * Operator affectation
         * @param iObject the object to copy
         */
        virtual const SKGAccountObject& operator= (const SKGObjectBase &iObject);

        /**
         * Destructor
         */
        virtual ~SKGAccountObject();

        /**
         * Get the parent bank
         * @param oBank the parent bank
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError getBank(SKGBankObject& oBank) const;

        /**
         * Set the parent bank
         * @param iBank the parent bank
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setBank(const SKGBankObject& iBank);

        /**
         * Set the number of the account
         * @param iNumber the number
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setNumber(const QString& iNumber);

        /**
         * Get the number of the account
         * @return the number
         */
        virtual QString getNumber() const;

        /**
         * Set the agency number of the account
         * @param iNumber the agency number
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setAgencyNumber(const QString& iNumber);

        /**
         * Get the agency number of the account
         * @return the number
         */
        virtual QString getAgencyNumber() const;

        /**
         * Set the agency address of the account
         * @param iAddress the agency address
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setAgencyAddress(const QString& iAddress);

        /**
         * Get the agency address of the account
         * @return the number
         */
        virtual QString getAgencyAddress() const;

        /**
         * Set the comment of account
         * @param iComment the comment
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setComment(const QString& iComment);

        /**
         * Get the comment of this account
         * @return the comment
         */
        virtual QString getComment() const;

        /**
         * Add a new operation to this account
         * @param oOperation the created operation
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError addOperation(SKGOperationObject& oOperation);

        /**
         * Get the current amount
         * @return the current amount
         */
        virtual double getCurrentAmount() const;

        /**
         * Get amount of the account at a date
         * @param iDate date
         * @return amount of the account
         */
        virtual double getAmount(const QDate& iDate) const;

        /**
         * Set the type of this account
         * @param iType the type
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setType(SKGAccountObject::AccountType iType);

        /**
         * Get the type of this account
         * @return the type
         */
        virtual SKGAccountObject::AccountType getType() const;

        /**
         * To set the closed attribute of an account
         * @param iClosed the closed attribute: true or false
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError setClosed(bool iClosed);

        /**
         * To know if the account has been closed or not
         * @return an object managing the error
         *   @see SKGError
         */
        virtual bool isClosed() const;

        /**
         * Get the unit
         * @param oUnit the unit
         * @return an object managing the error
         *   @see SKGError
         */
        virtual SKGError getUnit(SKGUnitObject& oUnit) const;
};

#endif
