/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGMAINPANEL_H
#define SKGMAINPANEL_H
/** @file
* This file is part of Skrooge and defines classes skrooge.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectbase.h"
#include "skgbasegui_export.h"
#include <kparts/mainwindow.h>
#include <ktabwidget.h>
#include <kurl.h>
#include <ksystemtrayicon.h>

#include "ui_skgmainpanel_base.h"
#include "ui_skgmainpanel_pref.h"

class SKGInterfacePlugin;
class SKGDocument;
class SKGTabWidget;

class QPrinter;
class QListWidgetItem;

class KToggleAction;
class KUrl;
class KAction;
class KSplashScreen;
class KListWidget;
class KTipDatabase;
class KComboBox;

/**
 * This class serves as the main window for skrooge.  It handles the
 * menus, toolbars, and status bars.
 */
class SKGBASEGUI_EXPORT SKGMainPanel : public KParts::MainWindow
{
        Q_OBJECT
public:
        /**
         * Default Constructor
         * @param iSplashScreen the splash screen
         * @param iDocument the document to manage data
         * @param iArguments the arguments
         */
        SKGMainPanel ( KSplashScreen* iSplashScreen, SKGDocument* iDocument, const QStringList& iArguments );

        /**
         * Default Destructor
         */
        virtual ~SKGMainPanel();

        /**
         * This function is called when the application is launched again with new arguments
         * @param iArgument the arguments
         */
        virtual void processArguments(const QStringList& iArgument);

        /**
         * Get the current selection
         * @return selected objects
         */
        virtual SKGObjectBase::SKGListSKGObjectBase getSelectedObjects();

        /**
         * Get the number of seleted object
         * @return number of selected objects
         */
        virtual int getNbSelectedObjects();

        /**
         * To know if the closure of the application is authorized
         * @return true if close is authorized else false
         */
        virtual bool queryClose();

        /**
         * To know if the closure of the file is authorized
         * @return true if close is authorized else false
         */
        virtual bool queryFileClose();

        /**
         * Return the plugin number by index
         * @param iIndex the index of the plugin
         * @return the plugin pointer. Can be NULL. Mustn't be deleted
         */
        virtual SKGInterfacePlugin * getPluginByIndex ( int iIndex );

        /**
         * Return the plugin number by name
         * @param iName the name of the plugin
         * @return the plugin pointer. Can be NULL. Mustn't be deleted
         */
        virtual SKGInterfacePlugin * getPluginByName ( const QString& iName );

        /**
         * Get the index of the current tab
         * @return index of the current tab
         */
        virtual int currentTabIndex() const;

        /**
         * Get the current tab
         * @return the current tab
         */
        virtual SKGTabWidget* currentTab() const;

        /**
         * Get the label for normal message in status bar
         * @return the label
         */
        virtual QLabel* statusNormalMessage() const;

        /**
         * Get the current splash screen. NULL is the splash screen is closed.
         * @return the splash screen
         */
        virtual KSplashScreen* splashScreen() const;

        /**
         * Get then number of tabs
         * @return the number of tabs
         */
        virtual int countTab() const;

        /**
         * Set the current page
         * @param iIndex the current page
         */
        virtual void setCurrentTabIndex ( int iIndex );

        /**
         * Open a plugin in a page
         * @param plugin the plugin
         * @param index index of the tab to replace or -1 to add a new one
         * @param parameters parameters of the plugin
         * @param title title of the page
         * @return the opened tab
         */
        virtual SKGTabWidget* setNewTabContent ( SKGInterfacePlugin* plugin, int index=-1, const QString& parameters="", const QString& title="" );

        /**
         * Register a global action
         * @param iIdentifier identifier of the action
         * @param iAction action pointer
         */
        virtual void registedGlobalAction ( const QString& iIdentifier, KAction* iAction );

        /**
         * Get a registered global action
         * @param iIdentifier identifier of the action
         * @return action pointer
         */
        virtual KAction* getGlobalAction ( const QString& iIdentifier );

        /**
         * Creates a modal file dialog and returns the selected filename or an empty string if none was chosen.
         * A confirmation message is displayed if needed
         * @param startDir this can either be
         * @li the URL of the directory to start in.
         * @li a KUrl() to start in the current working directory, or the last directory where a file has been selected.
         * @li an URL starting with 'kfiledialog:///\<keyword\>' to start in the directory last used by a filedialog in the same application that specified the same keyword.
         * @li an URL starting with 'kfiledialog:///\<keyword\>?global' to start in the directory last used by a filedialog in any application that specified the same keyword.
         * @param filter a shell glob or a mime-type-filter that specifies which files to display. The preferred option is to set a list of mimetype names, see setMimeFilter() for details. Otherwise you can set the text to be displayed for the each glob, and provide multiple globs, see setFilter() for details.
         * @param parent the widget the dialog will be centered on initially.
         * @param caption the name of the dialog widget.
         * @return the file name
         */
        static QString getSaveFileName ( const KUrl & startDir = KUrl(), const QString & filter = QString(), QWidget * parent = 0, const QString & caption = QString() );

        /**
         * Display an error message
         * @param iError the error
         */
        static void displayErrorMessage ( const SKGError& iError );

        /**
         * Fill a widget with distinct values
         * @param iWidget the widget
         * @param iDoc document
         * @param iTable table
         * @param iAttribut attribute
         * @param iWhereClause where clause
         */
        static void fillWithDistinctValue (
                QWidget* iWidget,
                const SKGDocument* iDoc,
                const QString& iTable,
                const QString& iAttribut,
                const QString& iWhereClause );

        /**
         * Return main panel
         */
        static SKGMainPanel* getMainPanel();

public slots:
        /**
         * Force the refresh of the object.
         */
        virtual void refresh();

        /**
         * Send notifications corresponding to the transaction.
         * @param iTransaction the transaction identifier
         */
        virtual void notify ( int iTransaction );

        /**
         * Close a page
         * @param iWidget the page to remove (NULL means current one)
         */
        virtual void closeTab ( QWidget* iWidget=NULL );

        /**
         * Close all other pages
         * @param iWidget the page to keep (NULL means current one)
         */
        virtual void closeAllOtherTabs ( QWidget* iWidget=NULL );

        /**
         * Close all pages
         */
        virtual void closeAllTabs();

protected:
        /**
         * event
         * @param e event
         */
        virtual void changeEvent ( QEvent * e );

private slots:
        void optionsPreferences();
        void onSettingsChanged();
        void onCancelCurrentAction();
        void onQuitAction();

        void addTab();
        void onBeforeOpenContext();
        void onOpenContext();
        void onTipOfDay();
        void onTrayActivated ( QSystemTrayIcon::ActivationReason reason );
        void onChangeStyleSheet ( const QString& iNewStryle );
        void saveDefaultState();
        void resetDefaultState();
        void overwriteBookmarkState();
private:
        Q_DISABLE_COPY ( SKGMainPanel );
        void setupActions();
        bool iconInSystemtray();
        void refreshTabPosition();
        bool eventFilter ( QObject *object, QEvent *event );

        static int progressBarCallBack ( int iPos, void* iProgressBar );
        static bool naturalLessThan(const QString &s1, const QString &s2);

        Ui::skgmainpanel_base ui;
        Ui::skgmainpanel_pref uipref;

        KTabWidget *tabView;
        KSplashScreen* spScreen;

        SKGDocument* currentBankDocument;

        QList<SKGInterfacePlugin *> pluginsList;
        QMap<QString, KAction*> registeredGlogalAction;

        KAction *addTabAction;
        KAction *tipOfDayAction;
        KAction *closePageAction;
        KAction *closeAllPagesAction;
        KAction *closeAllOtherPagesAction;
        KAction *saveDefaultStateAction;
        KAction *resetDefaultStateAction;
        KAction *overwriteBookmarkStateAction;
        KTipDatabase* tipDatabase;
        static bool currentActionCanceled;
        KPushButton *kCancelButton;
        KSystemTrayIcon* kSystemTrayIcon;
        QLabel* kNormalMessage;

        struct doublePointer {
                void* p1;
                void* p2;
        };

        doublePointer progressObjects;
        QString defaultStyle;
        bool middleClick;

        static SKGMainPanel* mainPanel;
};

#endif // SKGMAINPANEL_H
