/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for bank management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbankpluginwidget.h"
#include "skgsortfilterproxymodel.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgtransactionmng.h"
#include "skgbankobject.h"
#include "skgaccountobject.h"
#include "skgdocumentbank.h"
#include "skgtraces.h"

#include <kstandarddirs.h>

#include <QDomDocument>
#include <QFile>
#include <QDir>

SKGBankPluginWidget::SKGBankPluginWidget ( SKGDocumentBank* iDocument )
                : SKGTabWidget (iDocument )
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::SKGBankPluginWidget" );

        ui.setupUi ( this );

        timer = new QTimer( this );
        timer->setSingleShot( true );
        connect( timer, SIGNAL( timeout() ), this, SLOT( refreshInfoZone() ) );

        //Add Standard KDE Icons to buttons to Accounts
        ui.kAccountCreatorUpdate->setIcon ( KIcon ( "dialog-ok-apply" ) );
        ui.kAccountCreatorAdd->setIcon ( KIcon ( "list-add" ) );

        ui.kStandardBtn->setIcon (KIcon ( "flag-blue" ) );
        ui.kIntermediateBtn->setIcon (KIcon ( "flag-green" ) );
        ui.kAllBtn->setIcon (KIcon ( "flag-yellow" ) );

        //Add NLS values for type of account
        //C=current D=credit card P=passif (for objects) I=Investment O=other
        ui.kAccountCreatorType->addItem ( i18n ( "Current" ) );
        ui.kAccountCreatorType->addItem ( i18n ( "Credit card" ) );
        ui.kAccountCreatorType->addItem ( i18n ( "Investment" ) );
        ui.kAccountCreatorType->addItem ( i18n ( "Assets" ) );
        ui.kAccountCreatorType->addItem ( i18n ( "Other" ) );

        //Bind account creation view
        objectModel = new SKGObjectModel ( ( SKGDocumentBank* ) getDocument(), "v_account_display", "1=0", this, "", false);
        SKGSortFilterProxyModel* modelProxy = new SKGSortFilterProxyModel ( this );
        modelProxy->setSourceModel ( objectModel );
        modelProxy->setSortRole ( Qt::UserRole );
        modelProxy->setDynamicSortFilter ( true );

        ui.kFilterEdit->setProxy(modelProxy);

        ui.kAccountTableViewEdition->setModel ( modelProxy );
        ui.kAccountTableViewEdition->setWindowTitle ( i18n ( "Accounts" ) );

        //Add registered global action in contextual menu

        if (SKGMainPanel::getMainPanel()) {
                ui.kAccountTableViewEdition->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "edit_delete" ) );
                QAction* sep=new QAction ( this );
                sep->setSeparator ( true );
                ui.kAccountTableViewEdition->insertAction ( 0, sep );
                ui.kAccountTableViewEdition->insertAction ( 0, SKGMainPanel::getMainPanel()->getGlobalAction ( "open_report" ) );
        }
        connect ( ui.kAccountTableViewEdition, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );
        connect ( objectModel, SIGNAL ( beforeReset() ), ui.kAccountTableViewEdition, SLOT ( saveSelection() ) );
        connect ( objectModel, SIGNAL ( afterReset() ), ui.kAccountTableViewEdition, SLOT ( resetSelection() ) );

        ui.kAccountTableViewEdition->sortByColumn ( 0, Qt::AscendingOrder );

        //Logo for banks
        ui.kAccountCreatorIcon->addItem ( "" );
        QFile file ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/list_bank.txt" ) ) );
        if ( file.open ( QIODevice::ReadOnly | QIODevice::Text ) ) {
                QDir dirLogo ( KStandardDirs::locate ( "data", QString::fromLatin1 ( "skrooge/images/logo/" ) ) );

                QTextStream in ( &file );
                while ( !in.atEnd() ) {
                        QString line=in.readLine().trimmed();
                        if ( !line.isEmpty() ) {
                                QString bankName=line;
                                bankName.remove ( ".png" );
                                bankName.replace ( '_', ' ' );

                                QRegExp rx ( "(.+) {2,}(.+)" );
                                if ( rx.indexIn ( bankName ) !=-1 ) {
                                        //Icon is something like <bank name>__<banknumber>.png
                                        listBankNumbers.append ( rx.cap ( 2 ) );
                                        bankName=rx.cap ( 1 );
                                } else {
                                        listBankNumbers.append ( "" );
                                }

                                ui.kAccountCreatorIcon->addItem ( QIcon ( dirLogo.absoluteFilePath ( line ) ), bankName );
                        }
                }

                file.close();
        }

        setCurrentMode(-1);

        //Refresh
        connect ( ( const QObject* ) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection );
        connect ( ( const QObject* ) getDocument(), SIGNAL ( transactionSuccessfullyEnded ( int ) ), this, SLOT ( onSelectionChanged() ), Qt::QueuedConnection );
        dataModified("", 0);
}

SKGBankPluginWidget::~SKGBankPluginWidget()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::~SKGBankPluginWidget" );
        objectModel=NULL;
        timer=NULL;
}

void SKGBankPluginWidget::onSelectionChanged()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::onSelectionChanged" );
        //Mapping
        QItemSelectionModel *selModel=ui.kAccountTableViewEdition->selectionModel();
        if ( selModel ) {
                QModelIndexList indexes=selModel->selectedRows();
                if ( indexes.count() && objectModel ) {
                        QModelIndex idx=indexes[indexes.count()-1];

                        QSortFilterProxyModel* proxyModel= ( QSortFilterProxyModel* ) ui.kAccountTableViewEdition->model();
                        QModelIndex idxs=proxyModel->mapToSource ( idx );

                        //set the icon
                        SKGAccountObject account ( objectModel->getObject ( idxs ) );
                        SKGBankObject bank;
                        account.getBank ( bank );
                        QString iconName=bank.getIcon();
                        if ( iconName.length() ) {
                                iconName.remove ( ".png" );
                                iconName.replace ( '_', ' ' );

                                QRegExp rx ( "(.+) {2,}(.+)" );
                                if ( rx.indexIn ( iconName ) !=-1 ) {
                                        iconName=rx.cap ( 1 );
                                }
                                ui.kAccountCreatorIcon->setText ( iconName );
                        } else ui.kAccountCreatorIcon->setText ( "" );

                        ui.kAccountCreatorBank->setText ( account.getAttribute ( "t_BANK" ) );
                        ui.kAccountCreatorAccount->setText ( account.getAttribute ( "t_name" ) );
                        ui.kAccountCreatorBankNumber->setText ( account.getAttribute ( "t_BANK_NUMBER" ) );
                        ui.kAccountCreatorAgencyNumber->setText ( account.getAttribute ( "t_agency_number" ) );
                        ui.kAccountCreatorNumber->setText ( account.getAttribute ( "t_number" ) );
                        ui.kAccountCreatorType->setText ( account.getAttribute ( "t_TYPENLS" ) );
                        ui.kAccountCreatorAddress->setText ( account.getAttribute ( "t_agency_address" ) );
                        ui.kAccountCreatorComment->setText ( account.getAttribute ( "t_comment" ) );
                }

                onAccountCreatorModified();
                emit selectionChanged();
        }
}

int SKGBankPluginWidget::getCurrentMode()
{
        return mode;
}

void SKGBankPluginWidget::onBtnModeClicked()
{
        QWidget* sender=static_cast<QWidget*> ( this->sender() );

        int currentMode=getCurrentMode();
        int newMode=0;

        if (sender==ui.kStandardBtn) newMode=0;
        else if (sender==ui.kIntermediateBtn) newMode=1;
        else if (sender==ui.kAllBtn) newMode=2;

        if (currentMode==newMode) newMode=-1;
        setCurrentMode(newMode);
}

void SKGBankPluginWidget::setCurrentMode(int iMode)
{
        mode=iMode;

        ui.kAllFrm->hide();
        ui.kIntermediateFrm->hide();
        ui.kCommonFrm->setVisible(mode>=0);
        ui.kBtnFrm->setVisible(mode>=0);

        ui.kStandardBtn->setChecked(false);
        ui.kIntermediateBtn->setChecked(false);
        ui.kAllBtn->setChecked(false);

        if (mode==0) {
                ui.kStandardBtn->setChecked(true);
        } else if (mode==1) {
                ui.kIntermediateFrm->show();
                ui.kIntermediateBtn->setChecked(true);
        } else if (mode==2) {
                ui.kIntermediateFrm->show();
                ui.kAllFrm->show();
                ui.kAllBtn->setChecked(true);
        }

        onAccountCreatorModified();
}

void SKGBankPluginWidget::onAccountCreatorModified()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::onAccountCreatorModified" );

        bool activated=ui.kAccountCreatorBank->text().length() >0 &&
                       ui.kAccountCreatorAccount->text().length() >0;

        int nbSelect=getNbSelectedObjects();
        ui.kAccountCreatorAdd->setEnabled ( activated );
        ui.kAccountCreatorUpdate->setEnabled ( activated && nbSelect>0 );
        ui.kAccountCreatorAccount->setEnabled ( nbSelect<=1 );
        ui.kAccountCreatorNumber->setEnabled ( nbSelect<=1 );

        //Facilitate bank name
        if ( ui.kAccountCreatorBank->text().length() ==0 ) {
                ui.kAccountCreatorBank->setText ( ui.kAccountCreatorIcon->currentIndex() !=0 ? ui.kAccountCreatorIcon->currentText() : "" );
        }

        //Facilitate bank number
        if ( ui.kAccountCreatorBankNumber->text().length() ==0 ) {
                int pos=ui.kAccountCreatorIcon->currentIndex();
                ui.kAccountCreatorBankNumber->setText ( pos>=1 && pos-1<listBankNumbers.count() ? listBankNumbers[pos-1] : "" );
        }
}

void SKGBankPluginWidget::onAddAccountClicked()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGBankPluginWidget::onAddAccountClicked",err );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        {
                SKGAccountObject accountObj;

                QString bankname=ui.kAccountCreatorBank->text();
                QString accountname=ui.kAccountCreatorAccount->text();
                QString name=bankname+'-'+accountname;
                {
                        SKGBEGINTRANSACTION ( *getDocument(), i18n ( "Account creation [%1]", name ), err );

                        //Create bank object in case of missing
                        SKGBankObject bankObj ( getDocument() );
                        if ( err.isSucceeded() ) err=bankObj.setName ( ui.kAccountCreatorBank->text() );
                        if ( err.isSucceeded() ) {
                                //Build icon name
                                QString icon;
                                int pos=ui.kAccountCreatorIcon->currentIndex();
                                if ( pos!=0 ) {
                                        icon=ui.kAccountCreatorIcon->currentText();
                                        if ( listBankNumbers[pos-1].length() ) icon+="  "+listBankNumbers[pos-1];

                                        icon.replace ( ' ', '_' );
                                        icon+=".png";

                                }

                                err=bankObj.setIcon ( icon );
                        }
                        if ( err.isSucceeded() ) err=bankObj.setNumber ( ui.kAccountCreatorBankNumber->text() );

                        if ( err.isSucceeded() ) err=bankObj.save();
                        if ( err.isSucceeded() ) err=bankObj.load();

                        //Create account object in case of missing
                        if ( err.isSucceeded() ) err=bankObj.addAccount ( accountObj );
                        if ( err.isSucceeded() ) err=accountObj.setName ( accountname );
                        if ( err.isSucceeded() ) err=accountObj.setAgencyNumber ( ui.kAccountCreatorAgencyNumber->text() );
                        if ( err.isSucceeded() ) err=accountObj.setAgencyAddress ( ui.kAccountCreatorAddress->text() );
                        if ( err.isSucceeded() ) err=accountObj.setComment ( ui.kAccountCreatorComment->text() );
                        if ( err.isSucceeded() ) err=accountObj.setNumber ( ui.kAccountCreatorNumber->text() );
                        if ( err.isSucceeded() ) err=accountObj.setType ( ( SKGAccountObject::AccountType ) ( ui.kAccountCreatorType->currentIndex() ) );
                        if ( err.isSucceeded() ) err=accountObj.save ( false );
                }

                //status bar
                if ( err.isSucceeded() ) {
                        err=SKGError ( 0, i18n ( "Account [%1] created", name ) );
                        ui.kAccountTableViewEdition->selectObject ( accountObj.getUniqueID() );
                } else err.addError ( ERR_FAIL, i18n ( "Account creation failed" ) );
        }
        QApplication::restoreOverrideCursor();

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

SKGObjectBase::SKGListSKGObjectBase SKGBankPluginWidget::getSelectedObjects()
{
        return ui.kAccountTableViewEdition->getSelectedObjects();
}

int SKGBankPluginWidget::getNbSelectedObjects()
{
        return ui.kAccountTableViewEdition->getNbSelectedObjects();
}

void SKGBankPluginWidget::onModifyAccountClicked()
{
        SKGError err;
        SKGTRACEINRC ( 10, "SKGBankPluginWidget::onModifyAccountClicked",err );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        {
                //Get Selection
                SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

                int nb=selection.count();
                SKGBEGINPROGRESSTRANSACTION ( *getDocument(), i18n ( "Account update" ), err, nb );
                for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                        SKGAccountObject accountObj=selection[i];

                        //Update account if single selection
                        if ( nb==1 ) {
                                err=accountObj.setName ( ui.kAccountCreatorAccount->text() );
                                if ( err.isSucceeded() ) err=accountObj.setNumber ( ui.kAccountCreatorNumber->text() );
                        }
                        if ( err.isSucceeded() ) err=accountObj.setType ( ( SKGAccountObject::AccountType ) ( ui.kAccountCreatorType->currentIndex() ) );
                        if ( err.isSucceeded() ) err=accountObj.setAgencyNumber ( ui.kAccountCreatorAgencyNumber->text() );
                        if ( err.isSucceeded() ) err=accountObj.setAgencyAddress ( ui.kAccountCreatorAddress->text() );
                        if ( err.isSucceeded() ) err=accountObj.setComment ( ui.kAccountCreatorComment->text() );
                        if ( err.isSucceeded() ) err=accountObj.save();

                        //Update bank
                        SKGBankObject bankObj;
                        if ( SKGNamedObject::getObjectByName ( getDocument(), "bank", ui.kAccountCreatorBank->text(), bankObj ).isSucceeded() ) {
                                //The created bank already exist ==> update parent bank
                                if ( err.isSucceeded() ) err=accountObj.setBank ( bankObj );
                                if ( err.isSucceeded() ) err=accountObj.save();
                        } else {
                                //The bank does not exist ==> update this one
                                if ( err.isSucceeded() ) err=accountObj.getBank ( bankObj );
                                if ( err.isSucceeded() ) err=bankObj.setName ( ui.kAccountCreatorBank->text() );
                        }
                        if ( err.isSucceeded() ) {
                                //Build icon name
                                QString icon;
                                int pos=ui.kAccountCreatorIcon->currentIndex();
                                if ( pos!=0 ) {
                                        icon=ui.kAccountCreatorIcon->currentText();
                                        if ( listBankNumbers[pos-1].length() ) icon+="  "+listBankNumbers[pos-1];

                                        icon.replace ( ' ', '_' );
                                        icon+=".png";

                                }

                                err=bankObj.setIcon ( icon );
                        }
                        if ( err.isSucceeded() ) err=bankObj.setNumber ( ui.kAccountCreatorBankNumber->text() );
                        if ( err.isSucceeded() ) err=bankObj.save();

                        if ( err.isSucceeded() ) err=getDocument()->stepForward ( i+1 );
                }

                //status bar
                if ( err.isSucceeded() ) err=SKGError ( 0, i18n ( "Account updated." ) );
                else err.addError ( ERR_FAIL, i18n ( "Update failed" ) );
        }
        QApplication::restoreOverrideCursor();

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGBankPluginWidget::onDoubleClickedAccount()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::onDoubleClickedAccount" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        //Open in operation plugin
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nb=selection.count();
        for ( int i=0; i<nb; ++i ) {
                SKGAccountObject accountObj=selection[i];

                QDomDocument doc("SKGML");
                doc.setContent ( getDocument()->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
                QDomElement root = doc.documentElement();
                if (root.isNull()) {
                        root=doc.createElement("parameters");
                        doc.appendChild(root);
                }

                root.setAttribute ( "account", accountObj.getName() );
                SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );

        }
        QApplication::restoreOverrideCursor();
}

void SKGBankPluginWidget::onFilterChanged()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::onFilterChanged" );
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        //Compute where clause
        QString filter;

        if ( ui.kHideUseless->checkState() ==Qt::Checked ) {
                filter=" t_close='N'";
        }

        //Update model
        if ( objectModel ) {
                objectModel->setFilter ( filter );
                ui.kAccountTableViewEdition->setState ( ui.kAccountTableViewEdition->getState() );
        }

        QApplication::restoreOverrideCursor();
}

void SKGBankPluginWidget::onComputeRIB()
{
        QString sb=ui.kAccountCreatorBankNumber->text().rightJustified ( 5, '0', true );
        QString sg=ui.kAccountCreatorAgencyNumber->text().rightJustified ( 5, '0', true );
        QString sc=ui.kAccountCreatorNumber->text().rightJustified ( 11, '0', true );

        QString l1="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        QString l2="12345678912345678923456789";

        for ( int i=0; i<26; ++i ) {
                sc=sc.replace ( l1[i], l2[i] );
        }

        int b=SKGServices::stringToInt ( sb );
        int g=SKGServices::stringToInt ( sg );
        int d=SKGServices::stringToInt ( sc.left ( 6 ) );
        int c=SKGServices::stringToInt ( sc.right ( 5 ) );

        ui.kRib->setText ( SKGServices::intToString ( 97 - ( ( 89 * b + 15 * g + 76 * d + 3 * c ) % 97 ) ) );
}

void SKGBankPluginWidget::cleanEditor()
{
        if (getNbSelectedObjects()==0) {
                ui.kAccountCreatorIcon->setText ( "" );
                ui.kAccountCreatorBank->setText ( "" );
                ui.kAccountCreatorAccount->setText ( "" );
                ui.kAccountCreatorBankNumber->setText ( "" );
                ui.kAccountCreatorAgencyNumber->setText ( "" );
                ui.kAccountCreatorNumber->setText ( "" );
                //Not done because of it's a combo ui.kAccountCreatorType->setText("");
                ui.kAccountCreatorAddress->setText ( "" );
                ui.kAccountCreatorComment->setText ( "" );
        }
}

QString SKGBankPluginWidget::getState()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::getState" );
        QDomDocument doc ( "SKGML" );
        QDomElement root = doc.createElement ( "parameters" );
        doc.appendChild ( root );

        root.setAttribute ( "hideUseless", ui.kHideUseless->checkState() ==Qt::Checked ? "Y" : "N" );
        root.setAttribute ( "filter", ui.kFilterEdit->lineEdit()->text() );
        root.setAttribute ( "currentPage", getCurrentMode() );

        //Memorize table settings
        root.setAttribute ( "view", ui.kAccountTableViewEdition->getState() );
        return doc.toString();
}

void SKGBankPluginWidget::setState ( const QString& iState )
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::setState" );
        QDomDocument doc ( "SKGML" );
        doc.setContent ( iState );
        QDomElement root = doc.documentElement();

        QString hideUseless=root.attribute ( "hideUseless" );
        QString filter=root.attribute ( "filter" );
        QString currentPage=root.attribute ( "currentPage" );

        if ( !hideUseless.isEmpty() ) ui.kHideUseless->setCheckState ( hideUseless=="Y" ? Qt::Checked : Qt::Unchecked );
        if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );
        if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );

        if (objectModel) {
                bool previous=objectModel->blockRefresh(true);
                onFilterChanged();
                objectModel->blockRefresh(previous);
        }

        //!!! Must be done here after onFilterChanged
        ui.kAccountTableViewEdition->setState ( root.attribute ( "view" ) );
}

QString SKGBankPluginWidget::getDefaultStateAttribute()
{
        return "SKGBANK_DEFAULT_PARAMETERS";
}

void SKGBankPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)

{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::dataModified" );
        Q_UNUSED(iIdTransaction);

        //Refresh widgets
        QSqlDatabase* db = getDocument()->getDatabase();
        setEnabled ( db!=NULL );
        if ( db!=NULL && (iTableName=="account" || iTableName.isEmpty())) {
                //Correction bug 2299394 vvv
                if ( ui.kAccountTableViewEdition->isAutoResized() ) ui.kAccountTableViewEdition->resizeColumnsToContents();
                //Correction bug 2299394 ^^^

                //Refresh info area
                if (timer) timer->start( 300 );
        }
}

void SKGBankPluginWidget::refreshInfoZone()
{
        SKGTRACEIN ( 10, "SKGBankPluginWidget::refreshInfoZone" );

        //Refresh info area
        SKGStringListList listTmp;
        SKGServices::executeSelectSqliteOrder ( getDocument(),
                                                "SELECT TOTAL(f_CURRENTAMOUNT), TOTAL(f_CHECKED), TOTAL(f_COMING_SOON) from v_account_display",
                                                listTmp );
        if ( listTmp.count() ==2 ) {
                KLocale* locale=KGlobal::locale();

                QString pUnit= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                double v1=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 0 ) );
                double v2=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 1 ) );
                double v3=SKGServices::stringToDouble ( listTmp.at ( 1 ).at ( 2 ) );
                QString s1=QString ( "<font color=\"" ) + ( v1<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v1, pUnit,2 ) +"</font>";
                QString s2=QString ( "<font color=\"" ) + ( v2<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v2, pUnit,2 ) +"</font>";
                QString s3=QString ( "<font color=\"" ) + ( v3<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v3, pUnit,2 ) +"</font>";
                ui.kInfo->setText ( i18n ( "Balance: %1     Cleared: %2     In Transit: %3", s1, s2, s3 ) );

                QString secondaryUnit= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnit();
                double secondaryUnitValue= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnitValue();
                if ( !secondaryUnit.isEmpty() && secondaryUnitValue ) {
                        s1=QString ( "<font color=\"" ) + ( v1<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v1/secondaryUnitValue, secondaryUnit,2 ) +"</font>";
                        s2=QString ( "<font color=\"" ) + ( v2<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v2/secondaryUnitValue, secondaryUnit,2 ) +"</font>";
                        s3=QString ( "<font color=\"" ) + ( v3<0 ? "red" : "black" ) +"\">"+locale->formatMoney ( v3/secondaryUnitValue, secondaryUnit,2 ) +"</font>";
                }
                ui.kInfo->setToolTip ( i18n ( "<p>Balance: %1</p><p>Cleared: %2</p><p>In Transit: %3</p>", s1, s2, s3 ) );
        }
}

QWidget* SKGBankPluginWidget::getWidgetForPrint()
{
        return ui.kAccountTableViewEdition;
}
#include "skgbankpluginwidget.moc"
