/*
** Copyright (C) 2002-2009 Sourcefire, Inc.
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License Version 2 as
** published by the Free Software Foundation.  You may not use, modify or
** distribute this program under any other version of the GNU General
** Public License.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id$ */
/* spp_rpc_decode 
 * 
 * Purpose:
 *
 * This preprocessor normalizes the RPC requests from remote machines by
 * converting all fragments into one continous stream.
 * This is very useful for doing things like defeating hostile attackers
 * trying to stealth themselves from IDSs by fragmenting the request so the
 * string 0186A0 is broken up.
 *
 * Arguments:
 *   
 * This plugin takes a list of integers representing the TCP ports that the
 * user is interested in having normalized
 *
 * Effect:
 *
 * Changes the data in the packet payload and changes
 * p->dsize to reflect the new (smaller) payload size.
 *
 * Comments:
 *
 */

#ifdef HAVE_CONFIG_H	 
#include "config.h"	 
#endif

#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef HAVE_STRINGS_H	 
#include <strings.h>	 
#endif

#include "decode.h"
#include "plugbase.h"
#include "parser.h"
#include "log.h"
#include "debug.h"
#include "util.h"

#include "mstring.h"
#include "snort.h"
#include "detect.h"
#include "log.h"
#include "generators.h"
#include "event_queue.h"

#include "profiler.h"
#include "bounds.h"
#include "strlcatu.h"

#include "stream_api.h"

#ifdef TARGET_BASED
#include "sftarget_protocol_reference.h"
#endif
#include "sfPolicy.h"

extern uint8_t DecodeBuffer[DECODE_BLEN];

#define OPT_ALERT_FRAGMENTS "alert_fragments"
#define OPT_ALERT_MULTIPLE_REQUESTS "no_alert_multiple_requests"
#define OPT_ALERT_LARGE_FRAGMENTS "no_alert_large_fragments"
#define OPT_ALERT_INCOMPLETE "no_alert_incomplete"

#define TEXT_ALERT_MULTIPLE_REQUESTS "alert_multiple_requests"
#define TEXT_ALERT_LARGE_FRAGMENTS "alert_large_fragments"
#define TEXT_ALERT_INCOMPLETE "alert_incomplete"
#define RPC_DECODE_DEFAULT_PORTS "111 32271"

#define RPC_CLASS DECODE_CLASS /* use the same classification as the other decoder alerts */

typedef struct _RpcDecodeConfig
{
    char alert_fragments;    /* Alert when we see ANY fragmented RPC requests */
    char alert_incomplete;   /* Alert when we don't see all of a request in one packet */
    char alert_multi;        /* Alert when we see multiple requests in one packet */
    char alert_large;        /* Alert when we see multiple requests in one packet */
    uint8_t RpcDecodePorts[MAXPORTS/8];

} RpcDecodeConfig;

static tSfPolicyUserContextId rpc_decode_config = NULL;

#ifdef TARGET_BASED
static int16_t rpc_decode_app_protocol_id;
#endif

#ifdef PERF_PROFILING
PreprocStats rpcdecodePerfStats;
#endif

static void RpcDecodeInit(char *);
static void PreprocRpcDecode(Packet *, void *);
static void ParseRpcConfig(RpcDecodeConfig *, char *);
static int ConvertRPC(RpcDecodeConfig *, Packet *);
static void RpcDecodeFreeConfig(tSfPolicyUserContextId rpc);
static void RpcDecodeCleanExit(int, void *);
static void _addPortsToStream5Filter(RpcDecodeConfig *, tSfPolicyId);
#ifdef TARGET_BASED
static void _addServicesToStream5Filter(tSfPolicyId);
#endif
static void RpcDecodePortsAssign(uint8_t *, char *);

#ifdef SNORT_RELOAD
static tSfPolicyUserContextId rpc_decode_swap_config = NULL;
static void RpcDecodeReload(char *);
static void * RpcDecodeReloadSwap(void);
static void RpcDecodeReloadSwapFree(void *);
#endif


//function to assign the RpcDecodePorts array
static void RpcDecodePortsAssign(uint8_t *RpcDecodePorts, char *portlist)
{
    int num;
    int num_toks;
    char **toks;
    if( portlist == NULL || *portlist == '\0')
    {
        portlist = RPC_DECODE_DEFAULT_PORTS;
    }
    toks = mSplit(portlist, " \t", 0, &num_toks, 0);

    for(num = 0; num < num_toks; num++)
    {
        if(isdigit((int)toks[num][0]))
        {
            char *num_p = NULL; /* used to determine last position in string */
            long t_num;

            t_num = strtol(toks[num], &num_p, 10);

            if(*num_p != '\0')
            {
                ParseError("Port Number invalid format: %s.", toks[num]);
            }
            else if(t_num < 0 || t_num > MAXPORTS-1 )
            {
                ParseError("Port Number out of range: %d.", t_num);
            }

            RpcDecodePorts[(t_num/8)] |= 1<<(t_num%8);
        }
    }
    mSplitFree(&toks, num_toks);

}

/*
 * Function: SetupRpcDecode()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupRpcDecode(void)
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
#ifndef SNORT_RELOAD
    RegisterPreprocessor("rpc_decode", RpcDecodeInit);
#else
    RegisterPreprocessor("rpc_decode", RpcDecodeInit, RpcDecodeReload,
                         RpcDecodeReloadSwap, RpcDecodeReloadSwapFree);
#endif

    DEBUG_WRAP(DebugMessage(DEBUG_RPC,"Preprocessor: RpcDecode in setup...\n"););
}


/*
 * Function: RpcDecodeInit(char *)
 *
 * Purpose: Processes the args sent to the preprocessor, sets up the
 *          port list, links the processing function into the preproc
 *          function list
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void RpcDecodeInit(char *args)
{
    tSfPolicyId policy_id = getParserPolicy();
    RpcDecodeConfig *pPolicyConfig = NULL;

    if (rpc_decode_config == NULL)
    {
        rpc_decode_config = sfPolicyConfigCreate(); 

        AddFuncToPreprocCleanExitList(RpcDecodeCleanExit, NULL, PRIORITY_LAST, PP_RPCDECODE);
        AddFuncToPreprocRestartList(RpcDecodeCleanExit, NULL, PRIORITY_LAST, PP_RPCDECODE);

#ifdef PREF_PROFILING
        RegisterPreprocessorProfile("rpcdecode", &rpcdecodePerfStats, 0, &totalPerfStats);
#endif

#ifdef TARGET_BASED
        /* Find and cache protocol ID for packet comparison */
        rpc_decode_app_protocol_id = FindProtocolReference("sunrpc");
        if (rpc_decode_app_protocol_id == SFTARGET_UNKNOWN_PROTOCOL)
            rpc_decode_app_protocol_id = AddProtocolReference("sunrpc");

#endif
    }

     sfPolicyUserPolicySet (rpc_decode_config, policy_id);
     pPolicyConfig = (RpcDecodeConfig *) sfPolicyUserDataGetCurrent(rpc_decode_config);
     if (pPolicyConfig)
     {
        ParseError("RPC decode can only be configured once.\n");
     }
     pPolicyConfig = (RpcDecodeConfig *)SnortAlloc(sizeof(RpcDecodeConfig));
     if (!pPolicyConfig)
     {
         ParseError("RPC_DECODE preprocessor: memory allocate failed.\n");
     }

     sfPolicyUserDataSetCurrent(rpc_decode_config, pPolicyConfig);

    
    /* parse the argument list into a list of ports to normalize */
    ParseRpcConfig(pPolicyConfig, args);

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(PreprocRpcDecode, PRIORITY_APPLICATION, PP_RPCDECODE, PROTO_BIT__TCP);

    _addPortsToStream5Filter(pPolicyConfig, policy_id);

#ifdef TARGET_BASED
    _addServicesToStream5Filter(policy_id);
#endif

    DEBUG_WRAP(DebugMessage(DEBUG_RPC,"Preprocessor: RpcDecode Initialized\n"););
}

/*
 * Function: ParseRpcConfig(char *)
 *
 * Purpose: Reads the list of port numbers from the argument string and
 *          parses them into the port list data struct
 *
 * Arguments: portlist => argument list
 *
 * Returns: void function
 *
 */
void ParseRpcConfig(RpcDecodeConfig *rpc, char *portlist)
{
    char portstr[STD_BUF];
    char **toks;
    int is_reset = 0;
    int num_toks;
    int num;

    if (rpc == NULL)
        return;

    if(portlist == NULL || *portlist == '\0')
    {
        portlist = RPC_DECODE_DEFAULT_PORTS;
    }
    rpc->alert_multi = 1;
    rpc->alert_incomplete = 1;
    rpc->alert_large = 1;

    /* tokenize the argument list */
    toks = mSplit(portlist, " \t", 0, &num_toks, 0);

    LogMessage("rpc_decode arguments:\n");
    
    /* convert the tokens and place them into the port list */
    for(num = 0; num < num_toks; num++)
    {
        if(isdigit((int)toks[num][0]))
        {
            char *num_p = NULL; /* used to determine last position in string */
            long t_num;

            t_num = strtol(toks[num], &num_p, 10);

            if(*num_p != '\0')
            {
                ParseError("Port Number invalid format: %s.", toks[num]);
            }
            else if(t_num < 0 || t_num > MAXPORTS-1)
            {
                ParseError("Port Number out of range: %d\n", t_num);
            }

            /* user specified a legal port number and it should override the default
               port list, so reset it unless already done */
            if(!is_reset)
            {
                memset(rpc->RpcDecodePorts, 0, sizeof(rpc->RpcDecodePorts));
                portstr[0] = '\0';
                is_reset = 1;
            }

            /* mark this port as being interesting using some portscan2-type voodoo,
               and also add it to the port list string while we're at it so we can
               later print out all the ports with a single LogMessage() */
            rpc->RpcDecodePorts[(t_num/8)] |= 1<<(t_num%8);
            strlcat(portstr, toks[num], STD_BUF - 1);
            strlcat(portstr, " ", STD_BUF - 1);
        }
        else if(!strcasecmp(OPT_ALERT_MULTIPLE_REQUESTS,toks[num]))
        {
            rpc->alert_multi = 0;
        }
        else if(!strcasecmp(OPT_ALERT_INCOMPLETE,toks[num]))
        {
            rpc->alert_incomplete = 0;
        }
        else if(!strcasecmp(OPT_ALERT_LARGE_FRAGMENTS,toks[num]))
        {
            rpc->alert_large = 0;
        }
        else if(!strcasecmp(OPT_ALERT_FRAGMENTS,toks[num]))
        {
            rpc->alert_fragments = 1;
        }
        else
        {
            ParseError("Unknown argument to rpc_decode "
                       "preprocessor: \"%s\".", toks[num]);
        }
    }

    mSplitFree(&toks, num_toks);

    if( !is_reset )
    {
        RpcDecodePortsAssign(rpc->RpcDecodePorts, RPC_DECODE_DEFAULT_PORTS);
    }
	
    /* print out final port list */
    LogMessage("    Ports to decode RPC on: %s\n", is_reset ? portstr : RPC_DECODE_DEFAULT_PORTS); 
    LogMessage("    %s: %s\n", OPT_ALERT_FRAGMENTS, rpc->alert_fragments ? "ACTIVE": "INACTIVE");
    LogMessage("    %s: %s\n", TEXT_ALERT_LARGE_FRAGMENTS, rpc->alert_large ? "ACTIVE": "INACTIVE");
    LogMessage("    %s: %s\n", TEXT_ALERT_INCOMPLETE, rpc->alert_incomplete ? "ACTIVE": "INACTIVE");
    LogMessage("    %s: %s\n", TEXT_ALERT_MULTIPLE_REQUESTS, rpc->alert_multi ? "ACTIVE": "INACTIVE");
}                                                                               
   

/*
 * Function: PreprocRpcDecode(Packet *)
 *
 * Purpose: Inspects the packet's payload for fragment records and 
 *          converts them into one infragmented record.
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
static void PreprocRpcDecode(Packet *p, void *context)
{
    int ret = 0; /* return code for ConvertRPC */
#ifdef TARGET_BASED
    int16_t app_id = SFTARGET_UNKNOWN_PROTOCOL;
    int valid_app_id = 0;
#endif
    RpcDecodeConfig *rconfig = NULL; 
    PROFILE_VARS;
    sfPolicyUserPolicySet (rpc_decode_config, getRuntimePolicy());
    rconfig = (RpcDecodeConfig *)sfPolicyUserDataGetCurrent(rpc_decode_config);

    /* Not configured in this policy */
    if (rconfig == NULL)
        return;

    DEBUG_WRAP(DebugMessage(DEBUG_RPC,"rpc decoder init on %d bytes\n", p->dsize););

    /* check to make sure we're talking TCP and that the TWH has already
       completed before processing anything */
    if(!IsTCP(p))
    {
        DEBUG_WRAP(DebugMessage(DEBUG_RPC,"It isn't TCP session traffic\n"););
        return;
    }

    if (ScStateful() && (p->packet_flags & PKT_FROM_SERVER))
    {
        DEBUG_WRAP(DebugMessage(DEBUG_RPC,"This is from a server\n"););
        return;
    }


#ifdef TARGET_BASED
    /* check stream info, fall back to checking ports */
    if(stream_api)
    {
        app_id = stream_api->get_application_protocol_id(p->ssnptr);
        if (app_id > 0)
        {
            valid_app_id = 1;
        }
    }

    if(valid_app_id && app_id != rpc_decode_app_protocol_id)
    {
        return;
    }

    if(!valid_app_id && !(rconfig->RpcDecodePorts[(p->dp/8)] & (1<<(p->dp%8))))
    {
        return;
    }
#else
    if(!(rconfig->RpcDecodePorts[(p->dp/8)] & (1<<(p->dp%8))))
    {
        return;
    }
#endif

    PREPROC_PROFILE_START(rpcdecodePerfStats);

    ret = ConvertRPC(rconfig, p);
    DEBUG_WRAP(DebugMessage(DEBUG_RPC,"Got ret: %d from ConvertRPC\n", ret););
    
    if(ret != 0)
    {
        switch(ret)
        {
        case RPC_MULTIPLE_RECORD:
            if(rconfig->alert_multi)
            {
                SnortEventqAdd(GENERATOR_SPP_RPC_DECODE, RPC_MULTIPLE_RECORD, 
                        1, RPC_CLASS, 3, RPC_MULTIPLE_RECORD_STR, 0);
            }
            break;
        case RPC_LARGE_FRAGSIZE:
            if(rconfig->alert_large)
            {
                SnortEventqAdd(GENERATOR_SPP_RPC_DECODE, RPC_LARGE_FRAGSIZE, 
                        1, RPC_CLASS, 3, RPC_LARGE_FRAGSIZE_STR, 0);
            }
            break;
        case RPC_INCOMPLETE_SEGMENT:
            if(rconfig->alert_incomplete)
            {
                SnortEventqAdd(GENERATOR_SPP_RPC_DECODE, RPC_INCOMPLETE_SEGMENT, 
                        1, RPC_CLASS, 3, RPC_INCOMPLETE_SEGMENT_STR, 0);
            }
            break;
        case RPC_ZERO_LENGTH_FRAGMENT:
            if(rconfig->alert_multi)
            {
                SnortEventqAdd(GENERATOR_SPP_RPC_DECODE, RPC_ZERO_LENGTH_FRAGMENT, 
                        1, RPC_CLASS, 3, RPC_ZERO_LENGTH_FRAGMENT_STR, 0);
            }
            break;
        }
    }
    
    PREPROC_PROFILE_END(rpcdecodePerfStats);
    return;    
}

/* most significant bit */
#define MSB 0x80000000

/*
 * For proto ref, see rfc1831 section 10 and page 445 UNP vol2
 *  
 * check to make sure we've got enough data to process a record
 *
 * Where did the original 16 come from?  It seems that it could be
 * a last frag of 0 length according to spec.
 *
 * The minimum "valid" packet for us is 8 fields * 4 bytes
 *
 * This decoder is ignorant of TCP state so we'll have to assume
 * that reassembled TCP stuff is reinjected to the preprocessor
 * chain
 *
 * This decoder is also ignorant of multiple RPC requests in a
 * single stream.  To compensate, we can configure alerts
 *
 * Additionally, we don't do anything to verify that this is
 * really an RPC service port so we don't decode anything that
 * happens as a result
 *
 * From rfc1831:
 *
 *  Fragment Header ( 1 flag bit, 31 bit uint )
 *     RPC Body
 *  
 *        unsigned int xid 
 *        struct call_body {
 *             unsigned int rpcvers;  // must be equal to two (2) 
 *             unsigned int prog;
 *             unsigned int vers;
 *             unsigned int proc;
 *             opaque_auth  cred;
 *             opaque_auth  verf;
 *        }
 */

static int ConvertRPC(RpcDecodeConfig *rconfig, Packet *p)
{
    const uint8_t *data = p->data;   /* packet data */
    uint8_t *norm_index;
    uint8_t *data_index;     /* this is the index pointer to walk thru the data */
    uint8_t *data_end;       /* points to the end of the payload for loop control */
    uint16_t psize = p->dsize;     /* payload size */
    int length;          /* length of current fragment */
    int last_fragment = 0; /* have we seen the last fragment sign? */
    int decoded_len; /* our decoded length is always atleast a 0 byte header */
    uint32_t fraghdr;   /* Used to store the RPC fragment header data */
    int fragcount = 0;   /* How many fragment counters have we seen? */
    int ret;
    uint8_t *decode_buf_start = &DecodeBuffer[0];
    uint8_t *decode_buf_end = decode_buf_start + DECODE_BLEN;

    if(psize < 32)
    {
        DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Returning due to"
                                " small packet size: %d\n ", psize););
        return 0;
    }

    /* on match, normalize the data */
    DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Got RPC traffic (%d bytes)!\n", psize););

    /* cheesy alignment safe fraghdr = *(uint32_t *) data*/
    *((uint8_t *)  &fraghdr)      = data[0];
    *(((uint8_t *) &fraghdr) + 1) = data[1];
    *(((uint8_t *) &fraghdr) + 2) = data[2];
    *(((uint8_t *) &fraghdr) + 3) = data[3];
    

    /* The fragment header is 4 bytes in network byte order */
    fraghdr = ntohl(fraghdr);
    length = fraghdr & 0x7FFFFFFF;
    
    /* Check to see if we are on the last fragment */
    if(fraghdr & MSB)
    {
        /* on match, normalize the data */
        DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Found Last Fragment: %u!\n",fraghdr););

        if((length + 4 != psize) && !(p->packet_flags & PKT_REBUILT_STREAM))
        {
            DEBUG_WRAP(DebugMessage(DEBUG_RPC, "It's not the only thing in this buffer!"
                                    " length: %d psize: %d!\n", length, psize););            
            return RPC_MULTIPLE_RECORD;
        }
        else if ( length == 0 )
        {
            DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Zero-length RPC fragment detected."
                                    " length: %d psize: %d.\n", length, psize););            
            return RPC_ZERO_LENGTH_FRAGMENT;
        }
        return 0;
    }
    else if(rconfig->alert_fragments)
    {
        /* Log alert but continue processing */
        SnortEventqAdd(GENERATOR_SPP_RPC_DECODE, RPC_FRAG_TRAFFIC, 
                       1, RPC_CLASS, 3, RPC_FRAG_TRAFFIC_STR, 0);
    }

    norm_index = &DecodeBuffer[0]; 
    data_index = (uint8_t *)data;
    data_end = (uint8_t *)data + psize;

    /* now we know it's in fragmented records, 4 bytes of 
     * header(of which the most sig bit fragment (0=yes 1=no). 
     * The header is followed by the value move pointer up 4 
     * bytes, we need to stuff header in first 4 bytes.  
     * But the header has the total length...we don't know 
     * until the end 
     */
    
    /* This is where decoded data will be written */
    norm_index += 4;
    decoded_len = 4;

    /* always make sure that we have enough data to process atleast
     * the header and that we only process at most, one fragment
     */
    
    while(((data_end - data_index) >= 4) && (last_fragment == 0))
    {
        /* get the fragment length (31 bits) and move the pointer to
           the start of the actual data */

        *((uint8_t *) &fraghdr)       = data_index[0];
        *(((uint8_t *) &fraghdr) + 1) = data_index[1];
        *(((uint8_t *) &fraghdr) + 2) = data_index[2];
        *(((uint8_t *) &fraghdr) + 3) = data_index[3];

        fraghdr = ntohl(fraghdr);
        length = fraghdr & 0x7FFFFFFF;
        
        /* move the current index into the packet past the
           fragment header */
        data_index += 4; 
        
        if(fraghdr & MSB)
        {
            DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Last Fragment detected\n"););
            last_fragment = 1;
        }

        if((length + decoded_len) < decoded_len)
        {
            /* don't allow integer overflow to confuse us.  Should be
             * caught by length > psize but who knows when weird
             * psize's might be allowed */
            
            DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Integer Overflow"
                                    " field(%d) exceeds packet size(%d)\n",
                                    length, psize););
            return RPC_LARGE_FRAGSIZE;
        }

        decoded_len += length;

        if(length > psize)
        {
            DEBUG_WRAP(DebugMessage(DEBUG_RPC, "Length of"
                                    " field(%d) exceeds packet size(%d)\n",
                                    length, psize););
            return RPC_INCOMPLETE_SEGMENT;
        }
        else if(decoded_len > psize)
        {
            /* The entire request is larger than our current packet
             *  size
             */
            DEBUG_WRAP(DebugMessage(DEBUG_RPC, " Decoded Length (%d)"
                                    "exceeds packet size(%d)\n",
                                    decoded_len, psize););
            return RPC_LARGE_FRAGSIZE;
        }
        else if((data_index + length) > data_end)
        {
            DEBUG_WRAP(DebugMessage(DEBUG_RPC,
                                    "returning LARGE_FRAGSIZE"
                                    "since we'd read past our end\n"););
            return RPC_LARGE_FRAGSIZE;
        }
        else
        {
            fragcount++;
            
            DEBUG_WRAP(DebugMessage(DEBUG_RPC,
                                    "length: %d size: %d decoded_len: %d\n",
                                    length, psize, decoded_len););                        

            ret = SafeMemcpy(norm_index, data_index, length, decode_buf_start, decode_buf_end);
            if (ret != SAFEMEM_SUCCESS)
            {
                return 0;
            }

            norm_index += length;
            data_index += length;
        }
    }

    /* rewrite the header on the request packet */
    /* move the fragment header back onto the data */
    
    fraghdr = ntohl(decoded_len); /* size */

    DecodeBuffer[0] = *((uint8_t *) &fraghdr);
    DecodeBuffer[1] = *(((uint8_t *) &fraghdr) + 1);
    DecodeBuffer[2] = *(((uint8_t *) &fraghdr) + 2);
    DecodeBuffer[3] = *(((uint8_t *) &fraghdr) + 3);
    
    DecodeBuffer[0] |=  0x80;             /* Mark as unfragmented */

    /* is there another request encoded that is trying to evade us by doing
     *
     * frag last frag [ more data ]?
     */
    if(decoded_len + ((fragcount - 1) * 4) != psize)
    {
        DEBUG_WRAP(DebugMessage(DEBUG_RPC, "decoded len does not compute: %d\n",
                                decoded_len););
        return RPC_MULTIPLE_RECORD;
    }

    DEBUG_WRAP(DebugMessage(DEBUG_RPC, "New size: %d\n", decoded_len);
               DebugMessage(DEBUG_RPC, "converted data:\n");
               //PrintNetData(stdout, data, decoded_len);
               );

    p->alt_dsize = decoded_len;
    p->packet_flags |= PKT_ALT_DECODE;

    return 0;
}

static void _addPortsToStream5Filter(RpcDecodeConfig *rpc, tSfPolicyId policy_id)
{
    unsigned int portNum;

    if (rpc == NULL)
        return;

    if (stream_api)
    {
        for (portNum = 0; portNum < MAXPORTS; portNum++)
        {
            if(rpc->RpcDecodePorts[(portNum/8)] & (1<<(portNum%8)))
            {
                //Add port the port
                stream_api->set_port_filter_status
                    (IPPROTO_TCP, (uint16_t)portNum, PORT_MONITOR_SESSION, policy_id);
            }
        }
    }
}

#ifdef TARGET_BASED
static void _addServicesToStream5Filter(tSfPolicyId policy_id)
{
    if (stream_api)
        stream_api->set_service_filter_status
            (rpc_decode_app_protocol_id, PORT_MONITOR_SESSION, policy_id);
}
#endif

static int RpcDecodeFreeConfigPolicy(tSfPolicyUserContextId rpc,tSfPolicyId policyId, void* pData )
{
    RpcDecodeConfig *pPolicyConfig = (RpcDecodeConfig *)pData;
    sfPolicyUserDataClear (rpc, policyId);
    free(pPolicyConfig);
    return 0;
}

static void RpcDecodeFreeConfig(tSfPolicyUserContextId rpc)
{

    if (rpc == NULL)
        return;
    sfPolicyUserDataIterate (rpc, RpcDecodeFreeConfigPolicy);
    sfPolicyConfigDelete(rpc);
}

static void RpcDecodeCleanExit(int signal, void *unused)
{
    RpcDecodeFreeConfig(rpc_decode_config);
    rpc_decode_config = NULL;
}

#ifdef SNORT_RELOAD
static void RpcDecodeReload(char *args)
{
    int policy_id = (int)getParserPolicy();
    RpcDecodeConfig *pPolicyConfig = NULL;	

    if (rpc_decode_swap_config == NULL)
    {
        rpc_decode_swap_config = sfPolicyConfigCreate();
    }
    sfPolicyUserPolicySet (rpc_decode_swap_config, policy_id);
    pPolicyConfig = (RpcDecodeConfig *)sfPolicyUserDataGetCurrent(rpc_decode_swap_config);
    if (pPolicyConfig)
    {
       ParseError("RPC decode can only be configured once.\n");
    }

    pPolicyConfig = (RpcDecodeConfig *)SnortAlloc(sizeof(RpcDecodeConfig));
    if (!pPolicyConfig)
    {
        ParseError("RPC Decode preprocessor: memory allocate failed.\n");
    }
    sfPolicyUserDataSetCurrent(rpc_decode_swap_config, pPolicyConfig);

    /* parse the argument list into a list of ports to normalize */
    ParseRpcConfig(pPolicyConfig, args);

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(PreprocRpcDecode, PRIORITY_APPLICATION, PP_RPCDECODE, PROTO_BIT__TCP);

    _addPortsToStream5Filter(pPolicyConfig, policy_id);

#ifdef TARGET_BASED
    _addServicesToStream5Filter(policy_id);
#endif
}

static void * RpcDecodeReloadSwap(void)
{
    tSfPolicyUserContextId old_config = rpc_decode_config;

    if (rpc_decode_swap_config == NULL)
        return NULL;

    rpc_decode_config = rpc_decode_swap_config;
    rpc_decode_swap_config = NULL;
    return (void *)old_config;
}

static void RpcDecodeReloadSwapFree(void *data)
{
    if (data == NULL)
        return;

    RpcDecodeFreeConfig((tSfPolicyUserContextId)data);
}
#endif
