 /*
  *  Copyright (C) 2008 Marián Kyral
  *  Copyright (C) 2008 Ezequiel R. Aguerre
  *  Copyright (C) 2008 Pierpaolo Vittorini
  *  Copyright (C) 2008 Ruan Strydom
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of the GNU General Public License as published by
  * the Free Software Foundation; either version 2 of the License, or
  * (at your option) any later version.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
  */

#include "yawpwserv.h"
#include <QDomDocument>
#include <QRegExp>
#include <QFile>
#include <QDir>
#include <QMetaType>
#include <QResource>
#include <QDataStream>
#include <klocalizedstring.h>

#include <kio/netaccess.h>
#include <kio/job.h>


//-----------------------------------------------------------------------------
YaWPWeatherService::YaWPWeatherService(const QString &name, int dayCount) : m_image(), m_logo() {
    m_name = name;
    m_info = YaWPWeatherInfo(dayCount);
    m_dir = QDir::homePath() + "/.cache/yawp";
}

bool YaWPWeatherService::readIconMapping(const QString & resource) {
    //QTextStream stream(QByteArray((const char*)res.data(), res.size()));
    QFile file(resource);
    file.open(QIODevice::ReadOnly);
    QTextStream stream(&file);
    m_icons.clear();
    QString line;
    QString icon;
    QString value;
    while(!stream.atEnd()) {
        line = stream.readLine();
        if(line.length() < 1)
            continue;
        if(line[0] == '#')
            continue;
        QRegExp regx("\\s*([\\w-_]*)(\\s*[=]+\\s*)([\\w-_]*)\\s*");
        if(regx.indexIn(line) > -1) {
            icon = regx.cap(1);
            value = regx.cap(3);
            if(value.length() > 0)
                m_icons[icon] = value;
            else
                m_icons[icon] = "unknown";
        }
    }
    return true;
}

QString YaWPWeatherService::mapIcon(const QString &icon) const {
    QString value = m_icons[icon];
    if(value.length() < 1 || value.isNull())
        return "unknown";
    else
        return value;
}

/* QUrl and KUrl encode the strings in UTF-8 as the standard establishes,
 * however, accuweather is expecting strings in ISO-8859-1(Latin1). This is
 * a workaround, it percent-encodes the string using ISO-8859-1. This way we
 * can then create a KUrl object using the already encoded string, so KUrl
 * won't encode it again and we get the correct URL :-)
 *
 * TODO/FIXME: We have to analize this function with caution, because it's
 * a _very_ essential piece of code, if something goes wrong with this then
 * yaWP will be able to get NOTHING from the internet. Let's test it, let's
 * fix bugs and please, let's find another solution :-)
 */
QString YaWPWeatherService::toPercentEncoding(const QString &str) {
  QString out;
  int n = str.length ();
  for (int i = 0; i < n; i++) {
    unsigned char c = str[i].toLatin1 ();
    if ( c >= 128 || c == '|' || c == ' ' ) // Outside ASCII or | or space
      out.append (QString("%%1").arg((int)c, 2, 16, QLatin1Char('0')).toAscii());
    else
      out.append((char)c);
  }
  return out;
}

void YaWPWeatherService::update(const QMap<QString,QString> &data) {
  Q_UNUSED(data);

  //Create the config directory
  //We do it in config because we want the data to be permanent
  QDir dir(m_dir);
  if(!dir.exists())
    dir.mkpath(m_dir);

  KUrl xu(toPercentEncoding(m_xmlurl));
  KUrl iu(m_imageurl); // This doesn't have weird accents

  m_xmldata.clear();
  m_imagedata.clear();

  m_code = YaWPWeatherService::UNDEFINED;

  KIO::Job *jobXml = KIO::get(xu, KIO::NoReload, KIO::HideProgressInfo);
  connect(jobXml, SIGNAL(data(KIO::Job*, const QByteArray&)), this, SLOT(slotXmlData(KIO::Job *, const QByteArray &)));
  connect(jobXml, SIGNAL(result(KJob *)), this, SLOT(slotXmlResult(KJob*)));
  jobXml->start();

  KIO::Job *jobImage = KIO::get(iu, KIO::NoReload, KIO::HideProgressInfo);
  connect(jobImage, SIGNAL(data(KIO::Job*, const QByteArray&)), this, SLOT(slotImageData(KIO::Job *, const QByteArray &)));
  connect(jobImage, SIGNAL(result(KJob *)), this, SLOT(slotImageResult(KJob*)));

  jobImage->start();
}

void YaWPWeatherService::slotXmlData(KIO::Job *job, const QByteArray &data) {
  Q_UNUSED(job);
  m_xmldata += data;
}

void YaWPWeatherService::slotXmlResult(KJob *job) {
  QString fileName =  QString("%1/%2").arg(m_dir).arg(qHash(m_xmlurl)) + ".xml";
  if (job->error() == 0) {
    if (parseInfo(m_xmldata, m_info)) {

      QFile f(fileName);
      f.open(QIODevice::WriteOnly);
      QDataStream ds(&f);
      ds << m_xmldata;
      f.close();

      m_code = SUCCESS;
    }
    else
    {
      m_code = DATA_FAILED;
      QFile f(fileName + ".debug");
      f.open(QIODevice::WriteOnly);
      QDataStream ds(&f);
      ds << m_xmldata;
      f.close();
    }
  }
  else {
    QFile f(fileName);
    if (f.exists()) {
      f.open(QIODevice::ReadOnly);
      QDataStream ds(&f);
      ds >> m_xmldata;
      f.close();
      parseInfo(m_xmldata, m_info);
      m_code = CACHE;
    }
    else
      m_code = CONNECTION_FAILED;
  }
  emit completed(this, m_code);
}

void YaWPWeatherService::slotImageData(KIO::Job *job, const QByteArray &data) {
  Q_UNUSED(job);
  m_imagedata += data;
}

void YaWPWeatherService::slotImageResult(KJob *job) {
  QString fileName = QString("%1/%2").arg(m_dir).arg(qHash(m_xmlurl)) + ".jpg";
  if (job->error() == 0) {
    FILE *f = fopen(fileName.toUtf8().data(), "w");
    fwrite(m_imagedata.data(), 1, m_imagedata.size(), f);
    fclose(f);

    m_image.load(fileName);
  }
  else {
    QFile f(fileName);
    if (f.exists()) {
      m_image.load(fileName);
      m_code = CACHE;
    }
  }
}

//-----------------------------------------------------------------------------
YaWPGoogleWeatherService::YaWPGoogleWeatherService() : YaWPWeatherService("Google", 4) {
    if(!readIconMapping(":/googleicons.conf"))
        kDebug() << "Failed reading googleicons" << endl;
}


//-----------------------------------------------------------------------------
YaWPGoogleWeatherService::~YaWPGoogleWeatherService() {}


//-----------------------------------------------------------------------------
void YaWPGoogleWeatherService::update(const QMap<QString,QString> &data) {
    m_country = data["country"];
    m_city = data["city"];

    m_xmlurl = QString("http://www.google.com/ig/api?weather=%1,%2").arg(m_city).arg(m_country);
    m_xmlurl = m_xmlurl.replace(" ", "%20");

    YaWPWeatherService::update(data);
}

//-----------------------------------------------------------------------------
//Called by updateInfo
bool YaWPGoogleWeatherService::parseInfo(const QByteArray &data, YaWPWeatherInfo &info) {

    bool success = false;
    int units = YaWPDay::IMPERIAL;

    QDomDocument doc("weatherdoc");
    doc.setContent(data);

    info.reset();

    QDomNodeList nodeList = doc.elementsByTagName("forecast_information");
    for(int i = 0; i < nodeList.size(); i++) {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "forecast_information") {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    if(element.tagName() == "unit_system") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            if(attribute.value() == "US")
                                units = YaWPDay::IMPERIAL;
                            else
                                units = YaWPDay::METRIC;
                        }
                    }
                }
                node = node.nextSibling();
            }
        }
    }

    nodeList = doc.elementsByTagName("forecast_conditions");
    for(int i = 0; i < nodeList.size() && i < 4; i++) {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "forecast_conditions") {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    info.days()[i].setUnits(units);
                    if(element.tagName() == "low") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull())
                            info.days()[i].setLow(attribute.value().toInt());
                    }
                    if(element.tagName() == "icon") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            QRegExp regx("([/]?\\w*[/]?\\w*/)([\\w_]*)([.]+.*)");
                            if(regx.indexIn(attribute.value()) > -1) {
                                info.days()[i].setEmblem(mapIcon(regx.cap(2)));
                            }
                        }
                    }
                    if(element.tagName() == "high") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull())
                            info.days()[i].setHigh(attribute.value().toInt());
                    }
                    if(element.tagName() == "condition") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            info.days()[i].setDescription(attribute.value());
                        }
                    }
                    if(element.tagName() == "day_of_week") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            info.days()[i].setDate(attribute.value());
                        }
                    }
                }
                node = node.nextSibling();
            }
        }
    }

    nodeList = doc.elementsByTagName("current_conditions");
    for(int i = 0; i < nodeList.size(); i++) {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "current_conditions") {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    //SUCCESS
                    success = true;
                    info.days()[0].setUnits(units);
                    if(element.tagName() == "condition") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            info.days()[0].setDescription(attribute.value());
                        }
                    }
                    if(element.tagName() == "temp_f") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull())
                            info.days()[0].setCurrent(YaWPDay::convertDegrees(YaWPDay::IMPERIAL, units, attribute.value().toInt()));
                    }
                    if(element.tagName() == "temp_c") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull())
                            info.days()[0].setCurrent(YaWPDay::convertDegrees(YaWPDay::METRIC, units, attribute.value().toInt()));
                    }
                    if(element.tagName() == "humidity") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            QRegExp regx(".*([0-9]*).*");
                            if(regx.indexIn(attribute.value()) > -1)
                                info.days()[0].setHumidity(regx.cap(1).toInt());
                        }
                    }
                    if(element.tagName() == "wind_condition") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            QRegExp regx("(.*:\\s+)([A-Z]*)(\\s+\\w+\\s+)([0-9]*)(.*)");
                            if(regx.indexIn(attribute.value()) > -1) {
                                info.days()[0].setWindSpeed(regx.cap(4).toInt());
                                info.days()[0].setWindDirection(regx.cap(2));
                            }
                        }
                    }
                    if(element.tagName() == "icon") {
                        QDomAttr attribute = element.attributeNode("data");
                        if(!attribute.isNull()) {
                            QRegExp regx("([/]?\\w*[/]?\\w*/)([\\w_]*)([.]+.*)");
                            if(regx.indexIn(attribute.value()) > -1) {
                                info.days()[0].setEmblem(mapIcon(regx.cap(2)));
                            }
                        }
                    }
                }
                node = node.nextSibling();
            }
        }
    }
    return success;
}



//-----------------------------------------------------------------------------
YaWPForecastfoxWeatherService::YaWPForecastfoxWeatherService() : YaWPWeatherService("accuweather", 5) {
  m_logo.load(":/aw.png");
  if(!readIconMapping(":/accuweathericons.conf"))
      kDebug() << "Failed reading accuweathericons" << endl;
}


//-----------------------------------------------------------------------------
YaWPForecastfoxWeatherService::~YaWPForecastfoxWeatherService() {}


//-----------------------------------------------------------------------------
void YaWPForecastfoxWeatherService::update(const QMap<QString,QString> &data) {
    m_location = data["location"];

    m_xmlurl = QString("http://ruan.accu-weather.com/widget/ruan/weather-data.asp?location=%1").arg(m_location) ;
    m_xmlurl = m_xmlurl.replace(" ", "%20");

    m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/iswmerc.jpg";

    if (m_location.mid(3,1) != "|")
      m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/EI/iseun.jpg";
    else {
      QString area = m_location.mid(0,3);
      QString country = m_location.mid(4,2);

      if (area == "EUR")
        m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/iseurm.jpg";
      else
        if (area == "SAM") {
          m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/isnsam.jpg";
          if (country == "AR" || country == "CL")
            m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/iscsam.jpg";
        }
        else
          if (area == "NAM") {
            m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/ir/isun.jpg";
            if (country == "CA")
              m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/ir/iscanm.jpg";
            else
              if (country == "MX")
                m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/ismex.jpg";
          }
          else
            if (area == "CAC")
              m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/iscar.jpg";
            else
              if (area == "OCN")
                m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/isaust.jpg";
              else
                if (area == "ASI") {
                  m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/isasia.jpg";
                  if (country == "IN")
                    m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/isindia.jpg";
                }
                else
                  if (area == "MEA")
                    m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/ismide.jpg";
                  else
                    if (area == "AFR") {
                      m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/isafrs.jpg";
                      if (country == "MA" || country == "DZ" || country == "TN" || country == "LY" || country == "EG")
                        m_imageurl = "http://sirocco.accuweather.com/sat_mosaic_400x300_public/IR/isafrn.jpg";
                    }
    }

    YaWPWeatherService::update(data);
}

//-----------------------------------------------------------------------------
//Called by updateInfo
bool YaWPForecastfoxWeatherService::parseInfo(const QByteArray &data, YaWPWeatherInfo &info) {

    bool success = false;
    int temp_units = YaWPDay::IMPERIAL;
    int speed_units = YaWPDay::IMPERIAL;
    QString city, state;

    QDomDocument doc("weatherdoc");
    doc.setContent(data);

    info.reset();

    QDomNodeList nodeList = doc.elementsByTagName("units");
    for(int i = 0; i < nodeList.size(); i++) {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "units") {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    if(element.tagName() == "temp") {
                        if(element.text() == "F")
                            temp_units = YaWPDay::IMPERIAL;
                        else
                            temp_units = YaWPDay::METRIC;
                    }
                    if(element.tagName() == "speed") {
                        if(element.text() == "MPH")
                            speed_units = YaWPDay::IMPERIAL;
                        else
                            speed_units = YaWPDay::METRIC;
                    }
                }
                node = node.nextSibling();
            }
        }
    }

    nodeList = doc.elementsByTagName("local");
    for(int i = 0; i < nodeList.size(); i++) {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "local") {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    if(element.tagName() == "city") {
                        city = element.text();
                    }
                    if(element.tagName() == "state") {
                        state = element.text();
                    }
                    if(element.tagName() == "time") {
                        info.days()[0].setTime(QTime::fromString(element.text(),"hh:mm").toString("h:m ap"));
//                        info.days()[0].setTime(element.text());
                    }
                }
                node = node.nextSibling();
            }
        }
    }

    info.setMessage(i18nc("City and Country/State in the status message",
          "%1 %2", city, i18nc("Country or state", state.toUtf8().constData())));

    nodeList = doc.elementsByTagName("day");
    for(int i = 0; i < nodeList.size() && i < 5; i++)
    {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "day")
        {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    info.days()[i].setUnits(temp_units);
                    if(element.tagName() == "obsdate") {
                      info.days()[i].setDay(QDate::fromString(element.text(), "M/d/yyyy"));
                    }
                    if(element.tagName() == "daycode") {
                      info.days()[i].setDateLong(element.text());
                      info.days()[i].setDate(element.text().mid(0,3));
                    }
                    if(element.tagName() == "sunrise") {
                      info.days()[i].setSunrise(element.text());
                    }
                    if(element.tagName() == "sunset") {
                        info.days()[i].setSunset(element.text());
                    }
                    if(element.tagName() == "observationtime") {
                        info.days()[i].setObservation(element.text());
                    }
                    if(element.tagName() == "daytime") {
                        QDomNodeList daytimeList = node.childNodes();
                        for(int j = 0; j < daytimeList.size(); j++) {
                            QDomNode daytimeNode = daytimeList.at(j);
                            element = daytimeNode.toElement();
                            if(element.tagName() == "txtshort") {
                                info.days()[i].setDescription(element.text());
                                // A temporary solution to collect complete list,
                                // if it is possible
                                kDebug() << "WEATHER_SHORT" << element.text() << endl;
                            }
                            if(element.tagName() == "weathericon") {
                                info.days()[i].setEmblem(mapIcon(element.text()));
                            }
                            if(element.tagName() == "hightemperature") {
                              info.days()[i].setHigh(element.text().toInt());
                            }
                            if(element.tagName() == "realfeellow") {
                              info.days()[i].setRfLow(element.text().toInt());
                            }
                            if(element.tagName() == "realfeelhigh") {
                              info.days()[i].setRfHigh(element.text().toInt());
                            }
                            if(element.tagName() == "lowtemperature") {
                              info.days()[i].setLow(element.text().toInt());
                            }
                            if(element.tagName() == "windspeed") {
//                                info.days()[i].setWindSpeed(element.text().toInt());
                                info.days()[i].setWindSpeed(YaWPDay::convertDistance(speed_units, temp_units, element.text().toInt()));
                            }
                            if(element.tagName() == "winddirection") {
                                info.days()[i].setWindDirection(element.text());
                            }
                        }
                    }
                    if(element.tagName() == "nighttime") {
                        info.days()[i].setHasNightValues(true);
                        QDomNodeList daytimeList = node.childNodes();
                        for(int j = 0; j < daytimeList.size(); j++) {
                            QDomNode daytimeNode = daytimeList.at(j);
                            element = daytimeNode.toElement();
                            if(element.tagName() == "txtshort") {
                                info.days()[i].setNightDescription(element.text());
                                // A temporary solution to collect complete list,
                                // if it is possible
                                kDebug() << "WEATHER_SHORT" << element.text() << endl;
                            }
                            if(element.tagName() == "weathericon") {
                                info.days()[i].setNightEmblem(mapIcon(element.text()));
                            }
                            if(element.tagName() == "hightemperature") {
                                info.days()[i].setNightHigh(element.text().toInt());
                            }
                            if(element.tagName() == "lowtemperature") {
                                info.days()[i].setNightLow(element.text().toInt());
                            }
                            if(element.tagName() == "realfeellow") {
                              info.days()[i].setNightRfLow(element.text().toInt());
                            }
                            if(element.tagName() == "realfeelhigh") {
                              info.days()[i].setNightRfHigh(element.text().toInt());
                            }
                            if(element.tagName() == "windspeed") {
//                                info.days()[i].setNightWindSpeed(element.text().toInt());
                                info.days()[i].setNightWindSpeed(YaWPDay::convertDistance(speed_units, temp_units, element.text().toInt()));
                            }
                            if(element.tagName() == "winddirection") {
                                info.days()[i].setNightWindDirection(element.text());
                            }
                        }
                    }
                }
                node = node.nextSibling();
            }
        }
    }

    nodeList = doc.elementsByTagName("currentconditions");
    for(int i = 0; i < nodeList.size(); i++)
    {
        QDomNode node = nodeList.at(i);
        if(node.nodeName() == "currentconditions")
        {
            node = node.firstChild();
            while(!node.isNull()) {
                QDomElement element = node.toElement();
                if(!element.isNull()) {
                    //SUCCESS
                    success = true;
                    info.days()[0].setUnits(temp_units);
                    if(element.tagName() == "temperature") {
                        info.days()[0].setCurrent(element.text().toInt());
                    }
                    if(element.tagName() == "humidity") {
                        info.days()[0].setHumidity(element.text().replace("%","").toInt());
                    }
                    if(element.tagName() == "windspeed") {
//                        info.days()[0].setWindSpeedCurr(element.text().toInt());
                        info.days()[0].setWindSpeedCurr(YaWPDay::convertDistance(speed_units, temp_units, element.text().toInt()));
                    }
                    if(element.tagName() == "winddirection") {
                        info.days()[0].setWindDirectionCurr(element.text());
                    }
                    if(element.tagName() == "weathericon") {
                        info.days()[0].setEmblemCurr(mapIcon(element.text()));
                    }
                }
                node = node.nextSibling();
            }
        }
    }
    return success;
}


#include "moc_yawpwserv.cpp"
