/*
   Copyright (C) 2008-2011 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>.
*/

#include "tabproject.h"

#include "case.h"
#include "effectwidget.h"
#include "i18n.h"
#include "project.h"
#include "textstylewidget.h"

#include <QtGui/QComboBox>
#include <QtGui/QFontMetrics>
#include <QtGui/QSpinBox>
#include <QtGui/QTextEdit>
#include <QtGui/QVBoxLayout>

#include <assert.h>


TabProject::TabProject(KoverArtist::Project* aProject, QWidget* aParent)
:Inherited(aProject, aParent)
{
   createCaption(i18n("Number of Discs"));
   mSpbDiscs = new QSpinBox(this);
   mSpbDiscs->setRange(1, 100);
   mSpbDiscs->setSingleStep(1);
   mSpbDiscs->setFixedHeight(mSpbDiscs->sizeHint().height());
   connect(mSpbDiscs, SIGNAL(valueChanged(int)), SLOT(discsChanged()));
   mMainBox->addWidget(mSpbDiscs);

   createCaption(i18n("Case"));
   mCboCase = new QComboBox(this);
   connect(mCboCase, SIGNAL(currentIndexChanged(int)), SLOT(caseSelected(int)));
   mMainBox->addWidget(mCboCase);
   setupCasesComboBox();
   mCboCase->setMinimumWidth(mCboCase->sizeHint().width());

   //
   //  Title text and style options
   //
   createCaption(i18n("Title"));
   mTswTitle = new TextStyleWidget(&mProject->titleStyle(),
                                   Qt::Horizontal|Qt::Vertical, this);
   connect(mTswTitle, SIGNAL(changed()), SLOT(titleChanged()));
   mMainBox->addWidget(mTswTitle);

   QFontMetrics fontMetrics(font());
   int edtHeight = int(fontMetrics.lineSpacing()*2.5);

   mEdtTitle = new QTextEdit(this);
   mEdtTitle->setAcceptRichText(false);
   mEdtTitle->setWordWrapMode(QTextOption::NoWrap);
   mEdtTitle->setFixedHeight(edtHeight + mEdtTitle->height() - mEdtTitle->contentsRect().height());
   mEdtTitle->setMinimumWidth(mEdtTitle->sizeHint().width());
   connect(mEdtTitle, SIGNAL(textChanged()), SLOT(titleChanged()));
   mMainBox->addWidget(mEdtTitle);

   //
   //  Sub-title text and style options
   //
   createCaption(i18n("Sub Title"));
   mTswSubTitle = new TextStyleWidget(&mProject->subTitleStyle(),
                                      Qt::Horizontal, this);
   connect(mTswSubTitle, SIGNAL(changed()), SLOT(subTitleChanged()));
   mMainBox->addWidget(mTswSubTitle);

   mEdtSubTitle = new QTextEdit(this);
   mEdtSubTitle->setAcceptRichText(false);
   mEdtSubTitle->setWordWrapMode(QTextOption::NoWrap);
   mEdtSubTitle->setFixedHeight(edtHeight + mEdtSubTitle->height() - mEdtSubTitle->contentsRect().height());
   mEdtSubTitle->setMinimumWidth(mEdtSubTitle->sizeHint().width());
   connect(mEdtSubTitle, SIGNAL(textChanged()), SLOT(subTitleChanged()));
   mMainBox->addWidget(mEdtSubTitle);

   //
   //  Title and sub-title text effects
   //
   createCaption(i18n("Title and Sub Title Effect"));

   mEffTitle = new EffectWidget(&mProject->titleEffect(), this);
   connect(mEffTitle, SIGNAL(changed()), SLOT(titleChanged()));
   mMainBox->addWidget(mEffTitle);

   setMinimumSize(sizeHint());
   mMainBox->addStretch(100);
}


TabProject::~TabProject()
{
}


void TabProject::updateContents()
{
   mSpbDiscs->setValue(mProject->count());

   int idx = mCboCase->findData(mProject->box().name());
   if (mCboCase->currentIndex() != idx)
      mCboCase->setCurrentIndex(idx);

   mTswTitle->updateContents();
   mEdtTitle->setText(mProject->title());

   mTswSubTitle->updateContents();
   mEdtSubTitle->setText(mProject->subTitle());

   mEffTitle->updateContents();
}


void TabProject::setupCasesComboBox()
{
   const QVector<KoverArtist::Case>& cases = KoverArtist::Case::cases();
   const int num = cases.size();

   mCboCase->clear();

   for (int i=0; i<num; ++i)
      mCboCase->addItem(cases[i].label(), cases[i].name());
}


void TabProject::discsChanged()
{
   const int num = mSpbDiscs->value();
   if (num==mProject->count()) return;
   mProject->resize(num);
   if (updatesEnabled()) emit changed();
}


void TabProject::caseSelected(int aIdx)
{
   const KoverArtist::Case* box;
   
   if (aIdx >= 0) box = KoverArtist::Case::findCase(mCboCase->itemData(aIdx).toString());
   else box = &KoverArtist::Case::standardCase();
   assert(box!=0);

   mProject->setBox(*box);
   if (updatesEnabled()) emit changed();
}


void TabProject::titleChanged()
{
   mProject->setTitle(mEdtTitle->toPlainText());
   if (updatesEnabled()) emit changed();
}


void TabProject::subTitleChanged()
{
   mProject->setSubTitle(mEdtSubTitle->toPlainText());
   if (updatesEnabled()) emit changed();
}


void TabProject::setFocusTitle()
{
   mEdtTitle->setFocus();
}
