;;;
;;; gauche.cgen.cise - C in S expression
;;;  
;;;   Copyright (c) 2004-2008  Shiro Kawai  <shiro@acm.org>
;;;   
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;   
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;  
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;  
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;  
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;  
;;;  $Id: cise.scm,v 1.9 2008-05-10 13:35:57 shirok Exp $
;;;

(define-module gauche.cgen.cise
  (use srfi-1)
  (use gauche.sequence)
  (use gauche.parameter)
  (use gauche.cgen.unit)
  (use gauche.cgen.literal)
  (use gauche.experimental.ref)
  (use util.match)
  (use util.list)
  (export cise-render cise-render-rec
          cise-context cise-context-copy cise-register-macro! cise-lookup-macro
          cise-emit-source-line
          define-cise-macro
          define-cise-stmt
          define-cise-expr
          )
  )
(select-module gauche.cgen.cise)

;;=============================================================
;; Parameters
;;

;; If true, include #line directive in the output.
(define cise-emit-source-line (make-parameter #t))

;; Keeps the cise macro bindings.
(define cise-context (make-parameter (make-hash-table 'eq?)))

;;=============================================================
;; Environment
;;

;; Environment must be treated opaque from outside of CISE module.

(define-class <cise-env> ()
  ((context :init-keyword :context :init-value 'stmt) ; stmt or expr
   (decls   :init-keyword :decls   :init-value '())   ; list of extra decls
   ))

(define (make-env context decls)
  (make <cise-env> :context context :decls decls))
(define (env-ctx env)   [~ env'context])
(define (env-decls env) [~ env'decls])
(define (expr-ctx? env) (eq? (env-ctx env) 'expr))
(define (stmt-ctx? env) (eq? (env-ctx env) 'stmt))

(define (null-env)      (make-env 'stmt '()))

(define (expr-env env)
  (if (expr-ctx? env) env (make-env 'expr (env-decls env))))
(define (stmt-env env)
  (if (stmt-ctx? env) env (make-env 'stmt (env-decls env))))

(define (ensure-stmt-ctx form env)
  (unless (stmt-ctx? env)
    (error "cise: statment appears in an expression context:" form)))

(define (env-decl-add! env decl)
  (push! [~ env'decls] decl))

(define (wrap-expr form env)
  (if (stmt-ctx? env) `(,form ";") form))

(define (render-env-decls env)
  (map (match-lambda
         ((var type) `(,(cise-render-type type)" ",var";")))
       (env-decls env)))

;; Check source-info attribute of the input S-expr, and returns Stree
;; of "#line" line if necessary.
(define (source-info form env)
  (if (not (cise-emit-source-line))
    '()
    (match (debug-source-info form)
      [((? string? file) line)
       `((source-info ,file ,line))]
      [_ '()])))
   
;;=============================================================
;; Expander
;;
;;  Cgen expander knows little about C.  It handles literals
;;  (strings, numbers, booleans, and characters) and function calls.
;;  All other stuff is handled by "cise macros"

;;
;; cise-register-macro! NAME EXPANDER &optional CONTEXT
;;
;;   Register cise macro expander EXPANDER with the name NAME.
;;   EXPANDER takes twi arguments, the form to expand and a
;;   opaque cise environmen.
;;
(define (cise-register-macro! name expander . opts)
  (let-optionals* opts ((context (cise-context)))
    (hash-table-put! context name expander)))

;;
;; cise-lookup-macro NAME &optional CONTEXT
;;
;;   Lookup cise macro.
;;
(define (cise-lookup-macro name . opts)
  (let-optionals* opts ((context (cise-context)))
    (hash-table-get context name #f)))

;;
;; copy the current cise context
;;
(define (cise-context-copy . opts)
  (let-optionals* opts ((context (cise-context)))
    (hash-table-copy context)))

;;
;; define-cise-macro (OP FORM ENV) . BODY
;;
;;   Default syntax to add new cise macro to the current context.
;;
(define-syntax define-cise-macro
  (syntax-rules ()
    [(_ (op form env) . body)
     (cise-register-macro! 'op (lambda (form env) . body))]))

;;
;; define-cise-stmt OP [ENV] CLAUSE ... [:where DEFINITION ...]
;; define-cise-expr OP [ENV] CLAUSE ... [:where DEFINITION ...]
;;
(define-syntax define-cise-stmt
  (syntax-rules ()
    ;; recursion
    [(_ "clauses" op env clauses (:where defs ...))
     (define-cise-macro (op form env)
       defs ...
       (ensure-stmt-ctx form env)
       (match form . clauses))]
    [(_ "clauses" op env clauses ())
     (define-cise-stmt "clauses" op env clauses (:where))]
    [(_ "clauses" op env (clause ...) (x . y))
     (define-cise-stmt "clauses" op env (clause ... x) y)]
    ;; entry
    [(_ op (pat . body) .  clauses) ; (pat . body) rules out a single symbol
     (define-cise-stmt "clauses" op env ((pat . body)) clauses)]
    [(_ op env . clauses)
     (define-cise-stmt "clauses" op env () clauses)]))

(define-syntax define-cise-expr
  (syntax-rules ()
    ;; recursion
    [(_ "clauses" op env clauses (:where defs ...))
     (define-cise-macro (op form env)
       defs ...
       (let1 expanded (match form . clauses)
         (if (and (pair? expanded) (symbol? (car expanded)))
           (render-rec expanded env)
           (wrap-expr expanded env))))]
    [(_ "clauses" op env clauses ())
     (define-cise-expr "clauses" op env clauses (:where))]
    [(_ "clauses" op env (clause ...) (x . y))
     (define-cise-expr "clauses" op env (clause ... x) y)]
    ;; entry
    [(_ op (pat . body) .  clauses)
     (define-cise-expr "clauses" op env ((pat . body)) clauses)]
    [(_ op env . clauses)
     (define-cise-expr "clauses" op env () clauses)]))

;;
;; cise-render cise &optional port as-expr?
;;
;;   External entry of renderer
;;
(define (cise-render form . opts)
  (let-optionals* opts ((port (current-output-port))
                        (expr #f))
    (define current-file #f)
    (define current-line 1)
    (define (render-finish stree)
      (match stree
        [('source-info (? string? file) line)
         (cond ((and (equal? file current-file) (eqv? line current-line)))
               ((and (equal? file current-file) (eqv? line (+ 1 current-line)))
                (inc! current-line)
                (format port "\n"))
               (else
                (set! current-file file)
                (set! current-line line)
                (format port "\n#line ~a ~s\n" line file)))]
        [(x . y) (render-finish x) (render-finish y)]
        [(? (any-pred string? symbol? number?) x) (display x port)]
        [_ #f]))
    
    (let* ((env ((if expr expr-env identity) (null-env)))
           (stree (render-rec form env)))
      (render-finish `(,@(render-env-decls env) ,stree)))))

;;
;; cise-render-rec cise stmt/expr env
;;
;;   External interface to call back to cise expander recursively.
;;   stmt/expr should be either a symbol stmt or expr.
;;   env must be treated as opaque object.
;;
(define (cise-render-rec form stmt/expr env)
  (case stmt/expr
    [(stmt) (render-rec form (stmt-env env))]
    [(expr) (render-rec form (expr-env env))]
    [else (error "cise-render-rec: second argument must be either \
                  stmt or expr, but got:" stmt/expr)]))

;; render-rec :: Cise, Env -> Stree
;;   Recursively expands 
(define (render-rec form env)
  (match form
    [((? symbol? key) . args)
     (cond ((cise-lookup-macro key)
            => (lambda (expander)
                 `(,@(source-info form env)
                   ,@(render-rec (expander form env) env))))
           (else
            (let1 eenv (expr-env env)
              (wrap-expr
               `(,@(source-info form env)
                 ,(cise-render-identifier key) "("
                 ,@(intersperse "," (map (cut render-rec <> eenv) args))
                 ")")
               env))))]
    [(x . y)     form]   ; already stree
    [(? symbol?) (wrap-expr (cise-render-identifier form) env)]
    [(? identifier?) (wrap-expr (cise-render-identifier (unwrap-syntax form))
                                env)]
    [(? string?) (wrap-expr (write-to-string form) env)]
    [(? real?)   (wrap-expr form env)]
    [()          '()]
    [#\'         (wrap-expr "'\\''"  env)]
    [#\\         (wrap-expr "'\\\\'" env)]
    [#\newline   (wrap-expr "'\\n'"  env)]
    [#\return    (wrap-expr "'\\r'"  env)]
    [#\tab       (wrap-expr "'\\t'"  env)]
    [(? char?)   (wrap-expr `("'" ,(if (char-set-contains? #[[:alnum:]] form)
                                     (string form)
                                     (format "\\x~2'0x" (char->integer form)))
                              "'") env)]
    [_           (error "Invalid CISE form: " form)]))

;;=============================================================
;; Built-in macros
;;

;;------------------------------------------------------------
;; C function definition
;;
(define-cise-macro (define-cfn form env)
  (define (argchk args)
    (match args
      [() '()]
      [((var ':: type) . rest) `((,var . ,type) ,@(argchk rest))]
      [(var . rest) `((,var . ScmObj) ,@(argchk rest))]))
  ;; NB: we need to confine temporary decls within the function body,
  ;; hence the ugly nested cise-render, since the extra decls handling
  ;; is done at that level.  Hopefully this is an exception.
  (define (gen-cfn cls name args rettype body)
    `(,(cise-render-identifier cls) " "
      ,(cise-render-type rettype)
      " " ,(cise-render-identifier name) "("
      ,@(intersperse "," (map (lambda (a)
                                `(,(cise-render-type (cdr a)) " "
                                  ,(cise-render-identifier (car a))))
                              args))
      ")" "{"
      ,(call-with-output-string (cut cise-render `(begin ,@body) <>))
      "}"))
  ;; NB: this only works at toplevel.  The stmt check doesn't exclude
  ;; non-toplevel use, and will give an error at C compilation time.
  ;; Eventually we need to check better one.
  (ensure-stmt-ctx form env)
  (match form
    [(_ name (args ...) ':: ret-type ':static . body)
     (gen-cfn "static" name (argchk args) ret-type body)]
    [(_ name (args ...) ':: ret-type . body)
     (gen-cfn "" name (argchk args) ret-type body)]
    [(_ name (args ...) ':static . body)
     (gen-cfn "static" name (argchk args) 'ScmObj body)]
    [(_ name (args ...) . body)
     (gen-cfn "" name (argchk args) 'ScmObj body)]))

;;------------------------------------------------------------
;; Syntax
;;

;; [cise stmt]  begin STMT ...
;;    Grouping.
(define-cise-macro (begin form env)
  (ensure-stmt-ctx form env)
  (match form
    [(_ . forms)
     `("{" ,@(map (cut render-rec <> env) forms) "}")]))

;; [cise stmt]  let* ((VAR [:: TYPE] [INIT-EXPR]) ...) STMT ...
;;    Local variables.   Because of C semantics, we only support
;;    let*-style scoping.
;;    :: TYPE can be omitted if the type of VAR is ScmObj.
(define-cise-macro (let* form env)
  (ensure-stmt-ctx form env)
  (match form
    [(_ ((var . spec) ...) . body)
     (let1 eenv (expr-env env)
       `(begin ,@(map (lambda (var spec)
                        (receive (type has-init? init)
                            (match spec
                              [()         (values 'ScmObj #f #f)]
                              [(init)     (values 'ScmObj #t init)]
                              [(':: type) (values type #f #f)]
                              [(':: type init) (values type #t init)])
                          `(,(cise-render-type type)" "
                            ,(cise-render-identifier var)
                            ,@(cond-list (has-init?
                                          `("=",(render-rec init eenv))))
                            ";")))
                      var spec)
               ,@(map (cut render-rec <> env) body)))]
    ))

;; [cise stmt] if TEST-EXPR THEN-STMT [ELSE-STMT]
;;    Conditional.
(define-cise-macro (if form env)
  (ensure-stmt-ctx form env)
  (let1 eenv (expr-env env)
    (match form
      [(_ test then)
       `("if (",(render-rec test eenv)")"
         ,(render-rec then env))]
      [(_ test then else)
       `("if (",(render-rec test eenv)")"
         ,(render-rec then env)" else " ,(render-rec else env))]
      )))

;; [cise stmt] when TEST-EXPR STMT ...
;; [cise stmt] unless TEST-EXPR STMT ...
(define-cise-stmt when
  [(_ test . forms) `(if ,test (begin ,@forms))])

(define-cise-stmt unless
  [(_ test . forms) `(if (not ,test) (begin ,@forms))])

;; [cise stmt] cond (TEST STMT ...) ... [ (else STMT ...) ]
;;   Nested if.
(define-cise-macro (cond form env)
  (ensure-stmt-ctx form env)
  (let1 eenv (expr-env env)
    (define (a-clause test rest)
      `("(" ,(render-rec test eenv) ")" ,(render-rec `(begin ,@rest) env)))
    (match form
      [(_ (test . rest) ...)
       (fold-right (lambda (test rest r)
                     (cond
                      [(and (null? r) (eq? test 'else))
                       `(" else ",(render-rec `(begin ,@rest) env))]
                      [(eq? test (caadr form)) ; first form
                       `("if ",(a-clause test rest) ,@r)]
                      [else
                       `("else if" ,(a-clause test rest) ,@r)]))
                   '() test rest)]
      )))

;; [cise stmt] case EXPR ((VAL ...) STMT ...) ... [ (else STMT ...) ]
;; [cise stmt] case/fallthrough EXPR ((VAL ...) STMT ...) ... [ (else STMT ...) ]
;;    Expands to switch-case statement.   The 'case' form does not
;;    fallthrough, while 'case/fallthrough' does.
(define (case-generator form env fallthrough?)
  (let1 eenv (expr-env env)
    (match form
      [(_ expr (literalss . clauses) ...)
       `("switch (",(render-rec expr eenv)") {"
         ,@(map (lambda (literals clause)
                  `(,@(source-info literals env)
                    ,@(if (eq? literals 'else)
                        '("default: ")
                        (map (lambda (literal) `("case ",literal" : "))
                             literals))
                    ,@(render-rec `(begin ,@clause
                                          ,@(if fallthrough? '() '((break))))
                                  env)))
                literalss clauses)
         "}")]
      )))    

(define-cise-macro (case form env)
  (ensure-stmt-ctx form env)
  (case-generator form env #f))

(define-cise-macro (case/fallthrough form env)
  (ensure-stmt-ctx form env)
  (case-generator form env #t))

;; [cise stmt] for (START-EXPR TEST-EXPR UPDATE-EXPR) STMT ...
;; [cise stmt] for () STMT ...
;;   Loop.
(define-cise-macro (for form env)
  (ensure-stmt-ctx form env)
  (let1 eenv (expr-env env)
    (match form
      [(_ (start test update) . body)
       `("for (",(render-rec start eenv)"; "
         ,(render-rec test eenv)"; "
         ,(render-rec update eenv)")"
         ,(render-rec `(begin ,@body) env))]
      [(_ () . body)
       `("for (;;)" ,(render-rec `(begin ,@body) env))]
      )))

;; [cise stmt] loop STMT ...
;;   Alias of (for () STMT ...)
(define-cise-stmt loop
  [form `(for () ,@(cdr form))])

;; [cise stmt] while TEST-EXPR STMT ...
;;   Loop.
(define-cise-macro (while form env)
  (ensure-stmt-ctx form env)
  (let1 eenv (expr-env env)
    (match form
      [(_ test . body)
       `("while"
         "(",(render-rec test eenv)")"
         ,(render-rec `(begin ,@body) env))])))

;; [cise stmt] for-each (lambda (VAR) STMT ...) EXPR
;;   EXPR must yield a list.  Traverse the list, binding each element
;;   to VAR and executing STMT ....
;;   The lambda form is a fake; you don't really create a closure.
(define-cise-macro (for-each form env)
  (ensure-stmt-ctx form env)
  (let ((eenv (expr-env env))
        (tmp  (gensym "cise__")))
    (match form
      [(_ ('lambda (var) . body) list-expr)
       (env-decl-add! env `(,tmp ScmObj))
       `("SCM_FOR_EACH(" ,(cise-render-identifier tmp) ","
         ,(render-rec list-expr eenv) ") {"
         ,(render-rec `(let* ((,var :: ScmObj (SCM_CAR ,tmp)))
                         ,@body) env)
         "}")])))

;; [cise stmt] pair-for-each (lambda (VAR) STMT ...) EXPR
;;   Like for-each, but VAR is bound to each 'spine' cell instead of
;;   each element of the list.
(define-cise-macro (pair-for-each form env)
  (ensure-stmt-ctx form env)
  (let ((eenv (expr-env env)))
    (match form
      [(_ ('lambda (var) . body) list-expr)
       (env-decl-add! env `(,var ScmObj))
       `("SCM_FOR_EACH(" ,(cise-render-identifier var) ","
         ,(render-rec list-expr eenv) ")"
         ,(render-rec `(begin ,@body) env)
         )])))

;; [cise stmt] dotimes (VAR EXPR) STMT ...
;;   EXPR must yield an integer, N.  Repeat STMT ... by binding VAR from 0
;;   to (N-1).
(define-cise-macro (dotimes form env)
  (ensure-stmt-ctx form env)
  (let ((eenv (expr-env env))
        (n    (gensym "cise__")))
    (match form
      [(_ (var expr) . body)
       `(let* ((,var :: int 0) (,n :: int ,expr))
          (for [() (< ,var ,n) (post++ ,var)] ,@body))])))

;; [cise stmt] return EXPR
;;   Return statement.
(define-cise-macro (return form env)
  (ensure-stmt-ctx form env)
  (match form
    [(_ expr) `("return (" ,(render-rec expr (expr-env env)) ");")]))

;; [cise stmt] break
;; [cise stmt] continue
;;   Break and continue.
(define-cise-stmt break
  [(_) '("break;")])

(define-cise-stmt continue
  [(_) '("continue;")])

;; [cise stmt] label NAME
;; [cise stmt] goto NAME
;;   Label and goto.
(define-cise-stmt label
  [(_ name) `(,(cise-render-identifier name) " : ")])

(define-cise-stmt goto
  [(_ name) `("goto " ,(cise-render-identifier name) ";")])

;;------------------------------------------------------------
;; Operators
;;

;; [cise expr] + EXPR ...
;; [cise expr] - EXPR ...
;; [cise expr] * EXPR ...
;; [cise expr] / EXPR ...
;; [cise expr] % EXPR EXPR
;;   Same as C.
;;
;; [cise expr] and EXPR ...
;; [cise expr] or  EXPR ...
;; [cise expr] not EXPR
;;
;;   Boolean ops.  C's &&, ||, and !.
;;
;; [cise expr] logand EXPR EXPR
;; [cise expr] logior EXPR EXPR
;; [cise expr] logxor EXPR EXPR
;; [cise expr] lognot EXPR EXPR
;;
;;   Bitwise ops.
;;
;; [cise expr] * EXPR
;; [cise expr] & EXPR
;;
;;   Address ops.
;;
;; [cise expr] pre++ EXPR
;; [cise expr] post++ EXPR
;; [cise expr] pre-- EXPR
;; [cise expr] post-- EXPR
;;
;;   pre/post increment/decrement.
;;
;; [cise expr] <  EXPR EXPR
;; [cise expr] <= EXPR EXPR
;; [cise expr] >  EXPR EXPR
;; [cise expr] >= EXPR EXPR
;; [cise expr] == EXPR EXPR
;; [cise expr] != EXPR EXPR
;;
;;   comparison.
;;
;; [cise expr] << EXPR EXPR
;; [cise expr] >> EXPR EXPR
;;
;;   shift.
;;
;; [cise expr] set! LVALUE EXPR LVALUE EXPR ...
;; [cise expr] +=   LVALUE EXPR
;; [cise expr] -=   LVALUE EXPR
;; [cise expr] *=   LVALUE EXPR
;; [cise expr] /=   LVALUE EXPR
;; [cise expr] %=   LVALUE EXPR
;; [cise expr] <<=  LVALUE EXPR
;; [cise expr] >>=  LVALUE EXPR
;; [cise expr] logand= LVALUE EXPR
;; [cise expr] logior= LVALUE EXPR
;; [cise expr] logxor= LVALUE EXPR
;;
;;   assignment.
;;
;; [cise expr] ->  EXPR EXPR ...
;; [cise expr] ref EXPR EXPR ...
;;
;;   reference.  (ref is C's '.')
;;
;; [cise expr] aref EXPR EXPR ...
;;
;;   array reference.
;;
;; [cise expr] cast TYPE EXPR
;;
;;   cast.
;;
;; [cise expr] ?: TEST-EXPR THEN-EXPR ELSE-EXPR
;;
;;   conditional.

(define-macro (define-nary op sop)
  `(define-cise-macro (,op form env)
     (let1 eenv (expr-env env)
       (wrap-expr
        (match form
          [(_ a)
           (list ,sop "("(render-rec a eenv)")")]
          [(_ a b)
           (list "("(render-rec a eenv)")",sop"("(render-rec b eenv)")")]
          [(_ a b . x)
           (list* ',op (list ',op a b) x)])
        env))))
       
(define-nary + "+")
(define-nary - "-")
(define-nary * "*")
(define-nary / "/")

(define-nary and "&&")
(define-nary or  "||")

(define-macro (define-unary op sop)
  `(define-cise-macro (,op form env)
     (wrap-expr
      (match form
        [(_ a)   (list ,sop "("(render-rec a (expr-env env))")")])
      env)))

(define-unary not    "!")
(define-unary lognot "~")
(define-unary &      "&")               ; only unary op

(define-unary pre++  "++")
(define-unary pre--  "--")

(define-macro (define-post-unary op sop)
  `(define-cise-macro (,op form env)
     (wrap-expr
      (match form
        [(_ a)   (list "("(render-rec a (expr-env env))")" ,sop)])
      env)))

(define-post-unary post++ "++")
(define-post-unary post-- "--")

(define-macro (define-binary op sop)
  `(define-cise-macro (,op form env)
     (wrap-expr
      (match form
        [(_ a b)
         (list "("(render-rec a (expr-env env))")",sop
               "("(render-rec b (expr-env env))")")])
      env)))

(define-binary %       "%")
(define-binary logior  "|")
(define-binary logxor  "^")
(define-binary logand  "&")
(define-binary <       "<")
(define-binary <=      "<=")
(define-binary >       ">")
(define-binary >=      ">=")
(define-binary ==      "==")
(define-binary !=      "!=")
(define-binary <<      "<<")
(define-binary >>      ">>")

(define-binary +=      "+=")
(define-binary -=      "-=")
(define-binary *=      "*=")
(define-binary /=      "/=")
(define-binary %=      "%=")
(define-binary <<=     "<<=")
(define-binary >>=     ">>=")

(define-binary logior= "|=")
(define-binary logxor= "^=")
(define-binary logand= "&=")

(define-macro (define-referencer op sop)
  `(define-cise-macro (,op form env)
     (let1 eenv (expr-env env)
       (wrap-expr
        (match form
          [(_ a b ...)
           (list "("(render-rec a eenv)")",sop
                 (intersperse ,sop (map (cut render-rec <> eenv) b)))])
        env))))

(define-referencer ->  "->")
(define-referencer ref ".")

(define-cise-macro (aref form env)
  (let1 eenv (expr-env env)
    (wrap-expr
     (match form
       [(_ a b ...)
        `("(",(render-rec a eenv)")"
          ,(append-map (lambda (ind) `("[",(render-rec ind eenv)"]")) b))])
     env)))

(define-cise-macro (cast form env)
  (let1 eenv (expr-env env)
    (wrap-expr
     (match form
       [(_ type expr)
        `("((",(cise-render-type type)")(",(render-rec expr eenv)"))")])
     env)))

(define-cise-macro (?: form env)
  (let1 eenv (expr-env env)
    (wrap-expr
     (match form
       [(?: test then else)
        (list "(("(render-rec test eenv)")?("
              (render-rec then eenv)"):("
              (render-rec else eenv)"))")])
     env)))

(define-cise-macro (set! form env)
  (let1 eenv (expr-env env)
    (let loop ((args (cdr form)) (r '()))
      (match args
        [()  (wrap-expr (intersperse "," (reverse r)) env)]
        [(var val . more)
         (loop (cddr args)
               `((,(render-rec var eenv)
                  "=(",(render-rec val eenv)")") ,@r))]
        [_   (error "uneven args for set!:" form)]))))

;;------------------------------------------------------------
;; Convenience expression macros
;;

;; Embed raw c code.  NB: I'm not sure yet about the name.  It is
;; desirable to be consistent with genstub (currently it uses (c <expr>))
;; I'll think it a bit more.
(define-cise-expr C:
  [(_ stuff) (list (x->string stuff))])

(define-cise-expr result
  [(_ e) `(set! SCM_RESULT ,e)]
  [(_ e0 e1) `(set! SCM_RESULT0 ,e0 SCM_RESULT1 ,e1)]
  [(_ e0 e1 e2) `(set! SCM_RESULT0 ,e0 SCM_RESULT1 ,e1 SCM_RESULT2 ,e2)]
  )

(define-cise-expr list
  [(_)           '("SCM_NIL")]
  [(_ a)         `(SCM_LIST1 ,a)]
  [(_ a b)       `(SCM_LIST2 ,a ,b)]
  [(_ a b c)     `(SCM_LIST3 ,a ,b ,c)]
  [(_ a b c d)   `(SCM_LIST4 ,a ,b ,c ,d)]
  [(_ a b c d e) `(SCM_LIST5 ,a ,b ,c ,d ,e)]
  [(_ x ...)     (fold (lambda (elt r) `(Scm_Cons ,elt ,r)) '() x)])

(define-cise-expr values
  [(_)           '("Scm_Values(SCM_NIL)")]
  [(_ a)         a]
  [(_ a b)       `(Scm_Values2 ,a ,b)]
  [(_ a b c)     `(Scm_Values3 ,a ,b ,c)]
  [(_ a b c d)   `(Scm_Values4 ,a ,b ,c ,d)]
  [(_ a b c d e) `(Scm_Values5 ,a ,b ,c ,d ,e)]
  [(_ x ...)   `(Scm_Values ,(fold (lambda (elt r) `(Scm_cons ,elt ,r)) '() x))]
  )
;; Using quote is a convenient way to embed Scheme constant in C code.
(define-cise-expr quote
  [(_ cst)
   (unless (cgen-current-unit)
     (error "cise: quote can't be used unless cgen-current-unit is set: '"
            cst))
   (list (cgen-cexpr (cgen-literal cst)))])

;;=============================================================
;; Other utilities
;;

(define (cise-render-type typespec)  ; for the time being
  (if (list? typespec)
    (intersperse " " (map x->string typespec))
    (x->string typespec)))

(define (cise-render-identifier sym)
  (cgen-safe-name-friendly (x->string sym)))

(provide "gauche/cgen/cise")

