/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * DefaultTypeRegistry.java
 * ------------
 */
package org.pentaho.reporting.libraries.formula.typing;

import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Clob;
import java.sql.SQLException;
import java.sql.Time;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import org.pentaho.reporting.libraries.formula.EvaluationException;
import org.pentaho.reporting.libraries.formula.FormulaContext;
import org.pentaho.reporting.libraries.formula.LocalizationContext;
import org.pentaho.reporting.libraries.formula.lvalues.LValue;
import org.pentaho.reporting.libraries.formula.lvalues.TypeValuePair;
import org.pentaho.reporting.libraries.formula.lvalues.StaticValue;
import org.pentaho.reporting.libraries.formula.typing.coretypes.AnyType;
import org.pentaho.reporting.libraries.formula.typing.coretypes.DateTimeType;
import org.pentaho.reporting.libraries.formula.typing.coretypes.LogicalType;
import org.pentaho.reporting.libraries.formula.typing.coretypes.NumberType;
import org.pentaho.reporting.libraries.formula.typing.coretypes.TextType;
import org.pentaho.reporting.libraries.formula.typing.sequence.NumberSequence;
import org.pentaho.reporting.libraries.formula.typing.sequence.AnySequence;
import org.pentaho.reporting.libraries.formula.typing.sequence.DefaultNumberSequence;
import org.pentaho.reporting.libraries.formula.typing.sequence.AnyNumberSequence;
import org.pentaho.reporting.libraries.formula.util.DateUtil;
import org.pentaho.reporting.libraries.formula.util.HSSFDateUtil;
import org.pentaho.reporting.libraries.formula.util.NumberUtil;
import org.pentaho.reporting.libraries.base.config.Configuration;
import org.pentaho.reporting.libraries.base.util.ObjectUtilities;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Creation-Date: 02.11.2006, 12:46:08
 *
 * @author Thomas Morgner
 */
public class DefaultTypeRegistry implements TypeRegistry
{
  private static final Log logger = LogFactory.getLog(DefaultTypeRegistry.class);

  private static class ArrayConverterCallback implements ArrayCallback
  {
    private Object retval;
    private Type targetType;

    private ArrayConverterCallback(final Object retval, final Type targetType)
    {
      this.retval = retval;
      this.targetType = targetType;
    }

    public LValue getRaw(final int row, final int column)
    {
      return null;
    }

    public Object getValue(final int row, final int column) throws EvaluationException
    {
      if (row == 0 && column == 0)
      {
        return retval;
      }
      return null;
    }

    public Type getType(final int row, final int column) throws EvaluationException
    {
      if (row == 0 && column == 0)
      {
        return targetType;
      }
      return null;
    }

    public int getColumnCount()
    {
      return 1;
    }

    public int getRowCount()
    {
      return 1;
    }
  }

  private static final BigDecimal NUM_TRUE = new BigDecimal("1");
  private static final BigDecimal NUM_FALSE = new BigDecimal("0");
  private static final BigDecimal ZERO = NUM_FALSE;

  private FormulaContext context;
  private NumberFormat[] numberFormats;

  public DefaultTypeRegistry()
  {
  }

  /**
   * Returns an comparator for the given types.
   *
   * @param type1
   * @param type2
   * @return
   */
  public ExtendedComparator getComparator(final Type type1, final Type type2)
  {
    final DefaultComparator comparator = new DefaultComparator();
    comparator.inititalize(context);
    return comparator;
  }

  /**
   * converts the object of the given type into a number. If the object is not convertible, a NumberFormatException is
   * thrown. If the given value is null or not parsable as number, return null.
   *
   * @param sourceType
   * @param value
   * @return
   * @throws NumberFormatException if the type cannot be represented as number.
   */
  public Number convertToNumber(final Type sourceType, final Object value)
      throws EvaluationException
  {
    final LocalizationContext localizationContext = context.getLocalizationContext();

    if (value == null)
    {
      // there's no point in digging deeper - there *is* no value ..
      throw new TypeConversionException();
    }

    final boolean isAnyType = sourceType.isFlagSet(Type.ANY_TYPE);
    if (sourceType.isFlagSet(Type.NUMERIC_TYPE) || isAnyType)
    {
      if (sourceType.isFlagSet(Type.DATETIME_TYPE)
          || sourceType.isFlagSet(Type.TIME_TYPE)
          || sourceType.isFlagSet(Type.DATE_TYPE)
          || isAnyType)
      {
        if (value instanceof Date)
        {
          final BigDecimal serial = HSSFDateUtil.getExcelDate((Date) value);
          return DateUtil.normalizeDate(serial, sourceType);
        }
      }

      if (value instanceof Number)
      {
        return (Number) value;
      }
    }

    if (sourceType.isFlagSet(Type.LOGICAL_TYPE) || isAnyType)
    {
      if (value instanceof Boolean)
      {
        if (Boolean.TRUE.equals(value))
        {
          return NUM_TRUE;
        }
        else
        {
          return NUM_FALSE;
        }
      }
    }

    if (sourceType.isFlagSet(Type.TEXT_TYPE) || isAnyType)
    {
      final String val = computeStringValue(value);

      // first, try to parse the value as a big-decimal.
      try
      {
        return new BigDecimal(val);
      }
      catch (NumberFormatException e)
      {
        // ignore ..
      }

      // then checking for datetimes
      final Iterator datetimeIterator = localizationContext.getDateFormats(DateTimeType.DATETIME_TYPE).iterator();
      while (datetimeIterator.hasNext())
      {
        final DateFormat df = (DateFormat) datetimeIterator.next();
        try
        {
          final Date date = df.parse(val);
          return HSSFDateUtil.getExcelDate(date);
        }
        catch (ParseException e)
        {
          // ignore as well ..
        }
      }
      // then checking for datetimes
      final Iterator dateIterator = localizationContext.getDateFormats(DateTimeType.DATE_TYPE).iterator();
      while (dateIterator.hasNext())
      {
        final DateFormat df = (DateFormat) dateIterator.next();
        try
        {
          final Date date = df.parse(val);
          return HSSFDateUtil.getExcelDate(date);
        }
        catch (ParseException e)
        {
          // ignore as well ..
        }
      }
      // then checking for datetimes
      final Iterator timeIterator = localizationContext
          .getDateFormats(DateTimeType.TIME_TYPE).iterator();
      while (timeIterator.hasNext())
      {
        final DateFormat df = (DateFormat) timeIterator.next();
        try
        {
          final Date date = df.parse(val);
          return HSSFDateUtil.getExcelDate(date);
        }
        catch (ParseException e)
        {
          // ignore as well ..
        }
      }

      // then checking for numbers
      for (int i = 0; i < numberFormats.length; i++)
      {
        try
        {
          final NumberFormat format = numberFormats[i];
          return format.parse(val);
        }
        catch (ParseException e)
        {
          // ignore ..
        }
      }
    }

    throw new TypeConversionException();
  }

  /**
   *
   * @param configuration
   * @param formulaContext
   * @deprecated Use the single-argument function instead.
   */
  public void initialize(final Configuration configuration,
                         final FormulaContext formulaContext)
  {
    this.initialize(formulaContext);
  }

  public void initialize(final FormulaContext formulaContext)
  {
    if (formulaContext == null)
    {
      throw new NullPointerException();
    }
    this.context = formulaContext;
    this.numberFormats = loadNumberFormats();
  }


  protected NumberFormat[] loadNumberFormats()
  {
    final ArrayList formats = new ArrayList();
    final DecimalFormat defaultFormat = new DecimalFormat("#0.###",
        new DecimalFormatSymbols(Locale.US));
    activateBigDecimalMode(defaultFormat);
    formats.add(defaultFormat);

    return (NumberFormat[]) formats.toArray(new NumberFormat[formats.size()]);
  }

  private void activateBigDecimalMode(final DecimalFormat format)
  {
    if (ObjectUtilities.isJDK14())
    {
      try
      {
        final Method method = DecimalFormat.class.getMethod("setParseBigDecimal", new Class[]{Boolean.TYPE});
        method.invoke(format, new Object[] {Boolean.TRUE});
      }
      catch (Exception e)
      {
        // ignore it, as it will always fail on JDK 1.4 or lower ..
      }
    }
  }

  public String convertToText(final Type type1, final Object value)
      throws EvaluationException
  {
    if (value == null)
    {
      return "";
    }

    // already converted or compatible
    if (type1.isFlagSet(Type.TEXT_TYPE))
    {
      // no need to check whatever it is a String
      return computeStringValue(value);
    }

    if (type1.isFlagSet(Type.LOGICAL_TYPE))
    {
      if (value instanceof Boolean)
      {
        final Boolean b = (Boolean) value;
        if (Boolean.TRUE.equals(b))
        {
          return "TRUE";
        }
        else
        {
          return "FALSE";
        }
      }
      else
      {
        throw new TypeConversionException();
      }
    }

    // 2 types of numeric : numbers and dates
    if (type1.isFlagSet(Type.NUMERIC_TYPE))
    {
      if (type1.isFlagSet(Type.DATETIME_TYPE)
          || type1.isFlagSet(Type.DATE_TYPE) || type1.isFlagSet(Type.TIME_TYPE))
      {
        final Date d = convertToDate(type1, value);
        final List dateFormats = context.getLocalizationContext()
            .getDateFormats(type1);
        if (dateFormats != null && dateFormats.size() >= 1)
        {
          final DateFormat format = (DateFormat) dateFormats.get(0);
          return format.format(d);
        }
        else
        {
          // fallback
          return DateFormat.getDateTimeInstance(
              DateFormat.FULL, DateFormat.FULL).format(d);
        }
      }
      else
      {
        try
        {
          final Number n = convertToNumber(type1, value);
          final NumberFormat format = getDefaultNumberFormat();
          return format.format(n);
        }
        catch (EvaluationException nfe)
        {
          // ignore ..
        }
      }
    }

    return computeStringValue(value);
  }

  private String computeStringValue(final Object retval)
  {
    if (retval instanceof Clob)
    {
      return readClob((Clob) retval);
    }
    if (retval != null)
    {
      return retval.toString();
    }
    return null;
  }

  private String readClob(final Clob clob)
  {
    try
    {
      return clob.getSubString(0, (int) clob.length());
    }
    catch (SQLException e)
    {
      return null;
    }
  }

  public Boolean convertToLogical(final Type type1, final Object value)
      throws TypeConversionException
  {
    if (value == null)
    {
      return Boolean.FALSE;
    }

    // already converted or compatible
    if (type1.isFlagSet(Type.LOGICAL_TYPE) || type1.isFlagSet(Type.ANY_TYPE))
    {
      if (value instanceof Boolean)
      {
        return (Boolean) value;
      }

      // fallback
      if ("true".equalsIgnoreCase(String.valueOf(value)))
      {
        return Boolean.TRUE;
      }
      return Boolean.FALSE;
    }

    if (type1.isFlagSet(Type.NUMERIC_TYPE))
    {
      // no need to check between different types of numeric
      if (value instanceof Number)
      {
        final Number num = (Number) value;
        if (!ZERO.equals(num))
        {
          return Boolean.TRUE;
        }
      }

      // fallback
      return Boolean.FALSE;
    }

    if (type1.isFlagSet(Type.TEXT_TYPE))
    {
      // no need to convert it to String
      final String str = computeStringValue(value);
      if ("TRUE".equalsIgnoreCase(str))
      {
        return Boolean.TRUE;
      }
      else if ("FALSE".equalsIgnoreCase(str))
      {
        return Boolean.FALSE;
      }
    }

    throw new TypeConversionException();
  }

  public Date convertToDate(final Type type1, final Object value)
      throws EvaluationException
  {
    if (type1.isFlagSet(Type.NUMERIC_TYPE) || type1.isFlagSet(Type.ANY_TYPE))
    {
      if (type1.isFlagSet(Type.DATE_TYPE)
          || type1.isFlagSet(Type.DATETIME_TYPE)
          || type1.isFlagSet(Type.TIME_TYPE) || type1.isFlagSet(Type.ANY_TYPE))
      {
        if (value instanceof Date)
        {
          return DateUtil.normalizeDate((Date) value, type1);
        }
      }
    }
    final Number serial = convertToNumber(type1, value);
    final BigDecimal bd = NumberUtil.getAsBigDecimal(serial);
    return HSSFDateUtil.getJavaDate(bd);
  }

  protected NumberFormat getDefaultNumberFormat()
  {
    final Locale locale = context.getLocalizationContext().getLocale();
    return new DecimalFormat("#0.#########", new DecimalFormatSymbols(locale));
  }

  /**
   * A internal method that converts the given value-pair into a sequence.
   *
   * @param targetType
   * @param valuePair
   * @return
   * @throws TypeConversionException if there was a error while converting types.
   */
  private TypeValuePair convertToSequence(final Type targetType, final TypeValuePair valuePair)
      throws EvaluationException
  {
    if (targetType.isFlagSet(Type.NUMERIC_SEQUENCE_TYPE))
    {
      return new TypeValuePair
          (targetType, convertToNumberSequence(valuePair.getType(), valuePair.getValue(), true));
    }

    return new TypeValuePair(targetType, convertToSequence(valuePair.getType(), valuePair.getValue()));
  }

  public Sequence convertToSequence(final Type type, final Object value) throws EvaluationException
  {
    // sclar
    if (type.isFlagSet(Type.SCALAR_TYPE))
    {
      return new AnySequence(new StaticValue(value, type), context);
    }
    // else already a sequence
    else if (type.isFlagSet(Type.SEQUENCE_TYPE))
    {
      if (value instanceof AnySequence)
      {
        return (AnySequence) value;
      }
      else
      {
        logger.warn("Assertation failure: Type declared to be a sequence, but no sequence found inside.");
        throw new TypeConversionException();
      }
    }
    // else an array source
    else if (type.isFlagSet(Type.ARRAY_TYPE))
    {
      if (value instanceof ArrayCallback)
      {
        return new AnySequence((ArrayCallback) value, context);
      }
      else
      {
        logger.warn("Assertation failure: Type declared to be array, but no array callback found inside.");
        throw new TypeConversionException();
      }
    }
    throw new TypeConversionException();
  }

  public NumberSequence convertToNumberSequence(final Type type, final Object value, final boolean strict)
      throws EvaluationException
  {
    // sequence array
    if (type.isFlagSet(Type.NUMERIC_SEQUENCE_TYPE))
    {
      if (value instanceof DefaultNumberSequence)
      {
        return (NumberSequence) value;
      }
      else
      {
        // a empty sequence ...
        return new DefaultNumberSequence(context);
      }
    }
    // array
    else if (type.isFlagSet(Type.ARRAY_TYPE))
    {
      if (value instanceof ArrayCallback)
      {
        if (strict)
        {
          return new DefaultNumberSequence((ArrayCallback) value, context);
        }
        else
        {
          return new AnyNumberSequence((ArrayCallback) value, context);
        }
      }
      else
      {
        logger.warn("Assertation failure: Type declared to be array, but no array callback found inside.");
        throw new TypeConversionException();
      }
    }
    // else scalar
    if (type.isFlagSet(Type.SCALAR_TYPE) || type.isFlagSet(Type.NUMERIC_TYPE))
    {
      return new DefaultNumberSequence
          (new StaticValue(convertToNumber(type, value), NumberType.GENERIC_NUMBER), context);
    }
    else
    {
      return new DefaultNumberSequence(context);
    }
  }

  /**
   * Checks whether the target type would accept the specified value object and value type.<br/> This method is called
   * for auto conversion of fonction parameters using the conversion type declared by the function metadata.
   *
   * @param targetType
   * @param valuePair
   * @noinspection ObjectEquality is tested at the end of the method for performance reasons only. We just want to
   * detect whether a new object has been created or not.
   */
  public TypeValuePair convertTo(final Type targetType,
                                 final TypeValuePair valuePair) throws EvaluationException
  {
    if (targetType.isFlagSet(Type.ARRAY_TYPE))
    {
      // Array conversion requested.
      // todo: This code looks weird
      if (valuePair.getType().isFlagSet(Type.ARRAY_TYPE))
      {
        if (valuePair.getType().isFlagSet(Type.SEQUENCE_TYPE))
        {
          return convertToSequence(targetType, valuePair);
        }
        else
        {
          if (targetType.isFlagSet(Type.SEQUENCE_TYPE))
          {
            //System.out.println(targetType.isFlagSet(Type.ARRAY_TYPE) + " " + valuePair.getType().isFlagSet(Type.ARRAY_TYPE) + " " + valuePair.getValue());
            return convertToSequence(targetType, valuePair);
          }
          else
          {
            return convertArrayToArray(targetType, valuePair);
          }
        }
      }
      else // conversion of a scalar to an array
      {
        if (targetType.isFlagSet(Type.SEQUENCE_TYPE))
        {
          return convertToSequence(targetType, valuePair);
        }
        else
        {
          final Object retval = convertPlainToPlain(targetType, valuePair.getType(), valuePair.getValue());
          return new TypeValuePair(targetType, new ArrayConverterCallback(retval, targetType));
        }
      }
    }

    // else scalar
    final Object value = valuePair.getValue();
    final Object o = convertPlainToPlain(targetType, valuePair.getType(), value);
    if (value == o)
    {
      return valuePair;
    }
    return new TypeValuePair(targetType, o);
  }

  private Object convertPlainToPlain(final Type targetType, final Type sourceType,
                                     final Object value) throws EvaluationException
  {
    if (targetType.isFlagSet(Type.NUMERIC_TYPE))
    {
      if (targetType.isFlagSet(Type.LOGICAL_TYPE))
      {
        if (sourceType.isFlagSet(Type.LOGICAL_TYPE))
        {
          return value;
        }

        return convertToLogical(sourceType, value);
      }

      if (value instanceof Date)
      {
        if (targetType.isFlagSet(Type.DATE_TYPE)
            || targetType.isFlagSet(Type.DATETIME_TYPE)
            || targetType.isFlagSet(Type.TIME_TYPE))
        {
          final Date toJavaDate = (Date) value;
          return DateUtil.normalizeDate(toJavaDate, targetType, false);
        }
      }

      final Number serial = convertToNumber(sourceType, value);
      if (targetType.isFlagSet(Type.DATE_TYPE)
          || targetType.isFlagSet(Type.DATETIME_TYPE)
          || targetType.isFlagSet(Type.TIME_TYPE))
      {
        final BigDecimal fromAsBigDecimal = NumberUtil.getAsBigDecimal(serial);
        final BigDecimal normalizedSerial = DateUtil.normalizeDate(fromAsBigDecimal, targetType);
        final Date toJavaDate = HSSFDateUtil.getJavaDate(normalizedSerial);
        return DateUtil.normalizeDate(toJavaDate, targetType, false);
      }
      return serial;
    }
    else if (targetType.isFlagSet(Type.TEXT_TYPE))
    {
      return convertToText(sourceType, value);
    }

    // Unknown type - ignore it, crash later :)
    return value;
  }

  private TypeValuePair convertArrayToArray(final Type targetType,
                                            final TypeValuePair pair) throws TypeConversionException
  {
//    final Object value = pair.getValue();
//    if (value instanceof ArrayCallback)
//    {
//      final ArrayCallback array = (ArrayCallback) value;
//      for (int i = 0; i < array.getRowCount(); i++)
//      {
//        for (int j = 0; j < array.getColumnCount(); j++)
//        {
//          //TODO
//          throw new UnsupportedOperationException("Not implemented exception");
//        }
//      }
//    }

    throw new TypeConversionException();
  }

  public Type guessTypeOfObject(final Object o)
  {
    if (o instanceof Number)
    {
      return NumberType.GENERIC_NUMBER;
    }
    else if (o instanceof Time)
    {
      return DateTimeType.TIME_TYPE;
    }
    else if (o instanceof java.sql.Date)
    {
      return DateTimeType.DATE_TYPE;
    }
    else if (o instanceof Date)
    {
      return DateTimeType.DATETIME_TYPE;
    }
    else if (o instanceof Boolean)
    {
      return LogicalType.TYPE;
    }
    else if (o instanceof String)
    {
      return TextType.TYPE;
    }

    return AnyType.TYPE;
  }
}
