/**
 * ===========================================================
 * LibRepository : a free Java content repository access layer
 * ===========================================================
 *
 * Project Info:  http://reporting.pentaho.org/librepository/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * StreamRepository.java
 * ------------
 */

package org.pentaho.reporting.libraries.repository.stream;

import java.io.InputStream;
import java.io.OutputStream;

import org.pentaho.reporting.libraries.repository.ContentLocation;
import org.pentaho.reporting.libraries.repository.DefaultMimeRegistry;
import org.pentaho.reporting.libraries.repository.MimeRegistry;
import org.pentaho.reporting.libraries.repository.Repository;

/**
 * A repository that feeds a single source.
 *
 * @author Thomas Morgner
 */
public class StreamRepository implements Repository
{
  private MimeRegistry mimeRegistry;
  private WrappedOutputStream outputStream;
  private WrappedInputStream inputStream;
  private StreamContentLocation rootLocation;
  private String contentName;

  /**
   * Creates a new repository that potentially allows both read and write access. If an input stream is given, then
   * a content name must be given as well.
   *
   * @param inputStream the inputstream from which to read from.
   * @param outputStream the output stream to which to write to.
   * @param contentName the content name by which the content should be accessed.
   */
  public StreamRepository(final InputStream inputStream,
                          final OutputStream outputStream,
                          final String contentName)
  {
    if (contentName == null)
    {
      throw new NullPointerException();
    }
    if (inputStream != null)
    {
      this.inputStream = new WrappedInputStream(inputStream);
    }
    if (outputStream != null)
    {
      this.outputStream = new WrappedOutputStream(outputStream);
    }

    this.contentName = contentName;
    this.mimeRegistry = new DefaultMimeRegistry();
    this.rootLocation = new StreamContentLocation(this);
  }

  /**
   * Creates a new read/write repository with a hardcoded name for the input stream.
   *
   * @param inputStream the input stream from where to read the data (can be null).
   * @param outputStream the output stream where data is written to (can be null). 
   * @deprecated This constructor should not be used, as it hardcodes the filename for the input stream.
   * Use one of the other constructors instead. 
   */
  public StreamRepository(final InputStream inputStream, final OutputStream outputStream)
  {
    this(inputStream, outputStream, "content.data");
  }

  /**
   * Creates a new read-only repository.
   *
   * @param inputStream the input stream from where to read the data (can be null).
   * @param contentName the content name by which the content should be accessed.
   */
  public StreamRepository(final InputStream inputStream, final String contentName)
  {
    this(inputStream, null, contentName);
  }

  /**
   * Creates a new write-only repository.
   *
   * @param outputStream the output stream to which to write to.
   */
  public StreamRepository(final OutputStream outputStream)
  {
    this(null, outputStream, "content.data");
  }

  /**
   * Returns the optional content name by which the data in the input-stream should be accessed.
   *
   * @return the content name or null, if this repository is write-only.
   */
  public String getContentName()
  {
    return contentName;
  }

  /**
   * Returns the optional output stream.
   *
   * @return the stream or null, if this repository is read-only.
   */
  public WrappedOutputStream getOutputStream()
  {
    return outputStream;
  }

  /**
   * Returns the optional input stream.
   *
   * @return the stream or null, if this repository is write-only.
   */
  public WrappedInputStream getInputStream()
  {
    return inputStream;
  }

  /**
   * Returns the content root of this repository.
   *
   * @return the content root.
   */
  public ContentLocation getRoot()
  {
    return rootLocation;
  }

  /**
   * Returns the mime registry for this repository.
   *
   * @return the mime-registry.
   */
  public MimeRegistry getMimeRegistry()
  {
    return mimeRegistry;
  }
}
