/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: sidfquery.c 692 2009-03-03 00:55:34Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: sidfquery.c 692 2009-03-03 00:55:34Z takahiko $");

#include <stdio.h>
#include <stdlib.h>
#include <sysexits.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "ptrop.h"
#include "dnsresolv.h"
#include "sidf.h"

static void
usage(FILE *fp)
{
    fprintf(fp, "\nUsage: sidfquery [-46mps] username@domain IP-address\n\n");
    fprintf(fp, "Handling of IP address:\n");
    fprintf(fp, "  -4    handle \"IP-address\" as IPv4 address (default)\n");
    fprintf(fp, "  -6    handle \"IP-address\" as IPv6 address\n\n");
    fprintf(fp, "evaluation mode:\n");
    fprintf(fp, "  -s  SPF mode (default)\n");
    fprintf(fp, "  -m  Sender ID (mfrom) mode\n");
    fprintf(fp, "  -p  Sender ID (pra) mode\n");
    exit(EX_USAGE);
}   // end functiion : usage

static void
stdout_logger(int priority, const char *message, ...)
{
    va_list args;
    va_start(args, message);
    vfprintf(stdout, message, args);
    putc('\n', stdout);
    va_end(args);
}   // end functiion : stdout_logger

int
main(int argc, char **argv)
{
    sa_family_t sa_family = AF_INET;
    SidfRecordScope scope = SIDF_RECORD_SCOPE_SPF1;

    int c;
    while (-1 != (c = getopt(argc, argv, "46mpsh"))) {
        switch (c) {
        case '4':  // IPv4
            sa_family = AF_INET;
            break;
        case '6':  // IPv6
            sa_family = AF_INET6;
            break;
        case 'm':  // SIDF/mfrom
            scope = SIDF_RECORD_SCOPE_SPF2_MFROM;
            break;
        case 'p':  // SIDF/pra
            scope = SIDF_RECORD_SCOPE_SPF2_PRA;
            break;
        case 's':  // SPF
            scope = SIDF_RECORD_SCOPE_SPF1;
            break;
        case 'h':
            usage(stdout);
            break;
        default:
            fprintf(stdout, "[Error] illegal option: -%c\n", c);
            usage(stdout);
            break;
        }   // end switch
    }   // end while

    argc -= optind;
    argv += optind;

    if (argc < 2) {
        usage(stdout);
    }   // end if

    DnsResolver *resolver = DnsResolver_new();
    if (NULL == resolver) {
        fprintf(stdout, "[Error] resolver initialization failed: err=%s\n", strerror(errno));
        exit(EX_OSERR);
    }   // end if

    const char *mailbox = argv[0];
    const char *ipaddr = argv[1];

    SidfPolicy *policy = SidfPolicy_new();
    if (NULL == policy) {
        fprintf(stdout, "[Error] SidfPolicy_new failed: err=%s\n", strerror(errno));
        exit(EX_OSERR);
    }   // end if
    SidfPolicy_setSpfRRLookup(policy, true);
    SidfPolicy_setLogger(policy, stdout_logger);

    SidfRequest *request = SidfRequest_new(policy, resolver);
    if (NULL == request) {
        fprintf(stdout, "[Error] SidfRequest_new failed: err=%s\n", strerror(errno));
        exit(EX_OSERR);
    }   // end if

    if (!SidfRequest_setIpAddrString(request, sa_family, ipaddr)) {
        fprintf(stdout, "[Error] invalid IP address: ip-address=%s\n", ipaddr);
        usage(stdout);
    }   // end if
    const char *dummy;
    InetMailbox *envfrom = InetMailbox_build2822Mailbox(mailbox, STRTAIL(mailbox), &dummy, NULL);
    if (NULL == envfrom) {
        fprintf(stdout, "[Error] mailbox is not RFC2822 compliant: mailbox=%s\n", mailbox);
        usage(stdout);
    }   // end if

    SidfRequest_setSender(request, envfrom);
    SidfRequest_setHeloDomain(request, InetMailbox_getDomain(envfrom));

    // SPF/Sender ID evaluation
    SidfScore score = SidfRequest_eval(request, scope);
    const char *spfresultexp = SidfEnum_lookupScoreByValue(score);
    fprintf(stdout, "%s %s %s\n", mailbox, ipaddr, spfresultexp);

    // clean up
    SidfRequest_free(request);
    SidfPolicy_free(policy);
    DnsResolver_free(resolver);
    InetMailbox_free(envfrom);

    exit(EX_OK);
}   // end function : main
