/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimheaders.c 653 2009-02-28 20:35:27Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimheaders.c 653 2009-02-28 20:35:27Z takahiko $");

#include <assert.h>
#include <string.h>
#include <sys/types.h>
#include <stdbool.h>

#include "dkimlogger.h"
#include "inetmailbox.h"
#include "ptrop.h"
#include "xskip.h"
#include "strarray.h"
#include "dkim.h"
#include "dkimspec.h"
#include "dkimpolicy.h"
#include "mailheaders.h"
#include "dkimheaders.h"

static dkim_stat_t
DkimHeaders_buildAuthorMailbox(const DkimPolicy *policy, const char *head, const char *tail,
                               InetMailbox **mailbox)
{
    assert(NULL != head);
    assert(NULL != tail);
    assert(NULL != mailbox);

    const char *p, *errptr;
    InetMailbox *build_mailbox = InetMailbox_build2822Mailbox(head, tail, &p, &errptr);

    if (NULL == build_mailbox) {
        // parse に失敗した場合
        if (NULL != errptr) {
            DkimLogPermFail(policy, "Mailbox parse error: near %.50s", p);
            return DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS;
        } else {
            DkimLogNoResource(policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    }   // end if

    XSkip_fws(p, tail, &p); // 末尾の余計な FWS は許容する.
    if (p == tail) {
        *mailbox = build_mailbox;
        return DSTAT_OK;
    } else {
        // parse には成功したが, 入力列に余りがある (= RFC2822 の mailbox にマッチしない) 場合
        DkimLogPermFail(policy, "Author field has unused portion: %d bytes, near %.50s", tail - p,
                        head);
        InetMailbox_free(build_mailbox);
        return DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS;
    }   // end if
}   // end function : DkimHeaders_buildAuthorMailbox

/*
 * @param headeridx Author として採用したヘッダへのインデックスを受け取る変数へのポインタ.
 *        返値が DSTAT_OK 以外の場合は未定義.
 * @param refHeaderf Author として採用したヘッダ名を受け取る変数へのポインタ.
 *        返値が DSTAT_OK 以外の場合は未定義.
 * @param refHeaderv Author として採用したヘッダ値を受け取る変数へのポインタ.
 *        返値が DSTAT_OK 以外の場合は未定義.
 * @param mailbox Author として採用したヘッダから構築した InetMailbox オブジェクトを受け取る変数へのポインタ.
 *        返値が DSTAT_OK 以外の場合は未定義.
 */
dkim_stat_t
DkimHeaders_extractAuthor(const DkimPolicy *policy, const MailHeaders *headers,
                          const StrArray *headerPriority, size_t *headeridx,
                          const char **refHeaderf, const char **refHeaderv, InetMailbox **mailbox)
{
    assert(NULL != headers);
    assert(NULL != headerPriority);

    size_t num = StrArray_getCount(headerPriority);
    for (size_t i = 0; i < num; ++i) {
        const char *targetField = StrArray_get(headerPriority, i);

        bool multiple;
        int index = MailHeaders_getHeaderIndex(headers, targetField, &multiple);
        if (index < 0) {
            // 探していたヘッダは見つからなかったので, 次の Author 候補ヘッダを探す.
            // do nothing
            continue;
        }   // end if
        if (multiple) {
            // 探していたヘッダが複数見つかった.
            DkimLogPermFail(policy, "Multiple %s Header is found, unable to extract Author",
                            targetField);
            return DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS;
        }   // end if

        // 探していたヘッダが見つかった.
        // 見つかった Author ヘッダをパースして mailbox を抜き出す
        // 書式が間違っている場合は次の Author 候補ヘッダがあろうがなかろうがエラー
        const char *headerf, *headerv;
        MailHeaders_get(headers, index, &headerf, &headerv);
        dkim_stat_t dstat =
            DkimHeaders_buildAuthorMailbox(policy, headerv, STRTAIL(headerv), mailbox);
        if (DSTAT_OK == dstat) {
            // パースに成功した場合, パース前のデータへのリファレンスを返す
            SETDEREF(headeridx, index);
            SETDEREF(refHeaderf, headerf);
            SETDEREF(refHeaderv, headerv);
        }   // end if
        return dstat;
    }   // end for

    DkimLogPermFail(policy, "No Author header found");
    return DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS;
}   // end function : DkimHeaders_extractAuthor
