/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimsignpolicy.c 686 2009-03-02 04:58:41Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimsignpolicy.c 686 2009-03-02 04:58:41Z takahiko $");

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <limits.h>
#include <sys/types.h>

#include "xskip.h"
#include "dkimlogger.h"
#include "strarray.h"
#include "xbuffer.h"
#include "ptrop.h"
#include "dkim.h"
#include "dkimenum.h"
#include "dkimsignpolicy.h"

/**
 * DkimSignPolicy オブジェクトの構築
 * @return 空の DkimSignPolicy オブジェクト
 */
DkimSignPolicy *
DkimSignPolicy_new(void)
{
    DkimSignPolicy *self = (DkimSignPolicy *) malloc(sizeof(DkimSignPolicy));
    if (NULL == self) {
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimSignPolicy));

    // 初期化
    DkimPolicy_init((DkimPolicy *) self);
    self->signature_ttl = -1LL; // 無期限
    self->pubkeyalg = DKIM_PUBKEYALG_RSA;
    self->digestalg = DKIM_DIGESTALG_SHA256;
    self->canon_method_header = DKIM_CANONALG_RELAXED;
    self->canon_method_body = DKIM_CANONALG_SIMPLE;

    return self;
}   // end function : DkimSignPolicy_new

/**
 * DkimSignPolicy オブジェクトの解放
 * @param self 解放する DkimSignPolicy オブジェクト
 */
void
DkimSignPolicy_free(DkimSignPolicy *self)
{
    assert(NULL != self);

    DkimPolicy_cleanup((DkimPolicy *) self);
    free(self);
}   // end function : DkimSignPolicy_free

dkim_stat_t
DkimSignPolicy_setCanonAlgorithm(DkimSignPolicy *self, const char *headercanon,
                                 const char *bodycanon)
{
    assert(NULL != self);

    if (NULL == headercanon || NULL == bodycanon) {
        DkimLogConfigError(self, "empty value specified for %s canonicalization algorithm",
                           NULL == headercanon ? "header" : "body");
        return DSTAT_CFGERR_EMPTY_VALUE;
    }   // end if

    self->canon_method_header = DkimEnum_lookupCanonAlgByName(headercanon);
    if (DKIM_CANONALG_NULL == self->canon_method_header) {
        DkimLogConfigError(self, "undefined header canonicalization algorithm: canonalg=%s",
                           headercanon);
        return DSTAT_CFGERR_UNDEFINED_KEYWORD;
    }   // end if

    self->canon_method_body = DkimEnum_lookupCanonAlgByName(bodycanon);
    if (DKIM_CANONALG_NULL == self->canon_method_body) {
        DkimLogConfigError(self, "undefined body canonicalization algorithm: canonalg=%s",
                           bodycanon);
        return DSTAT_CFGERR_UNDEFINED_KEYWORD;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignPolicy_setCanonAlgorithm

dkim_stat_t
DkimSignPolicy_setDigestAlgorithm(DkimSignPolicy *self, const char *digestalg)
{
    assert(NULL != self);

    if (NULL == digestalg) {
        DkimLogConfigError(self, "empty value specified for digest algorithm");
        return DSTAT_CFGERR_EMPTY_VALUE;
    }   // end if

    self->digestalg = DkimEnum_lookupDigestAlgByName(digestalg);
    if (DKIM_DIGESTALG_NULL == self->digestalg) {
        DkimLogConfigError(self, "undefined digest algorithm: digestalg=%s", digestalg);
        return DSTAT_CFGERR_UNDEFINED_KEYWORD;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignPolicy_setDigestAlgorithm

dkim_stat_t
DkimSignPolicy_setPublicKeyAlgorithm(DkimSignPolicy *self, const char *pubkeyalg)
{
    assert(NULL != self);

    if (NULL == pubkeyalg) {
        DkimLogConfigError(self, "empty value specified for public key algorithm");
        return DSTAT_CFGERR_EMPTY_VALUE;
    }   // end if

    self->pubkeyalg = DkimEnum_lookupPubkeyAlgByName(pubkeyalg);
    if (DKIM_PUBKEYALG_NULL == self->pubkeyalg) {
        DkimLogConfigError(self, "undefined public key algorithm: pubkeyalg=%s", pubkeyalg);
        return DSTAT_CFGERR_UNDEFINED_KEYWORD;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignPolicy_setPublicKeyAlgorithm

void
DkimSignPolicy_setSignatureTTL(DkimSignPolicy *self, long long signature_ttl)
{
    assert(NULL != self);
    self->signature_ttl = signature_ttl;
}   // end function : DkimSignPolicy_setSignatureTTL
