/**
 * @file filter.c Message filter utility functions
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "filter.h"
#include "utils.h"

static void
path_unregistered_func(DBusConnection *dbus_conn, void *user_data)
{
	GalagodStateData *state_data = (GalagodStateData *)user_data;

	if (state_data->iface != NULL)
		g_free(state_data->iface);

	g_free(state_data);
}

static DBusHandlerResult
path_message_func(DBusConnection *dbus_conn, DBusMessage *message,
				  void *user_data)
{
	GalagodStateData *state_data = (GalagodStateData *)user_data;
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_SIGNAL)
		return result;

	galago_context_push(state_data->context);
	result = galagod_cmd_table_execute(state_data->commands, dbus_conn,
									   message, state_data->coco,
									   state_data->u.object,
									   state_data->iface);
	galago_context_pop();

	return result;
}

static DBusObjectPathVTable common_vtable =
{
	path_unregistered_func,
	path_message_func,
	NULL
};

void
galagod_filters_add(void *obj, const GalagodCommand *commands,
					const char *iface)
{
	GalagodStateData *state_data;
	GalagodCoCo *coco;
	GalagoContext *context;

	g_return_if_fail(obj      != NULL);
	g_return_if_fail(commands != NULL);
	g_return_if_fail(iface    != NULL);
	g_return_if_fail(*iface   != '\0');
	g_return_if_fail(GALAGO_IS_OBJECT(obj));

	coco = g_object_get_data(G_OBJECT(obj), "coco");

	context = (coco != NULL
			   ? galagod_coco_get_context(coco)
			   : galagod_get_context());

	state_data = g_new0(GalagodStateData, 1);
	state_data->coco     = coco;
	state_data->context  = context;
	state_data->u.object = obj;
	state_data->commands = commands;
	state_data->iface    = g_strdup(iface);

	if (!dbus_connection_register_object_path(galago_get_dbus_conn(),
											  galago_object_get_dbus_path(obj),
											  &common_vtable, state_data))
	{
		abort();
	}
}

void
galagod_filters_remove(void *obj)
{
	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(obj));

	dbus_connection_unregister_object_path(galago_get_dbus_conn(),
										   galago_object_get_dbus_path(obj));
}

DBusHandlerResult
galagod_cmd_table_execute(const GalagodCommand *cmd_table,
						  DBusConnection *dbus_conn, DBusMessage *message,
						  GalagodCoCo *coco, void *data, const char *iface)
{
	const GalagodCommand *cmd;

	if (dbus_message_get_path(message) == NULL)
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	g_message("%s: %s",
			  dbus_message_get_interface(message),
			  dbus_message_get_member(message));

	for (cmd = cmd_table; cmd->name != NULL; cmd++)
	{
		if (dbus_message_is_method_call(message, iface, cmd->name))
			return cmd->func(dbus_conn, message, coco, data);
	}

	g_warning("Unhandled message!");
	g_warning("\tobj_path  = '%s'", dbus_message_get_path(message));
	g_warning("\tinterface = '%s'", dbus_message_get_interface(message));
	g_warning("\tmethod    = '%s'", dbus_message_get_member(message));

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
