/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include "RepoView.h"
#include "OutputView.h"
#include "DlgEditRepo.h"
#include "DlgImportGPG.h"
#include "YumLogView.h"
#include "KYumPreferences.h"
#include "KYumConf.h"
#include "KYumView.h"

#include <qpainter.h>
#include <qlayout.h>
#include <qpopupmenu.h>

#include <kmessagebox.h>
#include <klocale.h>

static KYumView * g_pYumView = 0;


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

KYumView::KYumView(QWidget * parent)

  : QTabWidget(parent),
    m_pCurrentTab(0)

{
    g_pYumView = this;

    // setup our layout manager to automatically add our widgets
    QHBoxLayout *top_layout = new QHBoxLayout(this);
    top_layout->setAutoAdd(true);

    addTab(m_pActionsTab = new ActionsTab(this), i18n("Actions"));
    addTab(m_pGroupsTab  = new GroupsTab(this),  i18n("Groups"));
    addTab(m_pYumConfTab = new YumConfTab(this), i18n("yum.conf"));
    addTab(m_pReposTab   = new ReposTab(this),   i18n("Repositories"));

    connect(getRepoView(), SIGNAL(selectionChanged(QListViewItem *)),
            this, SLOT(slotRepoSelected(QListViewItem *)));

    connect(getRepoView(), SIGNAL(clicked(QListViewItem *)),
            this, SLOT(slotRepoSelected(QListViewItem *)));

    connect(getRepoView(), SIGNAL(doubleClicked(QListViewItem *, const QPoint &, int)),
            this, SLOT(slotEditRepo(QListViewItem *, const QPoint &, int)));

    connect(getYumConfView(), SIGNAL(configSaved(const QString &)),
            this, SLOT(slotYumConfSaved(const QString &)));

    connect(this, SIGNAL(currentChanged(QWidget *)),
            this, SLOT(slotTabChanged(QWidget *)));

    connect(getRepoView(),
            SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
            this,
            SLOT(slotRepoContextMenu(QListViewItem *, const QPoint &, int)));

    m_yumConf.loadAll();

    //
    // If there are repository definitions in yum.conf, offer the user
    // to have them moved to kyum.repo...
    //
    if ( 1 < m_yumConf.countGroups() )
    {
        QString strMsg = i18n("You seem to have repositories defined in yum.conf.\n"
                              "Unless you have an old version of yum, the preferred location\n"
                              "for defining your repositories is a *.repo file under\n"
                              "/etc/yum.repos.d.\n\n"
                              "Should kyum convert these entries to *.repo file items?");

        if ( 0 == QMessageBox::question(this, "KYum", strMsg, i18n("&Yes"), i18n("&No")) )
        {
            QFile confFile(KYumPreferences::getYumConfPath());

            if ( confFile.exists() && confFile.open(IO_ReadOnly) )
            {
                QFile confFileBak(confFile.name() + ".old");

                if ( confFileBak.exists() )
                {
                    int num = 1;

                    do {
                        confFileBak.setName(confFile.name() + ".old" + QString::number(num++));
                    }
                    while ( confFileBak.exists() );
                }

                if ( confFileBak.open(IO_WriteOnly) )
                {
                    QString strLine;
                    QTextStream out(&confFileBak);

                    while ( !confFile.atEnd() )
                    {
                        confFile.readLine(strLine, 2048);
                        out << strLine;
                    }

                    confFileBak.close();

                    QString strMsg;

                    strMsg  = i18n("Your old yum.conf has been backed up as ");
                    strMsg += confFileBak.name();

                    QMessageBox::information(this, "KYum", strMsg);
                }

                confFile.close();
            }

            m_yumConf.moveReposToRepoFile();
            m_yumConf.saveAll();

            getYumConfView()->configLoaded();
        }
    }

    m_pActionsTab->getYumLogView()->startThread();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

KYumView::~KYumView()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool KYumView::loadYumConf()
{
    return m_yumConf.loadMainCfg();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool KYumView::saveRepo(Repo::Ptr ptrRepo)
{
    return m_yumConf.saveRepo(ptrRepo);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::print(QPainter * pPainter, int height, int width)
{
    // do the actual printing, here
    // pPainter->drawText(etc..)
    (void) pPainter;
    (void) height;
    (void) width;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::slotSetTitle(const QString & title)
{
    emit signalChangeCaption(title);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::slotYumConfSaved(const QString & /*strConf*/)
{
    //
    // Reload the changed file, so we'll deal with the correct values
    //
    loadYumConf();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::slotTabChanged(QWidget * pNewTab)
{
    if ( m_pCurrentTab == m_pYumConfTab && getYumConfView()->isDirty() )
    {
        //getYumConfView()->saveConfig();
    }

    m_pCurrentTab = pNewTab;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::slotRepoSelected(QListViewItem * pItem)
{
    if ( pItem )
    {
        QString             strFileName;
        Repo::Ptr ptrRepo = m_yumConf.findRepo(pItem->text(0), &strFileName);

        if ( ptrRepo.isValid() )
        {
            QCheckListItem * pCheckItem = dynamic_cast<QCheckListItem *>(pItem);

            if ( pCheckItem )
            {
                if ( ptrRepo->isEnabled() != pCheckItem->isOn() )
                {
                    ptrRepo->setEnabled(pCheckItem->isOn());
                    m_yumConf.saveRepo(ptrRepo);
                }
            }

            getOutputView()->setText(ptrRepo->toString());
        }
        else
        {
            getOutputView()->setText("<not found>"); // This should never happen!
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::slotEditRepo(QListViewItem * pItem, const QPoint &, int)
{
    QString         strRepo     = pItem->text(0);
    Repo::Ptr       ptrRepo     = m_yumConf.findRepo(strRepo);
    RepoFile::Ptr   ptrRepoFile = m_yumConf.getFileWithRepo(ptrRepo);

    if ( ptrRepo.isValid() )
    {
        Repo        newRepo(*ptrRepo.getPtr());
        DlgEditRepo dlg(this, i18n("Edit Repository"), newRepo);

        if ( QDialog::Accepted == dlg.exec() )
        {
            QCheckListItem * pCheckItem = dynamic_cast<QCheckListItem *>(pItem);

            //
            // Same tag and *.repo file as before?
            //
            if ( strRepo == newRepo.getTag()
                    && ptrRepo->getFileName() == newRepo.getFileName() )
            {
                ptrRepo->copyFrom(newRepo);

                getOutputView()->setText(ptrRepo->toString());
                pCheckItem->setOn(ptrRepo->isEnabled());

                m_yumConf.saveRepo(ptrRepo);
                getRepoView()->repoUpdated(strRepo, ptrRepo);
            }
            //
            // No, new tag or different file name...
            //
            else
            {
                QString strFileName = newRepo.getFileName();

                //
                // Get rid of the old repository...
                //
                m_yumConf.deleteRepo(ptrRepo);

                //
                // ...and add the new one...
                //
                ptrRepo->copyFrom(newRepo);
                m_yumConf.addRepo(ptrRepo, strFileName);

                //
                // ... then update the views accordingly...
                //
                getRepoView()->changeItem(pCheckItem, ptrRepo);
                getOutputView()->setText(ptrRepo->toString());
            }
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::addRepo(Repo::Ptr ptrRepo, QString strFileName)
{
    getRepoView()->addRepo(ptrRepo, strFileName);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void KYumView::slotRepoContextMenu (QListViewItem * pItem,
                                    const QPoint &  pos,
                                    int             column)
{
    QPopupMenu * pPopup = new QPopupMenu(this);

    enum
      {
        c_id_Edit,
        c_id_Add,
        c_id_Delete,
        c_id_ImportGPG
      };

    pPopup->insertItem(i18n("&New Repository..."), c_id_Add);

    //
    // The following menu item are only valid if the mouse is over a
    // repository item.
    //
    if ( pItem )
    {
        pPopup->insertItem(i18n("&Edit Properties..."), c_id_Edit);
        pPopup->insertItem(i18n("&Delete"), c_id_Delete);
    }

    pPopup->insertItem(i18n("&Import public key..."), c_id_ImportGPG);

    switch ( pPopup->exec(pos) )
    {
      case c_id_ImportGPG:
                        {
                            QString strKeyPath;

                            if ( pItem )
                            {
                                QString   strRepo = pItem->text(0);
                                Repo::Ptr ptrRepo = m_yumConf.findRepo(strRepo);

                                if ( ptrRepo.isValid() )
                                {
                                    strKeyPath = ptrRepo->getValue("gpgkey");

                                    if ( strKeyPath.isEmpty() )
                                    {
                                        strKeyPath = ptrRepo->getBaseURL().url();

                                        if ( strKeyPath.isEmpty() )
                                            strKeyPath = ptrRepo->getMirrorList();
                                    }
                                }
                            }

                            DlgImportGPG(this, strKeyPath).exec();
                        }
                        break;

      case c_id_Edit:   slotEditRepo(pItem, pos, column);
                        break;

      case c_id_Add:    emit signalContextAddRepo();
                        break;

      case c_id_Delete: {
                            QString strMsg;

                            strMsg.sprintf(i18n("Repository [%s] will be deleted. Are you sure?"),
                                          pItem->text(0).ascii());

                            if ( 0 == QMessageBox::question(this,
                                                            "KYum",
                                                            strMsg,
                                                            i18n("Yes"),
                                                            i18n("No")) )
                            {
                                getRepoView()->deleteItem(pItem);

                                //
                                // Why isn't the respective signal fired automatically when
                                // selection changes due to a deleted item???
                                //
                                pItem = getRepoView()->currentItem();
                                if ( pItem )
                                    slotRepoSelected(pItem);
                            }
                        }
                        break;

      default:
                        break;
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool KYumView::saveAll()
{
    return (!getYumConfView()->isDirty() || m_yumConf.saveMainCfg())
               && m_yumConf.saveRepos();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

KYumView * KYumView::getKYumView()
{
    return g_pYumView;
}


#include "KYumView.moc"
