/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#ifndef OptionsWidget_H_Included
#define OptionsWidget_H_Included

#include <qlineedit.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qtable.h>

#include "AutoPtr.h"

class QLabel;
class QGridLayout;

class CfgGroup;

class OptionEditor : public QObject
  {
    Q_OBJECT

    public:
      typedef AutoPtr<OptionEditor> Ptr;

    private:
      QWidget *         m_pWidget;
      bool              m_bDirty;

    protected:
                        OptionEditor(QWidget * pWidget)
                          : m_pWidget(pWidget),
                            m_bDirty(false)
                        {
                        }

    public:
      virtual void      setOptionValue(const QString & strValue) = 0;
      virtual QString   getOptionValue() const = 0;

      const QWidget *   getWidget() const
                        {
                            return m_pWidget;
                        }

      QWidget *         getWidget()
                        {
                            return (QWidget *) ((const OptionEditor *) this)->getWidget();
                        }

    signals:
      void              optionChanged (const OptionEditor * pEditor, const QString & strValue);
      void              activated     (const OptionEditor * pEditor);

    protected slots:
      void              slotLostFocus ();
      void              slotGotFocus  ();

      virtual void      slotOptionChanged(const QString & /*strValue*/)
                        {
                            slotTextChanged();
                        }

     virtual void       slotTextChanged()
                        {
                            m_bDirty = true;
                        }
  };


//
//
//
class Option
  {
    public:
      typedef AutoPtr<Option> Ptr;
      typedef std::list<Ptr>  List;

    private:
      QString           m_strLabel,
                        m_strDefault,
                        m_strValue,
                        m_strComment;

    public:
                        Option(QString strLabel,
                               QString strDefault = "",
                               QString strComment = "");

      virtual         ~ Option () {}

      virtual QString   convertValueForEditor(QString strValue) const
                        {
                            return strValue;
                        }

      virtual QString   convertValueFromEditor(QString strValue) const
                        {
                            return strValue;
                        }

      void              setValue(QString strValue)
                        {
                            m_strValue = strValue;
                        }

      QString           getLabel    () const { return m_strLabel; }
      QString           getDefault  () const { return m_strDefault; }
      QString           getValue    () const { return m_strValue; }
      QString           getComment  () const { return m_strComment; }

      virtual
        OptionEditor *  createEditor(QWidget * pParent) const = 0;
  };



//
//
//
class StringOptionEditor : public OptionEditor
  {
    public:
                StringOptionEditor(QWidget    * pParent,
                                   const char * pName   = 0,
                                   QString      strMask = "");

      void      setOptionValue(const QString & strValue);
      QString   getOptionValue() const;
  };


class StringOption : public Option
  {
    private:
      QString           m_strInputMask;

    public:
                        StringOption(QString strLabel,
                                     QString strDefault   = "",
                                     QString strInputMask = "",
                                     QString strComment   = "")

                          : Option(strLabel, strDefault, strComment),
                            m_strInputMask(strInputMask)
                        {
                        }

      OptionEditor *    createEditor(QWidget * pParent) const
                        {
                            return new StringOptionEditor(pParent, 0, m_strInputMask);
                        }

      virtual QString   convertValueForEditor (QString strValue) const;
      virtual QString   convertValueFromEditor(QString strValue) const;
  };


//
//
//
class IntOption : public StringOption
  {
    public:
                        IntOption(QString strLabel,
                                  QString strDefault   = "",
                                  QString strInputMask = "",
                                  QString strComment   = "")

                          : StringOption(strLabel, strDefault, strInputMask, strComment)
                        {
                        }

      virtual QString   convertValueForEditor (QString strValue) const;
      virtual QString   convertValueFromEditor(QString strValue) const;
  };


//
//
//
class TextOptionEditor : public OptionEditor
  {
    public:
                TextOptionEditor(QWidget * pParent, const char * pName = 0);

      void      setOptionValue(const QString & strValue);
      QString   getOptionValue() const;
  };


//
//
//
class TextOption : public Option
  {
    public:
                        TextOption(QString strLabel,
                                   QString strDefault = "",
                                   QString strComment = "")

                          : Option(strLabel, strDefault, strComment)
                        {
                        }

      OptionEditor *    createEditor(QWidget * pParent) const
                        {
                            return new TextOptionEditor(pParent);
                        }

      virtual QString   convertValueForEditor (QString strValue) const;
      virtual QString   convertValueFromEditor(QString strValue) const;
  };


//
//
//
class BoolOptionEditor : public OptionEditor
  {
    public:
                BoolOptionEditor(QWidget * pParent, const char * pName = 0);

      void      setOptionValue(const QString & strValue);
      QString   getOptionValue() const;

    protected:
      void      slotOptionChanged(const QString & strValue)
                {
                    OptionEditor::slotOptionChanged(strValue);
                    emit OptionEditor::slotLostFocus();
                }
  };


//
//
//
class BoolOption : public Option
  {
    public:
                        BoolOption(QString strLabel,
                                   QString strDefault = "",
                                   QString strComment = "")

                          : Option(strLabel, strDefault, strComment)
                        {
                        }

      OptionEditor *    createEditor(QWidget * pParent) const
                        {
                            return new BoolOptionEditor(pParent);
                        }

      virtual QString   convertValueForEditor (QString strValue) const;
      virtual QString   convertValueFromEditor(QString strValue) const;
  };


//
//
//
class FileOption : public StringOption
  {
    public:
                        FileOption(QString strLabel,
                                   QString strDefault = "",
                                   QString strComment = "")

                          : StringOption(strLabel, strDefault, "", strComment)
                        {
                        }

      OptionEditor *    createEditor(QWidget * pParent) const;

//      virtual QString   convertValueForEditor (QString strValue) const;
//      virtual QString   convertValueFromEditor(QString strValue) const;
  };


//
//
//
class DirOption : public StringOption
  {
    public:
                        DirOption(QString strLabel,
                                  QString strDefault = "",
                                  QString strComment = "");

      OptionEditor *    createEditor(QWidget * pParent) const;

      virtual QString   convertValueForEditor (QString strValue) const;
      virtual QString   convertValueFromEditor(QString strValue) const;

      static QString    adjustPath(QString strDir);
  };


//
//
//
class OptionsWidget : public QWidget
  {
    Q_OBJECT

    private:
      struct OptionInfo
        {
          typedef AutoPtr<OptionInfo> Ptr;
          typedef std::list<Ptr>      List;

          Option::Ptr         m_ptrOption;
          OptionEditor::Ptr   m_ptrEditor;

          OptionInfo(Option::Ptr ptrOption, OptionEditor::Ptr ptrEditor)
          {
              m_ptrOption = ptrOption;
              m_ptrEditor = ptrEditor;
          }
        };

      typedef std::map<const OptionEditor *, OptionInfo::Ptr>  EditorMap;


    private:

      OptionInfo::List  m_optInfos;
      EditorMap         m_editorMap;
      QGridLayout     * m_pTopLayout;


    private:
      void          addOption     (Option::Ptr ptrOption);


    public:
                    OptionsWidget (const Option::List  & options,
                                   QWidget             * pParent = 0,
                                   const char          * pName   = 0);

      virtual     ~ OptionsWidget ();


      void          setOption         (QString strKey, QString strValue);
      void          setOptions        (const CfgGroup * pGroup);
      void          setDefaultOptions ();


    signals:
      void          optionChanged     (const QString & strLabel, const QString & strValue);
      void          optionHelp        (const QString & strHelp);


    protected slots:
      void          slotOptionChanged (const OptionEditor * pEditor, const QString & strValue);
      void          slotEditorFocus   (const OptionEditor * pEditor);
  };

#endif
