# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gst
import gtk

from player import Player
from plugins.generic import GenericPlugin

import utils
from config import config
from action_manager import ActionManager
from utils import odict
from logger import Logger
from xdg_support import get_xdg_pixmap_file

COMBO_EMPTY_TEXT = "-----------"
COMBO_MANDATORY = odict()
COMBO_MANDATORY[_("Disabled")] = _("Disabled") + "<###>0:0:0:0:0:0:0:0:0:0"
COMBO_MANDATORY[_("Classical")] = _("Classical") + "<###>0.0:0.0:0.0:0.0:0.0:0.0:-7.2:-7.2:-7.2:-9.6"
COMBO_MANDATORY[_("Club")] = _("Club") + "<###>0.0:0.0:8:5.6:5.6:5.6:3.2:0.0:0.0:0.0"
COMBO_MANDATORY[_("Dance")] = _("Dance") + "<###>9.6:7.2:2.4:0.0:0.0:-5.6:-7.2:-7.2:0.0:0.0"
COMBO_MANDATORY[_("Full Bass")] = _("Full Bass") + "<###>-8:9.6:9.6:5.6:1.6:-4:-8:-10.4:-11.2:-11.2"
COMBO_MANDATORY[_("Full Bass and Treble")] = _("Full Bass and Treble") + "<###>7.2:5.6:0.0:-7.2:-4.8:1.6:8:11.2:12:12"
COMBO_MANDATORY[_("Full Treble")] = _("Full Treble") + "<###>-9.6:-9.6:-9.6:-4:2.4:11.2:16:16:16:16.8"
COMBO_MANDATORY[_("Laptop Speakers/Headphones")] = _("Laptop Speakers/Headphones") + "<###>4.8:11.2:5.6:-3.2:-2.4:1.6:4.8:9.6:12.8:14.4"
COMBO_MANDATORY[_("Large Hall")] = _("Large Hall") + "<###>10.4:10.4:5.6:5.6:0.0:-4.8:-4.8:-4.8:0.0:0.0"
COMBO_MANDATORY[_("Live")] = _("Live") + "<###>-4.8:0.0:4:5.6:5.6:5.6:4:2.4:2.4:2.4"
COMBO_MANDATORY[_("Party")] = _("Party") + "<###>7.2:7.2:0.0:0.0:0.0:0.0:0.0:0.0:7.2:7.2"
COMBO_MANDATORY[_("Pop")] = _("Pop") + "<###>-1.6:4.8:7.2:8:5.6:0.0:-2.4:-2.4:-1.6:-1.6"
COMBO_MANDATORY[_("Reggae")] = _("Reggae") + "<###>0.0:0.0:0.0:-5.6:0.0:6.4:6.4:0.0:0.0:0.0"
COMBO_MANDATORY[_("Rock")] = _("Rock") + "<###>8:4.8:-5.6:-8:-3.2:4:8.8:11.2:11.2:11.2"
COMBO_MANDATORY[_("Ska")] = _("Ska") + "<###>-2.4:-4.8:-4:0.0:4:5.6:8.8:9.6:11.2:9.6"
COMBO_MANDATORY[_("Soft")] = _("Soft") + "<###>4.8:1.6:0.0:-2.4:0.0:4:8:9.6:11.2:12"
COMBO_MANDATORY[_("Soft Rock")] = _("Soft Rock") + "<###>4:4:2.4:0.0:-4:-5.6:-3.2:0.0:2.4:8.8"
COMBO_MANDATORY[_("Techno")] = _("Techno") + "<###>8:5.6:0.0:-5.6:-4.8:0.0:8:9.6:9.6:8.8"
COMBO_MANDATORY[COMBO_EMPTY_TEXT] = COMBO_EMPTY_TEXT + "<###>0.0:0.0:0.0:0.0:0.0:0.0:0.0:0.0:0.0:0.0"

class EqualizerDialog(gtk.Dialog, Logger):
    def __init__(self):
            
        title = _("Equalizer")
        dialog_opts = gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT
        #dialog_opts = gtk.DIALOG_DESTROY_WITH_PARENT
        buttons = (gtk.STOCK_CLOSE, gtk.RESPONSE_REJECT,)

        super(EqualizerDialog, self).__init__(title, utils.get_main_window(), dialog_opts, buttons)

        self.set_property('skip-taskbar-hint', True)


        self.__btnsave = gtk.Button()
        self.__btnsave.add(gtk.image_new_from_stock(gtk.STOCK_SAVE, gtk.ICON_SIZE_LARGE_TOOLBAR))
        self.__btnsave.connect("clicked", self.__save)
        self.__btndelete = gtk.Button()
        self.__btndelete.add(gtk.image_new_from_stock(gtk.STOCK_DELETE, gtk.ICON_SIZE_LARGE_TOOLBAR))
        self.__btndelete.connect("clicked", self.__delete)
        #btnimport = gtk.Button(stock=gtk.STOCK_IMPORT)

        self.__combo = gtk.combo_box_entry_new_text()
        for name in COMBO_MANDATORY.keys():
            self.__combo.append_text(name)
        opts = config.options("plugins")
        for opt in opts:
            if opt.find("equalizer-pref-") == 0:
                name = config.get("plugins", opt).split("<###>")[0]
                self.__combo.append_text(name)

        self.__combo.set_active(len(COMBO_MANDATORY.keys()) - 1)
        self.__select_name()
        self.__combo.connect("changed", self.__change)
        self.__combo.child.connect("changed", self.__edit_check)
        
        vboxcombo = gtk.VBox()
        vboxcombo.pack_start(self.__combo, True, False)

        top_box = gtk.HBox()
        top_box.set_property("border-width", 6)
        top_box.pack_start(vboxcombo, True, True)
        top_box.pack_start(self.__btnsave, False, False)
        top_box.pack_start(self.__btndelete, False, False)

        self.hbox = gtk.HBox(True, 3)
        self.hbox.set_property("border-width", 6)
    
        self.vbox.set_property("border-width", 12)
        self.vbox.pack_start(top_box, False, False)
        self.vbox.pack_start(self.hbox, True, True)

        labels = {
                0:_("20Hz"),
                1:_("45Hz"),
                2:_("90Hz"),
                3:_("200Hz"),
                4:_("430Hz"),
                5:_("930Hz"),
                6:_("2KHz"),
                7:_("4.3KHz"),
                8:_("9.3KHz"),
                9:_("20KHz")
                }

        self.__scales = {}
        for i in range(0, 10):
            try : value = float(config.get("plugins", "equalizer-band%s" % str(i)))
            except : value = 0.0
            adjust = gtk.Adjustment(value=value, lower= -24, upper=12, step_incr=0.1, page_incr=1, page_size=0)
            adjust.changed_id = adjust.connect("value-changed", self.__on_adjust_change, i)
            vs = gtk.VScale(adjust)
            vs.set_draw_value(False)
            vs.set_inverted(True)
            vs.set_update_policy(gtk.UPDATE_CONTINUOUS)
            self.__scales[i] = vs
            bvs = gtk.VBox()
            bvs.pack_start(vs, True, True)
            label = gtk.Label("<small>" + utils.xmlescape(labels[i]) + "</small>")
            label.set_use_markup(True)
            bvs.pack_start(label, False, False)
            self.hbox.pack_start(bvs, True, True)

        db = gtk.Label("<small>" + utils.xmlescape(_("12 dB\n\n\n0dB\n\n\n\n\n-24 dB")) + "</small>")
        db.set_use_markup(True)

        db_empty = gtk.Label("<small>  </small>")
        db_empty.set_use_markup(True)

        bvs = gtk.VBox()
        bvs.pack_start(db, True, True)
        bvs.pack_start(db_empty, True, True)
 
        self.hbox.pack_start(bvs, True, True)
    
        self.__edit_check()
        self.show_all()
        self.run()
        self.destroy()

    def __on_adjust_change(self, adjust, i):
        config.set("plugins", "equalizer-band%s" % str(i), str(adjust.get_value()))
        self.__select_name()

    def __select_name(self):
        values = []
        for i in range(0, 10):
            try : value = int(float(config.get("plugins", "equalizer-band%s" % str(i))))
            except : value = 0
            values.append(str(value))
        values = ":".join(values)

        for name, value in COMBO_MANDATORY.iteritems():
            name = value.split("<###>")[0]
            value = value.split("<###>")[1].split(":")
            value = ":".join([ str(int(float(v))) for v in value])
            if value == values:
                self.__combo.child.set_text(name)
                self.__edit_check()
                return

        for opt in config.options("plugins"):
            if opt.find("equalizer-pref-") == 0:
                name = config.get("plugins", opt).split("<###>")[0]
                value = config.get("plugins", opt).split("<###>")[1].split(":")
                value = ":".join([ str(int(float(v))) for v in value])
                if value == values:
                    self.__combo.child.set_text(name)
                    self.__edit_check()
                    return

        self.__combo.child.set_text(COMBO_EMPTY_TEXT)
        self.__edit_check()

    def __save(self, *args):
        text = self.__combo.get_active_text()
        if text in COMBO_MANDATORY.keys(): return
        values = []
        for i in range(0, 10):
            try : value = float(config.get("plugins", "equalizer-band%s" % str(i)))
            except : value = 0.0
            values.append(str(value))
        self.logdebug("Try to save pref %s with %s", text, ":".join(values))
        need_append = not config.has_option("plugins", "equalizer-pref-" + text)
        try: 
            config.set("plugins", "equalizer-pref-" + text, text + "<###>" + ":".join(values))
        except:
            self.logexception("Failed to save pref %s with %s", text, ":".join(values))
        else:
            if need_append:
                self.__combo.append_text(text)
                self.__combo.show_all()


    def __delete(self, *args):
        text = self.__combo.get_active_text()
        if text in COMBO_MANDATORY.keys(): return
        try: config.remove_option("plugins", "equalizer-pref-" + text)
        except: pass
        else:
            for i, name in enumerate(self.__combo.get_model()):
                if name[0] == text: 
                    self.__combo.remove_text(i)
                    break

    def __change(self, *args):
        text = self.__combo.get_active_text()
        if text == COMBO_EMPTY_TEXT: return
        if text in COMBO_MANDATORY.keys():
            values = COMBO_MANDATORY[text]
            self.__btndelete.set_sensitive(False)
            self.__btnsave.set_sensitive(False)
        else:
            try: 
                values = config.get("plugins", "equalizer-pref-" + text)
                self.__btndelete.set_sensitive(True)
                self.__btnsave.set_sensitive(True)
            except: 
                self.__combo.set_active(len(COMBO_MANDATORY.keys()) - 1)
                return
        values = values.split("<###>")[1].split(":")
        for i, value in enumerate(values):
            adj = self.__scales[i].get_adjustment()
            adj.handler_block(adj.changed_id)
            self.__scales[i].set_value(float(value))
            config.set("plugins", "equalizer-band%s" % str(i), str(adj.get_value()))
            adj.handler_unblock(adj.changed_id)

    def __edit_check(self, *args):
        text = self.__combo.get_active_text()
        if text in COMBO_MANDATORY.keys():
            self.__btndelete.set_sensitive(False)
            self.__btnsave.set_sensitive(False)
        else:
            self.__btndelete.set_sensitive(True)
            self.__btnsave.set_sensitive(True)
 

class Equalizer(GenericPlugin):
    PLUGIN_NAME = "Equalizer"
    PLUGIN_DESC = "Equalizer for listen"
    PLUGIN_VERSION = "0.1"
    PLUGIN_AUTHOR = "Mehdi ABAAKOUK <theli48@gmail.com>"
    PLUGIN_WEBSITE = "http://www.listen-project.org"

   
    def __init__(self):
        super(Equalizer, self).__init__()
        try: 
            self.__equalizer = gst.element_factory_make("equalizer-10bands")
        except gst.PluginNotFoundError:
            self.__equalizer = None
            self.logerror("Equalizer support requires gstreamer-plugins-bad (>= 0.10.5)")
        else:
            Player.bin.xfade_add_filter(self.__equalizer)
            for i in range(0, 10):
                try : value = float(config.get("plugins", "equalizer-band%s" % str(i)))
                except : value = 0
                self.__equalizer.set_property("band" + str(i), float(value))
            
            factory = gtk.IconFactory()
            pb = gtk.gdk.pixbuf_new_from_file(get_xdg_pixmap_file("equalizer.png"))
            factory.add("equalizer", gtk.IconSet(pb))
            factory.add_default()

            ActionManager.register_action("equalizer", 1, None, ("equalizer", self.open_equalizer), pack_end=True)
            self.autoconnect(Player.bin, "tee-removed", self.__on_remove)
            self.autoconnect(config, "config-changed", self.__on_config_change)

    def __on_config_change(self, dispacher, section, option, value):
        if section == "plugins" and option.find("equalizer-band") == 0:
            band_name = option.replace("equalizer-", "")
            self.__equalizer.set_property(band_name, float(value))

    def __on_remove(self, bin, tee, element):
        if element != self.__equalizer: return
        self.__equalizer.set_state(gst.STATE_NULL)
        self.__equalizer = None
        super(Equalizer, self).destroy()

    def delete_thyself(self):
        if self.__equalizer:
            Player.bin.xfade_remove_filter(self.__equalizer)
            ActionManager.unregister_action("equalizer")
        super(Equalizer, self).delete_thyself()

    def open_equalizer(self, *args):
        EqualizerDialog()



