# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

####
# Parts from "Magnatune Rhythmbox plugin" (stolen from rhythmbox's MagnatuneSource.py)
#     Copyright (C), 2006 Adam Zimmerman <adam_zimmerman@sfu.ca>
#
# Parts from "Jamendo Rhythmbox plugin" (stolen from rhythmbox's JamendoSource.py:)
#     Copyright (C) 2007 - Guillaume Desmottes
###

import os
import gtk.glade
import pango
import gobject
import xml.sax
import gzip

import utils
import stock

from source import Source, SourceItem
from widget.song_view import SortableSongView
from widget.song_menu import SongMenu, SongMenuManager
from widget.browser import Browser
from xdg_support import get_xdg_config_file, get_xdg_config_dir, get_xdg_data_file

from helper import Dispatcher

from library import ListenDB, ListenDBQuery

from MagnatuneSaxHandler import MagnatuneSaxHandler

import vfs

import gnomevfs
# URIs
magnatune_song_info_uri = gnomevfs.URI("http://www.magnatune.com/info/song_info_xml.gz")
local_song_info_uri = gnomevfs.URI(get_xdg_config_file("magnatune-dbdump.en.xml"))
local_song_info_temp_uri = gnomevfs.URI(get_xdg_config_file("magnatune-dbdump.en.xml.tmp"))

#artwork_url = "http://www.magnatune.com/get/album/id/album/artworkurl/redirect/%s/?artwork_size=200"


magnatune_url = "http://img.magnatune.com/data/dbdump.en.xml.gz"

class MagnatuneBrowser(Browser):
    def __init__(self):
        tree_songs = SortableSongView("magnatune")
        tree_songs.set_menu(SongMenu())

        Browser.__init__(self, ListenDBQuery("magnatune"), tree_songs, "magnatune", True)
        self.set_menu(SongMenu(for_browser=True))
        
class MagnatuneItem(SourceItem):
    widget_klass = MagnatuneBrowser
    label = _("Magnatune")
    config_code = "magnatunelibrary"
    stock = stock.SRC_MAGNATUNE

class MagnatuneLoadingItem(SourceItem):
    widget = gtk.VBox()
    label = _("Magnatune")
    config_code = "magnatunelibrary"
    stock = stock.SRC_MAGNATUNE
    def __init__(self):
        message = gtk.glade.XML(get_xdg_data_file("magnatune-loading.glade"), root="magnatune_loading_vbox").get_widget("magnatune_loading_vbox")

        self.btn_connect = gtk.Button(stock=gtk.STOCK_CONNECT)
        self.btn_connect.show_all()
        self.btn_connect.set_no_show_all(True)

        self.progress = gtk.ProgressBar()
        self.progress.set_text("")
        self.progress.set_fraction(0)
        self.progress.set_pulse_step(0.1)
        self.progress.set_ellipsize(pango.ELLIPSIZE_END)

        hbox = gtk.HBox(spacing=6)
        hbox.pack_start(self.progress, True, True)
        hbox.pack_start(self.btn_connect, False, False)
        self.widget.pack_start(message, True, True)
        self.widget.pack_start(hbox, False, False)


class MagnatuneSource(Source):
    PLUGIN_NAME = "Magnatune"
    PLUGIN_DESC = "Allow browse music from magnatune"

    has_top_separateur = True
    categorie = "shop"
    
    def __init__(self):
        Source.__init__(self)
        self.items = [ MagnatuneLoadingItem() ]
        ListenDB.register_type("magnatune")
        for menuitem in [ "play", "enqueue", "lastfm" ]:
            SongMenuManager.allow_item_for_type(menuitem, "magnatune")

        self.__p2plinks = {}
        self.__album_id = {}

        # catalogue stuff
        self.__db = None
        self.__saxHandler = None
        self.__activated = False
        self.__notify_id = 0
        self.__update_id = 0
        self.__xfer_handle = None
        self.__info_screen = None
        self.__browser = self.items[0].widget
        self.__updating = True
        self.__load_handle = None
        self.__load_current_size = 0
        self.__load_total_size = 0

        self.items[0].btn_connect.connect("clicked", self.start)

    def start(self, widget):
        self.items[0].btn_connect.hide()
        self.items[0].progress.set_text("Downloading...")
        self.__activated = True
        self.__show_loading_screen (True)
        #self.__load_catalogue()

        # start our catalogue updates
        self.__update_id = gobject.timeout_add(6 * 60 * 60 * 1000, self.__update_catalogue)
        self.__update_catalogue()

    def save(self):
        try: self.items[0].widget.save_config()
        except: pass

    def delete_thyself(self):
        if self.__update_id != 0:
            gobject.source_remove (self.__update_id)
            self.__update_id = 0

        if self.__notify_id != 0:
            gobject.source_remove (self.__notify_id)
            self.__notify_id = 0

        if self.__xfer_handle is not None:
            self.__xfer_handle.cancel()
            self.__xfer_handle = None
        ListenDB.unregister_type("magnatune")
        super(MagnatuneSource, self).delete_thyself()

    #
    # internal catalogue downloading and loading
    #
    def __load_catalogue_read_cb (self, handle, data, exc_type, bytes_requested, parser):
        if exc_type:
            if issubclass (exc_type, gnomevfs.EOFError):
                def finish_loadscreen():
                    # successfully loaded
                    gtk.gdk.threads_enter()
                    self.__load_db ()
                    self.__show_loading_screen (False)

                    in_progress_dir = gnomevfs.DirectoryHandle(gnomevfs.URI(get_xdg_config_dir("magnatune")))
                    in_progress = in_progress_dir.next()
                    while True:
                        if in_progress.name[0:12] == "in_progress_":
                            in_progress = vfs.read_entire_file(get_xdg_config_file(os.path.join("magnatune" + in_progress.name)))
                            for uri in in_progress.split("\n"):
                                if uri == '':
                                    continue
                                self.__download_album(gnomevfs.URI(uri))
                        try:
                            in_progress = in_progress_dir.next()
                        except:
                            break
                    gtk.gdk.threads_leave()
                gobject.idle_add (finish_loadscreen)
            else:
                # error reading file
                raise exc_type

            parser.close()
            handle.close(lambda handle, exc: None) # FIXME: report it?
            self.__load_handle = None
            self.__updating = False
        else:

            parser.feed(data)
            handle.read(64 * 1024, self.__load_catalogue_read_cb, parser)

    def __load_catalogue_open_cb (self, handle, exc_type):
        if exc_type:
            self.__load_handle = None

            if gnomevfs.exists(local_song_info_uri):
                raise exc_type
            else:
                return

        self.items[0].progress.set_text("Read data...")
        parser = xml.sax.make_parser()
        self.__saxHandler = MagnatuneSaxHandler()
        parser.setContentHandler(self.__saxHandler)
        handle.read (64 * 1024, self.__load_catalogue_read_cb, parser)

    def __load_catalogue(self):
        self.__load_handle = gnomevfs.async.open (local_song_info_uri, self.__load_catalogue_open_cb)


    def __download_update_cb (self, _reserved, info, moving):
        self.__load_current_size = float(info.bytes_copied)
        self.__load_total_size = float(info.bytes_total)

        if info.phase == gnomevfs.XFER_PHASE_COMPLETED:
            self.__xfer_handle = None
            # done downloading, unzip to real location
            catalog = gzip.open(local_song_info_temp_uri.path)
            out = create_if_needed(local_song_info_uri, gnomevfs.OPEN_WRITE)
            out.write(catalog.read())
            out.close()
            catalog.close()
            gnomevfs.unlink(local_song_info_temp_uri)
            self.__updating = False
            self.__load_catalogue()
        else:
            print self.__load_current_size, self.__load_total_size
            if self.__load_total_size != 0:
                print float(self.__load_current_size / self.__load_total_size)
                self.items[0].progress.set_fraction(float(self.__load_current_size / self.__load_total_size))
        return 1

    def __download_catalogue(self):
        self.__updating = True
        create_if_needed(local_song_info_temp_uri, gnomevfs.OPEN_WRITE).close()
        self.__xfer_handle = gnomevfs.async.xfer (source_uri_list=[magnatune_song_info_uri],
                              target_uri_list=[local_song_info_temp_uri],
                              xfer_options=gnomevfs.XFER_FOLLOW_LINKS_RECURSIVE,
                              error_mode=gnomevfs.XFER_ERROR_MODE_ABORT,
                              overwrite_mode=gnomevfs.XFER_OVERWRITE_MODE_REPLACE,
                              progress_update_callback=self.__download_update_cb,
                              update_callback_data=False)

    def __update_catalogue(self):
        def info_cb (handle, results):
            (_remote_uri, remote_exc, remote_info) = results[0]
            (_local_uri, local_exc, local_info) = results[1]

            if remote_exc:
                # error locating remote file
                print "error locating remote catalogue", remote_exc
            elif local_exc:
                if issubclass (local_exc, gnomevfs.NotFoundError):
                    # we haven't got it yet
                    print "no local copy of catalogue"
                    self.__download_catalogue()
                else:
                    # error locating local file
                    print "error locating local catalogue", local_exc
                    self.__download_catalogue()
            else:
                try:
                    if remote_info.mtime > local_info.mtime:
                        # newer version available
                        self.__download_catalogue()
                    else:
                        print "catalogue up to date"
                        # up to date
                        self.__updating = False
                        self.__load_catalogue()
                except ValueError, e:
                    # couldn't get the mtimes. download?
                    print "error checking times", e
                    self.__download_catalogue()
            return

        gnomevfs.async.get_file_info ((magnatune_song_info_uri, local_song_info_uri), info_cb)

    def __show_loading_screen(self, show):
        print "__show_loading_screen(", show, "):"
        return 
        if self.__info_screen is None:
            # load the glade stuff
            gladexml = gtk.glade.XML(get_xdg_data_file("magnatune-loading.glade"), root="magnatune_loading_vbox")
            self.__info_screen = gladexml.get_widget("magnatune_loading_vbox")
            self.pack_start(self.__info_screen)
            self.get_entry_view().set_no_show_all (True)
            self.__info_screen.set_no_show_all (True)

        self.__info_screen.set_property("visible", show)
        self.items.widget.set_property("visible", not show)


    @utils.threaded
    def __load_db(self):
        self.items[0].progress.set_text("Import data...")
        try:
            tracks = self.__saxHandler.tracks
        except: 
            print "Failed loading catalogue"
            return

        i = 0
        for track in tracks:
            i += 1
            if track.get("uri"):
                ListenDB.get_or_create_song(track, "magnatune")

        print "loaded"
        self.__saxHandler = None
        # Load info in the browser
        gobject.idle_add(self.post_load)

    def post_load(self):
        load_item = self.items[0]
        load_item.widget.destroy()
        self.items = [ MagnatuneItem() ]
        self.items[0].widget.connect_to_db()
        Dispatcher.update_source()
        del load_item



def create_if_needed(uri, mode):
    if not gnomevfs.exists(uri):
        for directory in URIIterator(uri):
            if not gnomevfs.exists(directory):
                gnomevfs.make_directory(directory, 0755)
        out = gnomevfs.create(uri, open_mode=mode)
    else:
        out = gnomevfs.open(uri, open_mode=mode)
    return out

class URIIterator(object):
    def __init__(self, uri):
        self.uri_list = uri.dirname.split("/")[1:] # dirname starts with /
        self.counter = 0
    def __iter__(self):
        return self
    def next(self):
        if self.counter == len(self.uri_list) + 1:
            raise StopIteration
        value = "file://"
        for i in range(self.counter):
            value += "/" + self.uri_list[i]
        self.counter += 1
        return gnomevfs.URI(value)


