# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gtk
from hashlib import md5

import os
import stock
from config import config
from utils import get_main_window
from widget.misc import ScrolledWindow
from vfs import FileMonitor
from xdg_support import get_xdg_music_dir


class BrowserColumnDialog(gtk.Window):
    def __init__(self, conf_prefix):
        gtk.Window.__init__(self)
        self.conf_prefix = conf_prefix

        self.set_title(_("Preferences"))
        self.set_resizable(False)
        self.set_position(gtk.WIN_POS_CENTER)
        self.set_border_width(12)
        self.set_property("skip-taskbar-hint", False)
        #self.set_modal(True)
        self.set_transient_for(get_main_window())
        self.connect('delete-event', self.quit)


        fields = gtk.ListStore(str, str)
        for index in range(50):
            try: 
                tag = config.get("id3", "num" + str(index) + "_tag")
                label = config.get("id3", "num" + str(index) + "_label")
            except: pass 
            else: 
                fields.append((tag, label))

        vbox = gtk.VBox(False, 2)
        
        combo = gtk.ComboBox()
        model = gtk.ListStore(int, str)
        for row in range(4):
            model.append((row, str(row) + " " + _("Cols")))
        combo.set_model(model)
        cell = gtk.CellRendererText()
        combo.pack_start(cell)
        combo.add_attribute(cell, "text", 1)        
        vbox.pack_start(combo, False, False)

        for _col in range(int(config.get("browser", self.conf_prefix + "_num_cols"))):
            combo = gtk.ComboBox()
            combo.set_model(fields)
            cell = gtk.CellRendererText()
            combo.pack_start(cell)
            combo.add_attribute(cell, "text", 1)   
            vbox.pack_start(combo, False, False)
            

        self.add(vbox)
        self.show_all()

    def quit(self, *arg):
        self.destroy()
        
class HelperConfigureDialog(gtk.Window):
    def __init__(self, title):
        gtk.Window.__init__(self)
        self.set_title(title)
        self.set_resizable(False)
        self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        self.set_border_width(12)
        self.set_property("skip-taskbar-hint", True)
        self.set_modal(True)
        self.set_transient_for(get_main_window())

        btn = gtk.Button(stock=gtk.STOCK_CLOSE)
        btn.connect("clicked", self.on_destroy)
        box_btn = gtk.HBox()
        box_btn.pack_end(btn, False, False)

        self.vbox = gtk.VBox(spacing=6)

        gtk.Window.add(self, gtk.VBox(spacing=12))
        self.child.pack_start(self.vbox)
        self.child.pack_start(box_btn)

        self.connect('delete-event', self.on_destroy)
    
    def add(self, widget):
        self.vbox.pack_start(widget, False, False)

    def on_destroy(self, *args):
        config.write()
        self.destroy()

    def make_list_edit(self, section, attribut, label_new, label_header, small=False):
        def save():
            str = ""
            filters = [row[0] for row in self.model_filter]
            config.set(section, attribut, "<###>".join(filters))

        def on_edit(cellrenderertext, path, new_text):
            #old_text = self.model_filter[path][0]
            self.model_filter[path][0] = new_text.lower()
            save()

        box = gtk.HBox(False, 12)
        self.model_filter = gtk.ListStore(str)
        tree = gtk.TreeView(self.model_filter)
        cell = gtk.CellRendererText()
        cell.set_property("editable", True)
        cell.connect("edited", on_edit)
        tree.insert_column_with_attributes(0, label_header, cell, text=0)

        vbox = gtk.VBox(False, 6)
        if not small:
            btn_add = gtk.Button(stock=gtk.STOCK_ADD)
            btn_edit = gtk.Button(stock=gtk.STOCK_EDIT)
            btn_remove = gtk.Button(stock=gtk.STOCK_REMOVE)
        else:
            def get_btn(stock):
                btn = gtk.Button()
                btn.add(gtk.image_new_from_stock(stock, gtk.ICON_SIZE_BUTTON))
                btn.show_all()
                return btn
            btn_add = get_btn(gtk.STOCK_ADD)
            btn_edit = get_btn(gtk.STOCK_EDIT)
            btn_remove = get_btn(gtk.STOCK_REMOVE)

        def add(w):
            self.model_filter.append((label_new,))
            tree.set_cursor(len(self.model_filter) - 1, tree.get_column(0), True)
            
        def edit(w):
            model, iter = tree.get_selection().get_selected()    
            if iter:
                tree.set_cursor(model.get_path(iter), tree.get_column(0), True)
        
        def remove(w):
            _model, iter = tree.get_selection().get_selected()
            if iter:
                del self.model_filter[self.model_filter.get_path(iter)]
                save()

        btn_add.connect("clicked", add)
        btn_edit.connect("clicked", edit)
        btn_remove.connect("clicked", remove)

        vbox.pack_start(btn_edit, False, False)
        vbox.pack_start(btn_add, False, False)
        vbox.pack_start(btn_remove, False, False)

        box.pack_start(ScrolledWindow(tree), True, True)
        box.pack_start(vbox, False, False)
        filters = config.get(section, attribut).split("<###>")
        for filter in filters:
            self.model_filter.append((filter,))
        return box


    def get_tab(self, label, stock):
        tab = gtk.HBox(False, 6)
        image = gtk.Image()
        image.set_from_stock(stock, gtk.ICON_SIZE_MENU)
        tab.pack_start(image, False, False)
        tab.pack_start(gtk.Label(label), False, False)
        tab.show_all()
        return tab


    def get_title_label(self, str):
        label = gtk.Label("<b>" + str + "</b>")
        label.set_use_markup(True)
        label.set_alignment(0, 1)
        label.set_size_request(0, 22)
        return label
    
    def get_label(self, str):
        label = gtk.Label("<i>" + str + "</i>")
        label.set_use_markup(True)
        label.set_alignment(0, 1)
        #label.set_size_request(0,22)
        return label

    def make_label(self, string):
        label = gtk.Label(string)
        label.set_use_markup(True)
        label.set_alignment(0, 0.5)
        return label

    def make_color_btn(self, section, option, default=None):
        def on_change(w):
            color = w.get_color()
            config.set(section, option, ("#%2s%2s%2s" % (hex(color.red)[2:4], hex(color.green)[2:4], hex(color.blue)[2:4])).replace(" ", "0"))
        try: value = gtk.gdk.color_parse(config.get(section, option))
        except ValueError:
            value = gtk.gdk.color_parse(default)
        btn = gtk.ColorButton(value)
        btn.connect("color-set", on_change)
        return btn
        
    def make_number(self, label, label2, section, attribut, step=10, max=(24 * 60)):
        def save(spin):
            config.set(section, attribut, spin.get_value())
        b = gtk.SpinButton(gtk.Adjustment(float(config.get(section, attribut)), 0, max, step), 1.0)
        b.connect("value-changed", save)

        box = gtk.HBox(False, 6)
        box.pack_start(gtk.Label(label), False, False)
        box.pack_start(b, False, False)
        box.pack_start(gtk.Label(label2), False, False)
        box.get_children()[0].set_alignment(0, 0.5)
        box.get_children()[2].set_alignment(0, 0.5)
        return box

    def make_file(self, section, attribut, default=None):
        dialog = gtk.FileChooserDialog(title="Select a directory", parent=self, action=gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, buttons=(gtk.STOCK_OPEN, gtk.RESPONSE_ACCEPT,
                          gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL), backend=None)
        file = gtk.FileChooserButton(dialog)
        path = config.get(section, attribut, default)

        if not path:
            path = "~"
        path = os.path.expanduser(path)
        if not os.path.exists(path):
            try: 
                os.makedirs(path)
            except:
                path = os.path.expanduser("~")

        file.set_current_folder(path)
        def save(dialog, response):
            if response == gtk.RESPONSE_ACCEPT:
                config.set(section, attribut, file.get_filename())
        
        dialog.connect("response", save)
        def change(*param):
            config.set(section, attribut, file.get_filename())
        file.connect("selection-changed", change)
        return file

    def make_lmenu(self, label, section, attribut, values, default):
        m = self.make_menu(section, attribut, values, default)
        l = self.get_label(label + ":")
        l.set_alignment(0, 0.5)
        b = gtk.HBox(spacing=12)
        b.pack_start(l, True, True)
        b.pack_start(m, False, False)
        return b

    def make_menu(self, section, attribut, mydict, default):
        combo = gtk.combo_box_new_text()
        selected = config.get(section, attribut, default)
        for key in mydict.keys():
            combo.append_text(mydict[key])
            if selected == key:
                combo.set_active(len(combo.get_model()) - 1)

        def save(widget, section, attribut, mydict):
            text = widget.get_active_text()
            for key in mydict.keys():
                if mydict[key] == text:
                    config.set(section, attribut, key)
                    break

        combo.connect("changed", save, section, attribut, mydict)
        return combo

    def make_check_box(self, label, section, attribut):
        check = gtk.CheckButton(label)
        if config.get(section, attribut) == "true":
            check.set_property("active", True)

        def save_check(widget, sect, attr):
            if widget.get_property("active"):
                config.set(sect, attr, "true")
            else:
                config.set(sect, attr, "false")

        check.connect("toggled", save_check, section, attribut)
        return check

    def make_lentry(self, label, section, attribut, default, expand_entry = False):
        e = self.make_entry(label, section, attribut, default)
        l = self.get_label(label + ":")
        l.set_alignment(0, 0.5)
        b = gtk.HBox(spacing=12)
        if expand_entry:
            b.pack_start(l, False, False)
            b.pack_start(e, True, True)
        else:
            b.pack_start(l, True, True)
            b.pack_start(e, False, False)
        return b
        
    def make_entry(self, label, section, attribut, default=None):
        def save_entry(widget, sect, attr):
            config.set(sect, attr, widget.get_text())
        entry = gtk.Entry()
        entry.connect("changed", save_entry , section, attribut)
        entry.set_text(config.get(section, attribut, default))
        return entry

    def make_bar(self, section, attribut, lower, upper, step, page_incr=0, page_size=0):
        label = gtk.Label(config.get(section, attribut) + "s")
        value = float(config.get(section, attribut))
        bar = gtk.HScale(gtk.Adjustment(value, lower, upper, step, page_incr, page_size))
        bar.set_digits(0)
        bar.set_draw_value(False)
        def on_bar_change(bar, sect, attr, label):
            value = bar.get_value()
            value = round(value, 2)
            label.set_text("%02ss" % value)
            config.set(sect, attr, "%.2f" % value)
        bar.connect("value-changed", on_bar_change, section, attribut, label)
        box = gtk.HBox(False, 6)
        box.pack_start(bar)
        box.pack_start(label, False, False)
        box.set_spacing(12)
        return box

    def make_radio_list(self, section, attribut, listradio = [], boxklass=gtk.VBox):
        box = boxklass(False, 6)
        def on_change_radio(r, sect, attr, value):
            config.set(sect, attr, value)
        b = None
        for label, value in listradio:
            b = gtk.RadioButton(b, label)
            b.connect("toggled", on_change_radio, section, attribut, value)
            box.pack_start(b, False, False)
        return box

class PreferenceDialog(HelperConfigureDialog):

    def __init__(self,):
        HelperConfigureDialog.__init__(self, _("Preferences"))

        self.win_parent = get_main_window()
        self.osd = self.win_parent.osd

        self.monitoring_update = False

        vbox_general = gtk.VBox(False, 2)
        vbox_general.set_border_width(12)

    
        vbox_general.pack_start(self.get_title_label(_("General")), False, False)
        vbox_general.pack_start(self.make_check_box(_("Show current playing media in window title"), "setting", "update_title"), False, False)

        vbox_general.pack_start(self.get_title_label(_("Systray")), False, False)
        vbox_general.pack_start(self.make_check_box(_("Enable trayicon"), "setting", "use_trayicon"), False, False)
        vbox_general.pack_start(self.make_check_box(_("Minimized on startup"), "setting", "start_trayicon"), False, False)
        vbox_general.pack_start(self.make_check_box(_("Close to systray"), "setting", "fermer_trayicon"), False, False)
        vbox_general.pack_start(self.get_title_label(_("Browser view")), False, False)

        listradio = [
                (_('Artist and album'), 0), 
                (_('Genre and artist'), 1),
                (_('Genre, artist and album'), 2),
            ]
        vbox_general.pack_start(self.make_radio_list("browser","view", listradio), False, False)
        vbox_general.pack_start(self.get_title_label(_("Lyrics fonts and colors")), False, False)

        font = gtk.FontButton(config.get("setting", "web_font"))
        font.connect('font-set', self.font_set)
        vbox_general.pack_start(font, False, False)
        

        color_text = self.make_color_btn("setting", "web_color_text", "#000000")
        color_link = self.make_color_btn("setting", "web_color_link", "#0000FF")
        color_bg = self.make_color_btn("setting", "web_color_bg", "#FFFFFF")
        
        box_color = gtk.HBox(False, 6)
        box_color.pack_start(gtk.Label(_("Text color")))
        box_color.pack_start(color_text)
        box_color.pack_start(gtk.Label(_("Link color")))
        box_color.pack_start(color_link)
        box_color.pack_start(gtk.Label(_("Background color")))
        box_color.pack_start(color_bg)
        
        vbox_general.pack_start(box_color, False, False)
        
        vbox_playback = gtk.VBox(False, 2)
        vbox_playback.set_border_width(12)
        vbox_playback.pack_start(self.get_title_label(_("Playback")), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Play lastest played song on startup"), "player", "play_on_startup"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Enable crossfade"), "player", "crossfade"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Gapless instead of crossfade between track of same album"), "player", "crossfade_gapless_album"), False, False)
        try: 
            crossfade = float(config.get("player", "crossfade_time"))
        except: 
            pass
        else:
            if crossfade > 50: config.set("player", "crossfade_time", "3.5")

        vbox_playback.pack_start(self.make_bar("player", "crossfade_time", 0, 10, 1.0), False, False)

        vbox_playback.pack_start(self.get_title_label(_("Now playing")), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Playlist on the right (need restart)"), "window", "right_layout"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Enqueue as default action"), "player", "click_enqueue"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Play/Enqueue whole album as default action"), "player", "queuewholealbum"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Compact mode"), "setting", "compact_playlist"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Color by album"), "setting", "dynamic_playlist_color"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Don't clear when play a new song"), "player", "enqueue"), False, False)
        vbox_playback.pack_start(self.make_check_box(_("Play a random file when it is empty"), "setting", "empty_random"), False, False)
        

        vbox_library = gtk.VBox(False, 2)
        vbox_library.set_border_width(12)
        vbox_library.pack_start(self.get_title_label(_("Library location")), False, False)
        file = self.make_file("library", "location", get_xdg_music_dir())
        vbox_library.pack_start(file, False, False)

        def filechange(*param): self.monitoring_update = True
        file.connect("selection-changed", filechange)
                
        vbox_library.pack_start(self.make_check_box(_("Watch my library for added/changed/deleted files"), "library", "watcher"), False, False)
        vbox_library.pack_start(self.make_check_box(_("Check my library for deleted files on startup"), "library", "startup_deleted"), False, False)
        vbox_library.pack_start(self.make_check_box(_("Check my library for added/changed files on startup"), "library", "startup_added"), False, False)

        vbox_library.pack_start(self.get_title_label(_("Podcast behavior")), False, False)
        vbox_library.pack_start(self.make_check_box(_("Refresh on startup"), "podcast", "startup"), False, False)
        vbox_library.pack_start(self.make_number(_("Refresh every"), _("minute") + " " + _("( <= 1 to disable )"), "podcast", "time_refresh", 10), False, False)
        vbox_library.pack_start(self.make_number(_("Download only"), _("lastest podcast"), "podcast", "nb_download", 1), False, False)
        vbox_library.pack_start(self.get_title_label(_("Podcast directory")), False, False)
        vbox_library.pack_start(self.make_file("podcast", "folder", "~/Podcasts"), False, False)

        vbox_web = gtk.VBox(False, 2)
        vbox_web.set_border_width(12)
        vbox_web.pack_start(self.get_title_label(_("Web service filter")), False, False)
        vbox_web.pack_start(self.get_label(_("Skipped words for the web queries")), False, False)
        vbox_web.pack_start(self.make_list_edit("webservice", "filter", _("New filter"), _("Filter"), small=True), True, True)


        vbox_web.pack_start(self.get_title_label(_("Wikipedia")), False, False)
        vbox_web.pack_start(self.make_lentry(_("Language"), "wikipedia", "lang","en;fr;de;es;it;nl;sv;pl;pt", True), True, True)

        vbox_web.pack_start(self.get_title_label(_("Audioscrobbler")), False, False)
        vbox_web.pack_start(self.make_check_box(_("Enable Audioscrobbler support"), "audioscrobbler", "enable"), False, False)

        servers_list = {
                "post.audioscrobbler.com": "Last.fm",
                "turtle.libre.fm": "Libre.fm",
                }
        vbox_web.pack_start(self.make_lmenu(_("Scrobbler service"), "audioscrobbler", "url", servers_list, "post.audioscrobbler.com"), False, False)

        label = gtk.Label(_("Username"))
        label.set_alignment(0, 0.5)
        t = gtk.Table(2, 2)
        t.attach(label, 0, 1, 0, 1, gtk.FILL, gtk.FILL)
        label = gtk.Label(_("Password"))
        label.set_alignment(0, 0.5)
        t.attach(self.make_entry(_("Username"), "audioscrobbler", "username"), 1, 2, 0, 1, gtk.EXPAND | gtk.FILL, gtk.SHRINK)
        t.attach(label, 0, 1, 1, 2, gtk.FILL, gtk.FILL)
        entry = self.make_entry(_("Password"), "audioscrobbler", "password")
        entry.set_invisible_char("*")
        entry.set_visibility(False)
        t.attach(entry, 1, 2, 1, 2, gtk.EXPAND | gtk.FILL, gtk.SHRINK)
        t.set_row_spacings(2)
        t.set_col_spacings(2)
        vbox_web.pack_start(t, False, False)

        l = gtk.Label("<i>%s</i>"%_("Note: password will be  hidden when window have been closed"))
        l.set_alignment(0, 0.5)
        l.set_use_markup(True)
        vbox_web.pack_start(l, False, False)


        notebook = gtk.Notebook()
        notebook.append_page(vbox_general, self.get_tab(_("View"), gtk.STOCK_PREFERENCES))
        notebook.append_page(vbox_playback, self.get_tab(_("Playback"), gtk.STOCK_MEDIA_PLAY))
        notebook.append_page(vbox_library, self.get_tab(_("Library"), gtk.STOCK_PREFERENCES))
        notebook.append_page(vbox_web, self.get_tab(_("Web service"), gtk.STOCK_NETWORK))
        notebook.append_page(self.osd.PluginPreferences(), self.get_tab(_("OSD"), stock.PREF_OSD))

        notebook.set_current_page(0)
        notebook.connect("switch-page", self.on_switch_page)

        self.vbox.pack_start(notebook)

        self.show_all()


    def font_set(self, font):
        config.set("setting", "web_font", font.get_font_name())

    def on_destroy(self, *args):
        if self.monitoring_update:
            FileMonitor.stop_monitoring()
            if config.getboolean("library", "watcher"):
                FileMonitor.start_monitoring()
        
        self.osd.pref_hide_panel()
        self.osd.hide_osd()
        config.write()
        self.destroy()

    def on_switch_page(self, notebook, page, page_num):
        if page_num == 4:
            self.osd.pref_show_panel()
        else:
            self.osd.pref_hide_panel()

    def make_check_box(self, label, section, attribut):
        check = gtk.CheckButton(label)
        if config.get(section, attribut) == "true":
            check.set_property("active", True)

        def save_check(widget, sect, attr):
            if widget.get_property("active"):
                config.set(sect, attr, "true")
            else:
                config.set(sect, attr, "false")

            if sect == "setting" and attr == "compact_playlist":
                    self.win_parent.playlist_ui.playlist.refresh()
            
            if sect == "setting" and attr == "dynamic_playlist_color":
                    self.win_parent.playlist_ui.playlist.refresh()
                
            if sect == "setting" and attr == "use_trayicon":
                if widget.get_property("active"):
                    self.win_parent.tray.tray.show_all()
                else:
                    self.win_parent.tray.tray.hide_all()
            if sect == "library" and attr == "watcher":
                self.monitoring_update = True
                
        check.connect("toggled", save_check, section, attribut)
        return check

    def make_entry(self, label, section, attribut, default=None):
        def save_entry(widget, sect, attr):
            if sect == "audioscrobbler" and attr == "password":
                if widget.get_text() != "":
                    md5_pass = md5(widget.get_text()).hexdigest()
                    config.set(sect, attr, md5_pass)
            else:
                config.set(sect, attr, widget.get_text())
        entry = gtk.Entry()
        if not (section == "audioscrobbler" and attribut == "password"):
            entry.set_text(config.get(section, attribut, default))
        entry.connect("changed", save_entry , section, attribut)

        return entry



