# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gtk
import gobject
from heapq import heappush, heapify, heappop
from library import ListenDB
from logger import Logger

SONG_POPULATE = 500

class IncorrectFunctionUse(Exception):
    pass

class SongModel(gtk.ListStore,Logger):
    def __init__(self,treeview,*types,**kwargs):
        if len(types)>0:
            gtk.ListStore.__init__(self,object,*types)
        else:
            gtk.ListStore.__init__(self,object)
        
        self.tree = treeview
        self.__id_idle_fill = None
        self.refcache_uri = {}
        
    def set_column_type(self,*types):
        super(SongModel,self).set_column_type(object,*types)
        
    def clear(self):
        self.refcache_uri = {}
        try: gobject.source_remove(self.__id_idle_fill)
        except: pass
        gtk.ListStore.clear(self)

    def fill(self,songs):
        self.clear()
        self.__id_idle_fill = gobject.idle_add(self.__idle_fill,songs)

    def __idle_fill(self,songs,i=0):
        try: gobject.source_remove(self.__id_idle_fill)
        except: pass
        for song in songs[:SONG_POPULATE]:
            self.append((song,))
            self.refcache_uri[song.get("uri")] = i
            i += 1
        songs = songs[SONG_POPULATE:]
        if songs:
            self.__id_idle_fill = gobject.idle_add(self.__idle_fill,songs,i)

    def change_songs(self,songs):
        """ this function only update the row
        but it doesn't resort the model 
        MUST be ONLY used when the modified tag of the song are not a sortable tag
        Like podcast the progress bar
        """
        for song in songs:
            try:self[self.refcache_uri[song.get("uri")]] = (song,)
            except KeyError: continue
 
    def remove_songs(self,songs):
        uris = self.refcache_uri.keys()
        for uri in [ song.get("uri") for song in songs if song.get("uri") in uris ]:
            pos  = self.refcache_uri[uri]
            del self[pos]
            del self.refcache_uri[uri]
            for uri,p in self.refcache_uri.items():
                if p >= pos: self.refcache_uri[uri] = p-1


class SimpleSongModel(SongModel):
    def __init__(self,treeview,*types,**kwargs):
        super(SimpleSongModel,self).__init__(treeview, *types, **kwargs)
        ListenDB.connect("simple-changed", self.__on_change)
        ListenDB.connect("removed", self.__on_remove)
    
    def __on_remove(self,db,type,songs):
        self.remove_songs(songs)

    def __on_change(self,db,songs):
        self.change_songs(songs)
        
class SortableSongModel(SongModel):
    def __init__(self,treeview,*types,**kwargs):
        super(SortableSongModel,self).__init__(treeview,*types,**kwargs)

        self.__songs_cache = []

        self.__sort_by_func = None
        self.__sort_by_param = None
        
    def set_songs(self):
        songs = [ info[4] for info in self.__songs_cache ]
        self.clear()
        gobject.idle_add(self.append_songs,songs)

    def clear(self):
        self.__songs_cache = []
        SongModel.clear(self)

    def fill(self,songs):
        self.clear()
        self.append_songs(songs)

    def set_get_sort_by_func(self,func,*param):
        self.__sort_by_func = func
        self.__sort_by_param = param

    def get_sort_by(self):
        return self.__sort_by_func(*self.__sort_by_param)

    def append_songs(self,songs):
        try: gobject.source_remove(self.__id_idle_fill)
        except :pass
        tag, reverse = self.get_sort_by()
        heap = self.__songs_cache[:]
        if tag:
            [ heappush(heap,(song.get_sortable(tag),song.sort_key,song.get("uri") , False, song)) for song in songs ]
        else:
            [ heappush(heap,(None,song.sort_key,song.get("uri") , False, song)) for song in songs ]
        self.__idle_append_songs(reverse,heap,tag,0)

    def __idle_append_songs(self,reverse,heap,tag,i):
        try: gobject.source_remove(self.__id_idle_fill)
        except :pass
        total = len(self.__songs_cache) - 1
        while heap:
            song_info = heappop(heap)
            song = song_info[4]
            is_in_the_model = song_info[3]
            if not is_in_the_model:
                if not reverse: pos = i
                else: pos = total - i + 1
                try: 
                    self.insert(pos, (song,))
                    total += 1
                except TypeError:
                    self.logexception("Failed to insert at %s: %s", pos, song)
                if tag:
                    heappush(self.__songs_cache, (song.get_sortable(tag),song.sort_key,song.get("uri") , True, song))
                else:
                    heappush(self.__songs_cache, (None,song.sort_key,song.get("uri") , True, song))
            self.refcache_uri[song.get("uri")] = i
            i += 1
            if i%SONG_POPULATE == 0 :
                self.__id_idle_fill = gobject.idle_add(self.__idle_append_songs,reverse,heap,tag,i)
                break
        
    def change_songs(self,songs):
        """ this function only update the row
        but it doesn't resort the model 
        MUST be ONLY used when the modified tag of the song are not a sortable tag
        Like podcast the progress bar
        """
        _tag, reverse = self.get_sort_by()
        total = len(self.__songs_cache) - 1
        for song in songs:
            try:
                i = self.refcache_uri[song.get("uri")]
                if not reverse: pos = i
                else: pos = total - i
                self[pos] = (song, )
            except KeyError: continue
 
    def remove_songs(self,songs):
        _tag, reverse = self.get_sort_by()
        total = len(self.__songs_cache) - 1
        for song in songs:
            try:
                try : del self.__songs_cache[list([ info[4] for info in self.__songs_cache ]).index(song)]
                except ValueError: pass
                try: i = self.refcache_uri[song.get("uri")]
                except KeyError: continue
                if not reverse: pos = i
                else: pos = total - i +1 
                del self[pos]
                del self.refcache_uri[song.get("uri")]
                for uri,p in self.refcache_uri.items():
                    if p >= pos: self.refcache_uri[uri] = p-1
                total -= 1
            except:
                self.logexception("failed remove %s from the model",song.get("uri"))
        heapify(self.__songs_cache)

class PlaylistSongModel(SongModel):
    def __init__(self,treeview,*types,**kwargs):
        super(PlaylistSongModel,self).__init__(treeview, *types, **kwargs)
        ListenDB.connect("simple-changed", self.__on_change)

    def __on_change(self,db,songs):
        self.change_songs(songs)
 
