# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###


import gtk
import gobject

import stock
from config import config
from widget.view import MultiDragTreeview

from widget.song_columns import TextColumn, TAG_COLUMNS
from widget.song_model import SortableSongModel, SimpleSongModel, SongModel, PlaylistSongModel
from widget.song_menu import SongMenu, SongMenuManager
from widget.header_menu import HeaderMenu
from helper import Dispatcher, SignalContainer

from library import ListenDB

        
class SongView(MultiDragTreeview, SignalContainer):
    show_all_columns = False

    _model_class = SongModel
    def __init__(self, conf_prefix, **kargs):
        MultiDragTreeview.__init__(self)
        SignalContainer.__init__(self)

        self.conf_prefix = conf_prefix

        if not kargs: kargs = self.__get_kargs()
        model_types = tuple()
        col_tag = tuple()
        i = 0
        try: 
            kargs = [ (int(config.get("song_view", self.conf_prefix + "_" + tag + "_order")), tag, label, col_type) for tag, (label, col_type) in kargs.iteritems()]
        except: 
            kargs = [ (self.get_default_position(tag), tag, label, col_type) for tag, (label, col_type) in kargs.iteritems()]
        kargs.sort()
        for _pos, tag, label, col_type in kargs:
            model_types += col_type,
            col_tag += tag,
            try: 
                width = int(config.get("song_view", self.conf_prefix + "_" + tag + "_width"))
            except: 
                width = self.get_default_width(tag)
            self.insert_column_by_tag(i, tag, label, width)
            i += 1

        self.model = self._model_class(self)
        #self.model = self._model_class(self,*model_types)
        #if kargs:
        #    self.model.set_column_tags(*col_tag)

        self.set_rubber_banding(True)
        self.set_fixed_height_mode(True)

        self.set_model(self.model)

        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1), ("text/uri-list", 0, 2)]
        self.drag_source_set(gtk.gdk.BUTTON1_MASK, targets, gtk.gdk.ACTION_COPY)
        self.add_events(gtk.gdk.KEY_PRESS_MASK | 
              gtk.gdk.POINTER_MOTION_MASK | 
              gtk.gdk.BUTTON_PRESS_MASK | 
              gtk.gdk.SCROLL_MASK)
        
        self.autoconnect(self, "row-activated", self.on_activated)
        self.autoconnect(self, "drag-data-get", self.on_drag_data_get)
        self.autoconnect(self, "popup-menu", self.__popup_menu)
        self.autoconnect(self, "popup-empty-menu", self.__popup_empty_menu)
        
        self.set_rules_hint(True)
        
        self.menu = None
        self.empty_menu = None
        self.headermenu = None
        
        try: self.headermenu = HeaderMenu(self.conf_prefix, True, False, True)
        except: pass
        self.set_search_equal_func(self.on_song_search)
        self.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        
        if not self.show_all_columns: 
            self.autoconnect(config, "config-changed", self.on_config_change)
            gobject.idle_add(self.set_visible_column)

        for col in self.get_columns():
            button = col.get_button()
            self.autoconnect(button, "button-press-event", self.__button_pressed)

        self.set_column_font()
        self.connect("destroy", self.on_destroy)
    
    def get_default_position(self, tag):
        if tag == "#track": return 0
        elif tag == "title": return 1
        elif tag == "feed": return 2
        elif tag == "artist": return 2
        elif tag == "album": return 3
        elif tag == "genre": return 4
        elif tag == "#duration": return 5
        elif tag == "#progress": return 6
        else: return 10000

    def get_default_width(self, tag):
        if tag == "#track": return 25
        elif tag == "title": return 150
        elif tag == "#duration": return 60
        elif tag == "#disc": return 25
        else: return 100

    def get_columns_info(self):
        columns = []
        for i in range(len(self.get_columns())):
            col = self.get_column(i)
            columns.append((i, col.tag, col.title))
        return columns

    def __get_kargs(self):
        kargs = {}         
        #for index in range(50):
        #    try: 
        #        tag = config.get("id3","num"+str(index)+"_tag")
        #        label = config.get("id3","num"+str(index)+"_label")
        #    except: pass 
        #    else: 
        #        kargs[tag] = (label,gobject.TYPE_STRING)  
        
        kargs["#track"] = (_("#"), gobject.TYPE_STRING)
        kargs["title"] = (_("Title"), gobject.TYPE_STRING)
        kargs["genre"] = (_("Genre"), gobject.TYPE_STRING)
        kargs["artist"] = (_("Artist"), gobject.TYPE_STRING)
        kargs["album"] = (_("Album"), gobject.TYPE_STRING)
        kargs["date"] = (_("Year"), gobject.TYPE_STRING)

        kargs["#bitrate"] = (_("Bitrate"), gobject.TYPE_STRING) 
        kargs["#duration"] = (_("Length"), gobject.TYPE_STRING) 
        kargs["#playcount"] = (_("Play Count"), gobject.TYPE_STRING) 
        kargs["#skipcount"] = (_("Skip Count"), gobject.TYPE_STRING) 
        kargs["#lastplayed"] = (_("Last Played"), gobject.TYPE_STRING)     
        kargs["#added"] = (_("Added"), gobject.TYPE_STRING)
        kargs["#disc"] = (_("Disc"), gobject.TYPE_STRING)

        return kargs


    def on_drag_data_get(self, treeview, drag_context, selection, info, timestamp):
        model, rows = treeview.get_selection().get_selected_rows()
        if len(rows) < 0: return
        list_uri = list([ model[row[0]][0].get("uri") for row in rows])
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)
    
    def on_config_change(self, helper, section, option, value):
        if section == "song_view":
            self.set_visible_column()
        if section == "font":
            self.set_column_font()

    def save_config(self):
        print "Saving column config for", self.conf_prefix
        for i in range(len(self.get_columns())):
            col = self.get_column(i)
            config.set("song_view", self.conf_prefix + "_" + col.tag + "_width", str(col.get_current_width()))
            config.set("song_view", self.conf_prefix + "_" + col.tag + "_order", str(i))

    def get_default_visibility(self, tag):
        return tag in [ "#track" , "feed", "station", "title", "artist", "album", "genre", "#duration", "#progress" ]

    def set_visible_column(self):
        for col in self.get_columns():
            try: visible = config.get("song_view", self.conf_prefix + "_" + col.tag) == "true"
            except: visible = self.get_default_visibility(col.tag)
            col.set_visible(visible)

    def set_column_font(self):
        try: new_font = config.get("font", self.conf_prefix)
        except: new_font = None

        for col in self.get_columns():
            col.set_font(new_font)
       
        
    def on_activated(self, treeview, path, view_column):
        if config.getboolean('player', 'queuewholealbum'):
            song = treeview.get_model()[path][0]
            songs = ListenDB.request(song.get_type(), "&(artist = /^%s$/, album = /^%s$/)" % (song.get("artist"), song.get("album")))
            if songs: Dispatcher.cur_playlist_play(songs)
        else:
            Dispatcher.cur_playlist_play([treeview.get_model()[path][0]])

    def append_column_by_tag(self, tag, title, width):
        if TAG_COLUMNS.has_key(tag):
            self.append_column(TAG_COLUMNS[tag](tag, title, width))
        else:
            self.append_column(TextColumn(tag, title, width))

    def insert_column_by_tag(self, pos, tag, title, width):
        if TAG_COLUMNS.has_key(tag):
            self.insert_column(TAG_COLUMNS[tag](tag, title, width), pos)
        else:
            self.insert_column(TextColumn(tag, title, width), pos)
            
    def populate(self):
        pass
    
    def set_menu(self, menu, empty_menu=None):
        self.menu = menu
        self.empty_menu = empty_menu
        
    def get_select_songs(self):
        model, rows = self.get_selection().get_selected_rows()
        if len(rows) <= 0:
            return []
        else:
            return list([ model[row[0]][0] for row in rows])

    def __button_pressed(self, widget, event):
        if event.button == 3 and self.headermenu != None:
            self.headermenu.popup(self)
            return True
     
    """""""""""""""""""""
        FUNC SEARCH
    """""""""""""""""""""
    def on_song_search(self, model, column, key, iterator):
        title = model.get_value(iterator, 0).get_str("title")
        if title.lower().find(key.lower()) == 0:
            return False
        return True
    
    """""""""""""""""""""""""""
        FUNC SELECTION
    """""""""""""""""""""""""""
    def __popup_empty_menu(self, widget):
        if self.empty_menu:
            self.empty_menu.popup([])

    def __popup_menu(self, widget):
        songs = self.get_select_songs()
        if self.menu and songs:
            self.menu.popup(songs)

    def on_destroy(self, widget):
        self.autodisconnect_all()

class SimpleSongView(SongView):
    _model_class = SimpleSongModel
        
class SortableSongView(SongView):
    _model_class = SortableSongModel
    _user_sort = True
    def __init__(self, conf_prefix, **kargs):
        super(SortableSongView, self).__init__(conf_prefix, **kargs)
        self.model.set_get_sort_by_func(self.get_sort_by)
        self.__loaded = False

    def get_model(self):
        return self.model

    def set_sort_order(self):
        col_tag = [ col.tag for col in self.get_columns() ]
        try: 
            sort_tag = config.get("browser", self.conf_prefix + "_sort_tag")
            sort_order = config.get("browser", self.conf_prefix + "_sort_order")
        except: return
        if self.conf_prefix and sort_tag in col_tag and sort_order in ["ASC", "DESC"]:
            if sort_order == "ASC" : 
                sens = gtk.SORT_ASCENDING 
            else: 
                sens = gtk.SORT_DESCENDING
            self.set_sort_by(None, sort_tag, sens, False)
        
    def set_model(self, model):
        super(SortableSongView, self).set_model(model)
        if model:
            self.set_sort_order()
        
    def set_sort_by(self, header, tag=None, order=None, refresh=True):
        if header != None and tag == None: tag = header.tag

        for h in self.get_columns():
            if h.tag == tag:
                if self.conf_prefix:    
                    config.set("browser", self.conf_prefix + "_sort_tag", tag)
                if order != None:
                    h.set_sort_order(order)
                    h.set_sort_indicator(True)
                elif h.get_sort_indicator() :
                    if h.get_sort_order() == gtk.SORT_ASCENDING:
                        if self.conf_prefix: config.set("browser", self.conf_prefix + "_sort_order", "DESC")    
                        h.set_sort_order(gtk.SORT_DESCENDING)
                        h.set_sort_indicator(True)
                    else:
                        if self.conf_prefix: config.set("browser", self.conf_prefix + "_sort_order", "")
                        h.set_sort_indicator(False)
                else: 
                    if self.conf_prefix: config.set("browser", self.conf_prefix + "_sort_order", "ASC")
                    h.set_sort_order(gtk.SORT_ASCENDING)
                    h.set_sort_indicator(True)
            else: 
                h.set_sort_indicator(False)
                
        if refresh:
            self.get_model().set_songs()
        
    def get_sort_by(self):
        if not self.__loaded : 
            self.set_sort_order()
            self.__loaded = True
        for __, header in enumerate(self.get_columns()):
            if header.get_sort_indicator():
                return (header.tag,
                        header.get_sort_order() == gtk.SORT_DESCENDING)
        else: return None, None
    
    def append_column_by_tag(self, tag, *ncol):
        super(SortableSongView, self).append_column_by_tag(tag, *ncol)
        col = self.get_column(len(self.get_columns()) - 1)
        col.__id_header_clicked = col.connect('clicked', self.set_sort_by)
        col.set_clickable(self._user_sort)
        col.set_reorderable(True)
        
    def insert_column_by_tag(self, pos, tag, *ncol):
        super(SortableSongView, self).insert_column_by_tag(pos, tag, *ncol)  
        col = self.get_column(pos)
        col.__id_header_clicked = col.connect('clicked', self.set_sort_by)
        col.set_clickable(self._user_sort)
        col.set_reorderable(True)

class PlaylistSongView(SongView):
    _model_class = PlaylistSongModel
    editable = False

    def __init__(self, config_code, pl, **kargs):
        self.__pl = pl
        super(PlaylistSongView, self).__init__(config_code, **kargs)

        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1), ("text/uri-list", 0, 2)]

        if self.editable:
            self.enable_model_drag_dest(targets, gtk.gdk.ACTION_COPY)
            self.connect("drag-data-received", self.__on_drag_data_reveived)

        if self.editable: 
            for_custom = "customplaylistmenu"
        else: for_custom = None

        menu = SongMenu(for_custom=for_custom)
        menu.disable(["delete", "deletedisk"])
        self.set_menu(menu)
    

        SongMenuManager.connect("action", self.__song_menu_action)
        SongMenuManager.register_item("playlist_delete", 30, stock.DELETE, custom_only="customplaylistmenu")
        SongMenuManager.allow_item_for_type("playlist_delete", "alltypes")

        self.__pl.connect("update", self.__update)
        
        #Interconnect song tuple cache
        self.get_model().fill(pl.get_songs())

    def __on_drag_data_reveived(self, treeview, context, x, y, selection, info, timestamp):
        drop_info = treeview.get_dest_row_at_pos(x, y)
        model = treeview.get_model()
        #print selection.data
        if selection.target in ["text/uri-list", "text/listen-songs"]:
            if context.get_source_widget() == self:
                position = gtk.TREE_VIEW_DROP_AFTER
                model, rows = treeview.get_selection().get_selected_rows()
                if len(rows) < 1:
                    return
                rows = [row[0] for row in rows]

                if drop_info:
                    new_pos, position = drop_info
                    new_pos = new_pos[0]
                else:
                    new_pos = len(model) - 1

                if new_pos < rows[0]:
                    rows.reverse()

                i = 0
                for row in rows:
                    row += i

                    iter = model.get_iter(row)
                    new_iter = model.get_iter(new_pos)
                    if position == gtk.TREE_VIEW_DROP_BEFORE or position == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE:
                        model.move_before(iter, new_iter);
                    if position == gtk.TREE_VIEW_DROP_AFTER or position == gtk.TREE_VIEW_DROP_INTO_OR_AFTER:
                        model.move_after(iter, new_iter);
                        
                    if new_pos < rows[0]:
                        i += 1
                    else:
                        i -= 1 
                self.reorder_playlist()
                
            elif selection.target in ["text/listen-songs", "text/uri-list" ]:
                if selection.data:
                    if drop_info:
                        new_pos, position = drop_info
                        new_pos = new_pos[0]
                        if position == gtk.TREE_VIEW_DROP_AFTER:
                            new_pos += 1
                    else:
                        new_pos = len(model)
    
                    songs = []
                    for uri in  selection.data.splitlines():
                        song = ListenDB.get_song(uri)
                        if song :
                            songs.append(song)
                    self.__pl.extend_insert(songs, new_pos)
    
    def __song_menu_action(self, song_menu_manager, id_menu, songs):
        # Won't work
        if id_menu == "playlist_delete":
            gobject.timeout_add(1, self.delete)

    def delete(self, *args, **kwargs):
        _model, rows = self.get_selection().get_selected_rows()
        positions = [row[0] for row in rows]
        self.__pl.remove_positions(positions)

    def __update(self, pl):
        self.get_model().fill(pl.get_songs())

    def reorder_playlist(self):
        model = self.get_model()
        uris = []
        for i in range(0, len(model)):
            uris.append(model[i][0].get("uri"))
        self.__pl.reorder(uris)

